/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeException.h"
#include "TeDecoderJPEG.h"
#include "TeAsciiFile.h"
#include "TeUtils.h"
#include <jpeg.h>
#include <io.h>

TeDecoderJPEG::TeDecoderJPEG (const TeRasterParams& par) : 
	TeDecoderMemory()
{	
	params_ = par; 
	// if openning an existent file, read its informations
	if (params_.mode_ == 'w' || params_.mode_ == 'r')
	{
		int nb;
		if (!Jpeg::ReadFileParams(params_.ncols_ , params_.nlines_ , nb , (char*)params_.fileName_.c_str()))
			return;

		params_.nBands(nb);
		params_.setDataType(TeUNSIGNEDCHAR);
		params_.setPhotometric(TeRASTERMULTIBAND);
		params_.decoderIdentifier_ = "JPEG";
		params_.blockWidth_ = params_.ncols_;
		params_.blockHeight_ = params_.nlines_;
		TeProjection* proj = new TeNoProjection();
		params_.projection(proj);
		delete proj;

		// Try reading box from auxiliary file
		string filePrefix = TeGetName(params_.fileName_.c_str());
		string jgw = filePrefix+".jgw";
		try
		{
			TeAsciiFile	pFile (jgw);
			double dyj, dxi, x0, y0;

			params_.resx_ = pFile.readFloat();
			dyj = pFile.readFloat();		// ignored
			dxi = pFile.readFloat();		// ignored
			params_.resy_  = -1 * pFile.readFloat();
			x0 = pFile.readFloat();
			y0 = pFile.readFloat();

			// IMPORTANT: we discard any rotation or translation defined in the
			// .jgw file! It should be treated properly later
			params_.topLeftResolutionSize(x0,y0,params_.resx_,params_.resy_,
				                          params_.ncols_,params_.nlines_,true);
		}
		catch (...)
		{
			// no .jgw inexistent or inconsistent, uses resolution of 1 unit 
			// and box lower left coordinate of (0.5,0.5)
			params_.lowerLeftResolutionSize(0.5,0.5,1.0,1.0,params_.ncols_,params_.nlines_,true);
		}
	}
}

TeDecoderJPEG::~TeDecoderJPEG ()
{
	if ( isModified_ && (params_.mode_ == 'w' || params_.mode_ == 'c') )	// save contents to disk
	{
		int nb = params_.nBands();
		int nc = params_.fileName_.size();
		char*  name = new char[nc+1];
		strcpy(name,params_.fileName_.c_str());
		int i = _access(params_.fileName_.c_str(),06);
		i = _unlink(params_.fileName_.c_str());
		Jpeg::WriteFile((unsigned char*)data_,params_.ncols_,params_.nlines_,nb,name);
	}
	TeDecoderMemory::clear();
}


bool
TeDecoderJPEG::clear()
{
	if (data_ == 0)
	    return true;

	if ( isModified_ && (params_.mode_ == 'w' || params_.mode_ == 'c') )	// save contents to disk
	{
		int nb = params_.nBands();
		int nc = params_.fileName_.size();
		char*  name = new char[nc+1];
		strcpy(name,params_.fileName_.c_str());
		Jpeg::WriteFile((unsigned char*)data_,params_.ncols_,params_.nlines_,nb,name);
		isModified_ = 0;
	}
	TeDecoderMemory::clear();
	return true;
}


void
TeDecoderJPEG::init()
{
	params_.status_= TeNOTREADY;
	int nb = params_.nBands();
	if (nb != 1 && nb != 3)
		return;

	// try to allocate enough memory to hold the data
	TeDecoderMemory::init();
	if (params_.status_== TeNOTREADY)
		return;

	params_.status_= TeNOTREADY;
	if (params_.mode_ == 'c')	// creating a new file
	{
		if (_access(params_.fileName_.c_str(),00) != -1) // remove existing file
		{
			if (_unlink(params_.fileName_.c_str()) == -1)
				return;
		}		
		// write georeference file
		string filePrefix = TeGetName (params_.fileName_.c_str());
		string jgw = filePrefix+".jgw";

		try {
			TeAsciiFile jgwFile(jgw,"w+");
			string line;
			line = Te2String(params_.resx_) + "\n";
			jgwFile.writeString(line);
			line = "0.0\n";
			jgwFile.writeString(line);
			jgwFile.writeString(line);
			line = Te2String(params_.resy_ * -1.) + "\n";
			jgwFile.writeString(line);
			line = Te2String(params_.box().x1_) + "\n";
			jgwFile.writeString(line);
			line = Te2String(params_.box().y2_) + "\n";
			jgwFile.writeString(line);
		}
		catch(...)
		{}
		// write data
		int nc = params_.fileName_.size();
		char*  name = new char[nc+1];
		strcpy(name,params_.fileName_.c_str());

		if (!Jpeg::WriteFile((unsigned char*)data_,params_.ncols_,params_.nlines_,nb,name))
			return;
		params_.status_ = TeREADYTOWRITE;
	}
	else if (params_.mode_ == 'w')
	{
		int i = _access(params_.fileName_.c_str(),06);
		if (i == -1)
			return;
//		if (!Jpeg::ReadFile(params_.ncols_, params_.nlines_, params_.nBands(), (char*)params_.fileName_.c_str(), (unsigned char*) data_))
//			return;
		params_.status_ = TeREADYTOWRITE;
		i = _access(params_.fileName_.c_str(),06);
	}
	else if (params_.mode_ == 'r')
	{
		if ((_access(params_.fileName_.c_str(),04) == -1) || !Jpeg::ReadFile(params_.ncols_, params_.nlines_, params_.nBands(), (char*)params_.fileName_.c_str(), (unsigned char*) data_))
			return;
		params_.status_ = TeREADYTOREAD;	
	}
}
