/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeDecoderTIFF.h"
#include "TeGeometryAlgorithms.h"
#include "TeRaster.h"
#include "TeDecoderMemory.h"
#ifdef WIN32
#include <io.h>
#endif

#if !defined(WIN32)
#include <sys/param.h>
#endif 

// ---- Public Methods ----- 

TeDecoderTIFF::TeDecoderTIFF(const TeRasterParams& par)
{
	params_ = par;
	TImage = -1;
	TCurLine_ = -1;
	first = 1;
	TBuffer_ = 0;
	TBufferTile_ = 0;
	nBands_ = par.nBands();
	params_.status_= TeNOTREADY;

	bytesperline_ = 0;
	rowtilesize_  = 0;
	tilew_ = 0;
	tileh_ = 0;
	bytespertile_ = 0;

	tif = 0;
	gtif = 0;
	
	char tifmode = 'r';
	if (params_.mode_=='c')
	{
		tifmode = 'a';
	}
	else if (params_.mode_ == 'r')
	{// if file should exist already read tiff/geotiff parameters
		tif=XTIFFOpen(params_.fileName_.c_str(),&tifmode);
		if (!tif)
			return;
		gtif = (GTIF*)GTIFNew(tif);
		ReadTiffDirectory();
		if (gtif)
		{
			GTIFFree(gtif);
			gtif = 0;
		}
		if (tif)
		{
			XTIFFClose(tif); 
			tif = 0;
		}
		params_.decoderIdentifier_ = "TIF";
	}
}

void
TeDecoderTIFF::init()
{
	clear();		// reset existent internal structures

	params_.status_= TeNOTREADY;
	char tifmode;	// libtiff mode

	// check the initial permissions on the file
	if (params_.mode_ == 'c')		// creating a new raster file
	{
		if (access(params_.fileName_.c_str(),00) != -1)  // try to remove existing file
			if (unlink(params_.fileName_.c_str()) == -1)
				return;
		tifmode = 'a';
		params_.status_ = TeREADYTOWRITE;
	}
	else if ( params_.mode_ == 'r' )
	{
		if (access(params_.fileName_.c_str(),04) == -1)
			return;							// there is no permission to read
		tifmode = 'r';
		params_.status_ = TeREADYTOREAD;
	}
	// TIFF image files may not be opened for both reading and writing; 
	// there is no support for altering the contents of a TIFF file. So
	// mode == 'w' never initializes this decoder.
	else if (params_.mode_ == 'w')
	{
		params_.status_= TeNOTREADY;
		return;
	}
	else	// unknown mode
		return;

	// try to open the tiff file with the desired mode
	tif=XTIFFOpen(params_.fileName_.c_str(),&tifmode);
	if (!tif)
	{
		params_.status_= TeNOTREADY;
		return;
	}

	// try to create a geotiff instance on the tiff file 
	gtif = (GTIF*)GTIFNew(tif);
	if (!tif)
	{
		params_.status_= TeNOTREADY;
		return;
	}

	if (params_.mode_ == 'c')	// if geotiff is being created we should set the geo flags
		SetGeoKeys();
	
	ReadTiffDirectory();		// read the tiff geokeys
	AllocateBuffer();				// allocate the internal buffers
	if (params_.mode_ == 'c' && 	// if a new image writes creates a dummy image
		!WriteDummyImage())			// if couldnt write dummy image there is an error
	{
		params_.status_= TeNOTREADY;		 
		return;
	}
}


bool 
TeDecoderTIFF::clear()
{
	if (tif && params_.mode_=='c' && TCurLine_ >= 0)	//  if we have been 
	{													//	writting save the last 
		tif->tif_curoff = 0;
		int cc = TIFFStripSize(tif);
		TIFFWriteStrip(tif,TCurLine_,TBuffer_,cc);
	}
	if (gtif)
		GTIFFree(gtif);
	if (tif)
		XTIFFClose(tif); 

	tif = 0;
	gtif = 0;
	first = 0;
	TImage = -1;
	TCurLine_ = -1;
	bytesperline_ = 0;
	rowtilesize_  = 0;
	tilew_ = 0;
	tileh_ = 0;
	bytespertile_ = 0;

	DeallocateBuffer();
	params_.status_ = TeNOTREADY;
	return true;
}

TeDecoderTIFF::~TeDecoderTIFF()
{
	clear();
}

bool
TeDecoderTIFF::getElement(int col,int lin, double &val,int band)
{
	if (TCurLine_ != (int)lin)		// if line is not the current
	{
		if (TImage == MONOIMAGE || TImage == PALLETEIMAGE)
		{
			if (TIFFReadScanline(tif,TBuffer_,lin,0) < 0)
				return false;
		}
		else // RGB image
		{
			if (planar_ == PLANARCONFIG_CONTIG)
			{
				if (isTiled_)
				{
					if(!ReadTileImageContig(lin,TBuffer_))
						return false;
				}
				else
				{
					if (TIFFReadScanline(tif,TBuffer_,lin,0) < 0)
						return false;
				}
			}
			else	// RGB Separate
			{
				if (isTiled_)  return false; // SEPARATE and tiled => Error!
				for (int s=0; s < nBands_; s++)
					if (TIFFReadScanline(tif,&TBuffer_[s*params_.ncols_*params_.elementSize(s)],lin,s) < 0)
						return false;
			}
		}
		TCurLine_ = lin;
	}
	int pos;
	if (planar_ == PLANARCONFIG_CONTIG)
		pos = col*nBands_+band;
	else
		pos = band*params_.ncols_+col;

	if (params_.dataType_[band] == TeFLOAT)			// 32 bits
		val = ((float*)TBuffer_)[pos];
	else if (params_.dataType_[band] == TeSHORT)	// 16 bits
		val = ((short*)TBuffer_)[pos];
	else											// 8 bits
		val = TBuffer_[pos];
	return true;
}


// We allow writting only to a strip based, non compressed tiff file
bool
TeDecoderTIFF::setElement(int col, int lin, double val, int band)
{
	int strip = lin/rowsperstrip_;
	if (strip != TCurLine_)
	{
		if (TCurLine_ >= 0)
		{
			tif->tif_curoff = 0;
			int cc = TIFFStripSize(tif);
			TIFFWriteStrip(tif,TCurLine_,TBuffer_,cc);
		}
		TCurLine_ = strip;

		int l = strip * rowsperstrip_;
		bool res = false;
		if (TImage == MONOIMAGE || TImage == PALLETEIMAGE)
		{
			tif->tif_curstrip = 0;
			res = ReadMonoImage(l);
		}
		else if(TImage == RGBIMAGE)
		{
			tif->tif_curstrip = 0;
			res = ReadRGBImage (l);
		}
		if (!res)
		{
			TCurLine_ = -1;
			return false;
		}
	}
	int offset = lin % rowsperstrip_ * bytesperline_;
	unsigned char* cp = TBuffer_ + offset;
	int pos = col*nBands_+band;
	if (params_.dataType_[band] == TeFLOAT)			// 32 bits
		((float*)cp)[pos] = (float)val;
	else if (params_.dataType_[band] == TeSHORT)	// 16 bits
		((short*)cp)[pos] = (short)val;
	else											// 8 bits
		cp[pos] = (unsigned char)val;
	return true;
}


// --- Internal Methods ---

// --- Methods that deal with Tiff/GeoTiff Flags

void TeDecoderTIFF :: SetGeoKeys()
{

// Image Size
	TIFFSetField(gtif->gt_tif, TIFFTAG_IMAGEWIDTH, params_.ncols_);
	TIFFSetField(gtif->gt_tif, TIFFTAG_IMAGELENGTH, params_.nlines_);

// Orientation
	TIFFSetField(gtif->gt_tif, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);

// Number of bits
	int nb = params_.nbitsperPixel_[0];
	TIFFSetField(gtif->gt_tif, TIFFTAG_BITSPERSAMPLE, nb);

// Number of bands
	TIFFSetField(gtif->gt_tif, TIFFTAG_SAMPLESPERPIXEL, nBands_);

// Compression
	TIFFSetField(gtif->gt_tif, TIFFTAG_COMPRESSION, COMPRESSION_NONE);

// Planar configuration
	TIFFSetField(tif,TIFFTAG_PLANARCONFIG,  PLANARCONFIG_CONTIG);

// Photometric type
	if (nBands_ == 1) // monobands
	{
		if (params_.photometric_[0] == TeRASTERPALETTE)
		{
			TIFFSetField(gtif->gt_tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_PALETTE);
			SaveLut();
			TImage = PALLETEIMAGE;
		}
		else
		{
			TIFFSetField(gtif->gt_tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISBLACK);
			TImage = MONOIMAGE;
		}
	}
	else
	{
		TIFFSetField(gtif->gt_tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
		TImage = RGBIMAGE;
	}

	// Rows per Strip
	int rowsperstrip = 3;
//	rowsperstrip = TIFFDefaultStripSize(gtif->gt_tif, rowsperstrip);
	TIFFSetField(gtif->gt_tif, TIFFTAG_ROWSPERSTRIP, rowsperstrip);

	TeProjectionParams paramas = params_.projection()->params();

	// Projection parameters
	GTIFKeySet(gtif, GTRasterTypeGeoKey, TYPE_SHORT, 1, RasterPixelIsArea);
	double tiepoint[6];
	TeBox infobox = params_.box();
	tiepoint[0] =0;
	tiepoint[1] =0;
	tiepoint[2] =0;
	tiepoint[3] = infobox.x1();
	tiepoint[4] = infobox.y2();
	tiepoint[5] = 0;
	double resolution[3];
	resolution[0] = params_.resx_;
	resolution[1] = params_.resy_;
	resolution[2] = 0;
	TIFFSetField(gtif->gt_tif, GTIFF_TIEPOINTS, 6,&tiepoint); 
	TIFFSetField(gtif->gt_tif, GTIFF_PIXELSCALE, 3,&resolution);

	string projname = params_.projection()->name();
	if (projname == "NoProjection")
	{
		GTIFWriteKeys(gtif);				// write so far defined geo keys 
		return;
	}

	if (projname == "LatLong")
	{
		GTIFKeySet(gtif, GTModelTypeGeoKey, TYPE_SHORT, 1, ModelTypeGeographic);
		GTIFKeySet(gtif, GeogAngularUnitsGeoKey, TYPE_SHORT, 1, Angular_Degree);
	}
	else
	{
		GTIFKeySet(gtif, GTModelTypeGeoKey, TYPE_SHORT, 1, ModelTypeProjected);
		GTIFKeySet(gtif, ProjLinearUnitsGeoKey, TYPE_SHORT, 1,Linear_Meter);
		if (projname == "UTM")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_TransverseMercator);
			GTIFKeySet(gtif, ProjNatOriginLatGeoKey, TYPE_DOUBLE,1,paramas.lat0*TeCRD);
			GTIFKeySet(gtif, ProjNatOriginLongGeoKey, TYPE_DOUBLE,1,paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjScaleAtNatOriginGeoKey, TYPE_DOUBLE,1,paramas.scale);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE,1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey, TYPE_DOUBLE,1,paramas.offy);
		}
		else if (projname == "Mercator")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_Mercator);
			GTIFKeySet(gtif, ProjNatOriginLongGeoKey, TYPE_DOUBLE,1,paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjNatOriginLatGeoKey, TYPE_DOUBLE,1,paramas.stlat1*TeCRD);
			GTIFKeySet(gtif, ProjScaleAtNatOriginGeoKey, TYPE_DOUBLE,1,paramas.scale);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE,1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey, TYPE_DOUBLE,1,paramas.offy);
		}
		else if (projname == "LambertConformal")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_LambertConfConic_2SP);
			GTIFKeySet(gtif, ProjFalseOriginLatGeoKey,TYPE_DOUBLE, 1, paramas.lat0*TeCRD);
			GTIFKeySet(gtif, ProjFalseOriginLongGeoKey, TYPE_DOUBLE, 1, paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjStdParallel1GeoKey, TYPE_DOUBLE, 1, paramas.stlat1*TeCRD);
			GTIFKeySet(gtif, ProjStdParallel2GeoKey, TYPE_DOUBLE, 1, paramas.stlat2*TeCRD);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE, 1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey,TYPE_DOUBLE, 1,paramas.offy);
		}
		else if (projname == "Polyconic")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_Polyconic);
			GTIFKeySet(gtif, ProjFalseOriginLatGeoKey,TYPE_DOUBLE, 1, paramas.lat0*TeCRD);
			GTIFKeySet(gtif, ProjFalseOriginLongGeoKey, TYPE_DOUBLE, 1, paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE, 1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey,TYPE_DOUBLE, 1,paramas.offy);
			GTIFKeySet(gtif, ProjScaleAtNatOriginGeoKey, TYPE_DOUBLE,1,paramas.scale);
		}
		else if (projname == "CylindricalEquidistant")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_Equirectangular);
			GTIFKeySet(gtif, ProjCenterLatGeoKey,TYPE_DOUBLE, 1, paramas.stlat1*TeCRD);
			GTIFKeySet(gtif, ProjCenterLongGeoKey, TYPE_DOUBLE, 1, paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE, 1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey,TYPE_DOUBLE, 1,paramas.offy);
		}
		else if (projname == "PolarStereographic")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_PolarStereographic);
			GTIFKeySet(gtif, ProjNatOriginLatGeoKey, TYPE_DOUBLE,1,paramas.lat0*TeCRD);
			GTIFKeySet(gtif, ProjNatOriginLongGeoKey, TYPE_DOUBLE,1,paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjScaleAtNatOriginGeoKey, TYPE_DOUBLE,1,paramas.scale);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE,1,paramas.offx);

			GTIFKeySet(gtif, ProjFalseNorthingGeoKey, TYPE_DOUBLE,1,paramas.offy);
		}
		else if (projname == "Albers")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_AlbersEqualArea);
			GTIFKeySet(gtif, ProjFalseOriginLatGeoKey,TYPE_DOUBLE, 1, paramas.lat0*TeCRD);
			GTIFKeySet(gtif, ProjFalseOriginLongGeoKey, TYPE_DOUBLE, 1, paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjStdParallel1GeoKey, TYPE_DOUBLE, 1, paramas.stlat1*TeCRD);
			GTIFKeySet(gtif, ProjStdParallel2GeoKey, TYPE_DOUBLE, 1, paramas.stlat2*TeCRD);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE, 1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey,TYPE_DOUBLE, 1,paramas.offy);
		}
		else if (projname == "Miller")
		{
			GTIFKeySet(gtif, ProjCoordTransGeoKey, TYPE_SHORT, 1, CT_MillerCylindrical);
			GTIFKeySet(gtif, ProjCenterLongGeoKey, TYPE_DOUBLE, 1, paramas.lon0*TeCRD);
			GTIFKeySet(gtif, ProjFalseEastingGeoKey, TYPE_DOUBLE, 1,paramas.offx);
			GTIFKeySet(gtif, ProjFalseNorthingGeoKey,TYPE_DOUBLE, 1,paramas.offy);
		}
	}

	// datum parameters
	if (paramas.datum.name() == "CorregoAlegre")
		GTIFKeySet(gtif, GeogGeodeticDatumGeoKey, TYPE_SHORT, 1,6225);
	else if (paramas.datum.name() == "WGS84")
		GTIFKeySet(gtif, GeogGeodeticDatumGeoKey, TYPE_SHORT, 1,6326);
	else if (paramas.datum.name() == "SAD69")
		GTIFKeySet(gtif, GeogGeodeticDatumGeoKey, TYPE_SHORT, 1,6291);
	else
		GTIFKeySet(gtif, GeogGeodeticDatumGeoKey, TYPE_SHORT, 1,32767);

	GTIFKeySet(gtif, GeogCitationGeoKey, TYPE_ASCII,1,paramas.datum.name().c_str());
	double sMajorAxis = paramas.datum.radius();
	double sMinorAxis = sMajorAxis*(1-paramas.datum.flattening());
	GTIFKeySet(gtif, GeogSemiMajorAxisGeoKey,TYPE_DOUBLE, 1, sMajorAxis);
	GTIFKeySet(gtif, GeogSemiMinorAxisGeoKey,TYPE_DOUBLE, 1, sMinorAxis);
	GTIFWriteKeys(gtif);
}

void
TeDecoderTIFF::ReadTiffDirectory()
{
	int ncols=0, nlins = 0;
	TIFFGetField(tif,TIFFTAG_IMAGEWIDTH,&ncols);
	TIFFGetField(tif,TIFFTAG_IMAGELENGTH,&nlins);
	params_.ncols_ = ncols;
	params_.nlines_ = nlins;
		
	short samplesperPixel;
	if (!TIFFGetField(tif,TIFFTAG_SAMPLESPERPIXEL,&samplesperPixel) )
		params_.nBands(1);
	else
		params_.nBands(samplesperPixel);
	nBands_ = params_.nBands();

	nbitsperpixel_ = 0;
	if( !TIFFGetField(tif,TIFFTAG_BITSPERSAMPLE,&nbitsperpixel_) )
	{
		nbitsperpixel_ = 1;
		params_.setDataType(TeBIT);
	}
	else
	{
		int nbytes = nbitsperpixel_/8;
		if (nbytes == 1)
			params_.setDataType(TeUNSIGNEDCHAR);
		else if (nbytes == 2)
		    params_.setDataType(TeSHORT);
		else if (nbytes == 4)
			params_.setDataType(TeFLOAT);
		else
			params_.setDataType(TeDOUBLE);
	}
	
	isTiled_ = (TIFFIsTiled(tif) != 0);			// return non-zero if is organized in tiles 
	if (isTiled_)
	{
		rowtilesize_= TIFFTileRowSize(tif);
		TIFFGetField(tif,TIFFTAG_TILEWIDTH,&tilew_);
		TIFFGetField(tif,TIFFTAG_TILELENGTH,&tileh_);
		bytespertile_ = TIFFTileSize(tif);
	}
	else
		TIFFGetField(tif, TIFFTAG_ROWSPERSTRIP, &rowsperstrip_);


	bytesperline_ = TIFFScanlineSize(tif);

	TIFFGetField(tif,TIFFTAG_PLANARCONFIG,&planar_);	// storage organization
	TIFFGetField(tif,TIFFTAG_PHOTOMETRIC,&photom_);     // photometric interpretation

	if (samplesperPixel == 3 && 
	   (photom_ == PHOTOMETRIC_MINISWHITE || 
	    photom_ == PHOTOMETRIC_MINISBLACK))
		photom_ = PHOTOMETRIC_RGB;
	
	switch (photom_)
	{
	case PHOTOMETRIC_PALETTE:
		params_.setPhotometric(TeRASTERPALETTE);
		TImage = PALLETEIMAGE;
		ReadLut();
		break;
	case PHOTOMETRIC_MINISWHITE:
	case PHOTOMETRIC_MINISBLACK:
		params_.setPhotometric(TeRASTERMULTIBAND);
		TImage = MONOIMAGE;
		break;
	case PHOTOMETRIC_RGB:
		params_.setPhotometric(TeRASTERRGB);
		TImage = RGBIMAGE;
		break;
	default:
		break;
	}

	TIFFGetField(tif, TIFFTAG_COMPRESSION, &compress_);
	if (compress_ != COMPRESSION_NONE)
		params_.setCompressionMode(TeTIFFCOMPRESSION);

	// Read georeference parameters
	if (GetGeoTIFF ()) //GeoTiff Images
	{
		TeCoord2D lleft(0.,params_.nlines_-1);
		TeCoord2D uleft(0.,0.);
		TeCoord2D uright(params_.ncols_-1,0.);
		TeCoord2D lright(params_.ncols_-1,params_.nlines_-1);

		lleft = index2Coord (lleft);
		uleft = index2Coord (uleft);
		uright = index2Coord (uright);
		lright = index2Coord(lright);
		
		double x1 = min(lleft.x(), uleft.x());
		double x2 = max(lright.x(), uright.x());
		double y1 = min(lleft.y(), lright.y());
        // double y2 = max(uleft.y(), uright.y());   variable not used

		params_.lowerLeftResolutionSize(x1,y1,params_.resx_,params_.resy_,params_.ncols_,params_.nlines_);

        TeCoord2D coord(0,0);
		TeCoord2D pt1 = index2Coord(coord);
		x2 = pt1.x()+params_.boundingBox().width();
        TeCoord2D tmpCoord =  TeCoord2D(x2,pt1.y());
		TeCoord2D pt2 = params_.coord2Index(tmpCoord);
		params_.nTilesInMemory_ = (abs((int)pt2.y())+1)*nBands_;
	}
	else if (!params_.box().isValid())
	{
		params_.lowerLeftResolutionSize(0.5,0.5,1,1,params_.ncols_,params_.nlines_);
	}
}
	

bool
TeDecoderTIFF::GetGeoTIFF()
{
	if (!gtif) 
		return false;

	int count;
	transMatrix_ = 0;
	pixelScale_ = 0;
	tiePoints_ = 0;

	double TIangle = 0.;
	params_.resx_ = params_.resy_ = 1;
	double TIx0=0, TIy0=0, TIi0=0, TIj0 = 0.;
//	double TIzone = 0;

	if(gtif->gt_num_keys > 0)
	{
		if ((gtif->gt_methods.get)(tif, GTIFF_TRANSMATRIX, &count, &transMatrix_ ))
		{
			if (transMatrix_[0] > 0.) 
			{
				TIangle = atan (transMatrix_[4]/transMatrix_[0]);
				params_.resx_ = TeRoundD(fabs(transMatrix_[0]/cos(TIangle)));
				params_.resy_ = TeRoundD(fabs(transMatrix_[1]/sin(TIangle)));
				TIangle = -1. * TIangle * 180. / 3.14159264359;

			}
		}
		else
		{
			if ((gtif->gt_methods.get)(tif, GTIFF_PIXELSCALE, &count, &pixelScale_ ))
			{
				params_.resx_ = TeRoundD(fabs (pixelScale_[0]));
				params_.resy_ = TeRoundD(fabs (pixelScale_[1]));
			}
		}
	}

	if ((gtif->gt_methods.get)(tif, GTIFF_TIEPOINTS, &count, &tiePoints_ ))
	{
		if (count > 0)
		{
			TIi0 = tiePoints_[0];
			TIj0 = tiePoints_[1];
			int di = (int)(TIi0 - 0);
			int dj = (int)(TIj0 - 0); 
			
			TIx0 = tiePoints_[3];
			TIy0 = tiePoints_[4];

			double x0 = TIx0 - di*params_.resx_;
			double y0 = TIy0 - dj*params_.resy_;
			
			params_.topLeftResolutionSize(x0,y0,params_.resx_,params_.resy_,params_.ncols_,params_.nlines_);
		}
	}

	GTIFDefn	defn;        
    if (GTIFGetDefn( gtif, &defn ) )
	{
		isGeoTiff_ = false;

		// default parameters of a spherical ellipsoid
		double TISemiMajor = 6.371000e+06, TISemiMinor = 0;
		if (defn.SemiMajor == 0. && defn.SemiMinor == 0.)
		{
			GTIFKeyGet(gtif, GeogSemiMajorAxisGeoKey,&TISemiMajor, 0, 1 );
			GTIFKeyGet(gtif, GeogSemiMinorAxisGeoKey,&TISemiMinor, 0, 1 );
		}
		else
		{
			TISemiMajor = defn.SemiMajor;
			TISemiMinor = defn.SemiMinor; 
		}

		TeDatum mDatum;
		if (defn.Datum == 6291)
			mDatum = TeDatumFactory::make("SAD69");
		else if (defn.Datum == 6326)
			mDatum = TeDatumFactory::make("WGS84");
		else
		{
			double f = (TISemiMajor - TISemiMinor)/TISemiMajor;
			if (!findDatum(TISemiMajor,f,mDatum))
				mDatum = TeDatum("UserDefined",TISemiMajor,f,0.,0.,0.);
		}

		double TIlat1 = 0., TIlat2 = 0., TIlat0 = 0., TIlon0 = 0.;
		double TIdeltax = 0., TIdeltay = 0., TIScale = 0.;

		// Try to get projection parameters
		TeProjectionParams mProjPars;	
		mProjPars.units = "Meters";		

		TeProjection* proj = 0;

		short mtype = ModelTypeProjected;
		GTIFKeyGet(gtif, GTModelTypeGeoKey,&mtype, 0, 1 );
		if (mtype == ModelTypeGeographic)
		{
			proj = new TeLatLong(mDatum);
		}
		else
		{
			switch (defn.CTProjection)
			{
				case CT_TransverseMercator : 
					TIlat0 = defn.ProjParm[0];
					TIlon0 = defn.ProjParm[1];
					TIScale = defn.ProjParm[4];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];

					TIy0 = TIy0 + 10000000. - TIdeltay;
					TIx0 = TIx0 + 500000. - TIdeltax;

					mProjPars.name = "UTM";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lat0 = TIlat0*TeCDR;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.scale = TIScale;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;

					break;
				case CT_Mercator :
					TIlat1 = defn.ProjParm[0];
					TIlon0 = defn.ProjParm[1];
					TIScale = defn.ProjParm[4];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];
				
					mProjPars.name = "Mercator";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.stlat1 = TIlat1*TeCDR;
					mProjPars.scale = TIScale;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;
				case CT_LambertConfConic_2SP :
					TIlat0 = defn.ProjParm[0]*defn.UOMAngleInDegrees;
					TIlat1 = defn.ProjParm[2]*defn.UOMAngleInDegrees;
					TIlat2 = defn.ProjParm[3]*defn.UOMAngleInDegrees;
					TIlon0 = defn.ProjParm[1]*defn.UOMAngleInDegrees;
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];
					TIy0 = TIy0 - 10000000.;
					TIx0 = TIx0 - 3000000.;

					mProjPars.name = "LambertConformal";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lat0 = TIlat0*TeCDR;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.stlat1 = TIlat1*TeCDR;
					mProjPars.stlat2 = TIlat2*TeCDR;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;
				case CT_Polyconic : 
					TIlat0 = defn.ProjParm[0];
					TIlon0 = defn.ProjParm[1];
					TIScale = defn.ProjParm[4];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];

					mProjPars.name = "Polyconic";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lat0 = TIlat0*TeCDR;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.scale = TIScale;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;

				case CT_Equirectangular :
					TIlat1 = defn.ProjParm[0];
					TIlon0 = defn.ProjParm[1];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];

					mProjPars.name = "CylindricalEquidistant";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.stlat1 = TIlat1*TeCDR;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;

				case CT_PolarStereographic :
					TIlat0 = defn.ProjParm[0];
					TIlon0 = defn.ProjParm[1];
					TIScale = defn.ProjParm[4];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];
					
					mProjPars.name = "PolarStereographic";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lat0 = TIlat0*TeCDR;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.scale = TIScale;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;
				case CT_AlbersEqualArea :
					TIlat1 = defn.ProjParm[0];
					TIlat2 = defn.ProjParm[1];
					TIlat0 = defn.ProjParm[2];
					TIlon0 = defn.ProjParm[3];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];

					mProjPars.name = "Albers";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lat0 = TIlat0*TeCDR;
					mProjPars.lon0 = TIlon0*TeCDR;
					mProjPars.stlat1 = TIlat1*TeCDR;
					mProjPars.stlat2 = TIlat2*TeCDR;

					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;
				case CT_MillerCylindrical:
					TIlon0 = defn.ProjParm[1];
					TIdeltax = defn.ProjParm[5];
					TIdeltay = defn.ProjParm[6];

					mProjPars.name = "Miller";
					mProjPars.offx = TIdeltax;
					mProjPars.offy = TIdeltay;
					mProjPars.lon0 = TIlon0*TeCDR;
					if (TIdeltay > 0.1)
						mProjPars.hemisphere = TeSOUTH_HEM;
					else
						mProjPars.hemisphere = TeNORTH_HEM;
					break;
				default :
					mProjPars.name = "NoProjection";
					break;
			}
			mProjPars.datum = mDatum;
			proj = TeProjectionFactory::make(mProjPars);
		}
		if (proj && proj->name() != "NoProjection")
			isGeoTiff_ = true;
		params_.projection(proj);
		delete proj;
	}

	// try find .tfw file
	string filename=params_.fileName_;
	long pos = filename.size() - 3;
	filename.replace(pos,3,"tfw");
	DecodeTFW(filename);
	return isGeoTiff_;
}


bool
TeDecoderTIFF::DecodeTFW(const string& filename)
{
	FILE *fp = fopen(filename.data(),"r");
	if( fp )
	{
		isGeoTiff_ = true;
		double dxj=0.,dyj=0.,dxi=0.,dyi=0.,x0=0.,y0=0.;
		char val[40];
		if( fscanf(fp,"%s",val) )
			dxj = TeRoundD(atof(val)); // dxJ=resx
		else
			isGeoTiff_ = false;
		if( fscanf(fp,"%s",val) )
			dyj = TeRoundD(atof(val)); // dyJ
		else
			isGeoTiff_ = false;
		if( fscanf(fp,"%s",val) )
			dxi = TeRoundD(atof(val)); // dxI 
		else
			isGeoTiff_ = false;
		if( fscanf(fp,"%s",val) )
			dyi = TeRoundD(atof(val)); // dyI = resy
		else
			isGeoTiff_ = false;
		if( fscanf(fp,"%s",val) )
			x0 = TeRoundD(atof(val)); // x0
		else
			isGeoTiff_ = false;
		if( fscanf(fp,"%s",val) )
			y0 = TeRoundD(atof(val)); // y0
		else
			isGeoTiff_ = false;
		fclose(fp);

		if (isGeoTiff_)
		{
			params_.resx_ = dxj;
			params_.resy_ = ABS(dyi);

			if (dxi != 0 || dyj != 0)	
			{
				params_.dyJ_ = dyj;
				params_.dyI_ = dyi;
				params_.dxI_ = dxi;
				params_.dxJ_ = dxj;
				params_.x0_ = x0;
				params_.y0_ = y0;
			}
			else					// there is no rotation or scaling defined
			{						// uses only box and resolution
				// double llx = x0; variable not used
				// double ury = y0; variable not used
				params_.topLeftResolutionSize(x0,y0,params_.resx_,params_.resy_,params_.ncols_,params_.nlines_,true);
			}
		}
		return true;
	}
	return false;
}

// --- Methods that deal with the internal buffer to store the image in memory

bool 
TeDecoderTIFF::AllocateBuffer()
{
	unsigned long bufsize;
	int n;

	if (TBufferTile_)
	{
		delete []TBufferTile_;
		TBufferTile_ = 0;
	}
	if (TBuffer_)
	{
		delete []TBuffer_;
		TBuffer_ = 0;
	}

	if (isTiled_)		// we dont write tiled TIFF
	{					// let libtigg manage its raw data buffer
		// allocate a buffer to encode/decode a tile
		bufsize = bytespertile_;
		TBufferTile_ = new unsigned char[bytespertile_];
		if (!TBufferTile_)
			return false;
/*		n = TIFFReadBufferSetup(tif,TBufferTile_,bufsize);
		n = TIFFWriteBufferSetup(tif,TBufferTile_,bufsize);
*/
		// allocate a buffer to hold a line of image
		// a line may be formed by more than one tile
		TBuffer_ = new unsigned char[bytesperline_];
		if (!TBuffer_)
		{
			delete []TBufferTile_;
			TBufferTile_ = 0;
			return false;
		}
	}
	else
	{
		if (params_.mode_ == 'r')		// if we are reading let libtiff handle
		{								// the raw buffer and keeps a copy of each line
			bufsize = TIFFScanlineSize(tif) * params_.nBands();
			TBuffer_ = new unsigned char[bufsize];
			if (!TBuffer_)
				return false;
		}
		else							// if we are writting
		{								// we control the raw buffer 
			bufsize = TIFFStripSize(tif);
			TBuffer_ = new unsigned char[bufsize];
			if (!TBuffer_)
				return false;
			n = TIFFReadBufferSetup(tif,TBuffer_,bufsize);
			n = TIFFWriteBufferSetup(tif,TBuffer_,bufsize);
		}
	}
	return true;
}

void 
TeDecoderTIFF::DeallocateBuffer()
{
	if (TBuffer_)
		delete [] TBuffer_;
	TBuffer_ = 0;

	if (TBufferTile_)
		delete [] TBufferTile_;
	TBufferTile_ = 0;
}


// ---- Internal routines
bool 
TeDecoderTIFF::WriteDummyImage()
{
	long ssize = TIFFScanlineSize(gtif->gt_tif);
	unsigned char* obuf = (unsigned char*)_TIFFmalloc(ssize);
	unsigned char* pp;
	pp = obuf;
	if (TImage == RGBIMAGE)	// RGB image
	{
		for (int x = 0; x < params_.ncols_; x++) 
		{
			if (params_.dataType_[0] == TeFLOAT)			// 32 bits
			{
				((float*)obuf)[x] = (float)params_.dummy_[0];
				((float*)obuf)[x+1] = (float)params_.dummy_[1];
				((float*)obuf)[x+2] = (float)params_.dummy_[2];
			}
			else if (params_.dataType_[0] == TeSHORT)	// 16 bits
			{
				((short*)obuf)[x] = (short)params_.dummy_[0];
				((short*)obuf)[x+1] = (short)params_.dummy_[1];
				((short*)obuf)[x+2] = (short)params_.dummy_[2];
			}
			else											// 8 bits
			{
				obuf[x] = (unsigned char)params_.dummy_[0];
				obuf[x+1] = (unsigned char)params_.dummy_[1];
				obuf[x+2] = (unsigned char)params_.dummy_[2];
			}
		}
	}
	else		// Mono or pallete image
	{
		for (int x = 0; x < params_.ncols_; x++) 
		{
			if (params_.dataType_[0] == TeFLOAT)			// 32 bits
				((float*)obuf)[x] = (float)params_.dummy_[0];
			else if (params_.dataType_[0] == TeSHORT)	// 16 bits
				((short*)obuf)[x] = (short)params_.dummy_[0];
			else											// 8 bits
				obuf[x] = (unsigned char)params_.dummy_[0];
		}
	}

 	for (int row=0; row<params_.nlines_; row++)
	{
		if (!TIFFWriteScanline(gtif->gt_tif, obuf, row, 0))
			return false;
	}
	TIFFFlushData(gtif->gt_tif);
	_TIFFfree(obuf);
	return true;
}

bool 
TeDecoderTIFF::ReadMonoImage(unsigned long lin)
{	
	if (isTiled_)
	{
		if (!ReadTileImageContig(lin,TBuffer_))
			return false;
	}
	else
	{
		//int offset = (lin % rowsperstrip_)* bytesperline_;
		if (TIFFReadScanline(tif,TBuffer_,lin,0) < 0)
			return false;
	}
	return true;
}

bool
TeDecoderTIFF::ReadRGBImage (unsigned long lin)
{
	if (planar_ == PLANARCONFIG_CONTIG)
	{
		if (isTiled_)
		{
			if(!ReadTileImageContig(lin,TBuffer_))
				return false;
		}
		else
		{
			if (TIFFReadScanline(tif,TBuffer_,lin,0) < 0)
				return false;
		}
	}
	else	// RGB Separate
	{
		if (isTiled_)
			return false;
		int s;
		for (s=0; s < nBands_; s++)
			if (TIFFReadScanline(tif,&TBuffer_[s*params_.ncols_],lin,s) < 0)
				return false;
	}
	return true;
}

//		|-   -|     |-                 -|  |-   -|
//		|  X  |     |   a   b   0   d   |  |  I  |
//		|     |     |                   |  |     |
//		|  Y  |     |   e   f   0   h   |  |  J  |
//		|     |  =  |                   |  |     |
//		|  Z  |     |   0   0   0   0   |  |  K  |
//		|     |     |                   |  |     |
//		|  1  |     |   0   0   0   1   |  |  1  |
//		|-   -|     |-                 -|  |-   -|

//	Direct transformation
//		X = a*I + b*j + d
//		Y = e*I + f*j + h

//	Inverse Transformation
//		I = [f(X-d) - b(Y-h)) / (af-eb)
//		J = [e(X-d) - a(Y-h)) / (eb-af)
//	where
//	a = transMatrix_[0]
//	b = transMatrix_[1]
//	d = transMatrix_[3]
//	e = transMatrix_[4]
//	f = transMatrix_[5]
//	h = transMatrix_[7]

TeCoord2D
TeDecoderTIFF::index2Coord (TeCoord2D& pt)
{
	if (transMatrix_)
	{
		double x,y;
		x = transMatrix_[3] + pt.x()*transMatrix_[0] + pt.y()*transMatrix_[1];
		y = transMatrix_[7] + pt.x()*transMatrix_[4] - pt.y()*transMatrix_[5];
		return TeCoord2D(x,y);
	}
	else if (tiePoints_)
	{
		double x,y;
		x = tiePoints_[3] + (pt.x() - tiePoints_[0]) * params_.resx_;
		y = tiePoints_[4] - (pt.y() - tiePoints_[1]) * params_.resy_;
		return TeCoord2D(x,y);
	}
	else
		return params_.index2Coord (pt);
}

TeCoord2D
TeDecoderTIFF::coord2Index (TeCoord2D& pt)
{
	if (transMatrix_)
	{
		double i,j;
		double f  = -transMatrix_[5];
		double af = transMatrix_[0]*f;
		double eb = transMatrix_[4]*transMatrix_[1];
		double xd = pt.x() - transMatrix_[3];
		double yh = pt.y() - transMatrix_[7];

		i = (f*xd - transMatrix_[1]*(yh)) / (af-eb);
		j = (transMatrix_[4]*xd - transMatrix_[0]*(yh)) / (eb-af);

		return TeCoord2D(i,j);
	}
	else if (tiePoints_)
	{
		double i,j;
		i = (pt.x() - tiePoints_[3]) / params_.resx_ + tiePoints_[0];
		j = (tiePoints_[4] - pt.y()) / params_.resy_ + tiePoints_[1];
		return TeCoord2D(i,j);
	}
	else
		return params_.coord2Index(pt);
}

void TeDecoderTIFF::Print()
{
	long t = 0L;
	if(tif)
		TIFFPrintDirectory(tif,stdout,t);
}

static long Tiffpow2(short bits)
{
	unsigned long ret = 1;
	unsigned short x;
	
	for(x=0; x < bits ; x++)
		ret *= 2;
		
	return ret;
}

static unsigned short CalculateMask(unsigned short bits)
{
	unsigned long ret;
	ret = Tiffpow2(bits) -1L;
	return (unsigned short)ret;
}

static void ShiftAndLoadTo8Bits(unsigned char *ret, unsigned char *buf, short bits, unsigned long col, unsigned short mask)
{
	long shift1 = (long) (8 - bits);
	long shift = shift1;
	unsigned long y = 0;
	unsigned long x;

	for(x=0; x < col; x++)
	{
		ret[x] = (buf[y] >> shift)&mask;
		shift -= bits;
		if( shift < 0 )
		{
			shift = shift1;
			y++;
		}
	}	
}

static void CreateLut(short bits,u_char lut[])
{
	u_long 	itens = Tiffpow2(bits);
	u_long 	x;

	for (x=0; x < itens; x++)
		lut[x] = (u_char) ( (255L*x)/(itens-1));
}

void TeDecoderTIFF:: To8Bits(unsigned char* buf,unsigned char *ret,short vmax,short vmin,short nx)
{
	if (params_.nbitsperPixel_[0] == 32)
	{
		unsigned long aux;

		long j=0;
		long ls = (long)params_.ncols_*4;

		for (long i=0; i< ls; i+=4)
		{
			memcpy(&aux,&buf[i],4);
	
			ret[j] = (unsigned char)aux;
 			j++;
		}
		return;
	}

	if (params_.nbitsperPixel_[0] == 16)
	{
		short aux,aux2,i,j;
		float div = (float)((vmax-vmin)/255.);

		j=0;

		for (i=0; i<nx*2; i+=2)
		{
			memcpy(&aux,&buf[i],2);
	
			aux2 = (short) ((float)(aux-vmin) /div);
			ret[j] = (unsigned char)aux2;
 			j++;
		}
		return;
	}

	if ( params_.nbitsperPixel_[0] == 8 )
	{
		memcpy(ret,buf,params_.ncols_);
		return;
	}

	if( first )
	{
		mask = CalculateMask(params_.nbitsperPixel_[0]);
		CreateLut(params_.nbitsperPixel_[0],Lut);
		first = 0;
	}
	
	ShiftAndLoadTo8Bits(ret,buf,params_.nbitsperPixel_[0],params_.ncols_,mask);
	unsigned long x;
	for(x=0; x < (unsigned long)params_.ncols_; x++)
	{
		ret[x] = Lut[ret[x]];
	}
}

// --- Tile Functions ---/
bool 
TeDecoderTIFF::ReadTileImageContig(unsigned long lin, unsigned char *line)
{
	unsigned long offset = lin%tileh_;
	unsigned long col, i, nbytes;	
	for (col=0; col < (unsigned long)params_.ncols_; col += tilew_)
	{
		if (TIFFReadTile(tif, TBufferTile_, col, lin, 0, 0) < 0 )
			return false;
		nbytes = params_.nbitsperPixel_[0]/8;
		if (col + tilew_ > (unsigned long)params_.ncols_ )
		{
			unsigned long diff = params_.ncols_-col;
			for (i=0; i < diff*nbytes; i++)
				line[i+col] = TBufferTile_[i+offset*rowtilesize_];
		}
		else
		{
			for (i=0; i < tilew_*nbytes; i++)
				line[col+i] = TBufferTile_[i+offset*rowtilesize_];
		}
	}
	return true;
}

// --- To read and set a LUT table ---

static int checkmap(int n, unsigned short *r, unsigned short *g, unsigned short *b,long val)
{
    while (n-- >= 0)
		if (*r++ > val || *g++ > val || *b++ > val)
			return (16);
    return (8);
}

static unsigned short CVT1 (unsigned long x,long value)	//CVT is used in DEC station
{
	unsigned long den = (1L << 16) -1L;
	unsigned short num = (unsigned short)((x*value)/den); // normalize lut value
	return num;
}

bool
TeDecoderTIFF::ReadLut()
{
	unsigned short *rmap,*gmap,*bmap;
	if (!TIFFGetField(tif, TIFFTAG_COLORMAP, &rmap, &gmap, &bmap))
		return false;
	
	int range = (1 << params_.nbitsperPixel_[0])-1;
	int x;
	long val = (long)pow(2.,params_.nbitsperPixel_[0]) - 1;
	params_.setNumberPalleteEntries(range);
	if (checkmap (range,rmap,gmap,bmap,val) == 16)
	{
		for(x = range; x >= 0; x--)
		{
			params_.lutr_[x] = CVT1(rmap[x],val); 
			params_.lutg_[x] = CVT1(gmap[x],val);
			params_.lutb_[x] = CVT1(bmap[x],val);
		}
	}
	else
	{
		for(x = range; x >= 0; x--)
		{
			params_.lutr_[x] = rmap[x];
			params_.lutg_[x] = gmap[x];
			params_.lutb_[x] = bmap[x];
		}
	}
	return true;
}

void
TeDecoderTIFF::SaveLut()
{
	if (!tif)
		return;

	int nentries = params_.lutr_.size();
	if (nentries <= 0) 
		return;

	unsigned short* lutr = new unsigned short[nentries];
	unsigned short* lutg = new unsigned short[nentries];
	unsigned short* lutb = new unsigned short[nentries];

	for (int i=0; i<nentries; i++)
	{
		lutr[i] = params_.lutr_[i]*255;
		lutg[i] = params_.lutg_[i]*255;
		lutb[i] = params_.lutb_[i]*255;
	}
	TIFFSetField(tif, TIFFTAG_COLORMAP,lutr,lutg,lutb);
}
