/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include <TeDecoderVirtualMemory.h>

//! Destructor
TeMemoryPage::~TeMemoryPage()
{
	if (this->data_ != 0)
	{
		switch (dataType_)
		{
		case (TeUNSIGNEDCHAR):
			delete [](unsigned char*)data_;
			break;
		case (TeCHAR) :
			delete (char*)data_;
			break;
		case (TeUNSIGNEDSHORT):
			delete [](unsigned short*)data_;
			break;
		case (TeSHORT):
			delete (short*) data_;
			break;
		case (TeUNSIGNEDLONG):
			delete [](unsigned long*)data_;
			break;
		case (TeLONG):
			delete [](long*) data_;
			break;
		case (TeFLOAT):
			delete [](float*)data_;
			break;
		case (TeDOUBLE):
			delete [](double*) data_;
			break;
		default:
			break;
		}
		data_ = 0;
	}
}

TeMemoryPage::TeMemoryPage(unsigned long size, double defValue, TeDataType dataType):
		used_(false),
		size_(size),
		defValue_(defValue),
		dataType_(dataType)
{
	data_ = 0;
	try 
	{
		unsigned int i;
		// Try to allocate the necessary memory
		switch (dataType_) {
		case (TeUNSIGNEDCHAR):
			data_ = new unsigned char [size_];
			for (i=0;i<size;((unsigned char*)data_)[i]=(unsigned char) defValue_,i++);
			break;
		case (TeCHAR) :
			data_ = new char [size_];
			for (i=0;i<size;((char*)data_)[i]=(char) defValue_,i++);
			break;
		case (TeUNSIGNEDSHORT):
			data_ = new unsigned short [size_];
			for (i=0;i<size;((unsigned short*)data_)[i]= (unsigned short) defValue_,i++);
			break;
		case (TeSHORT):
			data_ = new short [size_];
			for (i=0;i<size;((short*)data_)[i]=(short)defValue_,i++);
			break;
		case (TeUNSIGNEDLONG):
			data_ = new unsigned long [size_];
			for (i=0;i<size;((unsigned long*)data_)[i]=(unsigned long)defValue_,i++);
			break;
		case (TeLONG):
			data_ = new long [size_];
			for (i=0;i<size;((long*)data_)[i]=(long)defValue_,i++);
			break;
		case (TeFLOAT):
			data_ = new float [size_];
			for (i=0;i<size;((float*)data_)[i]=(float)defValue_,i++);
			break;
		case (TeDOUBLE):
			data_ = new double [size_];
			for (i=0;i<size;((double*)data_)[i]=(double)defValue_,i++);
			break;
		default:
			break;
		}
	}
	catch (...)
	{
	}
}

void
TeMemoryPage::clear()
{	
	unsigned int i;
	switch (dataType_) 
	{
	case (TeUNSIGNEDCHAR):
		for (i=0;i<size_;((unsigned char*)data_)[i]=(unsigned char) defValue_,i++);
		break;
	case (TeCHAR) :
		for (i=0;i<size_;((char*)data_)[i]=(char) defValue_,i++);
		break;
	case (TeUNSIGNEDSHORT):
		for (i=0;i<size_;((unsigned short*)data_)[i]= (unsigned short) defValue_,i++);
		break;
	case (TeSHORT):
		for (i=0;i<size_;((short*)data_)[i]=(short)defValue_,i++);
		break;
	case (TeUNSIGNEDLONG):
		for (i=0;i<size_;((unsigned long*)data_)[i]=(unsigned long)defValue_,i++);
		break;
	case (TeLONG):
		for (i=0;i<size_;((long*)data_)[i]=(long)defValue_,i++);
		break;
	case (TeFLOAT):
		for (i=0;i<size_;((float*)data_)[i]=(float)defValue_,i++);
		break;
	case (TeDOUBLE):
		for (i=0;i<size_;((double*)data_)[i]=(double)defValue_,i++);
		break;
	default:
		break;
	}
}

long
TeMemoryPage::pageSize()
{
	long s;
	switch (dataType_) 
	{
	case (TeCHAR) :
		s = size_*sizeof(char);
		break;
	case (TeUNSIGNEDSHORT):
		s = size_*sizeof(unsigned short);
		break;
	case (TeSHORT):
		s = size_*sizeof(short);
		break;
	case (TeUNSIGNEDLONG):
		s = size_*sizeof(unsigned long);
		break;
	case (TeLONG):
		s = size_*sizeof(long);
		break;
	case (TeFLOAT):
		s = size_*sizeof(float);
		break;
	case (TeDOUBLE):
		s = size_*sizeof(double);
		break;
	default:
		s = size_*sizeof(unsigned char);
	}
	return s;
}

double 
TeMemoryPage::getVal(int col,int lin, int nCols)
{
	unsigned long offset = (lin-ulLin_)*nCols+(col-ulCol_);
	double val = defValue_;
	if (offset < size_) // solucao temporaria, rever
	{
		switch (dataType_) 
		{
		case (TeUNSIGNEDCHAR):
			val = ((unsigned char*)data_)[offset];
			break;
		case (TeCHAR) :
			val = ((char*) data_)[offset];
			break;
		case (TeUNSIGNEDSHORT):
			val = ((unsigned short*)data_)[offset];
			break;
		case (TeSHORT):
			val = ((short*)data_)[offset];
			break;
		case (TeUNSIGNEDLONG):
			val = ((unsigned long*)data_)[offset];
			break;
		case (TeLONG):
			val = ((long*)data_)[offset];
			break;
		case (TeFLOAT):
			val = ((float*)data_)[offset];
			break;
		case (TeDOUBLE):
			val = ((double*)data_)[offset];
			break;
		default:
			break;
		}
	}
	return val;
}

void 
TeMemoryPage::setVal(int col,int lin,int nCols, double val)
{
	unsigned long offset = (lin-ulLin_)*nCols+(col-ulCol_);
	if (offset < size_) 
	{
		switch (dataType_) 
		{
		case (TeUNSIGNEDCHAR):
			((unsigned char*)data_)[offset] = (unsigned char) val;
			break;
		
		case (TeCHAR) :
			((char*) data_)[offset] = (char) val;
			break;
		
		case (TeUNSIGNEDSHORT):
			((unsigned short*)data_)[offset] = (unsigned short) val;
			break;
		
		case (TeSHORT):
			((short*)data_)[offset] = (short) val;
			break;
		
		case (TeUNSIGNEDLONG):
			((unsigned long*)data_)[offset] = (unsigned long) val;
			break;
		
		case (TeLONG):
			((long*)data_)[offset] = (long) val;
			break;
		
		case (TeFLOAT):
			((float*)data_)[offset] = (float) val;
			break;
		
		case (TeDOUBLE):
			((double*)data_)[offset] = val;

		default:
			break;
		}
	}
	return;
}

TeDecoderVirtualMemory::TeDecoderVirtualMemory( const TeRasterParams& par)
{
	params_ = par;
}

TeDecoderVirtualMemory::~TeDecoderVirtualMemory()
{
	if (!pagesQueue_.empty())
		TeDecoderVirtualMemory::clear();
}

bool
TeDecoderVirtualMemory::getElement(int col,int lin, double &val,int band)
{
	// identify the page has the pixel (col,lin,band)
	string index = codifyId(col,lin,band,1,0);
	TeMemoryPage* block = 0;

//	int nt = params_.nTilesInMemory_;

	// check band cache first
	if (indexCache_[band] == index)
		block = pageCache_[band];
	else
	{
		// check if page is already in memory
		MapMemoryPageIterator p = virtualMemory_.find(index);
		if (p != virtualMemory_.end())
		{
			// use it
			 block = p->second;
		}
		else
		{
			// page is not in memory
			// check whether there is space in memory to bring another page
			if (virtualMemory_.size() >= (unsigned int)params_.nTilesInMemory_)
			{
				// FIFO strategy: replace the oldest page in memory
				string first = pagesQueue_.front();
				pagesQueue_.pop();
				p = virtualMemory_.find(first); 
				if (p != virtualMemory_.end())
				{
					block = p->second;
					if (block->used_)
						putRasterBlock(first,block->data_,block->pageSize());
					block->clear();				// reuse the allocated memory
					virtualMemory_.erase(first);
				}
			}
			else
				// bring the page to memory
				block = new TeMemoryPage(params_.blockHeight_*params_.blockWidth_, params_.dummy_[band],params_.dataType_[band]);

			virtualMemory_.insert(MapMemoryPage::value_type(index,block));
			pagesQueue_.push(index);
			int ulLin=0, ulCol=0;
			getRasterBlock(index,block->data_,ulCol,ulLin);
			block->ulLin_ = ulLin;
			block->ulCol_ = ulCol;
		}
		indexCache_[band] = index;
		pageCache_[band] = block;
	}
	val = block->getVal(col,lin,params_.blockWidth_);
	return true;
}

bool
TeDecoderVirtualMemory::setElement(int col, int lin, double val,int band)
{
	string index = codifyId(col,lin,band,1,0);

	TeMemoryPage* block = 0;

	// check band cache first
	if (indexCache_[band] == index)
		block = pageCache_[band];
	else
	{
		MapMemoryPageIterator p = virtualMemory_.find(index);
		if (p != virtualMemory_.end())
		{
			 block = p->second;
		}
		else
		{
			if (virtualMemory_.size() >= (unsigned int)params_.nTilesInMemory_)
			{
				string first = pagesQueue_.front();
				pagesQueue_.pop();
				p = virtualMemory_.find(first);
				if (p != virtualMemory_.end())
				{
					block = p->second;
					if (block->used_ == true)
					{
						putRasterBlock(first,block->data_,block->pageSize());
					}
					block->clear();
					virtualMemory_.erase(first);
				}
			}
			else
				block = new TeMemoryPage(params_.blockHeight_*params_.blockWidth_, params_.dummy_[band],params_.dataType_[band]);
			virtualMemory_.insert(MapMemoryPage::value_type(index,block));
			pagesQueue_.push(index);
			int ulLin=0, ulCol=0;
			getRasterBlock(index,block->data_,ulCol,ulLin);
			block->ulLin_ = ulLin;
			block->ulCol_ = ulCol;
		}
		indexCache_[band] = index;
		pageCache_[band] = block;
	}
	block->setVal(col,lin,params_.blockWidth_,val);
	block->used_ = true;
	return true;
}


/*bool
TeDecoderVirtualMemory::create()
{
	clear();
	if ( params_.nTilesInMemory_ == 0 )
		params_.nTilesInMemory_ = params_.nBands() * (params_.ncols_ / params_.blockWidth_+1) ;
	pageCache_.resize(params_.nBands(),0);
	indexCache_.resize(params_.nBands(),"");
	return true;
}
*/
void
TeDecoderVirtualMemory::init()
{
	TeDecoderVirtualMemory::clear();
	if ( params_.nTilesInMemory_ == 0 )
		params_.nTilesInMemory_ = params_.nBands() * (params_.ncols_ / params_.blockWidth_+1) ;
	pageCache_.resize(params_.nBands(),0);
	indexCache_.resize(params_.nBands(),"");
}

bool
TeDecoderVirtualMemory::clear()
{
	TeMemoryPage* block;
	string index;

	MapMemoryPageIterator p;

	while (!pagesQueue_.empty() )
	{
		index = pagesQueue_.front();
		pagesQueue_.pop();
		p = virtualMemory_.find(index);
		if (p != virtualMemory_.end())
		{
			block = p->second;
			if (block->used_)
			{
				putRasterBlock(index,block->data_,block->pageSize());
			}
			delete block;
			virtualMemory_.erase(index);
		}
	}
	indexCache_.clear();
	pageCache_.clear();
	return true;
}

