/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifdef WIN32
#pragma warning ( disable: 4786 )
#endif

#include "TeProjection.h"
#include "TeException.h"
#include "TeUtils.h"
#include <stdio.h>

//=========================================================
//
//  PROJECTION INFORMATION 
//
//  Auxiliary functions which indicate the information
//  associated to a projection
//
//  Used for reading and writing information
//
//==========================================================


// Name						Units  Long  Lat  Par1  Par2  Sca  Eas  Nor 
char* teProjInfo[]= {
"Albers",					"1",  "1",  "1", "1",  "1",  "0", "1", "1", 
"LatLong",					"1",  "0",  "0", "0",  "0",  "0", "0", "0",   
"LambertConformal",			"1",  "1",  "1", "1",  "1",  "0", "1", "1", 
"Mercator",					"1",  "1",  "1", "1",  "0",  "0", "1", "1", 
"Miller",					"1",  "1",  "0", "0",  "0",  "0", "1", "1", 
"UTM",						"1",  "1",  "0", "0",  "0",  "1", "1", "1", 
"Polyconic",				"1",  "1",  "1", "0",  "0",  "0", "1", "1",
"Sinusoidal",				"1",  "1",  "0", "0",  "0",  "0", "1", "1",
"CylindricalEquidistant",	"1",  "1",  "0", "1",  "0",  "0", "1", "1",
"PolarStereographic",		"1",  "1",  "0", "0",  "0",  "0", "1", "1",
"NoProjection",				"1",  "0",  "0", "0",  "0",  "0", "0", "0"		
};

TeProjInfo
TeProjectionInfo ( const string& projName  )
{

	TeProjInfoMap pjMap;
	TeProjInfo pjInfo;

	int k = 0;

	for ( int i = 0; i < NUM_PROJ; i++ )
	{
		string name = teProjInfo [k++];
			
	    pjInfo.hasUnits  = atoi ( teProjInfo [k++] );
	    pjInfo.hasLon0   = atoi ( teProjInfo [k++] );
	    pjInfo.hasLat0   = atoi ( teProjInfo [k++] );
	    pjInfo.hasStlat1 = atoi ( teProjInfo [k++] );
		pjInfo.hasStlat2 = atoi ( teProjInfo [k++] );
		pjInfo.hasScale  = atoi ( teProjInfo [k++] );
		pjInfo.hasOffx   = atoi ( teProjInfo [k++] );
	    pjInfo.hasOffy   = atoi ( teProjInfo [k++] );

		pjMap [ name ] = pjInfo;
	}


	TeProjInfoMap::iterator it = pjMap.find ( projName );

	if ( it == pjMap.end() )
		throw TeException ( PROJECTION_NOT_AVAILABLE );

return (*it).second;
}

//========================================================
//
// PROJECTION FACTORY
//
// =======================================================

TeProjection*
TeProjectionFactory::make ( const TeProjectionParams& par )
{
	string punits;
	if (par.units.empty())
		punits = "Meters";
	else
		punits = par.units;

	if ( par.name == "UTM" )
		return new TeUtm ( par.datum, par.lon0, par.lat0,
		par.offx, par.offy, punits, par.scale, par.hemisphere );

	if ( par.name == "LambertConformal")
		return new TeLambertConformal ( par.datum, par.lon0, par.lat0,
		par.offx, par.offy, par.stlat1, par.stlat2, punits );
	
	if ( par.name == "Albers" )
		return new TeAlbers ( par.datum, par.lon0, par.lat0,
		par.offx, par.offy, par.stlat1, par.stlat2, punits );

	if ( par.name == "Miller")
		return new TeMiller ( par.datum, par.lon0, par.offx, par.offy, punits );

	if ( par.name == "LatLong" )
	{
		if (par.units.empty())
			punits = "DecimalDegrees";
		else
			punits = par.units;
		return new TeLatLong ( par.datum, par.units );
	}

	if ( par.name == "Polyconic" )
		return new TePolyconic ( par.datum, par.lon0, par.lat0, par.offx, par.offy, punits); 

	if ( par.name == "Mercator" )
		return new TeMercator ( par.datum, par.lon0, par.lat0, 
		par.offx, par.offy, par.stlat1, punits );

	if ( par.name == "Sinusoidal")
		return new TeSinusoidal ( par.datum, par.lon0, par.offx, par.offy, punits );

	if ( par.name == "CylindricalEquidistant")
		return new TeCylindricalEquidistant ( par.datum, par.lon0, par.offx, par.offy, par.stlat1, punits);

	if ( par.name == "PolarStereographic")
		return new TePolarStereographic ( par.datum, par.lon0, par.offx, par.offy,punits, par.hemisphere );

	if ( par.name == "NoProjection")
		return new TeNoProjection (punits);

return 0;
}

TeProjectionParams 
TeProjection::params ()
{
	TeProjectionParams par;
	
	par.name  = GPname;
	par.datum = GPdatum;
	par.lon0  = GPlon0;
	par.lat0  = GPlat0;
	par.offx  = GPoffx;
	par.offy  = GPoffy;
	par.stlat1 = GPstlat1;
	par.stlat2 = GPstlat2;
	par.units  = GPunits;
	par.scale  = GPscale;
	par.hemisphere =  GPhemisphere;

return par;
}


TeProjection::TeProjection(const TeProjection& rhs)
{
	GPname  = rhs.GPname;
	GPdatum = rhs.GPdatum;
	GPlon0  = rhs.GPlon0;
	GPlat0  = rhs.GPlat0;
	GPoffx  = rhs.GPoffx;
	GPoffy  = rhs.GPoffy;
	GPstlat1 = rhs.GPstlat1;
	GPstlat2 = rhs.GPstlat2;
	GPunits  = rhs.GPunits;
	GPscale  = rhs.GPscale;
	GPhemisphere = rhs.GPhemisphere;
}

TeProjection& 
TeProjection::operator=(const TeProjection& rhs)
{
	if ( this != &rhs )
	{
		GPname  = rhs.GPname;
		GPdatum = rhs.GPdatum;
		GPlon0  = rhs.GPlon0;
		GPlat0  = rhs.GPlat0;
		GPoffx  = rhs.GPoffx;
		GPoffy  = rhs.GPoffy;
		GPstlat1 = rhs.GPstlat1;
		GPstlat2 = rhs.GPstlat2;
		GPunits  = rhs.GPunits;
		GPscale  = rhs.GPscale;
		GPhemisphere = rhs.GPhemisphere;
	}
	return *this;
}



/*******************************************************************
	CHECKS IF A PROJECTION INSTANCE IS EQUAL TO ANOTHER
********************************************************************/

bool
TeProjection::operator== (const TeProjection& proj)
{
	if (!(GPdatum==proj.GPdatum))
		return false;
	if (GPlon0!=proj.GPlon0)
		return false;
	if (GPlat0!=proj.GPlat0)
		return false;
	if (GPoffx!=proj.GPoffx)
		return false;
	if (GPoffy!=proj.GPoffy)
		return false;
	if (GPstlat1!=proj.GPstlat1)
		return false;
	if (GPstlat2!=proj.GPstlat2)
		return false;
	if (GPhemisphere!=proj.GPhemisphere)
		return false;

	return true;
}

/*******************************************************************
	PRINTS INFORMATION ABOUT A PROJECTION
********************************************************************/


void
TeProjection::print ( FILE* file_ )
{
	TeProjInfo pjInfo = TeProjectionInfo ( GPname );

	fprintf(file_,"%s\n", "// Projection Information" );
	fprintf ( file_, "%s %s \n", "PROJECTION", GPname.c_str() );
	fprintf ( file_, "%s %s \n", "DATUM", GPdatum.name().c_str() );
	fprintf ( file_, "%s %s \n", "UNITS", GPunits.c_str() );
	
	if ( pjInfo.hasLon0 )
		fprintf ( file_, "%s %17.6f \n", "ORIGIN LONG", GPlon0 );

	if ( pjInfo.hasLat0 )
		fprintf ( file_, "%s %17.6f \n", "ORIGIN LAT", GPlat0 );

	if ( pjInfo.hasOffx )
		fprintf ( file_, "%s %17.6f \n", "FALSE EASTING", GPoffx );

	if ( pjInfo.hasOffy )
		fprintf ( file_, "%s %17.6f \n", "FALSE NORTHING", GPoffy );
	
	if ( pjInfo.hasStlat1 )
		fprintf ( file_, "%s %17.6f \n", "FIRST STANDARD PARALEL", GPstlat1 );
	
	if ( pjInfo.hasStlat2 )
		fprintf ( file_, "%s %17.6f \n", "SECOND STANDARD PARALEL", GPstlat2 );

	if ( pjInfo.hasScale )
		fprintf ( file_, "%s %17.6f \n", "SCALE", GPscale );

	fprintf(file_,"%s\n", "// End of Projection Information" );
}


string
TeProjection::describe ()
{
	string desc;
	if (GPname == "NoProjection")
	{
		desc = "NoProjection";
		return desc;
	}

	TeProjInfo pjInfo = TeProjectionInfo ( GPname );

	desc =  GPunits;
	if ( pjInfo.hasLon0 )
		desc += "," + Te2String(GPlon0*TeCRD,6);

	if ( pjInfo.hasLat0 )
		desc += "," + Te2String(GPlat0*TeCRD,6);

	if ( pjInfo.hasStlat1 )
		desc += ", " + Te2String(GPstlat1*TeCRD,6);
	
	if ( pjInfo.hasStlat2 )
		desc += "," + Te2String(GPstlat2*TeCRD,6);

	if ( pjInfo.hasOffx )
		desc += "," + Te2String(GPoffx,6);

	if ( pjInfo.hasOffy )
		desc += "," + Te2String(GPoffy,6);
	
	if ( pjInfo.hasScale )
		desc += "," + Te2String(GPscale,6);

	return desc;
}


bool decodifyDescription(const string& projDescription, TeProjectionParams& pars)
{
	vector<string> projDesc;
	if (TeSplitString(projDescription, ",", projDesc) <= 0)
		return false;
	unsigned int npar = projDesc.size();
	TeProjInfo pjInfo = TeProjectionInfo (projDesc[0]);
	pars.name = projDesc[0];
	if (npar > 1)
		pars.units = projDesc[1];
	else 
		return true;

	unsigned int nextp = 2;
	if (pjInfo.hasLon0)
	{
		 if (nextp < npar)
		 {
			pars.lon0 = atof(projDesc[nextp].c_str()) * TeCDR;
			nextp++;
		 }
		 else
			 return false;
	}

	if (pjInfo.hasLat0)
	{
		 if (nextp < npar)
		 {
			pars.lat0 = atof(projDesc[nextp].c_str()) * TeCDR;
			nextp++;
		 }
		 else
			 return false;
	}

	if (pjInfo.hasStlat1)
	{
		 if (nextp < npar)
		 {
			pars.stlat1 = atof(projDesc[nextp].c_str()) * TeCDR;
			nextp++;
		 }
		 else
			 return false;
	}

	if (pjInfo.hasStlat2)
	{
		 if (nextp < npar)
		 {
			pars.stlat2 = atof(projDesc[nextp].c_str()) * TeCDR;
			nextp++;
		 }
		 else
			 return false;
	}

	if (pjInfo.hasOffx)
	{
		 if (nextp < npar)
		 {
			pars.offx = atof(projDesc[nextp].c_str());
			nextp++;
		 }
		 else
			 return false;
	}

	if (pjInfo.hasOffy)
	{
		 if (nextp < npar)
		 {
			pars.offy = atof(projDesc[nextp].c_str());
			nextp++;
		 }
		 else
			 return false;
	}
	
	if (pjInfo.hasScale)
	{
		 if (nextp < npar)
		 {
			pars.scale = atof(projDesc[nextp].c_str());
			nextp++;
		 }
		 else
			 return false;
	}
	return true;
}

/********************************************************************
		Planimetric datum transformation 
********************************************************************/
void
TeProjection :: ChangeLL (double &lon1, double &lat1)		
{
	double 	equad1,	// Squared eccentricity - datum 1
			equad2,	// Squared eccentricity - datum 2
			n1,		// Great normal of ellipsoid - datum 1
			n2,		// Great normal od ellipsoid - datum 2
			x1,		// Geocentric cartesian coordinates - datum 1 
			y1,
			z1,
			x2,		// Geocentric cartesian coordinates - datum 2
			y2,
			z2,
			d,lat2,		// Ancillary variable
			lon2;
	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

 	if( GPdestination->GPdatum.radius() == 0.) return;

	int	flt1 = (int)(Pflt*1000000000.);
	int	flt2 = (int)(GPdestination->GPdatum.flattening()*1000000000.);
	if (Prd == GPdestination->GPdatum.radius() && flt1==flt2)
		return;

// Geocentric cartesian coordinates calculation - datum 1
	equad1 = 2.*Pflt;
	equad1 -= Pflt*Pflt;
	double a1,a2,a3;
	a1 = sin(lat1);
	a1 *= a1;
	a1 *= equad1;
	a2 = 1-a1;
	n1 = Prd/sqrt(a2);
	x1 = n1*cos(lat1)*cos(lon1);
	y1 = n1*cos(lat1)*sin(lon1);
	z1 = (n1*(1-equad1))*sin(lat1);

// Geocentric cartesian coordinates calculation - datum 2
	if (GPdatum.xShift() == TeMAXFLOAT || GPdestination->GPdatum.xShift() == TeMAXFLOAT)
	{
		x2 = x1; 
		y2 = y1;
		z2 = z1;
	}
	else
	{	
		x2 = x1 + (GPdatum.xShift() - GPdestination->GPdatum.xShift());
		y2 = y1 + (GPdatum.yShift() - GPdestination->GPdatum.yShift());
		z2 = z1 + (GPdatum.zShift() - GPdestination->GPdatum.zShift());
	}

// Geodetic coordinates calculation - datum 2
	equad2 = 2.*GPdestination->GPdatum.flattening();
	equad2 -= (GPdestination->GPdatum.flattening())*(GPdestination->GPdatum.flattening());
	lat2 = lat1;
	do
	{
		a1 = sin(lat2);
		a1 *= a1;
		a1 *= equad2;
		a2 = 1-a1;
		n2 = GPdestination->GPdatum.radius()/sqrt(a2);
		a1 = equad2*sin(lat2);
		a1 *= n2;
		a1 += z2;
		a2 = x2*x2;
		a2 += (y2*y2);
		a3 = sqrt(a2);
		lat2 = atan2(a1,a3);
		a1 = sin(lat2);
		a1 *= a1;
		a1 *= equad2;
		a2 = 1-a1;
		a3 = sqrt(a2);
		d = (GPdestination->GPdatum.radius()/a3)-n2;
	}
	while (fabs(d) > 0.0000001);
	lon2 = atan2(y2,x2);
	lat1 = lat2;
	lon1 = lon2;
}

/********************************************************************
		GEODETIC TO UTM COORDINATES
********************************************************************/


TeCoord2D
TeUtm :: LL2PC (TeCoord2D& ptll)
{

	double	k0,		// Scale factor
		equad,		// Squared eccentricity
		n,		// Great normal of ellipsoid
		elinquad,	// ancillary variables 
		aux1,aux2,aux3,aux4,aux5,aux6,aux7,aux8,aux9,
		aux10,aux11,aux12,aux13,t,c,ag,m,ptllx,ptlly;

	double Pflt = GPdatum.flattening();
	ptllx = ptll.x();
	ptlly = ptll.y();
	k0 = 1. - (1./2500.);
	equad = 2.*Pflt;	
	equad -= Pflt*Pflt;
	elinquad = equad/(1. - equad);
	aux1 = equad*equad;
	aux2 = aux1*equad;
	aux3 = sin((double)2*ptlly);
	aux4 = sin((double)4*ptlly);
	aux5 = sin((double)6*ptlly);

	double a1, a2, a3;
	a1 = equad/4.;
	a2 = 3.*aux1/64.;
	a3 = 5.*aux2/256.;
	aux6 = (1-a1-a2-a3)*ptlly;

	a1 = 3.*equad/8.;
	a2 = 3.*aux1/32.;
	a3 = 45.*aux2/1024.;
	aux7 = (a1+a2+a3)*aux3;

	a1 = 15.*aux1/256.;
	a2 = 45.*aux2/1024.;
	aux8 = (a1+a2)*aux4;

	aux9 = 35.*aux2;
	aux9 /= 3072.;
	aux9 *= aux5;

	a1 = sin(ptlly);	
	a1 *= a1;		
	a1 *= equad;		
	n = GPdatum.radius()/sqrt((double)1-a1);


	t = tan(ptlly);		
	t *= t;			

	c = cos(ptlly);		
	c *= c;			
	c *= elinquad;		
	ag = (ptllx-GPlon0)*cos(ptlly);
	m = GPdatum.radius()*(aux6 - aux7 + aux8 - aux9);


	a1 = ag*ag*ag;
	aux10 = (1.-t+c)*a1/6.;

	a1 = 5.-(18.*t);
	a2 = a1+t*t;
	a1 = a2+72.*c;
	a2 = a1-(58.*elinquad);
	a1 = ag*ag*ag;		
	a1 *= ag;		
	a1 *= ag;		
	aux11 = a2*a1/120.;	

	a1 = 5.-t+9.*c;
	a2 = 4.*c*c;
	a3 = ag*ag;
	a3 *= ag;
	a3 *= ag;
	aux12 = (a1+a2)*a3/24.;

	a1 = 61.-(58.*t) ;
	a2 = a1+(t*t);
	a1 = a2+(600.*c);
	a2 = a1-(330.*elinquad);
	a1 = ag*ag*ag;		
	a1 *= ag;		
	a1 *= ag;		
	a1 *= ag;		
	aux13 = a2*a1/720.;	

	ptllx = k0*n*(ag + aux10 + aux11);

	a1 = ag*ag/2;
	a2 = a1+aux12+aux13;
	a3 = n*tan(ptlly);
	a1 = a2*a3;
	ptlly = k0*(m+a1);

	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));
}


/********************************************************************
		UTM TO GEODETIC COORDINATES
********************************************************************/

TeCoord2D
TeUtm :: PC2LL (TeCoord2D& ptpc)
{
	double  k0,             	// Scale factor
		equad,			// Squared eccentricity
		n1,			// Great normal of ellipsoid
		elinquad,		// Ancillary variables
		e1,aux1,aux2,aux3,aux4,aux5,m,mi,aux6,aux7,aux8,lat1,
		c1,t1, quoc,r1,d,aux9,aux10,aux11,aux12,ptpcx,ptpcy;

	double Prd  = GPdatum.radius();
	double Pflt = GPdatum.flattening();
	
	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;		

	k0 = 1. - (1./2500.);
	equad = 2.*Pflt;		
	equad -= Pflt*Pflt;		
	elinquad = equad/(1. - equad);

	double a1,a2,a3;
	a1 = sqrt((double)1-equad);
	e1 = (1.-a1)/(1.+a1);

	aux1 = equad*equad;
	aux2 = aux1*equad;
	aux3 = e1*e1;
	aux4 = e1*aux3;
	aux5 = aux4*e1;

	m = ptpcy/k0;


	a1 = 1.-(equad/4.);
	a2 = 3.*(aux1/64.);
	a3 = 5.*(aux2/256.);
	mi = m/(Prd*(a1-a2-a3));


	a1 = 3.*(e1/2.);
	a2 = 27.*(aux4/32.);
	a3 = sin((double)2*mi);
	aux6 = (a1-a2)*a3;

	a1 = 21.*(aux3/16.);
	a2 = 55.*(aux5/32.);
	a3 = sin((double)4*mi);
	aux7 = (a1-a2)*a3;

	a1 = 151.*(aux4/96.);
	a2 = sin((double)6*mi);
	aux8 = a1*a2;

	lat1 = mi + aux6 + aux7 + aux8;

	c1 = cos(lat1);			
	c1 *= c1;			
	c1 *= elinquad;			

	t1 = tan(lat1);			
	t1 *= t1;			

	a1 = sin(lat1);			
	a1 *= a1;			
	a1 *= equad;			
	a2 = 1.-a1;
	n1 = Prd/sqrt(a2);

	a1 = sin(lat1);
	a1 *= a1;
	a1 *= equad;
	a2 = 1.-a1;
	quoc = a2*a2*a2;

	r1 = Prd*(1.-equad);
	r1 /= sqrt(quoc);

	d = ptpcx;
	d /= (n1*k0);


	a1 = 5.+(3.*t1);
	a1 += (10.*c1);
	a1 += (-4.*(c1*c1));
	a1 += (-9.*elinquad);
	a1 *= d;
	a1 *= d;
	a1 *= d;
	a1 *= d;
	aux9 = a1/24.;

	a1 = 61.+(90.*t1);
	a1 += (298.*c1);
	a1 += (45.*(t1*t1));
	a1 += (-252.*elinquad);
	a1 += (-3.*c1*c1);
	aux10 = d*d*d;			
	aux10 *= d;			
	aux10 *= d;			
	aux10 *= d;			
	aux10 *= a1;
	aux10 /= 720.;

	a1 = d*d*d;
	a1 /= 6.;
	a2 = 2.*t1;
	a2 += (1.+c1);
	aux11 = d-a2*a1;

	a1 = 5.-(2.*c1);
	a1 += (28.*t1);
	a1 += (-3.*c1*c1);
	a1 += (8.*elinquad);
	a1 += (24.*t1*t1);
	aux12 = d*d*d;			
	aux12 *= d;			
	aux12 *= d;			
	aux12 *= a1;			
	aux12 /= 120.;			

	a1 = d*d/2.;
	a1 += (- aux9 + aux10);
	a2 = tan(lat1)/r1;
	a2 *= n1;
	ptpcy = lat1-a2*a1;

	ptpcx= GPlon0 + (aux11 + aux12)/cos(lat1);

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return(TeCoord2D(ptpcx,ptpcy));
}

/********************************************************************
		GEODETIC TO MERCATOR COORDINATES
********************************************************************/
TeCoord2D
TeMercator :: LL2PC (TeCoord2D& ptll)
{
	double	equad,			//Squared eccentricity
		aux1,			// Ancillary variables
		aux2,aux3,aux4,aux5,aux6,ptllx,ptlly;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptllx = ptll.x();		
	ptlly = ptll.y();		

	equad = 2.*Pflt;		
	equad -= Pflt*Pflt;		
	double	a1,a2,a3;		
 
	a1 = tan(ptlly/(double)2);	
	a2 = 1.+a1;			
	a3 = 1.-a1;			
	aux1 = a2/a3;			
 
	a1 = equad*equad/4.;		
	a1 += equad;			
	a2 = equad*equad*equad/8.;	
	a3 = sin(ptlly);		
	aux2 = (a1+a2)*a3;		
 
	a1 = equad*equad/12.;		
	a2 = equad*equad*equad/16.;	
	a3 = sin((double)3*ptlly);	
	aux3 = (a1+a2)*a3;		
 
	a1 = equad*equad*equad/80.;	
	a2 = sin((double)5*ptlly);	
	aux4 = a1*a2;			
	aux5 = cos(GPstlat1);
 
	a1 = sin(GPstlat1);		
	a1 *= a1;			
	a1 *= equad;			
	a2 = sqrt((double)1-a1);	
	aux6 = 1./a2;			

	ptllx = Prd*(ptllx - GPlon0)*aux5*aux6;
	ptlly = Prd*(log(aux1) - aux2 + aux3 - aux4)*aux5*aux6;

	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}

/********************************************************************
		MERCATOR TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeMercator :: PC2LL (TeCoord2D& ptpc)
{
	double	equad,			//Squared eccentricity 
	        pi,			// PII value
		t,			// Ancillary variables
		xx,aux1,aux2,aux3,aux4,aux5,ptpcx,ptpcy;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	equad = 2.*Pflt;		
	equad -= Pflt*Pflt;		
	pi = 4.*atan((double)1);
	double	a1,a2;		
	aux1 = cos(GPstlat1);

	a1 = sin(GPstlat1);		
	a1 *= a1;			
	a1 *= equad;			
	a2 = sqrt((double)1-a1);	
	aux2 = 1./a2;			
	ptpcx = ptpcx/(aux1*aux2);
	ptpcy = ptpcy/(aux1*aux2);
	t = exp(-ptpcy/Prd);
	xx = pi/2. - 2.*atan(t);
 
	a1 = equad/2.;			
	a1 += 5.*equad*equad/24.;	
	a1 += equad*equad*equad/12.;	
	a2 = sin((double)4*atan(t));	
	aux3 = a1*a2;			
 
	a1 = 7.*equad*equad/48.;	
	a1 += 29.*equad*equad*equad/240.;	
	a2 = sin((double)8*atan(t));	
	aux4 = -a1*a2;			
 
	a1 = 7.*equad*equad*equad/120.;	
	a2 = sin((double)12*atan(t));	
	aux5 = a1*a2;		 

	ptpcy = xx + aux3 + aux4 + aux5;
	ptpcx = ptpcx/Prd + GPlon0;
	
	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return(TeCoord2D(ptpcx,ptpcy));
}




/********************************************************************
		GEODETIC TO LAMBERT CONIC COORDINATES
********************************************************************/
TeCoord2D
TeLambertConformal :: LL2PC (TeCoord2D& ptll)
{

	double	equad,			// Squared eccentricity
		e,			// Ancillary variables
		m1,m2,aux1,aux2,aux0,t1,t2,t0,n,efe,ro0,aux,
		t,ro,teta,ptllx,ptlly;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptllx = (double)ptll.x();
	ptlly = (double)ptll.y();

	equad = 2.*Pflt - pow(Pflt,(double)2);
	e = sqrt(equad);

	m1 = cos(GPstlat1)/sqrt((double)1-equad*pow(sin(GPstlat1),(double)2));
	m2 = cos(GPstlat2)/sqrt((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux1 = sqrt(((double)1-e*sin(GPstlat1))/((double)1+e*sin(GPstlat1)));
	aux2 = sqrt(((double)1-e*sin(GPstlat2))/((double)1+e*sin(GPstlat2)));
	aux0 = sqrt(((double)1-e*sin(GPlat0))/((double)1+e*sin(GPlat0)));
	t1 = ((1.-tan(GPstlat1/(double)2))/(1.+tan(GPstlat1/(double)2)))/pow(aux1,e);
	t2 = ((1.-tan(GPstlat2/(double)2))/(1.+tan(GPstlat2/(double)2)))/pow(aux2,e);
	t0 = ((1.-tan(GPlat0/(double)2))/(1.+tan(GPlat0/(double)2)))/pow(aux0,e);

	if (GPstlat1 == GPstlat2)
		n = sin(GPstlat1);
	else
		n = (log(m1)-log(m2))/(log(t1)-log(t2));

	efe = m1/(n*pow(t1,n));
	ro0 = Prd*efe*pow(t0,n);

	aux = sqrt(((double)1-e*sin(ptlly))/((double)1+e*sin(ptlly)));
	t = ((1.-tan(ptlly/(double)2))/(1.+tan(ptlly/(double)2)))/pow(aux,e);
	ro = Prd*efe*pow(t,n);
	teta = n*(ptllx - GPlon0);
	ptllx = ro*sin(teta);
	ptlly = ro0 - ro*cos(teta);
	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}



/********************************************************************
		LAMBERT CONIC TO GEODETIC COORDINATES
 ********************************************************************/
TeCoord2D
TeLambertConformal :: PC2LL (TeCoord2D& ptpc)
{
	double	equad,			// Squared eccentricity 
		pi,			// PI value
		e,m1,m2,aux1,aux2,aux0,t1,t2,t0,n,efe,ro0,ro,teta,
		t,xx,aux3,aux4,aux5,ptpcx,ptpcy;

	int	sinal;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();
	

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	equad = 2.*Pflt - pow(Pflt,(double)2);
	e = sqrt(equad);
	pi = 4.*atan((double)1);

	m1 = cos(GPstlat1)/sqrt((double)1-equad*pow(sin(GPstlat1),(double)2));
	m2 = cos(GPstlat2)/sqrt((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux1 = sqrt(((double)1-e*sin(GPstlat1))/((double)1+e*sin(GPstlat1)));
	aux2 = sqrt(((double)1-e*sin(GPstlat2))/((double)1+e*sin(GPstlat2)));
	aux0 = sqrt(((double)1-e*sin(GPlat0))/((double)1+e*sin(GPlat0)));
	t1 = ((1.-tan(GPstlat1/(double)2))/(1.+tan(GPstlat1/(double)2)))/pow(aux1,e);
	t2 = ((1.-tan(GPstlat2/(double)2))/(1.+tan(GPstlat2/(double)2)))/pow(aux2,e);
	t0 = ((1.-tan(GPlat0/(double)2))/(1.+tan(GPlat0/(double)2)))/pow(aux0,e);

	if (GPstlat1 == GPstlat2)
		n = sin(GPstlat1);
	else
		n = (log(m1)-log(m2))/(log(t1)-log(t2));

	efe = m1/(n*pow(t1,n));
	ro0 = Prd*efe*pow(t0,n);

	sinal = (int)(n/fabs(n));
	ro = sqrt(ptpcx*ptpcx + (ro0-ptpcy)*(ro0-ptpcy));
	ro *= sinal;
	teta = atan(ptpcx/(ro0-ptpcy));
	t = pow((ro/(Prd*efe)),(double)1/n);
	xx = pi/2. - 2.*atan(t);
	aux3 = equad/2. + 5.*equad*equad/24. + equad*equad*equad/12.;
	aux4 = 7.*equad*equad/48. + 29.*equad*equad*equad/240.;
	aux5 = (7.*equad*equad*equad/120.)*sin(12.*atan(t));

	ptpcy = xx + aux3*sin(4.*atan(t)) - aux4*sin(8.*atan(t)) + aux5;
	ptpcx = teta/n + GPlon0;
	
	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return (TeCoord2D(ptpcx,ptpcy));
}

/********************************************************************
		GEODETIC TO POLYCONIC COORDINATES
********************************************************************/
TeCoord2D
TePolyconic :: LL2PC (TeCoord2D& ptll)
{
	double	equad,			// Squared eccentricity 
		n,			// Great normal of ellipsoid
		aux01,			// Ancillary variables
		aux02,aux03,aux04,m0,aux1,aux2,aux3,
		aux4,m,e,ptllx,ptlly;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptllx = (double)ptll.x();
	ptlly = (double)ptll.y();

	equad = 2.*Pflt - pow(Pflt,(double)2);

	aux01 = (1.-equad/4.-3.*equad*equad/64.-5.*equad*equad*equad/256.)*GPlat0;
	aux02 = (3.*equad/8+3.*equad*equad/32.+45.*equad*equad*equad/1024.)*sin((double)2*GPlat0);
	aux03 = (15.*equad*equad/256.+45.*equad*equad*equad/1024.)*sin((double)4*GPlat0);
	aux04 = (35.*equad*equad*equad/3072.)*sin((double)6*GPlat0);
	m0 = Prd*(aux01 - aux02 + aux03 - aux04);

	if (ptlly == 0.)
	{
		ptllx = Prd*(ptllx - GPlon0);
		ptlly = -m0;
	}
	else
	{
		aux1 = (1.-equad/4.-3.*equad*equad/64.-5.*equad*equad*equad/256.)*ptlly;
		aux2 = (3.*equad/8+3.*equad*equad/32.+45.*equad*equad*equad/1024.)*sin((double)2*ptlly);
		aux3 = (15.*equad*equad/256.+45.*equad*equad*equad/1024.)*sin((double)4*ptlly);
		aux4 = (35.*equad*equad*equad/3072.)*sin((double)6*ptlly);
		m = Prd*(aux1 - aux2 + aux3 - aux4);
		n = Prd/sqrt((double)1-equad*pow(sin(ptlly),(double)2));
		e = (ptllx - GPlon0)*sin(ptlly);
		ptllx = n*sin(e)/tan(ptlly);
		ptlly = m - m0 + (n*(1. - cos(e))/tan(ptlly));
	}
	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}

/*******************************************************************
		POLYCONIC TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TePolyconic :: PC2LL (TeCoord2D& ptpc)
{	
	double	A,	
		B,
		C,
		equad,			// Squared eccentricity
		aux01,			// Ancillary variables
		aux02,aux03,aux04,m0,mn,mnl,ma,cp,lat1,lat2,aux05,
		aux06,aux07,aux08,aux09,aux10,aux11,aux12,aux21,
		aux22,aux23,aux24,ptpcx,ptpcy;


	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	equad = 2.*Pflt - pow(Pflt,(double)2);

	// Initialize latitude with latitude of origin
	aux01 = (1.-equad/4.-3.*equad*equad/64.-5.*equad*equad*equad/256.)*GPlat0;
	aux02 = (3.*equad/8+3.*equad*equad/32.+45.*equad*equad*equad/1024.)*sin((double)2*GPlat0);
	aux03 = (15.*equad*equad/256.+45.*equad*equad*equad/1024.)*sin((double)4*GPlat0);
	aux04 = (35.*equad*equad*equad/3072.)*sin((double)6*GPlat0);
	m0 = Prd*(aux01 - aux02 + aux03 - aux04);

	if (ptpcy == (-m0))
	{
		ptpcy= 0.;
		ptpcx = ptpcx/Prd + GPlon0;
	}
	else
	{
		A = (m0 + ptpcy)/Prd;
		B = ((ptpcx*ptpcx)/(Prd*Prd)) +(A*A);

		lat2 = A;

		do
		{
			C = (sqrt(1.- equad*sin(lat2)*sin(lat2)))*tan(lat2);

			// mn calculation	
			aux21 = (1.-equad/4.-3.*equad*equad/64.-5.*equad*equad*equad/256.)*lat2;
			aux22 = (3.*equad/8.+3.*equad*equad/32.+45.*equad*equad*equad/1024.)*sin((double)2*lat2);
			aux23 = (15.*equad*equad/256.+45.*equad*equad*equad/1024.)*sin((double)4*lat2);
			aux24 = (35.*equad*equad*equad/3072.)*sin((double)6*lat2);
			mn = Prd*(aux21 - aux22 + aux23 - aux24);
		
			// mnl calculation
			aux05 = 1.- equad/4.-3.*equad*equad/64.-5.*equad*equad*equad/256.;
			aux06 = 2.*(3.*equad/8.+3.*equad*equad/32.+45.*equad*equad*equad/1024.)*cos((double)2*lat2);
			aux07 = 4.*(15.*equad*equad/256.+45.*equad*equad*equad/1024.)*cos((double)4*lat2);
			aux08 = 6.*(35.*equad*equad*equad/3072.)*cos((double)6*lat2);
			mnl = aux05 - aux06 + aux07- aux08;

			// ma calculation
			ma = mn/Prd;

			aux09 = (A*(C*ma+1)-ma)-(0.5*(ma*ma+B)*C);
			aux10 = equad*sin((double)2*lat2)*(ma*ma+B-2.*A*ma);
			aux11 = 4.*C+(A-ma);
			aux12 = C*mnl-(2./sin((double)2*lat2));

			// New latitude calculation 
			lat1 = lat2 - (aux09/((aux10/(aux11*aux12)) - mnl));
			cp = fabs(lat1-lat2) ;
			lat2 = lat1;


		}while(cp > 0.0000000001);

		ptpcy = lat1;
		ptpcx = ((asin((ptpcx*C)/Prd))/(sin(lat1))) + GPlon0;
	}

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return (TeCoord2D(ptpcx,ptpcy));	

}

/********************************************************************
	GEODETIC TO LAT LONG
********************************************************************/
TeCoord2D
TeLatLong :: LL2PC (TeCoord2D& ptll)
{
	return TeCoord2D( ptll.x()*TeCRD, ptll.y()*TeCRD ); 
}


/********************************************************************
	LAT LONG TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeLatLong :: PC2LL (TeCoord2D& ptpc)
{
	double ptpcx, ptpcy;

	ptpcx = ptpc.x()*TeCDR;
	ptpcy = ptpc.y()*TeCDR;
	
	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return TeCoord2D(ptpcx,ptpcy); 
}



/********************************************************************
	GEODETIC TO ALBERS EQUAL-AREA COORDINATES
********************************************************************/
TeCoord2D
TeAlbers :: LL2PC (TeCoord2D& ptll)
{

	double	equad,			// Squared eccentricity
		e,			// Ancillary variables
		m1,m2,aux1,aux10,aux11,aux12,aux2,aux20,aux21,
		aux22,q0,q1,q2,q,n,c,ro0,teta,ro,ptllx,ptlly;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptllx = (double)ptll.x();
	ptlly = (double)ptll.y();

	equad = 2.*Pflt - pow(Pflt,(double)2);
	e = sqrt(equad);

	m1 = cos(GPstlat1)/sqrt((double)1-equad*pow(sin(GPstlat1),(double)2));
	m2 = cos(GPstlat2)/sqrt((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux1 = sin(ptlly)/((double)1-equad*pow(sin(ptlly),(double)2));
	aux10 = sin(GPlat0)/((double)1-equad*pow(sin(GPlat0),(double)2));
	aux11 = sin(GPstlat1)/((double)1-equad*pow(sin(GPstlat1),(double)2));
	aux12 = sin(GPstlat2)/((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux2 = log((1. - e*sin(ptlly))/(1. + e*sin(ptlly)));
	aux20 = log((1. - e*sin(GPlat0))/(1. + e*sin(GPlat0)));
	aux21 = log((1. - e*sin(GPstlat1))/(1. + e*sin(GPstlat1)));
	aux22 = log((1. - e*sin(GPstlat2))/(1. + e*sin(GPstlat2)));
	q0 = (1. - equad)*(aux10 - (1./(2.*e))*aux20);
	q1 = (1. - equad)*(aux11 - (1./(2.*e))*aux21);
	q2 = (1. - equad)*(aux12 - (1./(2.*e))*aux22);
	q = (1. - equad)*(aux1 - (1./(2.*e))*aux2);
	n = (m1*m1 - m2*m2)/(q2 - q1);
	c = m1*m1 + n*q1;
	ro0 = Prd*sqrt(c - n*q0)/n;
	teta = n*(ptllx - GPlon0);
	ro = Prd*sqrt(c - n*q)/n;

	ptllx = ro*sin(teta);
	ptlly = ro0 - ro*cos(teta);

	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}


/********************************************************************
		ALBERS EQUAL-AREA TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeAlbers :: PC2LL(TeCoord2D& ptpc)
{

	double	equad,			// Squared eccentricity
		e,			// Eccentricity
		m1,			// Ancillary variable
		m2,aux10,aux11,aux12,aux20,aux21,aux22,q0,q1,q2,
		n,c,ro0,ro,q,aux,beta,aux1,aux2,aux3,teta,ptpcx,ptpcy;

	int	sinal;			// Ancillary variable	

	
	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	sinal = (int)(GPstlat2/fabs(GPstlat2));
	equad = 2.*Pflt - pow(Pflt,(double)2);
	e = sqrt(equad);

	m1 = cos(GPstlat1)/sqrt((double)1-equad*pow(sin(GPstlat1),(double)2));
	m2 = cos(GPstlat2)/sqrt((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux10 = sin(GPlat0)/((double)1-equad*pow(sin(GPlat0),(double)2));
	aux11 = sin(GPstlat1)/((double)1-equad*pow(sin(GPstlat1),(double)2));
	aux12 = sin(GPstlat2)/((double)1-equad*pow(sin(GPstlat2),(double)2));
	aux20 = log((1. - e*sin(GPlat0))/(1. + e*sin(GPlat0)));
	aux21 = log((1. - e*sin(GPstlat1))/(1. + e*sin(GPstlat1)));
	aux22 = log((1. - e*sin(GPstlat2))/(1. + e*sin(GPstlat2)));
	q0 = (1. - equad)*(aux10 - (1./(2.*e))*aux20);
	q1 = (1. - equad)*(aux11 - (1./(2.*e))*aux21);
	q2 = (1. - equad)*(aux12 - (1./(2.*e))*aux22);
	n = (m1*m1 - m2*m2)/(q2 - q1);
	c = m1*m1 + n*q1;
	ro0 = Prd*sqrt(c - n*q0)/n;
	ro = sqrt(ptpcx*ptpcx + (ro0 - ptpcy)*(ro0 - ptpcy));
	q = (c - (ro*ro*n*n/(Prd*Prd)))/n;
	aux = ((1. - equad)/(2.*e))*log((1. - e)/(1. + e));
	beta = asin(q/(1. - aux));
	aux1 = (equad/3. + 31.*equad*equad/180.+517.*equad*equad*equad/5040.)*sin(2.*beta);
	aux2 = (23.*equad*equad/360.+251.*equad*equad*equad/3780.)*sin(4.*beta);
	aux3 = (761.*equad*equad*equad/45360.)*sin(6.*beta);
	teta = fabs(atan(ptpcx/(ro0 - ptpcy)));

	if (sinal == 1)
	{
		if (ptpcx < 0.)
			teta = -teta;
	}

	if (sinal == -1)
	{
		if (ptpcx > 0.)
			teta *= sinal;
	}

	ptpcy = beta + aux1 + aux2 + aux3;
	ptpcx = GPlon0 + (teta/n);

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return (TeCoord2D(ptpcx,ptpcy));
}

/*******************************************************************
		GEODETIC TO MILLER COORDINATES
********************************************************************/
TeCoord2D
TeMiller :: LL2PC (TeCoord2D& ptll)
{

	double	pi,		// PI value
		ptllx,ptlly;	// Ancillary variables 

	int	sinal;		// Ancillary variables

	double Prd  = GPdatum.radius();
	
	ptllx = (double)ptll.x();
	ptlly = (double)ptll.y();

	pi = 4.*atan((double)1);
	sinal = (int)(fabs(ptlly)/ptlly);
	
	ptllx = Prd*(ptllx - GPlon0);

	if (sinal == 1)
		ptlly = (Prd/0.8)*(log(tan(pi/4. + 0.4*ptlly)));
	else
	{
		ptlly *= sinal;
		ptlly = (Prd/0.8)*(log(tan(pi/4. + 0.4*ptlly)));
		ptlly *= sinal;
	}

	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}


/********************************************************************
		MILLER TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeMiller :: PC2LL (TeCoord2D& ptpc)
{

	double	e,		// Neperian logarithm base 
		pi,		// PI value
		ptpcx,ptpcy;	// Ancillary variables

	int	sinal;		// Ancillary variable

	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	e = 2.718281828;
	pi = 4.*atan((double)1);
	sinal = (int)(fabs(ptpcy)/ptpcy);

	ptpcx = GPlon0 + (ptpcx/Prd);

	if (sinal == 1)
		ptpcy = 2.5*atan(pow(e,(0.8*ptpcy/Prd)))-5.*pi/8.;
	else
	{
		ptpcy *= sinal;
		ptpcy = 2.5*atan(pow(e,(0.8*ptpcy/Prd))) - 5.*pi/8.;
		ptpcy *= sinal;
	}

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return(TeCoord2D(ptpcx,ptpcy));
}


/*******************************************************************
		GEODETIC TO SINUSOIDAL COORDINATES
********************************************************************/
TeCoord2D
TeSinusoidal :: LL2PC (TeCoord2D& ptll)
{

	double	ptllx,ptlly;	// Ancillary variables 

	double Prd  = GPdatum.radius();
	
	ptllx = (double)ptll.x();
	ptlly = (double)ptll.y();
	
	ptllx = Prd*(ptllx - GPlon0);
	ptllx *= cos(ptlly);

	ptlly = Prd*ptlly;
	
	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));	
}


/********************************************************************
		SINUSOIDAL TO GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeSinusoidal :: PC2LL (TeCoord2D& ptpc)
{

	double	ptpcx,ptpcy;	// Ancillary variables

	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	ptpcy = ptpcy/Prd;

	ptpcx = GPlon0 + (ptpcx/(Prd*cos(ptpcy)));
	
	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return(TeCoord2D(ptpcx,ptpcy));
}

/*******************************************************************
		GEODETIC TO CYLINDRICAL EQUIDISTANT COORDINATES
********************************************************************/
TeCoord2D
TeCylindricalEquidistant :: LL2PC (TeCoord2D& ptll)
{
	double	ptllx,ptlly;

	double Prd  = GPdatum.radius();

	ptllx = ptll.x();
	ptlly = ptll.y();

	ptllx = Prd*(ptllx - GPlon0)*cos(GPstlat1);
	ptlly = Prd*ptlly;

	return(TeCoord2D(ptllx+GPoffx,ptlly+GPoffy));
}


/*******************************************************************
	CYLINDRICAL EQUIDISTANT TO	GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TeCylindricalEquidistant :: PC2LL (TeCoord2D& ptpc)
{
	double	ptpcx,ptpcy;

	double Prd  = GPdatum.radius();

	ptpcx = ptpc.x()-GPoffx;
	ptpcy = ptpc.y()-GPoffy;

	ptpcy = ptpcy/Prd;
	ptpcx = GPlon0 + ptpcx/(Prd*cos(GPstlat1));

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return (TeCoord2D(ptpcx,ptpcy));
}

/*******************************************************************
		GEODETIC TO POLAR STEREOGRAPHIC COORDINATES
********************************************************************/
TeCoord2D
TePolarStereographic :: LL2PC (TeCoord2D& ptll)
{
	double	k0,				// Scale factor
		e,					// Eccentricity
		pi,					// Auxillary variables
		lon0,				// auxilliary origin longitude
		t,ro,aux1,aux2,aux3,ptllx,ptlly;
	
	int GPhemis;
	if (GPhemisphere == TeNORTH_HEM)
		GPhemis = 1;
	else
		GPhemis = -1;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();
	
	ptllx = ptll.x();
	ptlly = ptll.y();

	k0 = 0.933;	// Standard parallel 60 degrees 
	e = sqrt((double)2*Pflt - pow(Pflt,(double)2));
	pi = 4.*atan((double)1);

	ptlly *= GPhemis;
	ptllx *= GPhemis;
	lon0 = (GPhemis == 1) ? GPlon0 : -GPlon0;

	aux1 = (1. - e*sin(ptlly))/(1. + e*sin(ptlly));
	t = tan((pi/4.) - (ptlly/2.))/pow(aux1,(e/(double)2));
	aux2 = pow(((double)1 + e),((double)1 + e));
	aux3 = pow(((double)1 - e),((double)1 - e));
	ro = 2.*Prd*k0*t/sqrt(aux2*aux3);

	aux1   = ro*sin(ptllx - lon0);
	ptlly = -ro*cos(ptllx - lon0);
	aux1 *= GPhemis;
	ptlly *= GPhemis;
	
	if (GPhemis == -1)
	{
		aux1 *= GPhemis;
		ptlly *= GPhemis;
	}
//	return(Point((float)aux1,(float)ptlly));
	return (TeCoord2D(aux1+GPoffx,ptlly+GPoffy));

}


/*******************************************************************
	POLAR STEREOGRAPHIC TO	GEODETIC COORDINATES
********************************************************************/
TeCoord2D
TePolarStereographic :: PC2LL (TeCoord2D& ptpc)
{
	double	k0,			// Scale factor

		equad,			// Squared eccentricity
		e,		        // Eccentricity
		pi,			// Ancillary variables
		lon0,			// auxilliary origin longitude
		ro,t,xx,aux1,aux2,aux3,aux4,aux5,ptpcx,ptpcy,px,py;

	px = (double)ptpc.x()-GPoffx;
	py = (double)ptpc.y()-GPoffy;

	int GPhemis;
	if (GPhemisphere == TeNORTH_HEM)
		GPhemis = 1;
	else
		GPhemis = -1;

	double Pflt = GPdatum.flattening();
	double Prd  = GPdatum.radius();

//	k0 = 0.994;	// Standard parallel 80.1 degrees 
	k0 = 0.933;	// Standard parallel 60 degrees
	pi = 4.*atan((double)1);
	equad = 2.*Pflt - pow(Pflt,(double)2);
	e = sqrt(equad);

	if (GPhemis == -1)
	{
		px *= GPhemis;
		py *= GPhemis;
	}

	lon0 = (GPhemis == 1) ? GPlon0 : -GPlon0;

	ro = sqrt(px*px + py*py);
	aux1 = pow(((double)1 + e),((double)1 + e));
	aux2 = pow(((double)1 - e),((double)1 - e));
	t = (ro*sqrt(aux1*aux2))/(2.*Prd*k0);
	xx = pi/2. - 2.*atan(t);
	aux3 = equad/2. + 5.*equad*equad/24. + equad*equad*equad/12.;
	aux4 = 7.*equad*equad/48. + 29.*equad*equad*equad/240.;
	aux5 = 7.*equad*equad*equad/120.;

	ptpcy = xx + aux3*sin((double)2*xx) + aux4*sin((double)4*xx) + aux5*sin((double)6*xx);

	if (py != 0.)
		ptpcx = lon0 + atan(px/(-py));


	if (GPhemis == 1)
	{
		if (px > 0. && py > 0.)
			ptpcx = ptpcx + pi;
		else if (px < 0. && py > 0.)
			ptpcx = ptpcx - pi;
		else if (px > 0. && py == 0.)
			ptpcx = lon0 + pi / 2.;
		else if (px < 0. && py == 0.)
			ptpcx = lon0 - pi / 2.;
		else if (px == 0. && py == 0.)
			ptpcx = lon0;
	}
	else
	{
		ptpcy *= GPhemis;
		ptpcx *= GPhemis;

		if (px > 0. && py < 0.)
			ptpcx = ptpcx + pi;
		else if (px < 0. && py < 0.)
			ptpcx = ptpcx - pi;
		else if (px > 0. && py == 0.)
			ptpcx = lon0 + pi / 2.;
		else if (px < 0. && py == 0.)
			ptpcx = lon0 - pi / 2.;
		else if (px == 0. && py == 0.)
			ptpcx = lon0;
	}

	if (ptpcx < (-pi)) ptpcx += 2.*pi;
	else if (ptpcx > pi)    ptpcx -= 2.*pi;

	if( GPdestination && !(datum() == GPdestination->datum()))
		ChangeLL(ptpcx,ptpcy);

	return (TeCoord2D(ptpcx,ptpcy));
}

TeProjection* TeGetTeProjectionFromSProj(const string& proj4)
{

	// a map from sproj4 projections to TerraLib projections
	map<string,string> sprojToTeProj;
	sprojToTeProj["aea"] = "Albers";
	sprojToTeProj["latlong"] = "LatLong";
	sprojToTeProj["lcc"] = "LambertConformal";
	sprojToTeProj["merc"] = "Mercator";
	sprojToTeProj["mill"] = "Miller";
	sprojToTeProj["utm"] = "UTM";
	sprojToTeProj["sinu"] = "Sinusoidal";
	sprojToTeProj["poly"] = "Polyconic";
	sprojToTeProj["eqc"] = "CylindricalEquidistant";
	sprojToTeProj["ups"] = "PolarStereographic";

    unsigned int pos;
    string attribute;
    string value;

  	TeProjectionParams par;
	par.hemisphere = TeNORTH_HEM;
	par.offx = 500000;
	par.offy = 0;
	par.units = "";
	TeDatum dat("UserDefined");

    for(unsigned int i = 0; i < proj4.size(); i++)
    {
        for( pos = i; i != proj4.size() && proj4[i] != '=';i++);

        attribute = proj4.substr(pos+1, i-pos-1);

        for(pos = i; i != proj4.size() && proj4[i] != ' ';i++);

        if(proj4[i] == ' ') value = proj4.substr(pos+1, i-pos-1);
        else                value = proj4.substr(pos+1, i-pos);

        if(attribute == "proj")
		{
			map<string,string>::iterator it = sprojToTeProj.find(value);
			if (it != sprojToTeProj.end())
				par.name = sprojToTeProj[value];
			else
				par.name = "NoProjection"; 
		}
	// Currently this routine understands the following parameters from
	// a sproj description: lon_0, lat_0, lat_1, lat_2, k, zone, x_0, y_0

        else if(attribute == "lon_0") 
			par.lon0 = atof(value.c_str())*TeCDR;
        else if(attribute == "lat_0") 
			par.lat0 = atof(value.c_str())*TeCDR;
        else if(attribute == "lat_1") 
			par.stlat1 = atof(value.c_str())*TeCDR;
        else if(attribute == "lat_2") 
			par.stlat2 = atof(value.c_str())*TeCDR;
        else if(attribute == "k") 
			par.scale = atof(value.c_str());
        else if(attribute == "zone") 
			par.lon0 = (-183 + 6*(atoi(value.c_str()))*TeCDR);
		else if (attribute == "south") 
		{
			par.hemisphere = TeSOUTH_HEM;
			par.offy = 10000000;
		}
		else if(attribute == "x_0") 
			par.offx = atof(value.c_str());
 		else if(attribute == "y_0") 
			par.offy = atof(value.c_str());
		else if (attribute == "units")
			par.units = value;
		// Datum parameters
        else if(attribute == "R")	// radius for a spherical ellipsoid
			dat = TeDatum("Spherical",atof(value.c_str()));
	    else if(attribute == "a")   // Earth equatorial radius
			dat.radius(atof(value.c_str()));
	    else if(attribute == "f")	// Earth flattening
			dat.flattening(atof(value.c_str()));
	    else if(attribute == "rf")	// Earth  reverse flattening (f = 1/rf)
			dat.flattening(1/atof(value.c_str()));
		else if (attribute == "es") // eccentricity squared (f=1-(1-es)^0.5) 
			dat.flattening(1-sqrt(1-atof(value.c_str())));
		else if (attribute == "e")	// eccentricity (f=1-(1-es^2)^0.5)
			dat.flattening(1-sqrt(1-pow(atof(value.c_str()),2)));

// TODO: sproj4 also allowd the selection of standard, predefined
// ellipsoid figures using the option +ellps=acronym. But we are not decoding 
// them yet...
    }
	par.datum = dat;
	return TeProjectionFactory::make(par);
}

string TeGetSProjFromTeProjection(TeProjection* teproj)
{
	map<string,string> teProjToSProj;
	teProjToSProj["Albers"] = "aea";
	teProjToSProj["LatLong"] = "latlong";
	teProjToSProj["LambertConformal"] = "lcc";
	teProjToSProj["Mercator"] = "merc";
	teProjToSProj["Miller"] = "mill";
	teProjToSProj["UTM"] = "utm";
	teProjToSProj["Sinusoidal"] = "sinu";
	teProjToSProj["Polyconic"] = "poly";
	teProjToSProj["CylindricalEquidistant"] = "eqc";
	teProjToSProj["PolarStereographic"] = "ups";

    string sresp = "+proj=";
	map<string,string>::iterator it = teProjToSProj.find(teproj->name());
	if (it == teProjToSProj.end())
	{
		sresp += "noprojection";
		return sresp;
	}
	else
		sresp += it->second;

	TeProjInfo pjInfo = TeProjectionInfo(teproj->name());
	if ( pjInfo.hasLon0 )
		sresp += " +lon_0=" + Te2String(teproj->lon0()*TeCRD,6);

	if ( pjInfo.hasLat0 )
		sresp += " +lat_0=" + Te2String(teproj->lat0()*TeCRD,6);

	if ( pjInfo.hasStlat1 )
		sresp += " +lat_1=" + Te2String(teproj->stLat1()*TeCRD,6);
	
	if ( pjInfo.hasStlat2 )
		sresp += " +lat_2=" +Te2String(teproj->stLat2()*TeCRD,6);

	if (teproj->name() == "UTM" && teproj->hemisphere() == TeSOUTH_HEM)
		sresp += " +south";
	else
	{
		if ( pjInfo.hasOffx )
			sresp += " +x_0=" + Te2String(teproj->offX(),6);
		if ( pjInfo.hasOffy )
			sresp += " +y_0=" + Te2String(teproj->offY(),6);
	}
	
	if ( pjInfo.hasScale )
		sresp += " +k="  + Te2String(teproj->scale(),6);

    if (teproj->datum().name() == "Spherical")
		sresp += " +R="  + Te2String(teproj->datum().radius(),6);
	else
	{
		sresp += " +a="  + Te2String(teproj->datum().radius(),6);
		sresp += " +f="  + Te2String(teproj->datum().flattening(),6);
	}
	return sresp;
}
