/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeSTElementSet.h"
#include "TeQuerier.h" 
#include "TeQuerierParams.h"


TeSTElementSet::TeSTElementSet (const TeSTElementSet& other)
{
	totalTime_ = other.totalTime_;
	elements_ = other.elements_;
	if(other.querier_)
		querier_ = new TeQuerier(other.querier_->params());
	else
		querier_ = 0;
}

TeSTElementSet::TeSTElementSet(TeTheme* theme)
{	
	TeQuerierParams par; 
	par.setParams(theme); 
	querier_ = new TeQuerier(par); 
}

TeSTElementSet::TeSTElementSet(TeLayer* layer)
{	
	TeQuerierParams par; 
	par.setParams (layer);
	querier_ = new TeQuerier(par);
}

TeSTElementSet::TeSTElementSet(TeQuerier* querier)
{
	querier_ = querier;
}


TeSTElementSet&
TeSTElementSet::operator= (const TeSTElementSet& other)
{
	if ( this != &other )
	{
		totalTime_ = other.totalTime_;
		elements_ = other.elements_;
		if(other.querier_)
			querier_ = new TeQuerier(other.querier_->params());
		else
			querier_ = 0;
	}
	return *this;
}

TeSTElementSet::~TeSTElementSet()
{
	if(querier_)
		delete (querier_);
}
	
TeTheme* 
TeSTElementSet::theme()
{ 
	if(querier_)
		return (querier_->theme());
	
	return 0;
}

bool 
TeSTElementSet::insertSTInstance (TeSTInstance& object) 
{
	if (elements_.find (object.objectId()) == elements_.end())  
	{
		//ainda no existe
		TeSTElement stElem(object.objectId());
		stElem.insertSTInstance (object);
		elements_[object.objectId()] = stElem;
	}
	else
		elements_[object.objectId()].insertSTInstance(object);
		
	return true;
}


bool 
TeSTElementSet::setGeometry(const string& object_id, const TePolygonSet& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::setGeometry(const string& object_id, const TeLineSet& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::setGeometry(const string& object_id, const TePointSet& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::setGeometry(const string& object_id, const TeCellSet& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::setGeometry(const string& object_id, vector<TeGeometry*>& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::setGeometry(const string& object_id, const TeMultiGeometry& geomSet, TeTimeInterval time)
{
	STElementSetIterator itSet = elements_.find (object_id);
	if (itSet == elements_.end()) 
		return false;

	STElemIterator itGeom = (*itSet).second.begin();
	while (itGeom != (*itSet).second.end())
	{
		if ((!time.isValid()) || ((*itGeom).timeInterval() == time))
			(*itGeom).setGeometry (geomSet);
		++itGeom;
	}
	return true;
}

bool 
TeSTElementSet::getGeometry(const string& object_id, TePolygonSet& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}


bool 
TeSTElementSet::getGeometry(const string& object_id, TeLineSet& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}

	
bool 
TeSTElementSet::getGeometry(const string& object_id, TePointSet& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}

	
bool 
TeSTElementSet::getGeometry(const string& object_id, TeCellSet& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}


bool 
TeSTElementSet::getGeometry(const string& object_id, vector<TeGeometry*>& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}
	

bool 
TeSTElementSet::getGeometry(const string& object_id, TeMultiGeometry& geomSet, TeTimeInterval time)
{
	// Verify if object is in the map
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;
	
	STElemIterator geomIt = (*geomSetIt).second.begin();
	while (geomIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*geomIt).timeInterval() == time))
			return ((*geomIt).getGeometry (geomSet));

		++geomIt;
	}
	return false;  
}

bool 
TeSTElementSet::getAttributeValue (const string&  object_id, const string& attr_name, string& val, TeTimeInterval time)
{
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;

	STElemIterator stoIt = (*geomSetIt).second.begin();
	while (stoIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*stoIt).timeInterval() == time))
			return ((*stoIt).getPropertyValue (attr_name, val));
		++stoIt;
	}
	return false;
}

bool 
TeSTElementSet::getAttributeValue (const string& object_id, int attr_index, string& val, TeTimeInterval time)
{
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;

	STElemIterator stoIt = (*geomSetIt).second.begin();
	while (stoIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*stoIt).timeInterval() == time))
			return ((*stoIt).getPropertyValue (val, attr_index));
		++stoIt;
	}
	return false;
}

bool 
TeSTElementSet::getAttributeValue (const string& object_id, int attr_index, double& val, TeTimeInterval time)
{
	string vals;
	if(!getAttributeValue (object_id, attr_index, vals, time))
		return false;

	val = atof(vals.c_str());
	return true;
}

int 
TeSTElementSet::getAttributeIndex (const string& attrName)
{
	STElementSetIterator geomSetIt = elements_.begin();
	if (geomSetIt == elements_.end()) 
		return -1;

	STElemIterator stoIt = (*geomSetIt).second.begin();
	if (stoIt == (*geomSetIt).second.end()) 
		return -1;

	TePropertyVector vec = (*stoIt).getPropertyVector();
	for (unsigned int i=0; i<vec.size(); ++i)
	{
		if(TeConvertToUpperCase(attrName)==TeConvertToUpperCase(vec[i].attr_.rep_.name_))
			return i;
	}
	return -1;
}

bool 
TeSTElementSet::getPropertyVector (const string& object_id, TePropertyVector& propVec, TeTimeInterval time)
{
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;

	STElemIterator stoIt = (*geomSetIt).second.begin();
	while (stoIt != (*geomSetIt).second.end())
	{
		if ((!time.isValid()) || ((*stoIt).timeInterval() == time))
		{
			propVec = (*stoIt).getPropertyVector();
			return true;
		}
		++stoIt;
	}
	return false;

}

bool 
TeSTElementSet::addProperty (const string& object_id, TeProperty& prop)
{
	STElementSetIterator geomSetIt = elements_.find (object_id);
	if (geomSetIt == elements_.end()) 
		return false;

	STElemIterator stoIt = (*geomSetIt).second.begin();
	if (!(*stoIt).setPropertyValue(prop.attr_.rep_.name_, prop.value_))
		(*stoIt).addProperty(prop);

	return true;
}


bool 
TeSTElementSet::existMultiTemporalElem(const string& objId)
{
	STElementSetIterator geomSetIt = elements_.find (objId);
	if (geomSetIt != elements_.end()) 
		return ((elements_[objId].numSTInstance())>1);
	
	return false;
}


int 
TeSTElementSet::numSTInstance(const string& objectId)
{
	if (!objectId.empty())
	{
		STElementSetIterator geomSetIt = elements_.find (objectId);
		if (geomSetIt != elements_.end()) 
			return (elements_[objectId].numSTInstance());
		return 0;
	}
	else
	{
		int ntot = 0;
		STElementSetIterator geomSetIt = elements_.begin();
		while (geomSetIt != elements_.end())
		{
			ntot += geomSetIt->second.numSTInstance();
			++geomSetIt;
		}
		return ntot;
	}
}


bool 
TeSTElementSet::build(bool loadGeometries, bool loadAllAttributes, vector<string> attrNames, int slide)
{
	if(!querier_)
		return false;

	TeQuerierParams param = querier_->params();
	param.setFillParams(loadGeometries, loadAllAttributes, attrNames);

	querier_->refresh(param);

	return(buildImpl(slide));
}

bool 
TeSTElementSet::build(TeGroupingAttr& groupAttr, bool loadGeometries, int slide)
{
	if(!querier_)
		return false;

	TeQuerierParams param = querier_->params();
	param.setFillParams(loadGeometries, groupAttr);

	querier_->refresh(param);

	return(buildImpl(slide));
}

bool 
TeSTElementSet::buildImpl(int slide)
{
	int dt = CLOCKS_PER_SEC/4, steps = 0;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;

	if(!querier_)
		return false;

	if(!querier_->loadInstances(slide))
		return false;

	int tot = querier_->numElemInstances();
	if(TeProgress::instance())
	{
		string caption = "Building data";
		TeProgress::instance()->setCaption(caption.c_str());
		string msg = "Building in progress. Please, wait!";
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(tot);
		t2 = clock();
		t0 = t1 = t2;
	}
	
	TeSTInstance obj;
	while(querier_->fetchInstance(obj)) 
	{
		this->insertSTInstance(obj);
		obj.clear();
		if(TeProgress::instance())
		{
			steps++;
			t2 = clock();
			if (int(t2-t1) > dt)
			{
				t1 = t2;
				if(TeProgress::instance()->wasCancelled())
					return false;
				
				if((int)(t2-t0) > dt2)
					TeProgress::instance()->setProgress(steps);
			}
		}
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();

	querier_->clear(); //clear instances 
	return true;
}


TeSTElementSet::iterator& 
TeSTElementSet::iterator::operator++()
{
	if (++curSTO_ == (*curObj_).second.end())
	{
		if (++curObj_ !=  stoSet_->elements().end())
		{
			curSTO_ = (*curObj_).second.begin();
		}
	}
	return *this;
}

double 
TeSTElementSet::propertyIterator::operator*() 
{
	string val;
	STElemIterator itCur = getCurSTO();
	if ((*itCur).getPropertyValue (attrName_, val))
		return (atof(val.c_str()));
	else
		return 0.0;
}

bool 
TeSTElementSet::propertyIterator::getProperty(TeProperty& prop) 
{
	STElemIterator itCur = getCurSTO();
	if ((*itCur).getProperty (prop, attrName_))
		return true;
	else
		return false;
}

TeSTElementSet::iterator 
TeSTElementSet::begin()     
{ 
	return iterator(this, elements_.begin(), (*(elements_.begin())).second.begin());
}

TeSTElementSet::propertyIterator 
TeSTElementSet::begin(const string& attrName)     
{ 
	return propertyIterator(this, elements_.begin(), (*(elements_.begin())).second.begin(),attrName);
}

TeSTElementSet::iterator 
TeSTElementSet::end()       
{
	if(elements_.size()<1)
		return iterator(this, elements_.end(), (*(elements_.end())).second.end());
	
	return iterator(this, elements_.end(), (*(--elements_.end())).second.end());
}

TeSTElementSet::propertyIterator 
TeSTElementSet::end(const string& attrName)       
{
	if(elements_.size()<1)
		return propertyIterator(this, elements_.end(), (*(elements_.end())).second.end(), attrName);
	
	return propertyIterator(this, elements_.end(), (*(--elements_.end())).second.end(), attrName);
}
