/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/


#include "TeSTInstance.h"

TeSTInstance::TeSTInstance (const string& object_id, TeProperty& prop ) 
{
	object_id_ = object_id;
	time_ = TeTimeInterval();
	properties_.push_back(prop);
	slice_ = 0;
	theme_= 0;
}
		

TeSTInstance::TeSTInstance(const TeSTInstance& other) 
{	
	object_id_		= other.object_id_;
	time_			= other.time_;
	properties_     = other.properties_;
	slice_			= other.slice_;
	theme_			= other.theme_;
	unique_id_		= other.unique_id_;
	geometries_		= other.geometries_;  
}
			
TeSTInstance&					
TeSTInstance::operator= (const TeSTInstance& other)
{	
	if ( this != &other )
	{
		object_id_		= other.object_id_;
		time_			= other.time_;
		properties_     = other.properties_;
		slice_			= other.slice_;
		theme_			= other.theme_;
		unique_id_		= other.unique_id_;
		geometries_		= other.geometries_; 
	}
	return *this;
}
	
	
bool 
TeSTInstance::operator== (const TeSTInstance& other) const
{	
	if (this->object_id_ != other.object_id_ || 
		!(this->time_ == other.time_) )
		return false;

	return true; 
}

	
bool 
TeSTInstance::getProperty (TeProperty& prop, unsigned int i)	
{ 
	if (i < properties_.size())
	{
		prop = properties_[i];
		return true;
	}
	else
		return false;
}

bool 
TeSTInstance::getProperty (TeProperty& prop, string name)	
{ 
	string newName; 
	size_t pos = name.find(".", string::npos,1);
	if (pos != string::npos)
		newName = name.substr(pos+1);
		
	TePropertyVector::iterator it = properties_.begin();
	while(it!=properties_.end())
	{
		if( ((*it).attr_.rep_.name_ == name) ||
			((*it).attr_.rep_.name_ == newName))
		{ 
			prop = (*it);
			return true;
		}
		++it;
	}
	return false;
}


bool 
TeSTInstance::getPropertyValue (string& val, int i)	
{ 
	if ((i >=0) && (i< (int)properties_.size()))
	{
		val = properties_[i].value_;
		return true;
	}
	else
		return false;
}

	
bool 
TeSTInstance::getPropertyValue (const string& name, string& val)	
{ 
	string newName; 
	size_t pos = name.find(".", string::npos,1);
	if (pos != string::npos)
		newName = name.substr(pos+1);

	TePropertyVector::iterator it = properties_.begin();
	while(it!=properties_.end())
	{
		if(	((*it).attr_.rep_.name_ == name) ||
			((*it).attr_.rep_.name_ == newName)) 
		{
			val = (*it).value_; 
			return true;
		}
		++it;
	}
	return false;
}

bool 
TeSTInstance::getGeometry(TePolygonSet& result)
{
	if(geometries_.getGeometry(result))
		return true;

	if((theme_)  && (theme_->layer()->loadGeometrySet(object_id_, result)))
		return true;
			
	return false;
}
	
bool 
TeSTInstance::getGeometry(TeLineSet& result)
{
	if(geometries_.getGeometry(result))
		return true;

	if((theme_)  && (theme_->layer()->loadGeometrySet(object_id_, result)))
		return true;
			
	return false;
}

bool 
TeSTInstance::getGeometry(TePointSet& result)
{
	if(geometries_.getGeometry(result))
		return true;

	if((theme_)  && (theme_->layer()->loadGeometrySet(object_id_, result)))
		return true;
			
	return false;
}

bool 
TeSTInstance::getGeometry(TeCellSet& result)
{
	if(geometries_.getGeometry(result))
		return true;

	if((theme_)  && (theme_->layer()->loadGeometrySet(object_id_, result)))
		return true;
			
	return false;
}

bool 
TeSTInstance::getGeometry(vector<TeGeometry*>& result)
{
	if(geometries_.getGeometry(result))
		return true;

	if(theme_)
	{
		if(theme_->layer()->hasGeometry (TePOLYGONS))
		{
			TePolygonSet pols; 
			if(!theme_->layer()->loadGeometrySet(object_id_, pols))
				return false;
			
			for(int i=0; i<(int)pols.size(); ++i)
			{
				TePolygon* pol = new TePolygon;
				result.push_back (pol);
			}
		}
		if(theme_->layer()->hasGeometry (TeLINES))
		{
			TeLineSet lins; 
			if(!theme_->layer()->loadGeometrySet(object_id_, lins))
				return false;
			
			for(int i=0; i<(int)lins.size(); ++i)
			{
				TeLine2D* line = new TeLine2D;
				result.push_back (line);
			}
		}
		if(theme_->layer()->hasGeometry (TePOINTS))
		{
			TePointSet	points; 
			if(!theme_->layer()->loadGeometrySet(object_id_, points))
				return false;

			for(int i=0; i<(int)points.size(); ++i)
			{
				TePoint* point = new TePoint;
				result.push_back (point);
			}
		}
		if(theme_->layer()->hasGeometry (TeCELLS))
		{
			TeCellSet cells; 
			if(!theme_->layer()->loadGeometrySet(object_id_, cells))
				return false;

			for(int i=0; i<(int)cells.size(); ++i)
			{
				TeCell* cell = new TeCell;
				result.push_back (cell);
			}
		}
	}
	else
		return false;
			
	return true;
}

bool 
TeSTInstance::getGeometry(TeMultiGeometry& result)
{
	if(!geometries_.empty())
	{
		result = geometries_;
		return true; 
	}

	geometries_.getGeometry (result.polygons_);
	geometries_.getGeometry (result.lines_);
	geometries_.getGeometry (result.points_);
	geometries_.getGeometry (result.cells_);
	
	return (!result.empty());
}

void 
TeSTInstance::addProperty(const double& val, TeAttributeRep rep, bool /* attrIn */)
{
	TeProperty prop;
	prop.attr_.rep_ = rep; 
	prop.value_ = Te2String(val); 
	properties_.push_back(prop);	
}


bool 
TeSTInstance::addGeometry(const TePolygon& poly)  
{ 
	if(poly.objectId() == objectId())
	{
		geometries_.addGeometry (poly);
		return true;
	}
	return false;
}
	
bool 
TeSTInstance::addGeometry(const TeLine2D& line)     
{ 
	if(line.objectId() == objectId())
	{
		geometries_.addGeometry (line);
		return true;
	}
	return false; 
}
	
bool 
TeSTInstance::addGeometry(const TePoint& point)	
{ 
	if(point.objectId() == objectId())
	{
		geometries_.addGeometry (point);
		return true;
	}
	return false;  
} 
	
bool
TeSTInstance::addGeometry(const TeCell& cell)		
{ 
	if(cell.objectId() == objectId())
	{
		geometries_.addGeometry (cell);
		return true;
	}
	return false;  
}


bool 
TeSTInstance::setPropertyValue (const string& name, const string& val)	
{ 
	string newName; 
	size_t pos = name.find(".", string::npos,1);
	if (pos != string::npos)
		newName = name.substr(pos+1);

	TePropertyVector::iterator it = properties_.begin();
	while(it!=properties_.end())
	{
		if( ((*it).attr_.rep_.name_ == name) ||
			((*it).attr_.rep_.name_ == newName))
		{
			(*it).value_ = val; 
			return true;
		}
		++it;
	}
	return false;
}


double 
TeSTInstance::operator[](int i)   
{
	TeAttrDataType	attrType = properties_[i].attr_.rep_.type_;

	if ((i >= 0) && (i < (int)properties_.size()) && 
		(attrType == TeREAL || attrType == TeINT))
	{
		if (properties_[i].value_.empty()) 
			return TeMAXFLOAT;
		return (atof(properties_[i].value_.c_str()));
	}
	else 
		return 0.;
}

bool 
TeSTInstance::centroid(TeCoord2D& centroid, TeGeomRep geomRep)
{
	bool status = false;
	if(geomRep==TePOLYGONS)
	{ 
		TePolygonSet pols;
		if(getGeometry(pols))
		{
			centroid = TeFindCentroid(pols);
			status = true;
		}
	}
	if(geomRep==TeLINES)
	{
		TeLineSet lins; 
		if(getGeometry(lins))
		{
			centroid = TeFindCentroid(lins);
			status = true;
		}
	}
	if(geomRep==TePOINTS)
	{
		TePointSet points; 
		if(getGeometry(points))
		{
			centroid = TeFindCentroid(points);
			status = true;
		}
	}
	if(geomRep==TeCELLS)
	{
		TeCellSet cells; 
		if(getGeometry(cells))
		{
			centroid = TeFindCentroid(cells);
			status = true;
		}
	}
	if(geomRep==TeGEOMETRYNONE)
	{
		if(hasPolygons())
			status = this->centroid(centroid, TePOLYGONS);
		else if(hasLines())
			status = this->centroid(centroid, TeLINES);
		else if (hasPoints())
			status = this->centroid(centroid, TePOINTS);
		else if (hasCells())
			status = this->centroid(centroid, TeCELLS);
	}

	return status;
}

bool 
TeSTInstance::area(double& a, TeGeomRep geomRep)
{
	bool status = false;
	if(geomRep==TePOLYGONS)
	{ 
		TePolygonSet pols;
		if(getGeometry(pols))
		{
			a = TeGeometryArea(pols);
			status = true;
		}
	}
	if(geomRep==TeLINES)
	{
		TeLineSet lins; 
		if(getGeometry(lins))
		{
			a = TeGeometryArea(lins);
			status = true;
		}
	}
	if(geomRep==TePOINTS)
	{
		TePointSet points; 
		if(getGeometry(points))
		{
			a = TeGeometryArea(points);
			status = true;
		}
	}
	if(geomRep==TeCELLS)
	{
		TeCellSet cells; 
		if(getGeometry(cells))
		{
			a = TeGeometryArea(cells);
			status = true;
		}
	}
	if(geomRep==TeGEOMETRYNONE)
	{
		if(hasPolygons())
			status = this->area(a, TePOLYGONS);
		else if(hasLines())
			status = this->area(a, TeLINES);
		else if (hasPoints())
			status = this->area(a, TePOINTS);
		else if (hasCells())
			status = this->area(a, TeCELLS);
	}

	return status;
}

void 
TeSTInstance::clear()
{
	slice_ = 0;
	theme_ = 0;
	object_id_.clear();
	unique_id_.clear();
	properties_.clear();
	geometries_.clear();
}



