/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeSTInstance.h
	This file contains structures and definitions to deal with an instance
	in time of a spatial element, including their geometry and set of properties  
*/

#ifndef  __TERRALIB_INTERNAL_STINSTANCE_H
#define  __TERRALIB_INTERNAL_STINSTANCE_H

#include "TeAttribute.h"
#include "TeTimeInterval.h"
#include "TeTheme.h"
#include "TeMultiGeometry.h"

#include <string>
#include <map> 
#include <vector>
using namespace std;

//! An instance in a time of a spatial element
class TeSTInstance
{
	
protected:
	string				object_id_;		// object identification
	vector<string>		unique_id_;		// instance identification for each attribute table
	int					slice_;			// grouping identification
	
	TeTimeInterval		time_;			// validity time
	TePropertyVector	properties_;	// set of properties
	TeMultiGeometry	    geometries_;	// set of geometries

	TeTheme*			theme_;			// theme

public:		

	//! Default constructor
	TeSTInstance() { }

	//! Constructor to an ST object without geometries or time stamp 
	TeSTInstance (const string& object_id, TeProperty& prop ); 
			
	//! Copy constructor
	TeSTInstance(const TeSTInstance& other); 
						
	//! Assignment operator
	TeSTInstance& operator= (const TeSTInstance& other);
		
	//! Operator ==
	bool operator== (const TeSTInstance& other) const;
	
	//! Destructor
	~TeSTInstance() {} 
	
	//! Sets the vector of properties of this ST object
	void properties(TePropertyVector& p) 
	{	properties_ = p;	}

	//! Returns the property vector of this instance
	TePropertyVector& getPropertyVector()
	{	return properties_;	}

	//! Returns the properties of this instance that was generated for a function
	TePropertyVector& getOutPropertyVector()
	{	return properties_;	}

	//! Returns the geometries
	TeMultiGeometry& geometries() { return geometries_;}

	//! Gets the i-th property
	bool getProperty (TeProperty& prop, unsigned int i = 0);	
	
	//! Gets the property named "name
	bool getProperty (TeProperty& prop, string name);	
	
	//! Gets the value (as a string) of of the i-th property
	bool getPropertyValue (string& val, int i = 0);	

	//! Gets the value (as a string) of a property named 'name'
	bool getPropertyValue (const string& name, string& val);	

	//! Return true if this has polygons
	bool hasPolygons()	
	{ return (geometries_.hasPolygons()); } 
	
	//! Return true if this has lines
	bool hasLines()		
	{ return (geometries_.hasLines()); } 
	
	//! Return true if this has points
	bool hasPoints()		
	{ return (geometries_.hasPoints()); } 
	
	//! Return true if this has cells
	bool hasCells()		
	{ return (geometries_.hasCells()); } 
	
	//! Get geometries
	bool getGeometry(TePolygonSet& result); 
	bool getGeometry(TeLineSet& result);
	bool getGeometry(TePointSet& result);
	bool getGeometry(TeCellSet& result);
	bool getGeometry(vector<TeGeometry*>& result);
	bool getGeometry(TeMultiGeometry& result);

	//! Adds a property to this ST object
	void addProperty(TeProperty& prop) 
	{	properties_.push_back ( prop );		}

	//! Adds a property to this ST object
	void addProperty(const double& val, TeAttributeRep rep = TeAttributeRep(), bool attrIn = true); 

	//! Set geometries
	void setGeometry(const TePolygonSet& result)
	{ geometries_.setGeometry(result); }
	
	void setGeometry(const TeLineSet& result)
	{ geometries_.setGeometry(result); }

	void setGeometry(const TePointSet& result)
	{ geometries_.setGeometry(result); }

	void setGeometry(const TeCellSet& result)
	{ geometries_.setGeometry(result); }

	void setGeometry(vector<TeGeometry*>& result)
	{ geometries_.setGeometry(result); }

	void setGeometry(const TeMultiGeometry& geoms) 
	{	geometries_ = geoms; }

	//! Add geometries
	bool addGeometry(const TePolygon& poly);  
	bool addGeometry(const TeLine2D& line);
	bool addGeometry(const TePoint& point);	
	bool addGeometry(const TeCell& cell);		
		
	//! Sets the value (as a string) of a property named 'name' 
	bool setPropertyValue (const string& name, const string& val);	
	
	//! Returns the value (as a double) of the i-th property
	double operator[](int i);    
	
	//! Return the centroid of the geometry
	bool centroid(TeCoord2D& centroid, TeGeomRep geomRep=TeGEOMETRYNONE); 
	
	//! Return the area of the geometry
	bool area( double& a, TeGeomRep geomRep=TeGEOMETRYNONE);
	
	//! Returns the validity interval
	TeTimeInterval timeInterval () 
	{	return time_;	}
	
	//! Sets the validity interval
	void timeInterval (const TeTimeInterval& t) 
	{	time_ = t;	}

	//! Returns the initial time of the validity interval
	string getInitialDateTime(const string& mask="YYYYsMMsDDsHHsmmsSS") 
	{	return time_.getInitialDateTime(mask);	}
	
	//! Returns the final time of the validity interval
	string getFinalDateTime(const string& mask="YYYYsMMsDDsHHsmmsSS") 
	{	return time_.getFinalDateTime(mask);	}

	//! Returns the object identification
	string objectId () 
	{	return object_id_;	}
	
	//! Sets the object identification
	void objectId (const string& id) 
	{	object_id_ = id;	}

	//! Returns the instance identification
	vector<string>& uniqueId () 
	{	return unique_id_;	}

	//! Returns the instance identification
	string uniqueId (int index_) 
	{	return unique_id_[index_];	}
	
	//! Sets the instance identifications
	void uniqueId (const vector<string>& ids) 
	{	unique_id_ = ids;	}

	//! Adds the instance identifications
	void addUniqueId(const string& id)
	{	unique_id_.push_back(id); }

	//! Sets the grouping identification of this ST object
	void slice (int s) 
	{	slice_ = s; }

	//! Returns the grouping identification of this ST object
	int slice()
	{	return slice_; }

	//! Return the theme that the instance belongs 
	TeTheme* theme() { return theme_; }

	//! Set the theme that the instance belongs
	void theme(TeTheme* theme)  { theme_ = theme; }

	//! clear instance
	void clear(); 
};


#endif 
