/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeTin.h
    This provides support for a TIN structure
*/
#ifndef  __TERRALIB_INTERNAL_TIN_H
#define  __TERRALIB_INTERNAL_TIN_H
#include "TeGeometry.h"
#include <list>
#include <set>

using namespace std;

//!  Tin Vertex types 
enum TeTinVertexType 
{ NORMALVERTEX = 0, ONCONTOURVERTEX = 1, ONBREAKLINEVERTEX = 0x10 };

//!  A basic class for handling vertex of TIN
/*!
	\sa TeGeometry TeMeasure
*/
class  TeTinVertex : public TeGeomSingle<TeCoord2D>, public TeMeasure
{
public:

	//! Empty constructor
	TeTinVertex() : TeMeasure ( TeMAXFLOAT ), oneEdge_ (-1) {}

	//! Constructor with parameters
	TeTinVertex( TeCoord2D xy, float value, TeTinVertexType type,
		int edge ) :
		TeGeomSingle<TeCoord2D> (xy), TeMeasure ( value ),
		 type_ (type), oneEdge_ ( edge )
		{ if ( ! xy.tooBig() ) TeGeometry::setBox(TeBox(xy,xy)); }

// -- Methods
//
	int oneEdge () 
	{ return oneEdge_; }

	void setOneEdge (int oneEdge) 
	{ oneEdge_ = oneEdge; }

	TeTinVertexType type () 
	{ return type_; }

	void setType (TeTinVertexType type) 
	{ type_ = type; }

	void value (float value) 
	{ value_ = value; }

	TeGeomRep elemType() { return TeSAMPLES; }

private:

	TeTinVertexType type_;
	int oneEdge_;
};

//!  A basic class for handling a set of TeTinVertex
class TeTinVertexSet : public TeGeomComposite <TeTinVertex>
{
public:

// -- Constructors

	TeTinVertexSet() {}

// -- Methods

	bool loadFromFile (string &fileName);

	TeGeomRep elemType() { return TeSAMPLES; }
};

//! Tin Edges type 
enum TeTinEdgeType 
{ NORMALEDGE, ONCONTOUREDGE, ONBREAKLINEEDGE };

//!  A basic class for handling edges of a  TIN
/*!
	\sa TeGeometry
*/
class TeTinEdge : public TeGeometry
{
public:

// -- Constructors

	TeTinEdge() {}
	TeTinEdge(int from, int to, int left, int right,
		TeTinEdgeType type) :
	from_ (from), to_ (to), left_ (left), right_ (right), type_ (type)
	{}

// -- Methods

	int from () 
	{ return from_; }

	int to () 
	{ return to_; }

	void setVertices (int from, int to)
	{ 
		from_ = from;
		to_ = to;
	}

	void swapVertices ()
	{ 
		int aux = from_;
		from_ = to_;
		to_ = aux;
	}

	int left () 
	{ return left_; }

	int right ()
	{ return right_; }

	void setTriangles (int left, int right)
	{ 
		left_ = left;
		right_ = right;
	}

	void swapTriangles ()
	{ 
		int aux = left_;
		left_ = right_;
		right_ = aux;
	}
	
	void exchangeTriangle ( int t1, int t2 )
	{
		if (left_ == t1 )
			left_ = t2;
		else
			right_ = t2;
	}

	void exchangeVertex ( int v1, int v2 )
	{
		if (from_ == v1 )
			from_ = v2;
		else
			to_ = v2;
	}

	TeTinEdgeType type () 
	{ return type_; }

	void setType (TeTinEdgeType type) 
	{ type_ = type; }

	TeGeomRep elemType() { return TeSAMPLES; }

private:

	int	from_, to_;
	int	left_, right_;
	TeTinEdgeType type_;
};

//!  A basic class for handling a set of TeTinEdge
class TeTinEdgeSet : public TeGeomComposite<TeTinEdge>
{
public:

// -- Constructors

	TeTinEdgeSet()	{}

// -- Methods

	bool loadFromFile (string &fileName);

	TeGeomRep elemType() { return TeSAMPLES; }
};

//! TeTinTriangle: A class for handling a TIN triangle
class TeTinTriangle : public TeGeometry
{
public:

// -- Constructors

	TeTinTriangle()
	{
		triEdges_[0] = -1;
		triEdges_[1] = -1;
		triEdges_[2] = -1;
	}

	TeTinTriangle(int first, int second, int third) 
	{
		triEdges_[0] = first;
		triEdges_[1] = second;
		triEdges_[2] = third;
	}

// -- Methods

	void setEdges (int first, int second, int third)
	{
		triEdges_[0] = first;
		triEdges_[1] = second;
		triEdges_[2] = third;
	}

	void edges (int& first, int& second, int& third)
	{
		first  = triEdges_[0];
		second = triEdges_[1];
		third  = triEdges_[2];
	}

	int edgeAt (short num)
	{
		if ((num > -1) && (num < 3))
			return triEdges_[num];
		else
			return -1;
	}

	TeGeomRep elemType() { return TeSAMPLES; }

private:

	int	triEdges_[3];
};

//! TeTinTriangle: A class for handling a set of TIN triangles
class TeTinTriangleSet : public TeGeomComposite<TeTinTriangle>
{
public:

// -- Constructors

	TeTinTriangleSet()	{}

// -- Methods

	bool loadFromFile (string &fileName);

	TeGeomRep elemType() { return TeSAMPLES; }
};

//! A class to handle a TIN
class TeTin
{
public:

// -- Constructors

	TeTin() {}
	TeTin(TeBox & /* box */) {}

// -- Methods
	void clear()
	{
		vertexSet_.clear();
		triangleSet_.clear();
		edgeSet_.clear();
	}

	bool createInitialTriangles(TeBox &box);
	bool insertLineSamples ( TeContourLine& line );
	bool loadFromFile(string &fileName);
	bool getLines(TeLineSet &lineSet);
	bool isEmpty ()
	{ return ( vertexSet_.empty() && triangleSet_.empty() &&
	           edgeSet_.empty() ); } 

	typedef  TeTinEdgeSet::iterator TeTinEdgeIterator;

	TeTinEdgeIterator edgeBegin()
	{ return edgeSet_.begin(); }

	TeTinEdgeIterator edgeEnd()
	{ return edgeSet_.end(); }

	TeTinVertex& vertex (int i)
	{ return vertexSet_[i];	}

	TeTinEdge& edge (int i)
	{ return edgeSet_[i]; }

	bool triangleContainsPoint(int t, TeCoord2D& pt);
	int edgeWithVertexDiffFromEdge(int t, int v, int e);
	int oneTriangleWithVertex(int vertex);
	int oneEdgeWithVertex(int vertex);
	int edgeOppVertex(int triangle, int vertex);
	int vertexOppEdge(int triangle, int edge);
	bool vertexEdges(int vertex, list<int> &edges);
	bool vertexOppEdges(int vertex, list<int> &edges);
	bool vertexOppVertices(int vertex, list<int> &vertices);
	bool vertexTriangles(int vertex, list<int> &triangles);
	bool triangle3Neighbors(int t, int& neighbor0, int& neighbor1, int& neighbor2);
	int triangleAt (TeCoord2D& pt);

	bool triangleVertices(int t, int& vertex0, int& vertex1, int& vertex2);

	int triangleVertexAt(int t, int vertex);
	bool trianglePoints(int t, TeCoord2D& pt0, TeCoord2D& pt1, TeCoord2D& pt2);
	
	bool twoNewTriangles(int t, int v, set<int>& triangles);
	int duplicateTriangle(int t, int n, int v, set<int>& triangles);
	bool dupNeighTriangle (int t, int e, int v, set<int>& triangles);

	void testDelaunayForVertex(int v, set<int>& triangles);
	bool testDelaunayAt(int t, int v, set<int>& triangles);
	bool swapEdges( int t, int tv, int ai );

	bool insertPoint(double x, double y, float value);

	bool edgesInterBySegment( TeCoord2D& pf, TeCoord2D& pn, list<int> &aiset, list<int> &viset );
	bool insertSegment( TeCoord2D& pf, TeCoord2D& pn );

	TeGeomRep elemType() { return TeSAMPLES; }
private:

	TeTinVertexSet		vertexSet_;
	TeTinTriangleSet	triangleSet_;
	TeTinEdgeSet		edgeSet_;

};

#endif
