/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeSpatialStatistics.h
    This file provides support for spatial statistics functions
	\note for a general overview of spatial statistics, please see
  "Spatial Data Analysis by Example", Bailey and  Gattrell
*/
#ifndef  __TERRALIB_INTERNAL_SPATIALSTATISTICS_H
#define  __TERRALIB_INTERNAL_SPATIALSTATISTICS_H

#include <string>
#include <vector>
#include <map>
#include <set>
using namespace std;

class TeSTEventSet;
class TeSTElementSet;
class TeGeneralizedProxMatrix;

 
/** @defgroup SpatialStatistics Spatial Statistics Algorithms
 *  TerraLib spatial statistics algorithms.
 *  @{
*/

/*! \fn void TeLocalMean(TeSTEventSet* elemSet, ProxMatrix& proxMatrix, int indexAttr);
    \brief calculates the local mean (WZ) and the number of neighbors for each object 
	\param elemSet		a set of objects  
	\param proxMatrix	proximity matrix  
	\param indexAttr	the index of the objects which keeps the attributes  
*/
bool 
TeLocalMean (TeSTEventSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0); 


/*! \fn void TeLocalMean(TeSTElementSet* elemSet, ProxMatrix& proxMatrix, int indexAttr);
    \brief calculates the local mean (WZ) and the number of neighbors for each object 
	\param elemSet		a set of objects  
	\param proxMatrix	proximity matrix  
	\param indexAttr	the index of the objects which keeps the attributes  
*/
bool 
TeLocalMean (TeSTElementSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0); 


/** @defgroup TeMoranIndex Moran Index Algorithm
 *  @ingroup SpatialStatistics
 *  @{
*/

/*! \fn double TeMoranIndex2 (TeSTEventSet* elemSet, ProxMatrix& proxMatrix, int indexAttr)
    \brief calculates the local moran index for each object and returns the global moran index  
	\param elemSet		a set of objects 
	\param proxMatrix	proximity matrix  
	\param indexAttr	the index of the objects which keeps the attributes  
*/
double
TeMoranIndex2 (TeSTEventSet* elemSet, const double& mean, const double& var, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0); 

/*! \fn double TeMoranIndex2 (TeSTElementSet* elemSet, ProxMatrix& proxMatrix, int indexAttr)
    \brief calculates the local moran index for each object and returns the global moran index  
	\param elemSet		a set of objects 
	\param proxMatrix	proximity matrix  
	\param indexAttr	the index of the objects which keeps the attributes  
*/
double
TeMoranIndex2 (TeSTElementSet* elemSet, const double& mean, const double& var, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0); 


/*! \fn double TeMoranIndex (TeSTEventSet* elemSet, int indexZ, int indexWZ)
    \brief calculates the local moran index (WZ) for each object and returns the global moran index  
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexWZ		the index of the objects which keeps the local mean deviations (WZ) 
*/
double
TeMoranIndex (TeSTEventSet* elemSet, int indexZ, int indexWZ); 

/*! \fn double TeMoranIndex (TeSTElementSet* elemSet, int indexZ, int indexWZ)
    \brief calculates the local moran index (WZ) for each object and returns the global moran index  
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexWZ		the index of the objects which keeps the local mean deviations (WZ) 
*/
double
TeMoranIndex (TeSTElementSet* elemSet, int indexZ, int indexWZ); 

/*! \fn double TeGlobalMoranSignificance (TeSTEventSet* elemSet, ProxMatrix& proxMatrix, int indexZ, unsigned int permutationsNumber, double moranIndex)
    \brief calculates and returns the global moran significance    
	\param elemSet		a set of objects  
	\param proxMatrix	proximity matrix      
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param permutationsNumber	number of the permutations used to calculate the significance
	\param moranIndex	the global moran index 
*/
double
TeGlobalMoranSignificance (TeSTEventSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr,
						   unsigned int permutationsNumber, double moranIndex); 

/*! \fn double TeGlobalMoranSignificance (TeSTElementSet* elemSet, ProxMatrix& proxMatrix, int indexZ, unsigned int permutationsNumber, double moranIndex)
    \brief calculates and returns the global moran significance    
	\param elemSet		a set of objects  
	\param proxMatrix	proximity matrix      
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param permutationsNumber	number of the permutations used to calculate the significance
	\param moranIndex	the global moran index 
*/
double
TeGlobalMoranSignificance (TeSTElementSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr,
						   unsigned int permutationsNumber, double moranIndex); 



/*! \fn void TeMoranMap (TeSTEventSet* elemSet, int indexLISAMap, int indexBoxMap)
    \brief classifies the objects based in the scatterplot of moran index and statistical significance
	\param elemSet		a set of objects 
	\param indexLISAMap	the index of the objects which keeps the statistical significances of the moran local indexes (LISA) 
	\param indexBoxMap	the index of the objects which keeps the classifications of the statistical significance of the moran local indexes 
*/
void
TeMoranMap (TeSTEventSet* elemSet, int indexLISAMap, int indexBoxMap); 

/*! \fn void TeMoranMap (TeSTElementSet* elemSet, int indexLISAMap, int indexBoxMap)
    \brief classifies the objects based in the scatterplot of moran index and statistical significance
	\param elemSet		a set of objects 
	\param indexLISAMap	the index of the objects which keeps the statistical significances of the moran local indexes (LISA) 
	\param indexBoxMap	the index of the objects which keeps the classifications of the statistical significance of the moran local indexes 
*/
void
TeMoranMap (TeSTElementSet* elemSet, int indexLISAMap, int indexBoxMap); 


/** @} */ 

/*! \fn void TeGStatistics (TeSTEventSet* elemSet, ProxMatrix& proxMatrix, int indexAttr)
    \brief calculates the G and G* statistics for each object 
	\param elemSet		a set of objects 
	\param proxMatrix	proximity matrix      
	\param indexAttr	the index of the objects which keeps the attributes  
*/
bool 
TeGStatistics (TeSTEventSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0 ); 


/*! \fn void TeGStatistics (TeSTElementSet* elemSet, ProxMatrix& proxMatrix, int indexAttr)
    \brief calculates the G and G* statistics for each object 
	\param elemSet		a set of objects 
	\param proxMatrix	proximity matrix      
	\param indexAttr	the index of the objects which keeps the attributes  
*/
bool 
TeGStatistics (TeSTElementSet* elemSet, TeGeneralizedProxMatrix* proxMatrix, int indexAttr=0 ); 



/*! \fn void TeBoxMap (TeSTEventSet* elemSet, int indexZ, int indexWZ, double mean)
    \brief classifies the objects in quadrants based in the scatterplot of moran index 
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexWZ		the index of the objects which keeps the local mean deviations (WZ) 
	\param mean			the global mean
*/
void
TeBoxMap (TeSTEventSet* elemSet, int indexZ, int indexWZ, double mean); 


/*! \fn void TeBoxMap (TeSTElementSet* elemSet, int indexZ, int indexWZ, double mean)
    \brief classifies the objects in quadrants based in the scatterplot of moran index 
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexWZ		the index of the objects which keeps the local mean deviations (WZ) 
	\param mean			the global mean
*/
void
TeBoxMap (TeSTElementSet* elemSet, int indexZ, int indexWZ, double mean); 



/*! \fn void TeLisaStatisticalSignificance ( TeSTEventSet* elemSet, int indexZ, int indexLISA, int indexNeighNum, int permutationsNumber)
    \brief evaluates the statistical significance of the moran local indexes (LISA) 
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexLISA	the index of the objects which keeps the local moran indexes (LISA) 
	\param indexNeighNum		the index of the objects which keeps the numbers of neighbours  
	\param permutationsNumber	number of the permutations used to evaluate the significance
*/
bool TeLisaStatisticalSignificance ( TeSTEventSet* elemSet, int indexZ, int indexLISA, 
									 int indexNeighNum, int permutationsNumber);  
 
/*! \fn void TeLisaStatisticalSignificance ( TeSTElementSet* elemSet, int indexZ, int indexLISA, int indexNeighNum, int permutationsNumber)
    \brief evaluates the statistical significance of the moran local indexes (LISA) 
	\param elemSet		a set of objects 
	\param indexZ		the index of the objects which keeps the deviations (Z)    
	\param indexLISA	the index of the objects which keeps the local moran indexes (LISA) 
	\param indexNeighNum		the index of the objects which keeps the numbers of neighbours  
	\param permutationsNumber	number of the permutations used to evaluate the significance
*/
bool TeLisaStatisticalSignificance ( TeSTElementSet* elemSet, int indexZ, int indexLISA, 
									 int indexNeighNum, int permutationsNumber);  


/*! \fn void TeLisaMap ( TeSTEventSet* elemSet, int indexSignifLISA, int permutationNumber)
    \brief classifies the objects based in the statistical significance of the moran local indexes (LISA) 
	\param elemSet				a set of objects  
	\param indexSignifLISA		the index of the objects which keeps the statistical significances of the local moran indexes (LISA) 
	\param permutationNumber	number of the permutations used to evaluate the significance
*/
void
TeLisaMap (TeSTEventSet* elemSet, int indexSignifLISA, int /* permutationNumber */); 


/*! \fn void TeLisaMap ( TeSTElementSet* elemSet, int indexSignifLISA, int permutationNumber)
    \brief classifies the objects based in the statistical significance of the moran local indexes (LISA) 
	\param elemSet				a set of objects  
	\param indexSignifLISA		the index of the objects which keeps the statistical significances of the local moran indexes (LISA) 
	\param permutationNumber	number of the permutations used to evaluate the significance
*/
void
TeLisaMap (TeSTElementSet* elemSet, int indexSignifLISA, int /* permutationNumber */); 

/** @defgroup BayesEstimation Bayes estimation Algorithm
 *  @ingroup SpatialStatistics
 *  @{
*/
//! Calculates the empirical Bayes estimation
template <typename It> void
TeEmpiricalBayes (It rBegin, It rEnd, It n, double mean, double variance, It bayes)
{
	double w= 0.;
	while (rBegin != rEnd)
	{
		w = variance / ( variance + ( mean/(*n)) );
		*bayes = w*(*rBegin) + (1-w)*mean;
		rBegin++; n++; bayes++;
	}
}

//! Calculates the local Bayes estimation
template <typename It> void
TeSpatialEmpiricalBayes (It rBegin, It rEnd, It n, It mean, It variance, It bayes)
{
	double w = 0.;
	while (rBegin != rEnd)
	{
		w = (*variance)/( (*variance) + ( (*mean)/n ));	
		*bayes = w*(*rBegin) + (1-w)*(*mean);
		rBegin++; n++; mean++; variance++; bayes++;
	}
}
  
/** @} */ 
/** @} */ 

/** \example STObjectSetExample3.cpp
 *	This is an example of how to use spatial statistic algorithms from 
	a Spatial Temporal Object Set (STObjectSet) and proximity matrix.
 */


#endif

