/*
 * Copyright (C) 2005 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <locale.h>
#include <stdlib.h>
#include <gmodule.h>
#include <translate.h>

#define REVERSE_TYPE_SERVICE		(reverse_service_get_type())

typedef struct
{
  TranslateService	parent;
} ReverseService;

typedef struct
{
  TranslateServiceClass	parent;
} ReverseServiceClass;

static gboolean
reverse_service_get_pairs (TranslateService *service,
			   GSList **pairs,
			   TranslateProgressFunc progress_func,
			   gpointer user_data,
			   GError **err)
{
  /* initialize the list */
  *pairs = NULL;

  /* add the English -> English (reversed) pair */
  *pairs = g_slist_append(*pairs,
			  translate_pair_new(TRANSLATE_PAIR_TEXT,
					     "en", "en-x-rev"));

  /* add the English (reversed) -> English pair */
  *pairs = g_slist_append(*pairs,
			  translate_pair_new(TRANSLATE_PAIR_TEXT,
					     "en-x-rev", "en"));

  return TRUE;			/* success */
}

static char *
reverse_service_translate_text (TranslateService *service,
				const char *text,
				const char *from,
				const char *to,
				TranslateProgressFunc progress_func,
				gpointer user_data,
				GError **err)
{
  return g_utf8_strreverse(text, -1);
}

static void
reverse_service_class_init (ReverseServiceClass *class)
{
  TranslateServiceClass *service_class = TRANSLATE_SERVICE_CLASS(class);
  
  service_class->get_pairs = reverse_service_get_pairs;
  service_class->translate_text = reverse_service_translate_text;
}

static void
reverse_service_register_type (GType *type)
{
  static const GTypeInfo info = {
    sizeof(ReverseServiceClass),
    NULL,
    NULL,
    (GClassInitFunc) reverse_service_class_init,
    NULL,
    NULL,
    sizeof(ReverseService),
    0,
    NULL
  };
  
  *type = g_type_register_static(TRANSLATE_TYPE_SERVICE,
				 "ReverseService",
				 &info,
				 0);
}

static GType
reverse_service_get_type (void)
{
  static GType type;
  static GOnce once = G_ONCE_INIT;

  g_once(&once, (GThreadFunc) reverse_service_register_type, &type);

  return type;
}

static TranslateService *
reverse_service_new (void)
{
  return g_object_new(REVERSE_TYPE_SERVICE,
		      "name", "reverse",
		      "nick", "Reverse",
		      NULL);
}

static void
reverse_service_register_tag (void)
{
  /*
   * Register our en-x-rev custom tag. The tag name was chosen
   * according to rules specified in RFC 3066 section 2.2.
   *
   * Note that registering a language name is optional and has
   * aesthetical purposes only. If no name exists for a particular
   * tag, the tag itself will be used.
   */
  translate_add_language("en-x-rev", "English (reversed)");
}

#ifdef STANDALONE

int
main (int argc, char **argv)
{
  GError *err = NULL;
  GSList *services = NULL;
  TranslateSession *session;
  char *reversed;

  setlocale(LC_ALL, "");	/* use user's locale */

  if (argc != 2)
    {
      g_printerr("Usage: rev TEXT\n");
      exit(1);
    }
  
  /* initialize libtranslate */
  if (! translate_init(&err))
    {
      g_printerr("Unable to initialize libtranslate: %s\n", err->message);
      g_error_free(err);
      exit(1);
    }

  /* register our custom tag */
  reverse_service_register_tag();

  /* create a session which will use our reverse service */
  services = g_slist_append(services, reverse_service_new());
  session = translate_session_new(services);
  g_slist_foreach(services, (GFunc) g_object_unref, NULL);
  g_slist_free(services);

  /* reverse the given text */
  reversed = translate_session_translate_text(session, argv[1], "en", "en-x-rev",
					      NULL, NULL, &err);

  /* handle failure */
  if (! reversed)
    {
      g_printerr("Unable to reverse text: %s\n", err->message);
      g_error_free(err);
      exit(1);
    }

  /* success, print the reversed text, cleanup and exit */
  g_print("%s", reversed);
  g_free(reversed);
  g_object_unref(session);

  return 0;
}

#else /* ! STANDALONE */

G_MODULE_EXPORT gboolean
translate_module_init (GError **err)
{
  TranslateService *service;

  /* register our custom tag */
  reverse_service_register_tag();

  /* add the reverse service */
  service = reverse_service_new();
  translate_add_service(service);
  g_object_unref(service);

  return TRUE;			/* success */
}

#endif /* ! STANDALONE */
