/*
 * Copyright (C) 2002-5 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <dirent.h>

#include <errno.h>
#include <gmodule.h>

#include "constants.h"
#include "types.h"

#include "primary.h"
#include "actions_lib.h"


/* this function is misc, rather than actions... */
G_MODULE_EXPORT
const gchar *xffm_new_name(const gchar *in_path, const gchar *base)
{
    int max_digit = 0;
    DIR *directory;
    struct dirent *d;
    static gchar *nfile=NULL;
    gchar *path;

    if (!in_path) return NULL;
    /* get highest version number */
    if (!g_file_test(in_path,G_FILE_TEST_IS_DIR)) path=g_path_get_dirname(in_path);
    else path=g_strdup(in_path);
                   
    TRACE("new_name: path is %s",path); 

    if (nfile) {
	    g_free(nfile); 
	    nfile=NULL;
    }
    if (base) nfile = g_strdup_printf("%s-", base);
    else nfile = g_strdup_printf("%s-", _("New"));
 
    directory = opendir(path);
    if (directory) {
	while((d = readdir(directory)) != NULL){
	    if (strncmp(d->d_name,nfile,strlen(nfile))==0){
		    char *q,*p=d->d_name+strlen(nfile);
		    int digit;
		    gboolean ok=TRUE;
                    TRACE("hey--> %s",d->d_name); 
		    for (q=p;*q;q++) {
			    if (*q > '9' || *q < '0'){
				    ok=FALSE;
                                    TRACE("ignore--> %s\n",d->d_name); 
				    break;
			    }
		    }
		    digit=atoi(p);
		    if (digit > max_digit) max_digit=digit;
	    }
	}
	closedir(directory);
    }  else {
	DBG("could not open %s",path);
    }
    g_free(nfile);
    nfile = g_strdup_printf("%s-%d", base, ++max_digit);
    g_free(path);
    return (const gchar *) nfile;
}

/*****************  widget_t functions **************************/
G_MODULE_EXPORT
int
xffm_refresh(widgets_t *widgets_p){
    g_return_val_if_fail(widgets_p != NULL, 0);
    g_return_val_if_fail(widgets_p->refresh != NULL, 0);
    return (*widgets_p->refresh)(widgets_p);
}

G_MODULE_EXPORT
int
xffm_refresh_parent(widgets_t *widgets_p){
    g_return_val_if_fail(widgets_p != NULL, 0);
    g_return_val_if_fail(widgets_p->refresh_parent != NULL, 0);
    return (*widgets_p->refresh_parent)(widgets_p);
}

G_MODULE_EXPORT
record_entry_t	*xffm_get_selected_entry(struct widgets_t *widgets_p){
    g_return_val_if_fail(widgets_p != NULL, NULL);
    g_return_val_if_fail(widgets_p->get_selected_entry != NULL, NULL);
    return (*widgets_p->get_selected_entry)(widgets_p);
}

G_MODULE_EXPORT
record_entry_t	*xffm_get_selected_parent_entry(struct widgets_t *widgets_p){
    g_return_val_if_fail(widgets_p != NULL, NULL);
    g_return_val_if_fail(widgets_p->get_selected_parent_entry != NULL, NULL);
    return (*widgets_p->get_selected_parent_entry)(widgets_p);
}

G_MODULE_EXPORT
void xffm_open_with(widgets_t *widgets_p, record_entry_t *en){
    g_return_if_fail(en != NULL);
    g_return_if_fail(widgets_p != NULL);
    g_return_if_fail(widgets_p->open_with != NULL);
    (*widgets_p->open_with)(widgets_p, en);
    return;
}


/*************    actions **********************/

G_MODULE_EXPORT
int xffm_print(widgets_t *widgets_p, const gchar *cmd, const gchar *file)
{
    GError *error=NULL;
    gchar *base=g_path_get_basename(file);
    gchar *command=g_strdup_printf("%s \"%s\"",cmd, file);

    if (!widgets_p || !file || !cmd || !strlen(file) || !strlen(cmd)) return FALSE;
    
    print_diagnostics(widgets_p,NULL,"$",cmd," \"",file,"\"\n",NULL);
   if (!g_spawn_command_line_async (command,&error)){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics(widgets_p,"xffm/error",msg,":\n",command,"\n",NULL);
	print_status(widgets_p,"xffm/error", cmd, " ", base, NULL);
	g_error_free (error);
	g_free (msg);
	g_free(command);
	g_free(base);
	return 0;
   } else {
	gchar *base=g_path_get_basename(file);
	g_free(command);
	print_status(widgets_p,"xffm/info", cmd, " ", base, NULL);
	g_free(base);
	return 1;
   }
}


/**
 * xffm_touch:
 * @path: path to touch
 *
 * Returns: True if operation was successfully queued.
 **/

G_MODULE_EXPORT
int xffm_touch(widgets_t *widgets_p, const gchar *path){
    gchar *argv[]={"touch",(gchar *)path,NULL};
    int status;
    pid_t child;
	
    TRACE("touch_it");
    if (!widgets_p || !path) return FALSE;
    print_diagnostics(widgets_p,NULL,"$touch \"",path,"\"\n",NULL);
    child = fork();
    if(child < 0) return FALSE;
    if(!child)
    {
        execvp(argv[0], argv);
        g_warning("error executing %s",argv[0]); 
        _exit(1);
    }
    usleep(50000);
    wait(&status);
    return TRUE;
}


G_MODULE_EXPORT
int xffm_rename(widgets_t *widgets_p, const gchar *nfile, const gchar *ofile)
{

    struct stat st;
    gboolean overwrite=FALSE;
 
    TRACE("at rename");
    if (!widgets_p || !nfile || !ofile || !strlen(nfile) || !strlen(ofile)) return FALSE;

    if(lstat(nfile, &st) >= 0)
    {
	 /* dir over dir. file over dir, file over file, dir over file:
	 * keep retrying if cannot unlink... */
	/*if (!query_rm(nfile, ofile,overwrite)){*/
	if (!xffm_query_rm(widgets_p, nfile, ofile,TRUE)){
		    return FALSE;
	}
	overwrite=TRUE;
    }


    print_diagnostics(widgets_p,NULL,"$mv ", ofile, " ", nfile,  "\n", NULL);
    if(rename(ofile, nfile) == -1)
    {
	print_diagnostics(widgets_p,"xffm/error", strerror(errno), ":\n", ofile, " --> ", nfile,  "\n", NULL);
	print_status(widgets_p,"xffm/error", strerror(errno), NULL);
	return FALSE;
    }
    return TRUE;
}

G_MODULE_EXPORT
int xffm_duplicate (widgets_t *widgets_p, const gchar *nfile, const gchar *ofile)
{

    struct stat n_st;
    struct stat o_st;
    gchar *fname;
    if (!widgets_p || !nfile || !ofile || !strlen(nfile) || !strlen(ofile)) return FALSE;

    if(lstat(nfile, &n_st) >= 0)
    {
	/*printf ("TRACE:query_rm\n");*/
	gboolean overwrite;
	if (lstat(ofile, &o_st) != 0) {
	    g_warning("lstat(ofile, &o_st) != 0");
	    return FALSE;
	}
	 /* dir over dir, do not retry if cannot unlink... */
        if (S_ISDIR(o_st.st_mode) && S_ISDIR(n_st.st_mode)) 
	    overwrite=TRUE; 
	else overwrite=FALSE;
	/* file over dir, file over file, dir over file:
	 * keep retrying if cannot unlink... */
	if (!xffm_query_rm(widgets_p, nfile, ofile,overwrite)){
		    return FALSE;
	}	

    }
    fname = xffm_SimpleTmpList(nfile, ofile);
    xffm_IndirectTransfer(TR_DUPLICATE, fname, widgets_p);
    g_free(fname);
    return TRUE;
}

G_MODULE_EXPORT
int xffm_symlink(widgets_t *widgets_p, const gchar *nfile, const gchar *ofile)
{

    int status;
    char *argv[5];
    pid_t child;
    struct stat st;

    if (!widgets_p || !nfile || !ofile || !strlen(nfile) || !strlen(ofile)) return FALSE;
    if(lstat(nfile, &st) >= 0)
    {
	if (!xffm_query_rm(widgets_p, nfile, ofile,FALSE)) return FALSE;
    }

    argv[0] = "ln";
    argv[1] = "-s";
    argv[2] = (char *)ofile;
    argv[3] = (char *)nfile;
    argv[4] = 0;
    print_diagnostics(widgets_p,NULL,"$ln -s ", ofile, " ", nfile,  "\n", NULL);
    child = fork();
    if(child < 0) return FALSE;
    if(!child)
    {
        execvp(argv[0], argv);
        _exit(1);
    }
    usleep(50000);
    wait(&status);
    return TRUE;
}


G_MODULE_EXPORT
int xffm_new(widgets_t *widgets_p, const char *path, const char *in_name, gboolean directory)
{
    gchar *fullpath=NULL;
    FILE *fp;
    struct stat st;

    in_name=utf_2_local_string(in_name);

    TRACE ("in_name=%s\n",in_name);

    if (!widgets_p || !path || !in_name || !strlen(path) || !strlen(in_name)) return FALSE;
    fullpath = g_strdup_printf("%s%c%s", path,G_DIR_SEPARATOR, in_name);
    if(lstat(fullpath, &st) >= 0)
    {
	gboolean overwrite;
	 /* dir over dir, do not retry if cannot unlink... */
        if (S_ISDIR(st.st_mode) && directory) overwrite=TRUE; 
	else overwrite=FALSE;
	/* file over dir, file over file, dir over file:
	 * keep retrying if cannot unlink... */
	if (!xffm_query_rm(widgets_p, fullpath, fullpath,overwrite)){
    		    /*if (treeview) unset_load_wait();*/
	    g_free(fullpath);
	    return FALSE;
	}
    }

    TRACE ("making %s\n",fullpath);
    if(directory)
    {
	print_diagnostics(widgets_p,NULL,"$mkdir ", fullpath,  "\n", NULL);	
	if(mkdir(fullpath, 0xFFFF) < 0)
	{
	    print_diagnostics(widgets_p,"xffm/error", strerror(errno), " : ", fullpath, "\n", NULL);
	    print_status(widgets_p,"xffm/error",strerror(errno), NULL);
	    g_free(fullpath);
	    return FALSE;
	}
    }
    else
    {
	print_diagnostics(widgets_p,NULL,"$touch ", fullpath, "\n", NULL);
	fp = fopen(fullpath, "w");
	if(fp)
	    fclose(fp);
	else
	{
	    print_diagnostics(widgets_p,"xffm/error", strerror(errno), " : ", fullpath, "\n", NULL);
	    print_status(widgets_p,"xffm/error", strerror(errno), NULL);
	    g_free(fullpath);
	    return FALSE;
	}
    }
    print_status(widgets_p,"xffm/info",(directory)?_("Directory created"):_("File created"), NULL);
    g_free(fullpath);
    return TRUE;
}

G_MODULE_EXPORT
int xffm_diff(widgets_t *widgets_p, const gchar *file1, const gchar *file2)
{
    char *argv[4];
    GError *error = NULL;

    if (!widgets_p) return FALSE;
    TRACE("xfdiff %s %s",file1, file2);
    argv[0] = "xfdiff4";
    argv[1] = (char *)file1;
    argv[2] = (char *)file2;
    argv[3] = NULL;
    chdir(GETWD);
    print_diagnostics(widgets_p,NULL,"$xfdiff ",file1," ",file2,NULL);
    if (!g_spawn_async (NULL,argv,NULL, G_SPAWN_SEARCH_PATH,NULL,NULL,NULL,&error)){
	gchar *msg = g_strcompress (error->message);
	xffm_confirm(widgets_p,msg,NULL,NULL);
	/*print_diagnostics(widgets_p,"xffm/error",msg,":\n",argv[0],NULL);*/
	g_free (msg);
	g_error_free (error);
	return FALSE;
    }
    return TRUE;
}

int xffm_scramble(	widgets_t *widgets_p, 
			const gchar *file, 
			gboolean cypher)
{

    GError *error = NULL;
    gchar *nfile;
    gchar *command;
    int argc;
    gchar **argv;
    gchar *options,*p;
    
   
    if (!file || !widgets_p || !strlen(file)) return FALSE;
    p=g_find_program_in_path("scramble");
    if (!p) {
	p=g_strdup_printf(_("File not found: %s"),"scramble");
	print_diagnostics(widgets_p,"xffm/error",p,"\n");
	g_free(p);
	return FALSE;
    }
    g_free(p);
    
    
    nfile = g_strconcat(file,".cyt",NULL);
    if(access(nfile, F_OK) == 0)
    {
	if (!xffm_query_rm(widgets_p,nfile, nfile,FALSE)){
		    g_free(nfile);
		    return FALSE;
	}
    }
    g_free(nfile);
    nfile=NULL;
  
    if (cypher) options = g_strdup("-e");
    else options = g_strdup("-db");
    if (g_file_test(file,G_FILE_TEST_IS_DIR) ) {
	p = g_strconcat(options,"r",NULL);
	g_free(options);
	options=p;
	p=NULL;
    }
    /* VERBOSE*/
    {
	p = g_strconcat(options,"v",NULL);
	g_free(options);
	options=p;
	p=NULL;
    }
    
    command=g_strconcat(xffm_what_term()," -e scramble ",options," \"",file,"\"",NULL);
    g_free(options);
    
    
    
    g_shell_parse_argv (command, &argc,&argv,&error);
    if (error){
	gchar *msg = g_strcompress (error->message);
	print_diagnostics(widgets_p,"xffm/error",msg,":\n",command,"\n",NULL);
	g_error_free(error);
	g_free(command);   
	g_free (msg);
	return FALSE;
    }
    xffm_runvwd(widgets_p,NULL,(const gchar **)argv);
    g_strfreev (argv);
    g_free(command);   
    
    return TRUE;

}

GtkWidget*
xffm_create_remove (widgets_t *widgets_p)
{
  GtkWidget *remove;
  GtkWidget *vbox2;
  GtkWidget *hbox26;
  GtkWidget *question;
  GtkWidget *warning;
  GtkWidget *vbox12;
  GtkWidget *label16;
  GtkWidget *label20;
  GtkWidget *hbox9;
  GtkWidget *togglebutton1;
  GtkWidget *hbuttonbox1;
  GtkWidget *cancelbutton;
  GtkWidget *wastebutton;
  GtkWidget *alignment1;
  GtkWidget *hbox27;
  GtkWidget *adicon;
  GtkWidget *label21;
  GtkWidget *removebutton;
  GtkWidget *hbox28;
  GtkWidget *image1;
  GtkWidget *label22;
  GdkPixbuf *pb;

  remove = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (remove), _("xffm alert!"));
  gtk_window_set_modal (GTK_WINDOW (remove), TRUE);

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox2);
  gtk_container_add (GTK_CONTAINER (remove), vbox2);

  hbox26 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox26);
  gtk_box_pack_start (GTK_BOX (vbox2), hbox26, TRUE, TRUE, 0);

  pb = icon_tell (widgets_p, REAL_BIG,"xffm/question");
  question=gtk_image_new_from_pixbuf (pb);
  g_object_unref(pb);
/*  question = gtk_image_new_from_stock ("gtk-dialog-question", GTK_ICON_SIZE_DIALOG);*/
  gtk_widget_show (question);
  gtk_box_pack_start (GTK_BOX (hbox26), question, TRUE, TRUE, 0);
  gtk_misc_set_padding (GTK_MISC (question), 5, 0);

  pb = icon_tell (widgets_p, REAL_BIG,"xffm/warning");
  warning=gtk_image_new_from_pixbuf (pb);
  g_object_unref(pb);
  /*warning = gtk_image_new_from_stock ("gtk-dialog-warning", GTK_ICON_SIZE_DIALOG);*/
  gtk_widget_show (warning);
  gtk_box_pack_start (GTK_BOX (hbox26), warning, TRUE, TRUE, 0);
  gtk_misc_set_padding (GTK_MISC (warning), 5, 0);

  vbox12 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox12);
  gtk_box_pack_start (GTK_BOX (hbox26), vbox12, TRUE, TRUE, 0);

  label16 = gtk_label_new ("");
  gtk_widget_show (label16);
  gtk_box_pack_start (GTK_BOX (vbox12), label16, FALSE, FALSE, 0);

  label20 = gtk_label_new ("label20");
  gtk_widget_show (label20);
  gtk_box_pack_start (GTK_BOX (vbox12), label20, FALSE, FALSE, 0);

  hbox9 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox9);
  gtk_box_pack_start (GTK_BOX (vbox12), hbox9, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox9), 5);

  togglebutton1 = gtk_check_button_new_with_mnemonic (_("Apply to all"));
  gtk_widget_show (togglebutton1);
  gtk_box_pack_start (GTK_BOX (hbox9), togglebutton1, FALSE, FALSE, 0);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton1), TRUE);

  hbuttonbox1 = gtk_hbutton_box_new ();
  gtk_widget_show (hbuttonbox1);
  gtk_box_pack_start (GTK_BOX (vbox12), hbuttonbox1, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbuttonbox1), 5);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (hbuttonbox1), GTK_BUTTONBOX_END);
  gtk_box_set_spacing (GTK_BOX (hbuttonbox1), 5);

  cancelbutton = gtk_button_new ();  /*gtk_button_new_from_stock ("gtk-cancel");*/
  gtk_widget_show (cancelbutton);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), cancelbutton);
  GTK_WIDGET_SET_FLAGS (cancelbutton, GTK_CAN_DEFAULT);
  
  alignment1 = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_widget_show (alignment1);
  gtk_container_add (GTK_CONTAINER (cancelbutton), alignment1);

  hbox27 = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox27);
  gtk_container_add (GTK_CONTAINER (alignment1), hbox27);

  /*adicon = gtk_image_new_from_stock ("gtk-delete", GTK_ICON_SIZE_BUTTON);*/
  pb = icon_tell (widgets_p, MEDIUM,"xffm/stock_cancel");
  adicon=gtk_image_new_from_pixbuf (pb);
  gtk_widget_show (adicon);
  g_object_unref(pb);
  gtk_box_pack_start (GTK_BOX (hbox27), adicon, FALSE, FALSE, 0);

  label21 = gtk_label_new_with_mnemonic (_("Cancel"));
  gtk_widget_show (label21);
  gtk_box_pack_start (GTK_BOX (hbox27), label21, FALSE, FALSE, 0);
/****************/
  wastebutton = gtk_button_new ();
  gtk_widget_show (wastebutton);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), wastebutton);
  GTK_WIDGET_SET_FLAGS (wastebutton, GTK_CAN_DEFAULT);

  alignment1 = gtk_alignment_new (0.5, 0.5, 0, 0);
  gtk_widget_show (alignment1);
  gtk_container_add (GTK_CONTAINER (wastebutton), alignment1);

  hbox27 = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox27);
  gtk_container_add (GTK_CONTAINER (alignment1), hbox27);

  /*adicon = gtk_image_new_from_stock ("gtk-delete", GTK_ICON_SIZE_BUTTON);*/
  pb = icon_tell (widgets_p, MEDIUM,"xffm/stock_delete");
  adicon=gtk_image_new_from_pixbuf (pb);
  gtk_widget_show (adicon);
  g_object_unref(pb);
  gtk_box_pack_start (GTK_BOX (hbox27), adicon, FALSE, FALSE, 0);

  label21 = gtk_label_new_with_mnemonic (_("_Wastebasket"));
  gtk_widget_show (label21);
  gtk_box_pack_start (GTK_BOX (hbox27), label21, FALSE, FALSE, 0);

  removebutton = gtk_button_new ();
  gtk_widget_show (removebutton);
  gtk_container_add (GTK_CONTAINER (hbuttonbox1), removebutton);
  GTK_WIDGET_SET_FLAGS (removebutton, GTK_CAN_DEFAULT);

  hbox28 = gtk_hbox_new (FALSE, 2);
  gtk_widget_show (hbox28);
  gtk_container_add (GTK_CONTAINER (removebutton), hbox28);

  /*image1 = create_pixmap (remove, "xffm/broken");*/
  pb = icon_tell (widgets_p, MEDIUM,"xffm/broken");
  image1=gtk_image_new_from_pixbuf (pb);
  g_object_unref(pb);
  gtk_widget_show (image1);
  gtk_box_pack_start (GTK_BOX (hbox28), image1, FALSE, TRUE, 0);

  label22 = gtk_label_new_with_mnemonic (_("_Remove"));
  gtk_widget_show (label22);
  gtk_box_pack_start (GTK_BOX (hbox28), label22, TRUE, FALSE, 0);

  /* Store pointers to all widgets, for use by lookup_widget(). */
  GLADE_HOOKUP_OBJECT_NO_REF (remove, remove, "remove");
  GLADE_HOOKUP_OBJECT (remove, vbox2, "vbox2");
  GLADE_HOOKUP_OBJECT (remove, hbox26, "hbox26");
  GLADE_HOOKUP_OBJECT (remove, question, "question");
  GLADE_HOOKUP_OBJECT (remove, warning, "warning");
  GLADE_HOOKUP_OBJECT (remove, vbox12, "vbox12");
  GLADE_HOOKUP_OBJECT (remove, label16, "label16");
  GLADE_HOOKUP_OBJECT (remove, label20, "label20");
  GLADE_HOOKUP_OBJECT (remove, hbox9, "hbox9");
  GLADE_HOOKUP_OBJECT (remove, togglebutton1, "togglebutton1");
  GLADE_HOOKUP_OBJECT (remove, hbuttonbox1, "hbuttonbox1");
  GLADE_HOOKUP_OBJECT (remove, cancelbutton, "cancelbutton");
  GLADE_HOOKUP_OBJECT (remove, wastebutton, "wastebutton");
  GLADE_HOOKUP_OBJECT (remove, alignment1, "alignment1");
  GLADE_HOOKUP_OBJECT (remove, hbox27, "hbox27");
  GLADE_HOOKUP_OBJECT (remove, adicon, "adicon");
  GLADE_HOOKUP_OBJECT (remove, label21, "label21");
  GLADE_HOOKUP_OBJECT (remove, removebutton, "removebutton");
  GLADE_HOOKUP_OBJECT (remove, hbox28, "hbox28");
  GLADE_HOOKUP_OBJECT (remove, image1, "image1");
  GLADE_HOOKUP_OBJECT (remove, label22, "label22");

  gtk_widget_grab_focus (cancelbutton);
  return remove;
}

