/*
 * Copyright (C) 2002-5 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <limits.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <libxml/parser.h>
#include <libxml/tree.h>



#include "constants.h"
#include "types.h"

#include "primary.h"
#include "primary-icons.i"



G_MODULE_EXPORT
gboolean is_image (gchar *file){
    static GSList *pix_formats=NULL;
    GSList *l;
    gboolean result=FALSE;
    
    const gchar *mimetype = MIME_get_type(file,TRUE);
    /* check for image support types */

    if (!pix_formats) pix_formats = gdk_pixbuf_get_formats();
    for(l = pix_formats; l; l = l->next) {
	gchar **pix_mimetypes;
	int i;
	GdkPixbufFormat *fmt = l->data;
	pix_mimetypes = gdk_pixbuf_format_get_mime_types(fmt);
	for(i = 0; pix_mimetypes[i]; i++) {
		if(!g_ascii_strcasecmp(pix_mimetypes[i],mimetype)) {
		    result=TRUE;
		    break;
		}
	}
	g_strfreev(pix_mimetypes);
	if (result) break;
    }
    /* let's keep this one in memory to reduce overhead */
    /*g_slist_free(pix_formats);*/
    return result;
}
 
#define BIG_ENOUGH (96*96*3)
G_MODULE_EXPORT
GdkPixbuf *create_preview (char *file, int size)
{
    GdkPixbuf *src, *tgt;
    int max_w, w, h;
    int area,pw, ph;
    double scaleW, scaleH, scale;
    GError *error = NULL;
    const gchar *thumbnail_path=get_thumbnail_path(file,size);
    
    if (thumbnail_path && g_file_test(thumbnail_path,G_FILE_TEST_EXISTS)){
	TRACE("retrieving thumbnail for %s (%s)",file,thumbnail_path);
	tgt = gdk_pixbuf_new_from_file(thumbnail_path, NULL);
	if (tgt) return tgt;
    }

    switch (size)
    {
	default:
	    max_w = w = 96, h = 48;
	    break;
	case SMALL:
	    max_w = w = 48, h = 24;
	    break;
	case MEDIUM:
	    max_w = w = 68, h = 34;
	    break;
	case BIG:
	    max_w = w = 100, h = 50;
	    break;
	case REAL_BIG:
	    max_w = w = 200, h = 100;
	    break;
    }

	TRACE("getting pixbuf %s",file);
#if GTK_CHECK_VERSION (2,4,0)
    src = gdk_pixbuf_new_from_file_at_size (file,w,h,&error);
    if(error){
	g_warning(error->message);
	g_error_free(error);
	error=NULL;
	return NULL;
    }
#else /* gtk < 2.4 */
    src = gdk_pixbuf_new_from_file(file, &error);
    if(error || !src){
	if (error) {
	    g_warning("file=%s (%s)",file,error->message);
	    g_error_free(error);
	}
	error=NULL;
	return NULL;
    }
#endif

    ph = gdk_pixbuf_get_height(src);
    pw = gdk_pixbuf_get_width(src);
    area=ph*pw;
    if ((ph > h || pw > w) && pw > 0 && ph > 0){
      scaleH = (double)h / ph;
      scaleW = (double)w / pw;
      if (scaleW*pw > max_w) scaleW=(double)max_w / pw;
      scale = (scaleH < scaleW) ? scaleH : scaleW;
      h = scale * ph;
      w = scale * pw;
    } else {
	if (area > BIG_ENOUGH) save_thumbnail(thumbnail_path,file,src);
	return (src);
    }
    if(w < 10 || h < 10)
    {
	if (src) g_object_unref(G_OBJECT(src));
	return NULL;
    }

    if(!src)
	return NULL;

    TRACE("scaling pixbuf %s to %d,%d",file,w,h);

    /* if growing: */
    tgt = gdk_pixbuf_scale_simple(src, w, h, GDK_INTERP_BILINEAR);
    /* if contracting */
    tgt = gdk_pixbuf_scale_simple(src, w, h, GDK_INTERP_BILINEAR);
    if (tgt) g_object_unref(G_OBJECT(src));

    if (area > BIG_ENOUGH) save_thumbnail(thumbnail_path,file,tgt);
    return tgt;
}

G_MODULE_EXPORT
GdkPixbuf *create_full_pixbuf (char *file)
{
    GdkPixbuf *src;
    GError *error = NULL;

    src = gdk_pixbuf_new_from_file(file, &error);
    if(!src)
    {
	printf("TRACE: pixbuf error, file=%s\n",file); 
	return NULL;
    }
    if(error)
	g_error_free(error);

    return src;
}

G_MODULE_EXPORT
const gchar *resolve_folder_icon (record_entry_t * en)
{
    int type = en->type;
    const gchar *basic_type;
 
   /* if (IS_NOACCESS(en->type)) return "xffm/no-access";*/
    
    
    if (en->path && strcmp(g_get_home_dir(),en->path)==0){
	/*printf("%s==%s\n",g_get_home_dir(),en->path);*/
	basic_type="xffm/stock_home";
    }
   
    else if(IS_EXPANDED(type)) basic_type="xffm/open_folder";
    else basic_type="xffm/closed_folder";
    
    return (basic_type);

}
   
G_MODULE_EXPORT
void 
insert_pixbuf_tag (		widgets_t *widgets_p,
				const gchar *id, 
				GdkPixbuf **tgt,
			        int size,	
				double scale, 
				const gchar *where,
				int alpha){
    int x_tgt=0,y_tgt=0; /* default: NW */
    GdkPixbuf *tag,*otag;
    int width, height;
   
    if (!(*tgt)) return;
    width=gdk_pixbuf_get_width(*tgt);
    height=gdk_pixbuf_get_height(*tgt);

    /* only use stock gtk here: */
    if (strncmp(id,"gtk-",strlen("gtk-"))==0){
	otag = gtk_widget_render_icon(widgets_p->window, id ,size, NULL);
    }
    else  otag = icon_tell(widgets_p, size,id);
    if (!otag) return;
    tag = gdk_pixbuf_scale_simple(otag, width/scale,height/scale, GDK_INTERP_BILINEAR);	
    g_object_unref(G_OBJECT(otag));
    if (!tag) return;
    
    if (strcmp(where,"NE")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=0;
    else if (strcmp(where,"SW")==0)
       	x_tgt=0, y_tgt=(scale-1)*width/scale;
    else if (strcmp(where,"SE")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=(scale-1)*width/scale;
    else if (strcmp(where,"E")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=(scale-1)*width/scale/2;
  
      
    if (alpha) {
	otag=*tgt;
	*tgt=combine_pixbufs(tgt, &tag,
			x_tgt, y_tgt,
			width/scale,height/scale);
	g_object_unref(G_OBJECT(tag)); 
	g_object_unref(G_OBJECT(otag)); 
	return;
    }
    else {
	gdk_pixbuf_copy_area (tag,0,0,width/scale,height/scale,*tgt,x_tgt,y_tgt);
	g_object_unref(G_OBJECT(tag)); 
	return;
    }
}

    
  
G_MODULE_EXPORT
GdkPixbuf *icon_tell(	widgets_t *widgets_p,
			int size, 
			const gchar *id)
{
    return icon_tell_cut(widgets_p, size, id, FALSE);
}




G_MODULE_EXPORT
const gchar *resolve_icon_id (record_entry_t * en)
{
    int subtype;
    if(!en) {
	g_warning("resolve_icon_id(NULL)");
	return "xffm/default";
    }
    
    if (en->icon && !IS_CHANGED(en->type)) {
	return en->icon;
    }
    if (IS_ROOT_TYPE(en->type) && IS_FIND_TYPE(en->type)) {
	return ("xffm/stock_search");
    }
    if(en->module){
	const gchar *id=function_natural("plugins",en->module,en,"module_icon_id");
	if (id) return (id);
    }
    

    if(IS_DIR(en->type) && 
	(subtype=GPOINTER_TO_INT(FSTAB_is_in_fstab(en->path)))!=0){
	const gchar *t="xffm/disk";
	if (IS_NFS_TYPE(subtype)) t = "xffm/nfs";
	/*else if (IS_SMB_TYPE(subtype)) t = "xffm/share";
	else if (IS_PROC_TYPE(subtype)) t = "xffm/process";*/
	else if (IS_CDFS_TYPE(subtype)) {
	    t = "xffm/cdrom";
	    if (strstr(en->path,"cdrom")) t = "xffm/cdrom";
	    if (strstr(en->path,"cdrw")) t = "xffm/cdrom";
	    if (strstr(en->path,"dvd")) t = "xffm/dvd";
    	}
	else if (strstr(en->path,"floppy")) t = "xffm/floppy";
	/*else if (strstr(en->path,"cdrom")) t = "xffm/cdrom";
	else if (strstr(en->path,"cdrw")) t = "xffm/cdrom";
	else if (strstr(en->path,"dvd")) t = "xffm/dvd";*/
	return t;
    }
    

    /* broken links : */
    if(IS_BROKEN_LNK(en->type)) return "xffm/broken";

    /* find results */
    if(IS_XF_FND(en->type)) return "xffm/find_result";

    if(IS_DIR(en->type))
    {
	/*printf("folder %s link=%d\n",en->path,IS_XF_LNK(en->type)); */
	/* strstr makes all folders inside wastebasket use this icon */
	if(strstr(en->path, "/..Wastebasket"))
	{
	    if(en->count)
		return "xffm/waste_basket_full";
	    return "xffm/waste_basket_empty";
	}

	return resolve_folder_icon(en);
    }

    /* non-broken links: */
    /*printf("file %s link=%d\n",en->path,IS_XF_LNK(en->type)); */
    /*if(IS_XF_LNK(en->type)) return "inode/symlink";*/

    /* d_types: */
    else if(IS_XF_CHR(en->type)) return ("inode/chardevice");
    else if(IS_XF_BLK(en->type)) return ("inode/blockdevice");
    else if(IS_XF_FIFO(en->type)) return ("inode/fifo");
    else if(IS_XF_SOCK(en->type)) return ("inode/socket");

    if (IS_NOACCESS(en->type)) return ("xffm/no-access");

    if (en->path) return MIME_get_type(en->path,FALSE);

    /*if all else fails: */
    return NULL;
}

 
G_MODULE_EXPORT
GdkPixbuf *resolve_icon_size (	widgets_t *widgets_p,
				record_entry_t * en,int size)
{
    int gtksize;
    static GtkStyle *style = NULL;
    char *loc;
    const gchar *id;


    if(!en || !en->path) return NULL;

    if(!style) style = gtk_style_new();
    gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
    /*size = SMALL;*/

    id = resolve_icon_id(en);
    if(!id && IS_EXE(en->type)) id = "xffm/executable";
    if(id) {
	return icon_tell(widgets_p, size, id);
    }

    /* mime type: */
    loc = strrchr(en->path, '/');
    /*printf("path=%s, loc=%s\n",en->path,(loc)?loc:"null");*/
    if(loc)
    {
	GdkPixbuf *Icon;
	GtkIconSet *iconset;
	const gchar *stock_id = NULL;
	stock_id = MIME_get_type(en->path,FALSE);
	if(stock_id)
	{
	    iconset = ICON_get_iconset(stock_id,widgets_p->window);
	    if(!iconset) return NULL;
	    Icon = gtk_icon_set_render_icon(iconset, style, GTK_TEXT_DIR_LTR, GTK_STATE_NORMAL, gtksize, NULL, NULL);
	    return Icon;
	}
	/*else printf("key=%s not found loco\n",loc); */
    }
    return icon_tell(widgets_p, size, "xffm/default");

}


G_MODULE_EXPORT
GdkPixbuf *get_icon (	widgets_t *widgets_p,
			record_entry_t *en,
			record_entry_t *p_en)
{
    GdkPixbuf *Icon;
    if (!en) return NULL;  
    if (p_en) TRACE("shows images=%d",SHOWS_IMAGES(p_en->type)); 
    Icon = resolve_icon(widgets_p, en, p_en,-1);
    return Icon;
}

G_MODULE_EXPORT
GtkWidget *icon_image (widgets_t *widgets_p, char *id)
{

    if(id)
    {
	GdkPixbuf *Icon;
	GtkIconSet *iconset;
	if(!style) style = gtk_style_new();
	if (strncmp(id,"gtk-",strlen("gtk-"))==0) {
	  Icon = load_stock_icon(widgets_p, id, GTK_ICON_SIZE_SMALL_TOOLBAR);
	} else {
	  iconset = ICON_get_iconset(id,widgets_p->window);
	  if(!iconset) return NULL;
	  Icon = gtk_icon_set_render_icon(iconset, style, GTK_TEXT_DIR_LTR, GTK_STATE_NORMAL, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL, NULL);
	}
	if(Icon){
	   GtkWidget *image=gtk_image_new_from_pixbuf(Icon);
	   g_object_unref(G_OBJECT(Icon));
	   return image;
	}
    }
    return NULL;
}


G_MODULE_EXPORT
GdkPixbuf *load_stock_icon (	widgets_t *widgets_p,
				const gchar *id, int size){
    /*gchar *s,*p=NULL;*/
    GdkPixbuf *pixbuf=NULL;
    if (strncmp(id,"gtk-",strlen("gtk-"))!=0){
	g_warning("load_stock_icon called on non stock item %s",id);
	return NULL;
    }
    pixbuf=gtk_widget_render_icon(widgets_p->window,id,size, NULL);
    return pixbuf;
}

/* 
  *options for icons... use GTK_STATE_INSENSITIVE for files that
  have been cut to the pasteboard, until the pasteboard has been
  found invalid... 
  *  GTK_STATE_NORMAL,
  GTK_STATE_ACTIVE,
  GTK_STATE_PRELIGHT,
  GTK_STATE_SELECTED,
  GTK_STATE_INSENSITIVE*/
/* use parameter for iconview size */
G_MODULE_EXPORT
GdkPixbuf *resolve_icon (	widgets_t *widgets_p,
				record_entry_t * en, 
				record_entry_t * p_en, 
				int s)
{
    int gtksize, size;
    char *loc = NULL;
    const gchar *id=NULL;
    int max_preview_size=256;
    gboolean cut = FALSE;
    /*GdkPixbuf *Icon;
    GtkIconSet *iconset;
    char *l, *stock_id = NULL;*/
       const gchar *mimetype;

    if (!en){
	g_warning("critical: en=NULL at resolve_icon");
	return NULL;
    }
    if(!style) style = gtk_style_new();

    if (s<0) {
	gtksize = GTK_ICON_SIZE_DIALOG;
	size = REAL_BIG;
    }
    else {
	gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
	size = MEDIUM;
    }
#if 0
    else switch (xffm_details->icon_size){
      case 3:
	gtksize = GTK_ICON_SIZE_DIALOG;
	size = REAL_BIG;
	break;
      case 2:
	gtksize = GTK_ICON_SIZE_DND;
	size = BIG;
	break;
      case 1:
	gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
	size = MEDIUM;
	break;
      case 0:
      default:
	gtksize = GTK_ICON_SIZE_BUTTON;
	size = SMALL;
	break;
    }
#endif
    if(valid_pasteboard() == 2)
    {
	cut = in_pasteboard(en);
	if(cut==2){  SET_CUT(en->type); }
    }
    else UNSET_CUT(en->type);


    /* image previews before executable */
    if(en->path) loc = strrchr(en->path, '/');
    if (getenv("XFFM_MAX_PREVIEW_SIZE") && strlen( getenv("XFFM_MAX_PREVIEW_SIZE"))){
	if (is_number(getenv("XFFM_MAX_PREVIEW_SIZE"))) {   
	   max_preview_size=(atoi(getenv("XFFM_MAX_PREVIEW_SIZE")));
	}	   
    }
    
    if ( !cut && loc && IS_FILE(en->type) && !IS_NETWORK_TYPE(en->type) &&
	 !IS_TRASH_TYPE(en->type) && !strstr(en->path, "/..Wastebasket") &&
	 en->st->st_size <= max_preview_size*1024 && 
	  ((p_en && SHOWS_IMAGES(p_en->type)) 
	 ) && (IS_IMAGE(loc))
	)
    {
 	GdkPixbuf *tgt;
        process_pending_gtk(); 
	if (s>=0) tgt = create_preview(en->path, size);
	else tgt = create_preview(en->path, s);
	if(tgt) return tgt;
    }
    
    if (!id) id = resolve_icon_id(en);
    if(id){
	TRACE("ICON RESOLVED: %s --> %s\n",en->path,id);
     	return icon_tell_cut(widgets_p, size, composite_type_id(en->path,en->type,en->subtype,id), cut);
    }

    /* mime type (before executables) : */
#if 0
       gchar *p=strrchr(en->path,G_DIR_SEPARATOR);
       if (p && strchr(p,'-')){
	       p=g_strdup(en->path);
	       *strrchr(p,'-')=0;
	       mimetype = MIME_get_type(p,FALSE); /* don't try magic here, too expensive */
	       g_free(p);
       } 
       else 
#endif
       mimetype = MIME_get_type(en->path,FALSE);
       if ((strcmp("application/octet-stream",mimetype)==0 ||strcmp("text/plain",mimetype)==0) && IS_EXE(en->type)){
	   return icon_tell_cut(widgets_p, size, composite_type_id(en->path,en->type,en->subtype,"xffm/executable"), cut); 
       } else if (strcmp("undetermined type",mimetype)==0) {
    /* icon determination has failed, return the default icon */
	   return icon_tell_cut(widgets_p, size, composite_type_id(NULL,en->type,en->subtype,"xffm/default"), cut);
       }
       
 
    return  icon_tell_cut(widgets_p, size, composite_type_id(en->path,en->type,en->subtype,mimetype), cut);
    
}


