/* sel-onequery.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-onequery.h"
#include <libgnomedb/libgnomedb.h>
#include "mg-query.h"
#include "mg-entity.h"
#include "mg-qfield.h"
#include "mg-field.h"
#include "mg-renderer.h"
#include "mg-server-data-type.h"
#include "mg-qf-all.h"
#include "mg-qf-field.h"
#include "mg-qf-value.h"
#include "mg-target.h"
#include "sel-onetarget.h"

/*
 *
 * Modules for the fields, targets, joins, etc of a given MgQuery
 *
 */
typedef struct {
	ModFlatData  data;
	GdkPixbuf   *field_pixbuf;
} ModOneQueryData;
#define QUERY_DATA(x) ((ModOneQueryData *)(x->mod_data))

static void         module_onequery_fill_model (Module *module);
static void         module_onequery_free (Module *module);
static const gchar *module_onequery_col_name (Module *module, guint colno);
static Module      *module_onequery_obj_manager (Module *module, GtkTreeIter *iter, GObject *object);
static void         module_onequery_model_store_data (Module *module, GtkTreeIter *iter);
static GSList      *module_onequery_get_objects_list (Module *module);
static const gchar *module_onequery_render_qfield_type (MgQfield *field);

Module *
sel_module_onequery_new (MgSelector *mgsel, gboolean insert_header, 
			 GtkTreeIter *iter, gpointer data)
{
	Module *module;
	MgQuery *query;
	GdkPixbuf *pixbuf_query = NULL;
	GdkPixbuf *pixbuf_field = NULL;
	GdkPixbuf *pixbuf_target = NULL;

	g_assert (data && IS_MG_QUERY (data));
	pixbuf_query = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-query_16x16.png");
	pixbuf_field = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-field_16x16.png");
	pixbuf_target = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-table_16x16.png");

	/* module structure */
	query = MG_QUERY (data);
	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_onequery_fill_model;
	module->free = module_onequery_free;
	module->col_name = module_onequery_col_name;
	module->obj_manager = module_onequery_obj_manager;
	module->model_store_data = module_onequery_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	/* Module's private data */
	module->mod_data = g_new0 (ModOneQueryData, 1);
	FLAT_DATA (module)->manager = G_OBJECT (query);
	FLAT_DATA (module)->manager_weak_refed = FALSE;
	FLAT_DATA (module)->pixbuf_hash = g_hash_table_new_full (NULL, NULL, NULL, 
								 (GDestroyNotify) g_object_unref);
	FLAT_DATA (module)->fallback_obj_pixbuf = pixbuf_query;
	FLAT_DATA (module)->get_objects_list = module_onequery_get_objects_list;
	QUERY_DATA (module)->field_pixbuf = pixbuf_field;

	g_hash_table_insert (FLAT_DATA (module)->pixbuf_hash, GUINT_TO_POINTER (MG_TARGET_TYPE), pixbuf_target);

	/* model settings */
	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, 
				    NAME_COLUMN, mg_base_get_name (MG_BASE (query)), 
				    PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;	
}

static void
module_onequery_fill_model (Module *module)
{
	GObject *manager;
	GtkTreeModel *model;

	manager = FLAT_DATA (module)->manager;

	/* Initial model filling */
	model = module->selector->priv->model;
	flat_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (manager, "target_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "target_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "target_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);

	g_signal_connect (manager, "join_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "join_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "join_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);

	g_signal_connect (manager, "field_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "field_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "field_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);
	g_signal_connect (manager, "fields_order_changed",
			  G_CALLBACK (flat_objs_order_changed_cb), module);

	g_signal_connect (manager, "sub_query_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "sub_query_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "sub_query_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);
}

static GSList *
module_onequery_get_objects_list (Module *module)
{
	GSList *retval = NULL;
	GSList *tmplist;

	g_return_val_if_fail (FLAT_DATA (module)->manager, NULL);
	g_return_val_if_fail (IS_MG_QUERY (FLAT_DATA (module)->manager), NULL);

	/* sub queries first, if applicable */
	if (module->selector->priv->mode & MG_SELECTOR_SUB_QUERIES) 
		retval = g_slist_concat (retval, 
					 mg_query_get_sub_queries (MG_QUERY (FLAT_DATA (module)->manager)));

	/* targets, if applicable */
	if ((module->selector->priv->mode & MG_SELECTOR_TARGETS) ||
	    (module->selector->priv->mode & MG_SELECTOR_TARGETS_CTS))
		retval = g_slist_concat (retval, 
					 mg_query_get_targets (MG_QUERY (FLAT_DATA (module)->manager)));

	/* fields, if applicable.
	 * note: if targets are shown then the fields "belonging" to the targets
	 * will be shown as children of the targets and must not appear there */
	if (!(module->selector->priv->mode & MG_SELECTOR_TARGETS_CTS)) {
		tmplist = NULL;
		if (module->selector->priv->mode & MG_SELECTOR_QVIS_FIELDS)
			tmplist = mg_entity_get_visible_fields (MG_ENTITY (FLAT_DATA (module)->manager));
		else
			if (module->selector->priv->mode & MG_SELECTOR_QALL_FIELDS) 
				tmplist = mg_entity_get_all_fields (MG_ENTITY (FLAT_DATA (module)->manager));
		if (tmplist && (module->selector->priv->mode & MG_SELECTOR_TARGETS)) {
			GSList *ptr, *newlist = NULL;
			ptr = tmplist;
			while (ptr) {
				if (!IS_MG_QF_FIELD (ptr->data) && !IS_MG_QF_ALL (ptr->data))
					newlist = g_slist_prepend (newlist, ptr->data);
				ptr = g_slist_next (ptr);
			}
			g_slist_free (tmplist);
			tmplist = g_slist_reverse (newlist);
		}
		if (tmplist)
			retval = g_slist_concat (retval, tmplist);
	}

	return retval;
}

static void
module_onequery_free (Module *module)
{
	GObject *manager = FLAT_DATA (module)->manager;
	GSList *list = module->sub_modules;

	g_assert (manager);

	/* free the extra pixbufs */
	if (QUERY_DATA (module)->field_pixbuf)
		g_object_unref (G_OBJECT (QUERY_DATA (module)->field_pixbuf));

	/* Free the sub modules */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}


	/* free this module */
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_updated_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);
	flat_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_onequery_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Field");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}

/*
 * module_onequery_obj_manager
 *
 * This function is called when a new object is created (MgQuery). It simply creates a new
 * Module (which will be appended to the list of modules of the module for the queries) for 
 * for the contents of that new query.
 *
 * The same applies for MgTarget objects, with the module for targets.
 */
static Module *
module_onequery_obj_manager (Module *module, GtkTreeIter *iter, GObject *object)
{
	Module *sub_module = NULL;

	g_assert (object);

	if (IS_MG_QUERY (object)) {
		if (module->selector->priv->mode & MG_SELECTOR_QVIS_FIELDS) 
			sub_module = sel_module_onequery_new (module->selector, FALSE, iter, object);
	}

	if (IS_MG_TARGET (object)) {
		if (module->selector->priv->mode & MG_SELECTOR_TARGETS) 
			sub_module = sel_module_onetarget_new (module->selector, FALSE, iter, object);
		if (module->selector->priv->mode & MG_SELECTOR_TARGETS_CTS) 
			sub_module = sel_module_onetarget_new_all (module->selector, FALSE, iter, object);
	}

	return sub_module;
}

static void
module_onequery_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && IS_MG_QFIELD (obj)) {
		MgQfield *field;
		const gchar *str1, *str2;
		gchar *str3;
		MgServerDataType *type;
		
		field = MG_QFIELD (obj);

		/* data type */
		type = mg_field_get_data_type (MG_FIELD (obj));
		if (type)
			str1 = mg_server_data_type_get_sqlname (type);
		else
			str1 = _("-");
		/* other */
		str2 = module_onequery_render_qfield_type (MG_QFIELD (obj));
		str3 = mg_renderer_render_as_str (MG_RENDERER (obj), NULL);
		
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, str1,
				    EXTRA6_COLUMN, str3,
				    EXTRA7_COLUMN, str2,
				    PIXBUF_COLUMN, QUERY_DATA (module)->field_pixbuf,
				    -1);
		if (str3)
			g_free (str3);

		str1 = mg_base_get_name (MG_BASE (field));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, _("Field <no name>"),
				    -1);
		}
	}

	if (obj && IS_MG_QUERY (obj)) {
		const gchar *str1;
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, mg_query_get_query_type_string (MG_QUERY (obj)),
				    -1);
		str1 = mg_base_get_name (MG_BASE (obj));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, _("Query <no name>"),
				    -1);
		}
	}
}

static const gchar *
module_onequery_render_qfield_type (MgQfield *field)
{
	GType ftype = G_OBJECT_TYPE (field);

	if (ftype == MG_QF_ALL_TYPE)
		return _("entity.*");
	if (ftype == MG_QF_FIELD_TYPE)
		return _("entity.field");
	if (ftype == MG_QF_VALUE_TYPE) {
		if (mg_qf_value_is_parameter (MG_QF_VALUE (field)))
			return _("parameter");
		else
			return _("value");
	}

	/* for other types */
	TO_IMPLEMENT;
	return "???";
}
