/////////////////////////////////////////////////////////////////////////////
// Name:        DScratch.cpp
// Purpose:
// Author:      Stefan Langhammer, Thomas Rogg
// Modified by:
// Created:     4.12.2005
// Modified:    16.04.2006
// Copyright:   (c) Stefan Langhammer, Thomas Rogg
// Licence:     LGPL
/////////////////////////////////////////////////////////////////////////////

// -----------------------------------------------------------------------------
//	DScratch.cpp - Final Scratch vinyl analyser
// -----------------------------------------------------------------------------

#include "DScratch.h"
#include "DAnalyse.h"

#include <set>
#include <string>


//Force scratchlib to use RtAudio!
#define __DSCRATCH_USE_RTAUDIO__


// -----------------------------------------------------------------------------
//	DScratch::DScratch - constructs the DScratch object
// -----------------------------------------------------------------------------

DScratch::DScratch()
{
	
	// Initialize variables
	mActiveDevice = -1;

	// Get all devices capable of being used by Digital Scratch
#ifdef __DSCRATCH_USE_RTAUDIO__
	mAudio = NULL;
#ifdef __WINDOWS_ASIO__
	ProbeDrivers(RtAudio::WINDOWS_ASIO);
#endif
#ifdef __WINDOWS_DS__
	ProbeDrivers(RtAudio::WINDOWS_DS);
#endif
#ifdef __LINUX_ALSA__
	ProbeDrivers(RtAudio::LINUX_ALSA);
#endif
#ifdef __LINUX_JACK__
	ProbeDrivers(RtAudio::LINUX_JACK);
#endif

#endif /* __DSCRATCH_USE_RTAUDIO__ */
}

// -----------------------------------------------------------------------------
//	DScratch::~DScratch - destructs the DScratch object
// -----------------------------------------------------------------------------

DScratch::~DScratch()
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	if (mAudio)
	{
		CloseRtAudio();
		delete mAudio;
	}
#endif /* __DSCRATCH_USE_RTAUDIO__ */
}

// -----------------------------------------------------------------------------
//	DScratch::SetActiveDevice - sets the device to use
// -----------------------------------------------------------------------------

bool DScratch::SetActiveDevice(int index)
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	if(index >= 0 && index < (int)mDeviceNames.size())
	{
		this->InitRtAudio (m_DeviceInfo.at(index).api);
		return OpenRtAudio(m_DeviceInfo.at(index).deviceID);
	}
	else
		return false;
#else
	return false;
#endif /* __DSCRATCH_USE_RTAUDIO__ */
}


// -----------------------------------------------------------------------------
//	DScratch::GetActiveDevice - returns the used device, -1 if no device
// -----------------------------------------------------------------------------

int DScratch::GetActiveDevice()
{
	return mActiveDevice;
}


void DScratch::SetTimecodeVinyl(TimecodeVinyl vinylType)
{
	mAnalyser.SetVinyl(int(vinylType));
}
// -----------------------------------------------------------------------------
//	DScratch::SetUseData - initialises to give data yourself, do not use device
// -----------------------------------------------------------------------------

void DScratch::SetUseData(int frequency)
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	CloseRtAudio();
#endif /* __DSCRATCH_USE_RTAUDIO__ */

	mAnalyser.SetFrequency(frequency);
}


// -----------------------------------------------------------------------------
//	DScratch::AnalyseData - analyses the given data
// -----------------------------------------------------------------------------

void DScratch::AnalyseData(short *data, int numData)
{
	mAnalyser.Analyse(data, numData);
}


// -----------------------------------------------------------------------------
//	DScratch::GetSpeed - returns the current speed of the vinyl
// -----------------------------------------------------------------------------

double DScratch::GetSpeed()
{
	return mAnalyser.GetSpeed();
}


// -----------------------------------------------------------------------------
//	DScratch::GetPosition - returns the current position of the vinyl
// -----------------------------------------------------------------------------

double DScratch::GetPosition()
{
	return mAnalyser.GetPosition();
}

// -----------------------------------------------------------------------------
//	DScratch::GetTimecodesPerSecond - returns detected timecodes/s
// -----------------------------------------------------------------------------
int	   DScratch::GetTimecodesPerSecond()
{
	return mAnalyser.GetTimecodesPerSecond();
}


// -----------------------------------------------------------------------------
//	DScratch::GetDeviceName - returns the name of the given device
// -----------------------------------------------------------------------------

bool DScratch::GetDeviceName(int index, string &name)
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	if (index < 0 || index >= int(mDeviceNames.size()))
		return false;
	name = mDeviceNames.at(index);
	return true;
#else
	return false;
#endif /* __DSCRATCH_USE_RTAUDIO__ */
}


// -----------------------------------------------------------------------------
//	DScratch::GetDeviceIndex - returns the index of the given device
// -----------------------------------------------------------------------------

int DScratch::GetDeviceIndex(const char *name)
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	size_t i;

	for(i = 0; i < mDeviceNames.size(); i++)
	{
		if(name == mDeviceNames.at(i))
			return (int)i;
	}
#endif /* __DSCRATCH_USE_RTAUDIO__ */

	return -1;
}


// -----------------------------------------------------------------------------
//	DScratch::GetNumDevices - returns the number of devices found
// -----------------------------------------------------------------------------

int DScratch::GetNumDevices()
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	return (int)mDeviceNames.size();
#else
	return 0;
#endif /* __DSCRATCH_USE_RTAUDIO__ */
}


#ifdef __DSCRATCH_USE_RTAUDIO__

// -----------------------------------------------------------------------------
//	DScratch::ProbeDrivers - probes the drivers
// -----------------------------------------------------------------------------
void DScratch::ProbeDrivers(RtAudio::RtAudioApi api)
{
	RtAudio rt(api);
	std::set<std::string> names;
	RtAudioDeviceInfo info;
	int numDevices, i;
	string name;

	numDevices = rt.getDeviceCount();

	for(i = 1; i <= numDevices; i++)
	{
		try
		{
			info = rt.getDeviceInfo(i);
		}
		catch(RtError &)
		{
			continue;
		}

		if(info.probed && info.inputChannels >= 2)
		{
			if(names.find(info.name) == names.end())
			{
				char temp[400];
				
				sprintf( temp,     "%-40s %s\n  %i input channels | %10s | highest freq: %i hz",	info.name.c_str(),
					GetRtAudioApiString(api).c_str(), info.inputChannels, GetAudioFormatAsString(&info.nativeFormats).c_str(),
					info.sampleRates.at(info.sampleRates.size()-1));
				DeviceInfo devInfo;
				devInfo.api = api;
				devInfo.deviceID = i;
				m_DeviceInfo.push_back(devInfo);
				name = temp;
				mDeviceNames.push_back(name);
				names.insert(info.name);
			}
		}
	}
}


// -----------------------------------------------------------------------------
//	DScratch::InitRtAudio - set the driver type ASIO, DSOUND,...
// -----------------------------------------------------------------------------

void DScratch::InitRtAudio(RtAudio::RtAudioApi driver)
{
#ifdef __DSCRATCH_USE_RTAUDIO__
	if (mAudio)
		delete mAudio;
	
	mAudio = new RtAudio(driver);
#endif /* __DSCRATCH_USE_RTAUDIO__ */
}

// -----------------------------------------------------------------------------
//	DScratch::OpenRtAudio - opens the RtAudio stream
// -----------------------------------------------------------------------------

bool DScratch::OpenRtAudio(int device)
{
	int bufferSize;
	
	printf("BLAAAAAAAAAAAAAAH!!!!!!!!!*********\n\n");
	
	// Close audio if open
	CloseRtAudio();

	// Open audio
	try
	{
		bufferSize = DSCRATCH_RTAUDIO_BUFFER_SIZE;
		RtAudioDeviceInfo info = mAudio->getDeviceInfo(device);
		
		// Use the highest sample frequency
		int freq = info.sampleRates.at(info.sampleRates.size()-1);
		printf("info.sampleRAte.size(): %d\n", info.sampleRates.size());
		mAnalyser.SetFrequency(freq);

		try {
			// SampleFormat note
			// we need RTAUDIO_SINT16 as output format for the analysing part.
		mAudio->openStream(0, 0, device, 2,
			RTAUDIO_SINT16, freq, &bufferSize,
			DSCRATCH_RTAUDIO_NUM_BUFFERS);
		}
		catch (RtError &error) {
			error.printMessage();
			// Perhaps try other parameters?
		}

		mActiveDevice = device;
		
		mAudio->setStreamCallback(RtAudioCallback, &mAnalyser);
		mAudio->startStream();
	}
	catch(RtError &)
	{
		CloseRtAudio();
		return false;
	}
	return true;
}


// -----------------------------------------------------------------------------
//	DScratch::CloseRtAudio - closes the RtAudio stream
// -----------------------------------------------------------------------------

void DScratch::CloseRtAudio()
{
	if(mActiveDevice == -1)
		return;

	try
	{
		mAudio->stopStream();
		mAudio->closeStream();
	}
	catch(RtError &)
	{
	}

	mActiveDevice = -1;
}


// -----------------------------------------------------------------------------
//	DScratch::RtAudioCallback - RtAudio callback
// -----------------------------------------------------------------------------

int DScratch::RtAudioCallback(char *buffer, int bufferSize, void *data)
{	
	((DAnalyse *)data)->Analyse((short *)buffer, bufferSize);

	return 0;
}
// -----------------------------------------------------------------------------
//	DScratch::GetRtAudioApiString - return the api name
// -----------------------------------------------------------------------------
string DScratch::GetRtAudioApiString(RtAudio::RtAudioApi api)
{
	if (api == RtAudio::IRIX_AL)
		return "IRIX_AL";
	if (api == RtAudio::LINUX_ALSA)
		return "LINUX_ALSA";
	if (api == RtAudio::LINUX_JACK)
		return "LINUX_JACK";
	if (api == RtAudio::LINUX_OSS)
		return "LINUX_OSS";
	if (api == RtAudio::MACOSX_CORE)
		return "MACOSX_CORE";
	if (api == RtAudio::WINDOWS_ASIO)
		return "WINDOWS_ASIO";
	if (api == RtAudio::WINDOWS_DS)
		return "WINDOWS_DS";
	return "UNSPECIFIED";
}

string DScratch::GetAudioFormatAsString(RtAudioFormat* format)
{
	string strSampleFormat = "Unknown";
		if ( *format & RTAUDIO_FLOAT64 )
			strSampleFormat = "RTAUDIO_FLOAT64";
		else if ( *format & RTAUDIO_FLOAT32 )
			strSampleFormat = "RTAUDIO_FLOAT32";
		else if ( *format & RTAUDIO_SINT32 )
			strSampleFormat = "RTAUDIO_SINT32";
		else if ( *format & RTAUDIO_SINT24 )
			strSampleFormat = "RTAUDIO_SINT24";
		else if ( *format & RTAUDIO_SINT16 )
			strSampleFormat = "RTAUDIO_SINT16";
		else if ( *format & RTAUDIO_SINT8 )
			strSampleFormat = "RTAUDIO_SINT8";
		return strSampleFormat;
}
#endif /* __DSCRATCH_USE_RTAUDIO__ */
