/***************************************************************************
                          vinylcontrol.cpp  -  description
                             -------------------
    begin                : Tue June 6 2006
    copyright            : (C) 2006 by Albert Santoni
    email                : 
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include "vinylcontrol.h"
#include "dscratch.h"
#include "enginebuffer.h"
#include "reader.h"
#include "controlobject.h"

VinylControl::VinylControl()
{

}

VinylControl::~VinylControl()
{
	free(scratch);
}

VinylControl::VinylControl(const char *_group, EngineBuffer* _buffer)
{
	dVinylPosition = 0.0f;
	dOldPos	= 0.0f;
	dOldDiff = 0.0f;
	bNeedleDown = true;
	bQuit = false;
	group = _group;
	
	string strDeviceName;
	
	//Get the control objects we need to manipulate the playback in Mixxx.
	playPos = ControlObject::getControl(ConfigKey(group, "playposition"));
	controlScratch = ControlObject::getControl(ConfigKey(group, "scratch"));
	rateSlider = ControlObject::getControl(ConfigKey(group, "rate")); //Range -1.0 to 1.0
    
	qDebug("VinylControl: Created new VinylControl object!\n");
	
	// Get pointer to play button
    playButton = ControlObject::getControl(ConfigKey(group, "play"));
	
	//Save the parameters inside this object.
	buffer = _buffer;
	
	//Create the "DScratch" object that interacts with scratchlib.
	scratch = new DScratch();
 
	//Choose which type of timecoded vinyl we're using.
	scratch->SetTimecodeVinyl(TimecodeVinyl(TIMECODE_VINYL_FINALSCRATCH));
	
	//Enable or disable RIAA correction
	scratch->EnableRIAACorrection(MIXXX_RIAA_CORRECTION);
	
	//Set the calibration value
	scratch->SetCalibration(MIXXX_CALIBRATION_VALUE);
	
	// Recording devices available ?
	int iNumDevices = scratch->GetNumDevices();
	if (iNumDevices == 0)
	{
		printf("VinylControl: Error: No recording device found!\n");
		printf("VinylControl: Press <Enter> to quit the application.\n");
		getchar();
		terminate();
	}	
	 
	//Get the device's name, print it out.
	scratch->GetDeviceName(MIXXX_SOUNDDEVICE,strDeviceName);
	printf("VinylControl: %i. %s\n", MIXXX_SOUNDDEVICE, strDeviceName.c_str());
	
	if(!scratch->SetActiveDevice(MIXXX_SOUNDDEVICE, MIXXX_ACTIVECHANNELS))
	{
		qDebug("VinylControl: Soundcard initialization failed!\n Press any key to bomb.");
		getchar();
		terminate();
	}
	else
	{
		//Start reading from the soundcard and processing the timecoded vinyl. 
		//(This ends up calling-back the function "run()" below.)
    	start();
	}
}

void VinylControl::run()
{
	int	iTPS = 0;	// Timecodes per second
	double diff;
	dVinylPosition = 0.0f;
	dVinylPitch = 0.0f;
	double dTemp = 0.0f;
	double dPlayPos = 0.0f;
	
	qDebug("VinylControl: Starting VC monitoring thread");
	
	while(!bQuit)
	{
		// Dump position and speed every 1/10 second
		dVinylPosition = scratch->GetPosition(); //Get the position of the needle on the vinyl.
		diff = fabs((double)dVinylPosition - (double)dOldPos);
		dVinylPitch = scratch->GetSpeed(); //Get the playback speed of the vinyl.
		
		if (diff > 0.1)
		{ 
			dTemp = 0;
			
			//If it looks like the user is seeking the vinyl... (ie. moving the
			//vinyl really fast in either direction)
			if ((dVinylPitch > 1.10 || dVinylPitch < 0.90) && isNeedleInBounds())
			{
				bSeeking = true;
				playButton->queueFromThread(0.0f);
				rateSlider->queueFromThread(0.0f);
				controlScratch->queueFromThread(dVinylPitch);
				qDebug("VinylControl: dVinylPitch = %f\n", dVinylPitch);
			}
			else { //we're not seeking... just regular playback
				if (isNeedleInBounds())
				{
					bSeeking = false;
					controlScratch->queueFromThread(0.0f);
					playButton->queueFromThread(1.0f);
				}
			}
			
			//If the needle just got placed on the record, or playback just resumed
			//from a standstill...
			if (bNeedleDown == false && bSeeking == false)
			{
				controlScratch->queueFromThread(0.0f);
				
				if (isNeedleInBounds()) //Reposition Mixxx and start playback if we're in bounds...
				{
					syncPosition();
					playButton->queueFromThread(1.0f);
					bNeedleDown = true; //The needle is now down/the record is playing.
					dTemp = 0;		
				}
			}
						
			iTPS = scratch->GetTimecodesPerSecond();
			qDebug("VinylControl: %10.2fs %10.2f%%  %3i", dVinylPosition, 100.0 * scratch->GetSpeed(), iTPS);
			
			//If we're not seeking, sync Mixxx's pitch with the turntable's pitch.
			if (!bSeeking) {
				syncPitch();
			}
				
			qDebug("VinylControl: Setting rate slider to: %f", ((scratch->GetSpeed()*100) - 100) / 8.0f);
			dOldPos = dVinylPosition;
		}
		else //Either the needle is stopped, up off the record, or we simply skipped some timecodes...
		{ 
			dTemp++;
			if (dTemp > 15) //If the needle is actually stopped/off the record...
			{ 
				//qDebug("VinylControl: Needle is up/stopped?");
				if (bNeedleDown == true)
               		syncPosition();	
				controlScratch->queueFromThread(0.0f);				
				bNeedleDown = false;
				dTemp = 16; //Make sure we don't overflow eventually..				
			}
		}
	
#ifdef WIN32
		Sleep(10);
#else
		usleep(10000);
#endif
	}
	
	qDebug("VinylControl: VC monitoring loop terminated successfully");
}


//Synchronize the pitch of the external turntable with Mixxx's pitch.
void VinylControl::syncPitch()
{
	//controlScratch->queueFromThread(0.0f); //Can't be seeking and adjusting rate at the same time.
	rateSlider->queueFromThread(((scratch->GetSpeed()*100) - 100) / 8.0f);
}

//Synchronize the position of the timecoded vinyl with Mixxx's position.
//Returns false is the position is out-of-bounds (before the start of the song...), otherwise returns true.
bool VinylControl::syncPosition()
{
	playButton->queueFromThread(0.0f); //Pause playback	
	dVinylPitch = 0.0f;
	controlScratch->queueFromThread(0.0f);
	playPos->queueFromThread(((dVinylPosition - MIXXX_VINYL_LEADIN) * 2) / ((buffer->getFileLength() / 44100)));
	qDebug("VinylControl: ***********SYNCHRONIZING PLAY POSITION******************");
	return true;
}

//Checks to see if the needle is on the vinyl within the area that the current track would occupy (if it were
//actually on the vinyl.) It also accounts for the lead-in time specified.
bool VinylControl::isNeedleInBounds()
{
	if (dVinylPosition - MIXXX_VINYL_LEADIN < 0)
		return false;
	//TODO: Check to see if the needle is beyond the end of the track...
	return true; 
}

//Shuts down the vinyl control object/interface.
void VinylControl::close()
{
	qDebug("VinylControl: Shutting down...");
	
	if (this->running())
	{
		bQuit = true; //Signal the run() thread to finish
		wait(); //Block until the run() thread is finished before proceeding...
	}	
}
