/***************************************************************************
                          wvisualwaveform.cpp  -
                             -------------------
    begin                : Thu Oct 9 2003
    copyright            : (C) 2002 by Tue & Ken Haste Andersen
    email                : haste@diku.dk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "wvisualwaveform.h"
#include "visual/visualchannel.h"
#include "visual/visualdisplay.h"
#include <qdragobject.h>
#include <qcursor.h>
#include <iostream>

WVisualWaveform::WVisualWaveform(QWidget *pParent, const char *pName, const QGLWidget *pShareWidget) : QGLWidget(pParent,pName,pShareWidget),
		m_GrabSensitivity(20.),
		m_Speed(64.),
		m_SpeedWearOff(0.2),
		m_iSpeedTimerID(0)
{
    setAcceptDrops(true);
    m_pVisualController = new VisualController();

    installEventFilter(this);

#ifdef __MACX__
    // Hack to reduce load in GUI thread. This makes the system behave
    // "correctly" on MacOS X, where it would otherwise stall the system
    // for some seconds now and then.
    m_iTimerID = startTimer(100);
#endif
#ifdef __WIN__
    m_iTimerID = startTimer(15);
#endif
#ifdef __LINUX__
    m_iTimerID = startTimer(30);
#endif

    m_qlList.setAutoDelete(false);
}

WVisualWaveform::~WVisualWaveform()
{
    // Stop timers
    killTimer(m_iTimerID);
    killTimer(m_iSpeedTimerID);

    // Delete associated VisualChannels
    while (m_qlList.remove());

    // Finally delete the VisualController
    delete m_pVisualController;
}

bool WVisualWaveform::directRendering()
{
    return 1; //format().directRendering();
}

void WVisualWaveform::dragEnterEvent(QDragEnterEvent *event)
{
    event->accept(QUriDrag::canDecode(event));
}

void WVisualWaveform::dropEvent(QDropEvent *event)
{
    QStringList lst;
    if (!QUriDrag::canDecode(event))
    {
        event->ignore();
        return;
    }

    event->accept();
    QUriDrag::decodeLocalFiles(event, lst);
    QString name = (*lst.begin());

    emit(trackDropped(name));
}

void WVisualWaveform::setup(QDomNode node)
{
    // Colors
    colorBack.setNamedColor(WWidget::selectNodeQString(node, "BgColor"));
    m_pVisualController->setBackgroundColor(colorBack);
    colorSignal.setNamedColor(WWidget::selectNodeQString(node, "SignalColor"));
    colorHfc.setNamedColor(WWidget::selectNodeQString(node, "HfcColor"));
    colorCue.setNamedColor(WWidget::selectNodeQString(node, "CueColor"));
    colorMarker.setNamedColor(WWidget::selectNodeQString(node, "MarkerColor"));
    colorBeat.setNamedColor(WWidget::selectNodeQString(node, "BeatColor"));
    colorFisheye.setNamedColor(WWidget::selectNodeQString(node, "FisheyeColor"));

    // Set position
    QString pos = WWidget::selectNodeQString(node, "Pos");
    int x = pos.left(pos.find(",")).toInt();
    int y = pos.mid(pos.find(",")+1).toInt();
    move(x,y);

    // Size
    QString size = WWidget::selectNodeQString(node, "Size");
    x = size.left(size.find(",")).toInt();
    y = size.mid(size.find(",")+1).toInt();
    setFixedSize(x,y);
}

bool WVisualWaveform::eventFilter(QObject *o, QEvent *e)
{
    // Handle mouse press events
    if (e->type() == QEvent::MouseButtonPress)
    {
        QMouseEvent *m = (QMouseEvent *)e;
        // Store current position of mouse pointer
	m_iButtonState = m->button();
	m_StartPos = m->pos();
        if (m_iButtonState==Qt::LeftButton)
        {
	    m_Speed = 64.;
	    qDebug( "SPEED: RESET" );
            emit(valueChangedLeftDown(m_Speed));
        } else 
	// Real Turntable behaviour code
	if (m_iButtonState==Qt::RightButton)
	{
	    // Start the speed wear off timer if it isn't running already
	    if (!m_iSpeedTimerID) {
		    m_iSpeedTimerID = startTimer(10);
		    qDebug("SPEED: START TIMER");
	    }
	}
/*
        else if (m->button()==Qt::RightButton)
        {
            // Toggle fish eye mode on each channel associated
            VisualChannel *c;
            for (c=m_qlList.first(); c; c=m_qlList.next())
                c->toggleFishEyeMode();
        }
*/
    }
    else if (e->type() == QEvent::MouseMove)
    {
	QMouseEvent *m = (QMouseEvent *)e;
        // Only process mouse move if it was initiated by a left click
        if (m_iButtonState==Qt::LeftButton)
        {
            m_Speed = 64.+(double)(m->x()-m_StartPos.x())/m_GrabSensitivity;
            if (m_Speed<0.)
                m_Speed = 0.;
            else if (m_Speed>127.)
                m_Speed= 127.;
            emit(valueChangedLeftDown(m_Speed));
        }
	if (m_iButtonState==Qt::RightButton) 
	{
            m_Speed += (double)(m->x()-m_StartPos.x())/m_GrabSensitivity;
            if (m_Speed<0.)
                m_Speed = 0.;
            else if (m_Speed>127.)
                m_Speed = 127.;
	    emit(valueChangedLeftDown(m_Speed));
	    QCursor::setPos(mapToGlobal(m_StartPos));
	}
    }
    else if (e->type() == QEvent::MouseButtonRelease)
    {
	if (((QMouseEvent *)e)->button()==Qt::LeftButton) {
		m_Speed = 64.;
	        emit(valueChangedLeftDown(64.));
	}
	m_iButtonState = Qt::NoButton;
    }
    else
    {
        // standard event processing
        return QObject::eventFilter(o,e);
    }
    return true;
}

void WVisualWaveform::slotNewTrack()
{
    // Call each channel associated
    VisualChannel *c;
    for (c=m_qlList.first(); c; c=m_qlList.next())
        c->setupBuffer();
}

VisualChannel *WVisualWaveform::add(const char *group)
{
    VisualChannel *c = new VisualChannel(m_pVisualController, group);

    // Position coding... hack
    //if (m_qlList.isEmpty())
    {
/*
        c->setPosX(-(width()/2));
        c->setLength(800); //width());
        c->setHeight(50); //height());
        c->setZoomPosX(50);
*/
        c->setColorBack((float)colorBack.red()/255., (float)colorBack.green()/255., (float)colorBack.blue()/255.);
        c->setColorSignal((float)colorSignal.red()/255., (float)colorSignal.green()/255., (float)colorSignal.blue()/255.);
        c->setColorHfc((float)colorHfc.red()/255., (float)colorHfc.green()/255., (float)colorHfc.blue()/255.);
        c->setColorCue((float)colorCue.red()/255., (float)colorCue.green()/255., (float)colorCue.blue()/255.);
        c->setColorMarker((float)colorMarker.red()/255., (float)colorMarker.green()/255., (float)colorMarker.blue()/255.);
        c->setColorBeat((float)colorBeat.red()/255., (float)colorBeat.green()/255., (float)colorBeat.blue()/255.);
        c->setColorFisheye((float)colorFisheye.red()/255., (float)colorFisheye.green()/255., (float)colorFisheye.blue()/255.);
    }
    /* else
    {
        c->setPosX(50);
        c->setZoomPosX(50);
    } */

    m_qlList.append(c);
    return c;
}

void WVisualWaveform::initializeGL()
{
    m_pVisualController->init();
    //m_pVisualBackplane = new VisualBackplane();
//    controller->add(m_pVisualBackplane);

    m_Picking.init(m_pVisualController);
}


void WVisualWaveform::paintGL()
{
    // Display stuff
    makeCurrent();
    m_pVisualController->display();
}

void WVisualWaveform::resizeGL(int width, int height)
{
    m_pVisualController->resize((GLsizei)width,(GLsizei)height);
}

void WVisualWaveform::timerEvent(QTimerEvent* e)
{
	if (e->timerId() == m_iTimerID)
	{
		updateGL();
	} else if (e->timerId() == m_iSpeedTimerID) {
		if (m_Speed < 56. && m_Speed > 55.)
			m_Speed = 57.;
		else if (m_Speed < 64.)
		{
			//            m_Speed += .1;
			if (fabs (64. - m_Speed) > 2.0)
				m_Speed += log10 ((64. - m_Speed) / 1.0) / 1.;
			else
				m_Speed += .1;
		}
		else if (m_Speed > 64.)
		{
			//            m_Speed -= .1;
			if (fabs (m_Speed -64.) > 2.0)
				m_Speed -= log10 ((m_Speed - 64.) / 1.5);
			else
				m_Speed -= .1;
		}
		if (fabs (m_Speed - 64.) < .2)
		{
			killTimer (m_iSpeedTimerID);
			m_iSpeedTimerID = 0;
			qDebug ("KILL");
			m_Speed = 64.;
		}
		emit(valueChangedLeftDown(m_Speed));
		qDebug (QString ("m_Speed: %1").arg (m_Speed));
	}
}
