#include <cppunit/TestFixture.h>
#include <cppunit/extensions/HelperMacros.h>

#include "StopWatch.h"
#include "Configuration.h"
#include "Tiles.h"
#include "Ship.h"
#include "GameControlBase.h"
#include "PlayGround.h"


//----------------------------------------------------------------------------
class PlayGroundTest : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(PlayGroundTest);

    CPPUNIT_TEST(testGetGravity);
    CPPUNIT_TEST(testCollidesWith);
    CPPUNIT_TEST(testIsBackgroundBetween);

    CPPUNIT_TEST_SUITE_END();

public:

    //------------------------------------------------------------------------
    void setUp()
    {
        Configuration::getInstance()->searchDataDir();
        Tiles::init();
    }

    //------------------------------------------------------------------------
    void tearDown() 
    {
        Tiles::destroy();
    }


protected:

    //------------------------------------------------------------------------
    void testGetGravity()
    {
        GameControlBase::init("test", "test_collision.xml");

        SDL_Rect r; r.w = 32; r.h = 32;

        PlayGround *playGround = PlayGround::getInstance();
        playGround->addYGravity(1, 1,  50);
        playGround->addYGravity(2, 2,  10);
        playGround->addYGravity(2, 3, -10);
        playGround->addYGravity(3, 2, -10);
        playGround->addYGravity(3, 3,  10);
        playGround->addYGravity(4, 4, -50);

        for (r.y = -36; r.y < 6*16+4; r.y += 2)
        {
            for (r.x = -36; r.x < 6*16+4; r.x += 2)
            {
                CPPUNIT_ASSERT_EQUAL(
                    helperGetGravity(playGround->m_yGravity, r),
                    playGround->getYGravity(r));
            }
        }

        GameControlBase::destroy();
    }

    //------------------------------------------------------------------------
    void testCollidesWith()
    {
        setUp_testCollidesWith();

        Ship *ship = new Ship(ShipSurfaces::S_DEFAULT);
        const SDL_Surface *ss = ship->getSurface();
        const SDL_Rect &sp = ship->getPosition();

        PlayGround *playGround = PlayGround::getInstance();
        const SDL_Surface *pgs = playGround->getMapSurface();
        const SDL_Rect &pgp = playGround->getBoundingBox();


        // Sweep through the playground and use SDL_TOOLS::isCollision()
        // as reference implementation.
        for (Sint16 y = 0; y < pgs->h - ss->h; y++)
        {
            for (Sint16 x = 0; x < pgs->h - ss->h; x++)
            {
                ship->setPosition(x, y);
                CPPUNIT_ASSERT_EQUAL(
                    SDL_TOOLS::isCollision(pgs, pgp, ss, sp),
                    playGround->collidesWith(ship));
            }
        }


        // Test, that the PlayGround's collision check is faster than the
        // reference implementation, if the object lies on background tiles.
        ship->setPosition(32, 32);
        StopWatch w1("SDL_TOOLS::isCollision()");
        StopWatch w2("PlayGround::collidesWith()");

        w1.start();
        for (int i=0; i<1000; i++)
        {
            SDL_TOOLS::isCollision(pgs, pgp, ss, sp);
        }
        w1.stop();

        w2.start();
        for (int i=0; i<1000; i++)
        {
            playGround->collidesWith(ship);
        }
        w2.stop();

        CPPUNIT_ASSERT_EQUAL(1, w1.compare(w2));


        ZAP_POINTER(ship);
        tearDown_testCollidesWith();
    }

    //------------------------------------------------------------------------
    void testIsBackgroundBetween()
    {
        GameControlBase::init("test", "test_background.xml");

        PlayGround *playGround = PlayGround::getInstance();

        // Test start point == end point.
        CPPUNIT_ASSERT( playGround->isBackgroundBetween(1, 1, 1, 1));
        CPPUNIT_ASSERT(!playGround->isBackgroundBetween(0, 0, 0, 0));
        
        // Test around the outermost path.
        CPPUNIT_ASSERT(helperIsBackgroundBetween(3, 1, 8, 1));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 3, 8, 8));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(6, 8, 1, 8));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 6, 1, 1));

        // Test 45 diagonals.
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 7, 2, 8));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(1, 6, 3, 8));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(1, 5, 4, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 4, 5, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 3, 6, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 2, 7, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 1, 8, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(2, 1, 8, 7));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(3, 1, 8, 6));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(4, 1, 8, 5));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(5, 1, 8, 4));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(6, 1, 8, 3));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(7, 1, 8, 2));

        CPPUNIT_ASSERT(!helperIsBackgroundBetween(2, 1, 1, 2));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(3, 1, 1, 3));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(4, 1, 1, 4));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(5, 1, 1, 5));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(6, 1, 1, 6));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(7, 1, 1, 7));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 1, 1, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 2, 2, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 3, 3, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 4, 4, 8));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(8, 5, 5, 8));
        CPPUNIT_ASSERT( helperIsBackgroundBetween(8, 6, 6, 8));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 7, 7, 8));

        // Test other-angle diagonals.
        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 1, 4, 6));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 1, 6, 4));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(5, 1, 1, 2));

        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 1, 3, 4));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 1, 5, 6));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 5, 7, 1));

        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 8, 3, 5));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(8, 8, 5, 3));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(4, 8, 8, 7));

        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 8, 4, 3));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 8, 6, 5));
        CPPUNIT_ASSERT(helperIsBackgroundBetween(1, 4, 2, 8));

        CPPUNIT_ASSERT(!helperIsBackgroundBetween(1, 2, 5, 4));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(7, 1, 5, 5));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(8, 7, 4, 5));
        CPPUNIT_ASSERT(!helperIsBackgroundBetween(2, 8, 4, 4));

        GameControlBase::destroy();
    }

protected:

    int helperGetGravity(const PlayGround::GravityMatrix &matrix,
                         const SDL_Rect &r)
    {
        // This is a reference implementation for PlayGround::getGravity()
        // used for comparision in testGetGravity().
        // Since it iterates through all pixels, it's much slower
        // than the real implementation in PlayGround::getGravity(),
        // which iterates through tiles of constant gravity.
        //
        // Don't modify this implementation!

        int sum = 0;
        for (Sint16 y=0; y<r.h; y++)
        {
            for (Sint16 x=0; x<r.w; x++)
            {
                Sint16 rxx = r.x + x;
                Sint16 ryy = r.y + y;

                Uint16 tileX = 0;
                Uint16 tileY = 0;

                if (rxx >= 0 && (unsigned)rxx < matrix.getXSize()*16)
                {
                    tileX = rxx/16;
                }
                if (ryy >= 0 && (unsigned)ryy < matrix.getYSize()*16)
                {
                    tileY = ryy/16;
                }

                sum += matrix.get(tileX, tileY);
            }
        }

        return r.w == 0 || r.h == 0 ? matrix.get(0, 0) : sum / (r.w * r.h);
    }


    //------------------------------------------------------------------------
    void setUp_testCollidesWith()
    {
        ShipSurfaces::init();
        GameControlBase::init("test", "test_collision.xml");
    }

    //------------------------------------------------------------------------
    void tearDown_testCollidesWith()
    {
        GameControlBase::destroy();
        ShipSurfaces::destroy();
    }


    //------------------------------------------------------------------------
    bool helperIsBackgroundBetween(Uint16 x1, Uint16 y1, Uint16 x2, Uint16 y2)
    {
        PlayGround *playGround = PlayGround::getInstance();

        bool rc1 = playGround->isBackgroundBetween(x1, y1, x2, y2);
        bool rc2 = playGround->isBackgroundBetween(x2, y2, x1, y1);

        CPPUNIT_ASSERT_EQUAL(rc1, rc2);

        return rc1;
    }
};

CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(PlayGroundTest, "PlayGround");
