/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Builds a "dot" file from a dump of LSP's output from a Mobile Mesh
 * Router and then calls "dotneato" to convert the dot file to a GIF.  A
 * "dot" file contains a description of a graph using the "dot"
 * language.  The dot language was created by ATT; more can be learned
 * at www.graphviz.org.
 *
 * The Mobile Mesh Router can be configured to output its entire link
 * state database when it updates routing tables. The LSP's in the
 * database are written to a well known Unix socket
 * (eg. mmrp-t20470). Client programs may register to hear link state
 * database outputs.
 *
 * This class acts as a client and registers to hear link state
 * databases and converts them into a "dot" file which describes the
 * network graph. To make the network diagram more intuitive, all the
 * interfaces running MMRP on a single platform are "clustered"
 * together in what "dot" calls a subgraph. For platforms with only a
 * single MMRP interface, no clustering is done.
 *
 * In general, each link has a metric associated with it and different
 * links can have different metrics. Thus, in some circumstances we
 * may want to see all links and there corresponding metrics. However,
 * the diagram can become very cluttered; so, we optionally omit the
 * metric. Also, if we omit the metric, then we are free to combine a
 * pair of uni-directional links from A to B and B to A into a single
 * bi-directional link between A and B.
 *
 * A simple GUI can then display the "dot" file or the corresponding
 * GIF file.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id$
 *   
 */
#ifndef __ToDot_h
#define __ToDot_h 

#include <fstream.h>
#include <strstream.h>
#include <string.h>
#include <signal.h>
#include <stdio.h>
#include <list.h>
#include <set.h>
#include <map.h>
#include <algo.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <net/route.h>

#include <MmMsg.h>
#include <UtDebug.h>
#include <UtString.h>
#include <UtTime.h>
#include <UtUnixSocket.h>
#include <UtUtil.h>


/* 
 List nodes:
   Lsp's with a single interface:
      "interfaceAddr";
   Lsp's with multiple interfaces:
      "subgraph cluster_<uid> {"
      "   interfaceAddr1;      "
      "   interfaceAddrN;
      "}                       "

 List edges:
   Uni-directional:
       "txInterfaceAddr -> rxInterfaceAddr; "
   Bi-directional:
       "interfaceAddrA -> interfaceAddrB; "

*/

class Link {
  unsigned int cTxAddr;
  unsigned int cRxAddr;
  unsigned int cMetric;
public:
  Link(unsigned int txAddr, unsigned int rxAddr, unsigned int metric) :
    cTxAddr(txAddr), cRxAddr(rxAddr), cMetric(metric) {};
  Link() : cTxAddr(0), cRxAddr(0), cMetric(0) {};
  unsigned int TxAddr() const { return cTxAddr; }
  unsigned int RxAddr() const { return cRxAddr; }
  unsigned int Metric() const { return cMetric; }
  
  unsigned int& TxAddr() { return cTxAddr; }
  unsigned int& RxAddr() { return cRxAddr; }
  unsigned int& Metric() { return cMetric; }
  
  int operator ==(const Link& rhs) const {
    if((cTxAddr  == rhs.cTxAddr)  &&
       (cRxAddr  == rhs.cRxAddr)) {
	 return(1);
       }

    return(0);
  }
  
  int operator <(const Link& rhs) const {
    if(cTxAddr <  rhs.cTxAddr)     return(1);
    if(cTxAddr != rhs.cTxAddr) return(0);
    
    // Here if the same tx addr
    if(cRxAddr <  rhs.cRxAddr)     return(1);
    
    return(0);
  }
  
  String Dump() const {
    String s;
    s += String("txAddr=") + DumpAddr(cTxAddr) + " ";
    s += String("rxAddr=") + DumpAddr(cRxAddr) + " ";
    s += String("metric=") + DumpAddr(cMetric) + " ";
    s += "\n";
    return(s);
  }
};



class ToDot {
  UnixSocket* cLspSocket;
  String      cLspSocketName;
  bool        cExpectReply;
  String      cOutputDotFileName;
  String      cOutputGifFileName;
  bool        cFirstDatabase;
  bool        cShowMetrics;
  bool        cShowAge;
  bool        cShowSeq;
  bool        cShowExtRoutes;

  fd_set cFds;
  int cFdLimit;
  
  typedef set<Link, less<Link> > LinkSet;
  LinkSet cLinks;

  // Keys are interface addresses, values are router id's
  typedef map<unsigned int, unsigned int, less<unsigned int> > IdMap;
  IdMap cIdMap;

  typedef set<ExtRoute, less<ExtRoute> > ExtRouteSet;


  // Keys are router Id's, values are Lsp's generated by corresponding router
  typedef map< unsigned int, Lsp, less< unsigned int> > LspMap;
  LspMap cLspMap;

  
  /* When we start up, the router may have a database of LSP's.
   * We can get the database by asking for it.
   */
  void QueryRouter() {
    String err = String("mmrp is not listening on Unix socket. ") + 
      "Please restart mmrp.";
    
    // Send a list request to mmrp program to get its Lsp database
    String s = "l\n";
    UnixDatagram q(s,cLspSocketName);
    if(!cLspSocket->Send(q)) {
      Report::Error(err);
    }
  }

  /* A crude test to determine if the router is still operational
   * Send a ping request...we harvest ping replies asynchronously
   * later. If we didn't hear a ping reply since last time we checked, we exit.
   */
  void CheckRouter() {
    String err = String("mmrp is not listening on Unix socket. ") + 
      "Please restart mmrp.";

    // Send a list request to mmrp program to get its Lsp database
    String s = "q\n";
    UnixDatagram q(s,cLspSocketName);
    if(!cLspSocket->Send(q)) {
      Report::Error(err);
    }
    
    if(cExpectReply) {
      Report::Error(err);
    }
    cExpectReply = true;
  }


  
  /* Determine what type of message this is, and handle it */
  void HandleReceive(UnixDatagram& dg) {
    String d = dg.GetData();
    
    if(d.length() < 2) {
      Report::Warn("ToDot::HandleReceive() - Invalid message, discarding.");
      return;
    }


    unsigned char type = d[0];
    switch(type) {
    case '{': 
      {
	if(Debug("ToDot")) {
	  Report::Debug("ToDot::HandleReceive() - got '{' \n");
	}
	// This marks the beginning of an LSP database dump
	cLspMap.clear();
	break;
      }
    case '}':
      {
	if(Debug("ToDot")) {
	  Report::Debug("ToDot::HandleReceive() - got '}' \n");
	}
	// This marks the end of an LSP database dump
	BuildDotFile();
	break;
      }
    case 'r':
      {
	// Ping reply
	cExpectReply = false;
	break;
      }
    default:
      {
	if(type == MsgTypes::Version) {
	  if(d[1] == MsgTypes::tLsp) {
	    Lsp lsp;
	    if(lsp.Decode(d)) {
	      cLspMap[lsp.RouterId()] = lsp;
	      if(Debug("ToDot")) {
		Report::Debug(String("ToDot::HandleReceive() - got lsp: \n") + lsp.Dump());
	      }
	    }
	    else {
	      Report::Warn(String("ToDot::HandleReceive() - unable to decode lsp! \n"));
	    }
	    break;
	  }
	}
	Report::Warn("ToDot::HandleReceive() - Invalid message type, discarding.");
      }
    }
  }
  
  String DumpLinks(const LinkSet& links) {
    LinkSet::const_iterator it;
    int i = 0;
    String s;
    for(it = links.begin(), i=0; it != links.end(); it++, i++) {
      s += String("\n[") + String(i) + "] " + (*it).Dump();
    }
    return(s);
  }

  enum LinkType { tFound, tStale, tStable }; 

  void WriteLink(ostream& f, Link& link, LinkType t) {
    String color;
    switch(t) {
    case tFound:
      color = "color=green4";
      break;
    case tStale:
      color = "color=red style=dashed";
      break;
    case tStable:
      color = "color=black";
    }
    if(cFirstDatabase) {
      color = "color=black";
    }

    String tId = (cIdMap[link.TxAddr()] ? DumpAddr(cIdMap[link.TxAddr()]) : DumpAddr(link.TxAddr()));
    String rId = (cIdMap[link.RxAddr()] ? DumpAddr(cIdMap[link.RxAddr()]) : DumpAddr(link.RxAddr()));
    String label;
    if(cShowMetrics) label += String(" ") + String((int)link.Metric()) + " ";
    if(label.length()) label = String("label=") + label;
    String arrow = " arrowsize=1.8 arrowhead=normal ";
    f << "   \"" << tId << "\":\"" << DumpAddr(link.TxAddr()) << "\" -> \"" << 
      rId << "\":\"" << DumpAddr(link.RxAddr()) << "\" [ " << color << " " << label << arrow << "]; \n";
  }

  void BuildDotFile() {
    LinkSet links;
    
    cIdMap.clear();

    // Iterate across Lsp map and build a set of links and populate router id map
    LspMap::iterator it;
    for(it = cLspMap.begin(); it != cLspMap.end(); it++) {
      Lsp& lsp = (*it).second;
      list<Lsp::Face>& faceList = lsp.FaceList();
      list<Lsp::Face>::iterator ii;
      for(ii = faceList.begin(); ii != faceList.end(); ii++) {
	Lsp::Face& face = (*ii);
	cIdMap[face.Addr()] = lsp.RouterId();
	list<Neighbor>& neighbors = face.NeighborList();
	list<Neighbor>::iterator jj;
	for(jj = neighbors.begin(); jj != neighbors.end(); jj++) {
	  Neighbor& neighbor = (*jj);
	  links.insert(Link(neighbor.Addr(),face.Addr(),neighbor.Metric()));
	}
      }
    }

    if(Debug("ToDot")) {
      DumpLinks(links);
    }
    

    // Figure out which links are new by performing a set difference:
    //   found = links - cLinks
    list<Link> found;
    back_insert_iterator< list<Link> > fi(found);
    set_difference(links.begin(),links.end(),
		   cLinks.begin(),cLinks.end(), 
		   fi);


    // Figure out which links are now stale by performing a set difference:
    //   stale = cLinks - links
    list<Link> stale;
    back_insert_iterator< list<Link> > si(stale);
    set_difference(cLinks.begin(),cLinks.end(),
		   links.begin(),links.end(), 
		   si);


    // Now compute stable links (links INTERSECT cLinks)
    list<Link> stable;
    back_insert_iterator< list<Link> > gi(stable);
    set_intersection(cLinks.begin(),cLinks.end(),
		     links.begin(),links.end(), 
		     gi);
    
    
    
    {
      ofstream f(cOutputDotFileName.str());
      f << "digraph mmrp_topology { \n";
      f << " bgcolor=LemonChiffon2; \n";
      f << " rankdir=LR; \n";
      //      f << " node [color=white style=filled fontname=arialbd]; \n";
      f << " node [color=black style=solid shape=record ]; \n";
      
      // Iterate across Lsp map and build nodes
      for(it = cLspMap.begin(); it != cLspMap.end(); it++) {
	Lsp& lsp = (*it).second;
	list<Lsp::Face>& faceList = lsp.FaceList();
	list<ExtRoute>& extRouteList = lsp.ExtRouteList();
	String id =  DumpAddr(lsp.RouterId());
	String label;
	if(cShowAge) {
	  if(label.length()) 
	    label += String(" | ");
	  label += String(" Age: ") + String((int)lsp.Age());
	}
	if(cShowSeq) {
	  if(label.length()) 
	    label += String(" | ");
	  label += String(" Seq: ") + String((int)lsp.Seq());
	}
	if(cShowExtRoutes) {
	  list<ExtRoute>::const_iterator k;
	  for(k = extRouteList.begin(); k != extRouteList.end(); k++) {
	    String addr = DumpAddr((*k).Addr());
	    String netmask = DumpAddr((*k).Netmask());
	    if(label.length()) 
	      label += String(" | ");
	    label += " Ext: " + addr  + "\\n" + netmask + "  ";
	  }
	}
	// Create a node for each local interface
	list<Lsp::Face>::iterator ii;
	for(ii = faceList.begin(); ii != faceList.end(); ii++) {
	  Lsp::Face& face = (*ii);
	  String addr = DumpAddr(face.Addr());
	  if(label.length()) 
	    label += String(" | ");
	  label += String("  <") + addr  + "> \\n " + addr + " \\n  ";
	}
	f << "  \"" << id << "\" [label=\"" << label << "\"]; \n";
      }


      list<Link>::iterator ii;
      for(ii = found.begin(); ii != found.end(); ii++) {
	WriteLink(f,(*ii),tFound);
      }
      
      for(ii = stale.begin(); ii != stale.end(); ii++) {
	WriteLink(f,(*ii),tStale);
      }
      

      for(ii = stable.begin(); ii != stable.end(); ii++) {
	WriteLink(f,(*ii),tStable);
      }
      
      f << "} \n";
    }
    String cmd = String("dotneato -Tgif -o ") + cOutputGifFileName + " " + cOutputDotFileName;
    int n = system(cmd.str());
    if((n == -1) || (n == 127)) {
      Report::Error("Could not run 'dot' command...check to ensure 'dot' is the PATH.\n");
    }

    // Update link set
    cLinks = links;
    cFirstDatabase = false;
    if(Debug("ToDot")) {
      Report::Debug("BuildDotFile!");
    }
  }
  
public:
  
  ToDot(unsigned short port, const String& debugFile, bool showMetrics, 
	bool showAge, bool showSeq, bool showExtRoutes) 
  {

    FD_ZERO(&cFds);
    cFdLimit = 0;
    
    
    cLspSocket = new UnixSocket();
    cLspSocketName = String("mmrp-t") + String((int)port);
    cOutputDotFileName = String("topology-") + String((int)port) + String(".dot");
    cOutputGifFileName = String("topology-") + String((int)port) + String(".gif");
    cFirstDatabase = true;
    cShowMetrics = showMetrics;
    cShowAge     = showAge;
    cShowSeq     = showSeq;
    cShowExtRoutes = showExtRoutes;

    FD_SET(cLspSocket->Fd(),&cFds);
    if(cLspSocket->Fd() >= cFdLimit) cFdLimit = cLspSocket->Fd() + 1;
    
    cExpectReply = false;

    if(debugFile.length()) {
      Debug::Load(debugFile);
    }
  }

  ~ToDot() {

    // Delete our Unix sockets
    delete(cLspSocket);
    cLspSocket = 0;
    cLinks.clear();
  }

  /* Main event loop...
   *
   */
  void Run() {
    double checkPeriod = 3.0;
    Time now = Time::Now();
    Time checkRouterTime = now + checkPeriod;  

    // Ask for the Lsp database immediately 
    QueryRouter();

    while(1) {
      
      Time timeleft = checkRouterTime - Time::Now();


      fd_set fds = cFds;      
      timeval tv = timeval(timeleft);
      int n;
      if((n=select(cFdLimit, &fds, 0, 0, &tv)) == -1) {
	char buf[1024];
	sprintf(buf,"ToDot::Run() - select failed: %s",strerror(errno));
	Report::Error(buf);
      }
      
      if(n > 0) {
	if(FD_ISSET(cLspSocket->Fd(),&fds)) {
	  UnixDatagram dg = cLspSocket->Recv();
	  HandleReceive(dg);
	}
      }

      Time now = Time::Now();
      if(double(checkRouterTime) <= double(now)) {
	  checkRouterTime = checkRouterTime + checkPeriod;
	  CheckRouter(); // Covenient place to do this
	  Debug::Update();  // Convenient place to check if debug flags have changed
      }
    }
  }
  
  
};

#endif
