/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Class for reading and writing an ethertap
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id$
 */
#ifndef _EtherTap_h
#define _EtherTap_h

#include <UtReport.h>
#include <UtString.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <netinet/ip.h>		// for struct iphdr, sockaddr_in
#include <sys/ioctl.h>		// for ioctl(), ...
#include <netinet/if_ether.h>	// for struct ethhdr, ...
#include <net/if.h>		// for struct ifreq
#include <fcntl.h>		// for O_RDWR, ...


class EtherTap { 
  
private:
  int cFd;
  struct ethhdr cHwHeader;	// Hardware header (needed?)
  int cHeaderPad;		// Padding before header...
  struct sockaddr_in cIfAddr;	// Interface's IP address...
  String cDevice;

  // Open() - Open and initialize ethertap device...
  bool Open() {
 
    do {			// Control only, never loops...
      cFd = open(cDevice.c_str(), O_RDWR | O_NONBLOCK);
      if(cFd < 0) break;
      
      // Now that device open, find it's interface address...
      int s = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
      struct ifreq ifr;

      sprintf(ifr.ifr_ifrn.ifrn_name, "%s", "tap0");
      if(ioctl(s, SIOCGIFADDR, &ifr)) { close(s); break; }
      memset(&cIfAddr, 0, sizeof(cIfAddr));
      struct sockaddr sa = ifr.ifr_ifru.ifru_addr;
      if(sa.sa_family == AF_INET) {
	struct sockaddr_in *sin = (struct sockaddr_in *)&sa;
	cIfAddr = *sin;
      }
      // Get tap's hardware addresses...
      memset(&cHwHeader, 0, sizeof(cHwHeader));
      if(ioctl(s, SIOCGIFHWADDR, &ifr) < 0) { close(s); break; }
      close(s);
      
      memcpy(&cHwHeader.h_source, &ifr.ifr_ifru.ifru_hwaddr.sa_data, ETH_ALEN);
      memcpy(&cHwHeader.h_dest, &cHwHeader.h_source, ETH_ALEN);
      cHwHeader.h_proto = ETH_P_PPP_MP;  
      
      // Compute the hardware header padding...
      // Appears to be word aligned, right justified for efficiency...
      cHeaderPad = ((sizeof(cHwHeader) + 4) & ~0x3) - sizeof(cHwHeader);
      
      // Happy return...
      return(true);
      
    } while(false);

    return(false);
  }


public:
  EtherTap(const String& device = "/dev/tap0")
    : cFd(-1), cDevice(device)
    { 
      Open();
    }

  ~EtherTap() {
    if(cFd >= 0) {
      close(cFd);
      cFd = -1;
    }
  }

  int Fd() const { return cFd; }

  // Write() - Send string into kernel
  void Write(const String &s) {
    char buf[2048];

    // Stick reasonable hardware header on packet...
    struct ethhdr *hwh = (struct ethhdr *)(buf + cHeaderPad);
    memcpy(hwh, &cHwHeader, sizeof(cHwHeader));

    struct iphdr *ipHdr = (struct iphdr *)s.c_str();
    if(IN_MULTICAST(ntohl(ipHdr->daddr)))
      ETHER_MAP_IP_MULTICAST(&ipHdr->daddr, hwh->h_dest);

    s.Export(buf + cHeaderPad + sizeof(cHwHeader));
    int length = s.length() + cHeaderPad + sizeof(cHwHeader);
    int len = write(cFd, buf, length);
    if(len < 0) {
      Report::Error(String("EtherTap::Write() - ") + strerror(errno));
    }
    if(len < length) {
      Report::Error(String("EtherTap::Write() - short write!"));
    }
  }

  // Read() - Reads one input unit from the kernel
  String Read() {
    String out;
    char buf[2048];		// Hmmm...  maybe from the heap?
    int len = read(cFd, buf, sizeof(buf));

    if(len < 0) {
      Report::Error(String("EtherTap::Read() - ") + strerror(errno));
    }
    if(len > 0) {
      int hlen = cHeaderPad + sizeof(cHwHeader);
      out = String(buf, hlen, len - hlen);
    }
    return(out);
  }

};

#endif
