/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
/* other libs */
#include <db.h>
/* local */
#include <libbttracker.h>

/* Private Functions */

static int btt_tracker_setup_env(btt_tracker* tracker, DB_ENV* dbenv) {
    int ret = 0;

    if ((ret = dbenv->set_data_dir(dbenv, tracker->db.dir)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_data_dir: %s",
            tracker->db.dir
        );
        goto err;
    }

    if ((ret = dbenv->set_tmp_dir(dbenv, tracker->db.dir)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_tmp_dir: %s",
            tracker->db.dir
        );
        goto err;
    }
 
    if ((ret = dbenv->set_cachesize(dbenv, 0, tracker->db.cache, 0)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_cachesize(%d)",
            tracker->db.cache
        );
        goto err;
    }

    if ((ret = dbenv->set_tx_max(dbenv, 2048)) != 0) {
        dbenv->err(dbenv, ret, "btt_tracker_setup_env(): set_tx_max(2048)");
        goto err;
    }

    if ((ret = dbenv->set_lk_detect(dbenv, DB_LOCK_DEFAULT)) != 0) {
        dbenv->err(
            dbenv, ret,
            "btt_tracker_setup_env(): set_lk_detect(DB_LOCK_DEFAULT)"
        );
        goto err;
    }

    /* need one locker for every
     * db handle / transaction / cursor open at any given time
     */
    if((ret = dbenv->set_lk_max_lockers(dbenv, 1024)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_lk_max_locks(1024)"
        );
        goto err;
    }

    /* if we get lots of peers in a torrent this could be trouble */
    if((ret = dbenv->set_lk_max_locks(dbenv, 2048)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_lk_max_locks(2048)"
        );
        goto err;
    }

    /* page size is average 4 records */
    if((ret = dbenv->set_lk_max_objects(dbenv, 65536)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_setup_env(): set_lk_max_objects(65536)"
        );
        goto err;
    }

    if((ret = dbenv->set_verbose(dbenv, DB_VERB_RECOVERY, 1)) != 0) {
        dbenv->err(
            dbenv, ret,
            "btt_tracker_setup_env(): set_verbose(DB_VERB_RECOVERY)"
        );
        goto err;
    }
 
    dbenv->set_errpfx(dbenv, "libbtt");
 
    err:
    return ret;
}

static int btt_tracker_connect_db_env(
    btt_tracker* tracker, int master
) {
    DB_ENV* dbenv = NULL;
    int ret = 0;
    u_int32_t flags = 0;
    char** dummy = NULL;

    if(master)
        flags = DB_RECOVER;
  
    /* Create an environment & initialize it for additional error reporting */
    if ((ret = db_env_create(&dbenv, 0)) != 0) {
        fprintf(
            stderr,
            "btt_tracker_connect_db_env(): db_env_create: %s\n",
            db_strerror(ret)
        );
        fflush(stderr);
        return 0;
    }

    if ((ret = btt_tracker_setup_env(tracker, dbenv)) != 0)
        goto err;

    if ((ret = dbenv->set_flags(dbenv, DB_LOG_AUTOREMOVE, 1)) != 0) {
        dbenv->err(
            dbenv, ret,
            "btt_tracker_connect_db_env(): set_flags(DB_LOG_AUTOREMOVE)"
        );
        goto err;
    }

    flags = flags | DB_INIT_TXN | DB_INIT_LOG;

    /* Open the environment with full transactional support. */
 
    if ((ret = dbenv->open(
        dbenv, tracker->db.dir, 
        DB_CREATE | DB_INIT_LOCK | DB_INIT_MPOOL | flags, S_IRUSR | S_IWUSR
    )) != 0) {
        dbenv->err(
            dbenv, ret,
            "btt_tracker_connect_db_env(): dbenv->open(%s)",
            tracker->db.dir
        );
        goto err;
    }

    if ((ret = dbenv->log_archive(dbenv, &dummy, DB_ARCH_REMOVE)) != 0) {
        dbenv->err(
            dbenv, ret, "btt_tracker_connect_db_env(): log_archive()"
        );
        goto err;
    }

    tracker->db.env = dbenv;
    return 1;

    err:
 
    dbenv->close(dbenv, 0);
    return 0;
}

static int btt_tracker_connect_db_index(
    btt_tracker* tracker,
    DB_TXN* ptxn,
    char* table,
    DB** tableref,
    DB* source,
    int (*callback)(DB *, const DBT *, const DBT *, DBT *),
    int master
) {
    DB *sdbp = NULL;
    DB_TXN *txn = NULL;
    int ret = 0;
    unsigned int ndel = 0;
    int dbflags = 0;
    DBTYPE dbtype;

    if(tracker->c->flags & BTT_TRACKER_BTREE) {
        dbtype = DB_BTREE;
        dbflags = DB_REVSPLITOFF;
    } else {
        dbtype = DB_HASH;
        dbflags = DB_REVSPLITOFF;
    }
 
    if(
        (ret = tracker->db.env->txn_begin(tracker->db.env, ptxn, &txn, 0))
        != 0
    ) {
        tracker->db.env->err(
            tracker->db.env, ret,
            "btt_tracker_connect_db_index(%s): txn_begin", table
        );
        goto err;
    }

    if ((ret = db_create(&sdbp, tracker->db.env, 0)) != 0) {
        tracker->db.env->err(
            tracker->db.env, ret,
            "btt_tracker_connect_db_index(%s): second db_create", table
        );
        goto err;
    }

    if ((ret = sdbp->set_pagesize(sdbp, 512)) != 0) {
        sdbp->err(
            sdbp, ret, "btt_tracker_connect_db_index(%s): set_pagesize", table
        );
        goto err;
    }

    if ((ret = sdbp->set_flags(sdbp, DB_DUP | DB_DUPSORT | dbflags)) != 0) {
        sdbp->err(
            sdbp, ret, "btt_tracker_connect_db_index(%s): set_flags", table
        );
        goto err;
    }

    if ((ret = sdbp->open(
            sdbp, txn, table, table, dbtype, DB_CREATE, 0600
    )) != 0) {
        sdbp->err(sdbp, ret, "btt_tracker_connect_db_index(%s): open", table);
        goto err;
    }

    if(master)
        if((ret = sdbp->truncate(sdbp, txn, &ndel, 0)) != 0) {
            sdbp->err(sdbp, ret, "btt_tracker_connect_db_index(): truncate");
            goto err;
        }

    /* Associate the secondary with the primary. */
    if (
        (ret = source->associate(source, txn, sdbp, callback, DB_CREATE)) != 0
    ) {
        source->err(
            sdbp, ret, "btt_tracker_connect_db_index(%s): associate", table
        );
        goto err;
    }

    if (txn) {
        if ((ret = txn->commit(txn, 0)) != 0) {
            tracker->db.env->err(
                tracker->db.env, ret,
                "btt_tracker_connect_db_index(%s): commit", table
            );
            goto err;
        }
    }

    *tableref = sdbp;
  
    return 1;

    err:
 
    if(sdbp)
        sdbp->close(sdbp, 0);
 
    if(txn)
        if ((ret = txn->abort(txn)) != 0)
            tracker->db.env->err(
                tracker->db.env, ret, "btt_tracker_connect_db_index(%s): abort",
                table
            );

    return 0;
}

static int btt_tracker_connect_db_table(
    btt_tracker* tracker, DB_TXN* ptxn, char* table, DB** tableref, int inflags,
    int truncate
) {
    DB* dbp = NULL;
    int ret = 0;
    unsigned int ndel = 0;
    DB_TXN* txn = NULL;
    DBTYPE dbtype = (tracker->c->flags & BTT_TRACKER_BTREE) ? DB_BTREE : DB_HASH;
    int dbflags = inflags;
    
    if(dbtype == DB_BTREE) {
        dbflags = dbflags | DB_REVSPLITOFF;
    }

    if(!(dbflags & DB_DUP)) {
        dbflags = dbflags | DB_RECNUM;
    }
  
    if ((ret = tracker->db.env->txn_begin(
        tracker->db.env, ptxn, &txn, 0
    )) != 0) {
        tracker->db.env->err(
            tracker->db.env, ret,
            "btt_tracker_connect_db_table(%s): txn_begin", table
        );
        goto err;
    }
	
    /* Create and initialize database object, open the database. */
    if ((ret = db_create(&dbp, tracker->db.env, 0)) != 0) {
        tracker->db.env->err(
            tracker->db.env, ret,
            "btt_tracker_connect_db_table(%s): db_create", table
        );
        goto err;
    }
 
    if ((ret = dbp->set_pagesize(dbp, 2048)) != 0) {
        dbp->err(
            dbp, ret, "bt_tracker_connect_db_table(%s): set_pagesize(2048)",
            table
        );
        goto err;
    }

    if(dbflags)
        if ((ret = dbp->set_flags(dbp, dbflags)) != 0) {
            dbp->err(
                dbp, ret, "btt_tracker_connect_db_table(%s): set_flags", table
            );
            goto err;
        }

    if ((ret = dbp->open(
        dbp, txn, table, table, dbtype, DB_CREATE, 0664)) != 0
    ) {
        dbp->err(dbp, ret, "btt_tracker_connect_db_table(%s): open", table);
        goto err;
    }

    if(truncate)
        if((ret = dbp->truncate(dbp, txn, &ndel, 0)) != 0) {
            dbp->err(dbp, ret, "btt_tracker_connect_db_table(): truncate");
            goto err;
        }
    
    if ((ret = txn->commit(txn, 0)) != 0) {
        tracker->db.env->err(
            tracker->db.env, ret, "btt_tracker_connect_db_table(%s): commit",
            table
        );
        goto err;
    }

    *tableref = dbp;
    return 1;

    err:

    if(dbp)
        dbp->close(dbp, 0);

    if(txn)
        if ((ret = txn->abort(txn)) != 0)
            tracker->db.env->err(
                tracker->db.env, ret,
                "btt_tracker_connect_db_table(%s): abort", table
            );

    return 0;
}

/* Public Functions */

int btt_tracker_connect_db(btt_tracker* tracker, int master) {
    DB_TXN* txn = NULL;
    int ret = 0;

    if(tracker->db.open) {
        fprintf(
            stderr,
            "btt_tracker_connect_db(): "
            "Attempted to connect to database while already connected!\n"
        );
        return 0;
    }
 
    if(btt_tracker_connect_db_env(tracker, master)) {
        if ((ret = tracker->db.env->txn_begin(
            tracker->db.env, NULL, &txn, 0
        )) != 0) {
            tracker->db.env->err(
                tracker->db.env, ret, "btt_tracker_connect_db: txn_begin"
            );
            goto err;
        }
  
        if(!btt_tracker_connect_db_table(
            tracker, txn, "hashes", &(tracker->db.hashes), 0, 0
        ))
            goto err;
  
        if(!btt_tracker_connect_db_table(
            tracker, txn, "peers", &(tracker->db.peers), 0, master
        ))
            goto err;

        if(!btt_tracker_connect_db_index(
            tracker, txn, "peersindex", &(tracker->db.index),
            tracker->db.peers, btt_peer_crossref, master
        ))
            goto err;
    } else {
        fprintf(
            stderr,
            "btt_tracker_connect_db(): Failed to create a database environment!\n"
        );
        fflush(stderr);
        goto err;
    }

    if ((ret = txn->commit(txn, 0)) != 0) {
        tracker->db.env->err(
            tracker->db.env, ret, "btt_tracker_connect_db: commit"
        );
        goto err;
    }

    tracker->db.open = 1;
    return 1;
 
    err:

    if(txn)
        if ((ret = txn->abort(txn)) != 0)
            tracker->db.env->err(
                tracker->db.env, ret, "btt_tracker_connect_db: abort"
        );

    return 0; 
}

int btt_tracker_connect_mem(btt_tracker* tracker, int master) {
    void* shm;
    int rv;
 
    if((shm = apr_shm_baseaddr_get(tracker->m))) {
        tracker->c = (btt_tracker_config*)shm;
        tracker->s = (btt_tracker_stats*)(shm + sizeof(btt_tracker_config));
        if(master) {
            *(tracker->c) = new_btt_tracker_config;
            *(tracker->s) = new_btt_tracker_stats;
            strcpy(tracker->c->db_dir, tracker->homedir);
        }
        tracker->s->num_children++;
        rv = 1;
    } else {
        fprintf(
            stderr,
            "btt_tracker_connect_mem: Failed to find shared memory segment!\n"
        );
        fflush(stderr);
        rv = 0;
    }

    return rv;
}

int btt_tracker_connect(btt_tracker* tracker, int master) {
    if(btt_tracker_connect_mem(tracker, master)) {
        strcpy(tracker->db.dir, tracker->homedir);
        if(btt_tracker_connect_db(tracker, master)) {
            if(btt_tracker_refresh_stats(tracker)) {
                return 1;
            } else {
                btt_tracker_disconnect_db(tracker);
                btt_tracker_disconnect_mem(tracker);
            }
        } else {
            btt_tracker_disconnect_mem(tracker);
        }
    }
 
    return 0;
}
