/* 
 *   Creation Date: <2002/10/23 20:26:40 samuel>
 *   Time-stamp: <2004/03/21 21:48:18 samuel>
 *   
 *	<splash.c>
 *	
 *	Mac-on-Linux splash screen
 *   
 *   Copyright (C) 2002, 2004 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "of.h"
#include "osi_calls.h"
#include "ofmem.h"
#include "boothelper_sh.h"
#include "fs.h"
#include "pseudofs_sh.h"
#include "prom.h"
#include "video_sh.h"

static struct {
	int		has_video;
	osi_fb_info_t	fb;
	ulong		*pal;		/* 256 elements */
} video;


int
video_get_res( int *w, int *h )
{
	if( !video.has_video ) {
		*w = *h = 0;
		return -1;
	}
	*w = video.fb.w;
	*h = video.fb.h;
	return 0;
}

static void
startup_splash( void )
{
	int s, i, y, x, dx, dy;
	int width, height;
	char *pp, *p;
	char buf[64];
	file_desc_t fd;
	
	/* only draw logo in 24-bit mode (for now) */
	if( video.fb.depth < 15 )
		return;
	for( i=0; i<2; i++ ) {
		if( !BootHGetStrResInd( "bootlogo", buf, sizeof(buf), 0, i) )
			return;
		*(!i ? &width : &height) = atol(buf);
	}
	if( (s=width * height * 3) > 0x20000 )
		return;

	if( (fd=pseudo_open("bootlogo")) ) {
		p = malloc( s );
		if( pseudo_read( fd, p, s ) != s )
			printm("bootlogo size error\n");

		dx = (video.fb.w - width)/2;
		dy = (video.fb.h - height)/3;
		
		pp = (char*)video.fb.mphys + dy * video.fb.rb + dx * (video.fb.depth >= 24 ? 4 : 2);
		
		for( y=0 ; y<height; y++, pp += video.fb.rb ) {
			if( video.fb.depth >= 24 ) {
				ulong *d = (ulong*)pp;
				for( x=0; x<width; x++, p+=3, d++ )
					*d = ((int)p[0] << 16) | ((int)p[1] << 8) | p[2];
			} else if( video.fb.depth == 15 ) {
				ushort *d = (ushort*)pp;
				for( x=0; x<width; x++, p+=3, d++ ) {
					int col = ((int)p[0] << 16) | ((int)p[1] << 8) | p[2];
					*d = ((col>>9) & 0x7c00) | ((col>>6) & 0x03e0) | ((col>>3) & 0x1f);
				}
			}
		}
		free( p );
	}
}

static ulong
get_color( int col_ind )
{
	ulong col;
	if( !video.has_video || col_ind < 0 || col_ind > 255 )
		return 0;
	if( video.fb.depth == 8 )
		return col_ind;
	col = video.pal[col_ind];
	if( video.fb.depth == 24 || video.fb.depth == 32 )
		return col;
	if( video.fb.depth == 15 )
		return ((col>>9) & 0x7c00) | ((col>>6) & 0x03e0) | ((col>>3) & 0x1f);
	return 0;
}

void 
draw_pixel( int x, int y, int colind )
{
	char *p = (char*)video.fb.mphys + video.fb.rb * y;
	int color, d = video.fb.depth;

	if( x < 0 || y < 0 || x >= video.fb.w || y >=video.fb.h )
		return;
	color = get_color( colind );

	if( d >= 24 )
		*((ulong*)p + x) = color;
	else if( d >= 15 )
		*((short*)p + x) = color;
	else
		*(p + x) = color;
}

void
fill_rect( int col_ind, int x, int y, int w, int h )
{
	char *pp;
	ulong col = get_color(col_ind);

	if( !video.has_video || x < 0 || y < 0 || x+w > video.fb.w || y+h > video.fb.h )
		return;

	pp = (char*)video.fb.mphys + video.fb.rb * y;
	for( ; h--; pp += video.fb.rb ) {
		int ww = w;
		if( video.fb.depth == 24 || video.fb.depth == 32 ) {
			ulong *p = (ulong*)pp + x;
			while( ww-- )
				*p++ = col;
		} else if( video.fb.depth == 16 || video.fb.depth == 15 ) {
			ushort *p = (ushort*)pp + x;
			while( ww-- )
				*p++ = col;
		} else {
			char *p = (char*)pp + x;
			while( ww-- )
				*p++ = col;
		}
	}
}

void
init_video( void )
{
	mol_phandle_t ph;
	int i, s, size;
	
	if( OSI_GetFBInfo( &video.fb ) ) {
		printm("init_video: No video display\n");
		return;
	}
	if( (ph=prom_find_device_type( "display", 0 )) != -1 ) {
		prom_set_prop( ph, "width", (char*)&video.fb.w, 4 );
		prom_set_prop( ph, "height", (char*)&video.fb.h, 4 );
		prom_set_prop( ph, "depth", (char*)&video.fb.depth, 4 );
		prom_set_prop( ph, "linebytes", (char*)&video.fb.rb, 4 );
		prom_set_prop( ph, "address", (char*)&video.fb.mphys, 4 );
	}
	video.has_video = 1;
	video.pal = malloc( 256 * sizeof(int) );

	s = (video.fb.mphys & 0xfff);
	size = ((video.fb.h * video.fb.rb + s) + 0xfff) & ~0xfff;
	s = video.fb.mphys - s;

	ofmem_claim_phys( video.fb.mphys, size, 0 );
	ofmem_claim_virt( video.fb.mphys, size, 0 );
	ofmem_map( video.fb.mphys, video.fb.mphys, size, -1 );
		
	for( i=0; i<256; i++ )
		set_color( i, i * 0x010101 );
	
	set_color( 254, 0xffffcc );
	fill_rect( 254, 0, 0, video.fb.w, video.fb.h );

	refresh_palette();
	startup_splash();
}

void
set_color( int ind, ulong color )
{
	if( !video.has_video || ind < 0 || ind > 255 )
		return;
	video.pal[ind] = color;

	if( video.fb.depth == 8 )
		OSI_SetColor( ind, color );
}

void
refresh_palette( void )
{
	if( video.fb.depth == 8 )
		OSI_RefreshPalette();
}
