/* dir- and filerequester

   Written by Matthias Hensler
   Copyright WSPse 1999+2000
   eMail: wsp@gmx.de

Created: 2000/01/24
Updated: 2000/02/28
*/


/* Copying:
   This program is free software; you can redistribute it and/or modify it under
   the terms of the GNU Gerneral Public License as published by the Free Soft-
   ware Foundation; either version 2 of License, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILTY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 675 Mass
   Ave, Cambridge, MA 02139, USA.
 */


/* Requester (Dir and File) for ncurses-library - V2
 * Written by Matthias Hensler
 * (C) WSPse 2000
 * Licenced under GNU General Public Licence
 *
 * Interface:
 *  char *req_get_dir (const char *path)
 *  if(path == NULL) -> homedirectory will be used
 *  returnvalue is dirname, or NULL if user quit
 *
 *  char *req_get_file (const char *path)
 *  if(path == NULL) -> homedirectory will be used
 *  returnvalue is  filename with complete path, or NULL if user quit
 *
 * Before calling get_dir/get_file, ncurses must be initialised (eg. initsrc(),
 * start_color(), cbreak(), etc). get_dir uses own window, you should
 * rebuild your windows after get_dir exits.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <ncurses.h>
#include "mp3creat.h"

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

extern WINDOW *c_newwin(int h, int l, int y, int x, void *proc, int arg1, int arg2);
extern void   c_delwin(WINDOW *win);
extern void   wuuush(int);
extern char   *input_line(int y_line, char *def_str, char *usage_str, BOOL select);
extern void   popup_error_win(char *tx);
extern char   *input_line2(int y_line, char *def_str, char *usage_str,
		      	   char *des1, char *des2, char *des3, char *des4,
			   char *des5, char *des6, BOOL select);
extern char *file_glob_out_name(char *name);
extern char *file_glob_in_name(char *name);
extern char *kill_double_slashs(char *string);
extern char *copy_char_str(char *old);
extern BOOL  config_curs_dir;
extern BOOL  config_req_hidden;

typedef struct _content {
  char *name;
  struct _content *prev;
  struct _content *next;
} content;

BOOL req_refresh_needed = FALSE;

/* build complete filename from current directory */
char *req_correct_dir(char *in_file)
{
  char *cwd_name, *pat;

  /* if none file is given, just return */
  if(! in_file) return NULL;

  if(*in_file != '/') {
    cwd_name = getcwd(NULL, 0);
    if(! cwd_name) {
      popup_error_win(_("getcwd() does not work"));
      free(in_file);
      in_file = NULL;
      return NULL;
    }
    
    pat = (char *) malloc(sizeof(char) * (strlen(in_file) + strlen(cwd_name) + 2));
    if(! pat) {
      wuuush(1);
    }
    
    sprintf(pat, "%s/%s", cwd_name, in_file);
    free(cwd_name);
    free(in_file);
    cwd_name = NULL;
    in_file  = NULL;
    in_file  = pat;
  }

  in_file = kill_double_slashs(in_file);

  while(1) {
    cwd_name = strstr(in_file, "/./");
    if(! cwd_name) break;
    memmove(cwd_name, cwd_name+2, strlen(cwd_name)-1);
  }
  
  while(1) {
    cwd_name = strstr(in_file, "/../");
    if(! cwd_name) break;
    *cwd_name = 0;
    pat = strrchr(in_file, '/');
    if(! pat) {
      popup_error_win(_("illegal directoryname"));
      free(in_file);
      in_file = NULL;
      return NULL;
    }
    memmove(pat, cwd_name+3, strlen(cwd_name+1)-1);
  }

  return in_file;  
}

/* free content list */
void req_free_content_list(content **list)
{
  content *curr, *next;

  curr = *list;
  while(curr) {
    if(curr->name) free(curr->name);
    next = curr;
    curr = curr->next;
    free(next);
  }

  *list = NULL;
}

/* insert a new entry into content list */
void req_insert_content(content *new, content **data)
{
  content *curr, *next;

  curr = *data;
  if((! curr) || strcmp(new->name, curr->name) < 0) {
    new->next = curr;
    if(curr) curr->prev = new;
    *data = new;
  } else {
    next = curr->next;
    while(next && strcmp(new->name, next->name) > 0) {
      curr = next;
      next = next->next;
    }
    new->next = next;
    if(next) next->prev = new;
    new->prev  = curr;
    curr->next = new;
  }
}

/* read data
   **file_data might be NULL, if only directorys are requested
 */
int req_read_dir_data(const char *path, content **dir_data, int *dir_count,
		      content **file_data, int *file_count, BOOL hidden,
		      char *old_dir_name, content **old_dir, BOOL *dir_found)
{
  DIR           *direc;
  struct stat   st;
  struct dirent *de;
  content       *new;
  char          *name;
  
  *dir_count  = 0;
  *file_count = 0;
  *dir_data   = NULL;
  *dir_found  = FALSE;
  if(file_data) *file_data = NULL;

  direc = opendir(path);
  if(! direc) return 1;

  while((de = readdir(direc)) != NULL) {
    /* skip "." and ".." */
    if((strcmp(de->d_name, ".") != 0) && (strcmp(de->d_name, "..") != 0) &&
       ((de->d_name[0] != '.') || hidden)) {
      name = (char *) malloc(sizeof(char) * (strlen(path) + strlen(de->d_name) + 2));
      if(! name) {
	wuuush(1);
      }
      sprintf(name, "%s/%s", path, de->d_name);
      if(stat(name, &st) != -1) {
	if(S_ISDIR(st.st_mode)) {
	  /* handle directory entry */
	  new = (content *) malloc(sizeof(content));
	  if(! new) {
	    wuuush(1);
	  }
	  new->name = copy_char_str(de->d_name);
	  new->next = NULL;
	  new->prev = NULL;
	  if(old_dir_name && (strcmp(de->d_name, old_dir_name) == 0)) {
	    *old_dir   = new;
	    *dir_found = TRUE;
	  }
	  req_insert_content(new, &(*dir_data));
	  (*dir_count)++;
	} else if(file_data) {
	  /* handle file entry */
	  new = (content *) malloc(sizeof(content));
	  if(! new) {
	    wuuush(1);
	  }
	  new->name = copy_char_str(de->d_name);
	  new->next = NULL;
	  new->prev = NULL;
	  req_insert_content(new, &(*file_data));
	  (*file_count)++;
	}
      }
      free(name);
      name = NULL;
    }
  }

  closedir(direc);
  if(strcmp(path, "/") != 0) {
    /* not root dir, so insert dir-up points */
    new = (content *) malloc(sizeof(content));
    if(! new) {
      wuuush(1);
    }
    new->name = copy_char_str("..");
    new->next = *dir_data;
    if(new->next) new->next->prev = new;
    new->prev = NULL;
    *dir_data = new;
    (*dir_count)++;
  }

  return 0;
}
 
/* draw statusline
   win = window
   ys  = first useable y-position
   ye  = last useable y-position
   x   = x-position
   abs = absolute position in list
   rel = relative position
   tot = total entries
 */
void req_draw_stat(WINDOW *win, int ys, int ye, int x, int abs, int rel, int tot)
{
  int i, y1, y2, yb;

  yb = (ye-ys) + 1;
  if(yb >= tot) {
    wmove(win, ys, x);
    wvline(win, ACS_VLINE, yb);
    return;
  }
  ys--;

  if(! tot) {
    y1  = 0;
    y2  = 0;
    rel = 0;
  } else {
    y1  = (abs * yb) / tot;
    y2  = y1 + ((yb*yb) / tot) + 1;
    y1++;
    if(y2 > yb) y1 = (yb - (y2-y1));
    if(tot <= yb) {
      rel = (yb*rel) / tot;
    } else {
      rel = y1 + ((((yb*yb)/tot)*rel)/yb);
    }
    if(rel < y1) rel = y1;
    if(rel > y2) rel = y2;
    if(rel < 1)  rel = 1;
    if(rel > yb) rel = yb;
  }

  for(i=1;i<=yb;i++) {
    if((i>=y1) && (i<=y2)) mvwaddch(win, ys+i, x, ACS_BLOCK);
    else                   mvwaddch(win, ys+i, x, ACS_VLINE);
    if(i == rel)           mvwaddch(win, ys+i, x, ACS_BOARD);
  }
}

/* mark one line
   win = window
   y   = line
   xe  = last useable x-position
   name= file name
   ch  = color
 */
void req_draw_mark(WINDOW *win, int y, int xe, char *name, const chtype ch)
{
  int    i;
  chtype old_ch;

  old_ch = getbkgd(win);
  if(ch == A_REVERSE) {
    wbkgdset(win, COLOR_PAIR(3));
  } else {
    wbkgdset(win, old_ch ^ ch);
  }
  if(name) {
    i = strlen(name);
    if(i > xe) i = xe;
    mvwaddnstr(win, y, 1, name, i);
  } else {
    i = 0;
  }
  for(;i<xe;i++) waddch(win, ' ');
  wbkgdset(win, old_ch);
}

void req_draw_help(WINDOW *win, int y1, int y2, BOOL hid)
{
  wmove(win, y1-1, 1);
  if(y2) {
    waddstr(win, _("F1: New File * "));
  } else {
    waddstr(win, _("F1: OK * "));
  }
  
  waddstr(win, _("F3: Home * F4: hidden "));
  if(hid) waddstr(win, _("off"));
  else    waddstr(win, _("on"));
  waddstr(win, _(" * F7: MKdir * F12: Abort   "));
}

/* draw layout
   win = window
   y1  = last useable y-position
   y2  = split line (!= 0 for file requester, 0 for dir)
   x   = last useable x-position
   hid = hidden flag
   f1  = show F1-text (only if y2 != 0)
   des = some description about what the use should do 
 */
void req_draw_layout(WINDOW *win, int y1, int y2, int x, BOOL hid,
		     const char *description)
{
  int i;
  chtype old_ch;

  wclear(win);
  wmove(win, 0, 0); waddch(win, ACS_ULCORNER); whline(win, ACS_HLINE, x-1);
  wmove(win, 0, x); waddch(win, ACS_URCORNER);

  for(i=1;i<y1;i++) {
    wmove(win, i, 0); waddch(win, ACS_VLINE);
    wmove(win, i, x); waddch(win, ACS_VLINE);
  }

  wmove(win, y1, 0);   waddch(win, ACS_LLCORNER); whline(win, ACS_HLINE, x-1);
  wmove(win, y1, x);   waddch(win, ACS_LRCORNER);
  wmove(win, y1-2, 0); waddch(win, ACS_LTEE); whline(win, ACS_HLINE, x-1);
  wmove(win, y1-2, x); waddch(win, ACS_RTEE);
  if(y2) {
    wmove(win, y2, 0); waddch(win, ACS_LTEE); whline(win, ACS_HLINE, x-1);
    wmove(win, y2, x); waddch(win, ACS_RTEE);
  }
  req_draw_help(win, y1, y2, hid);
  
  if(description) {
    i = strlen(description);
    if(i > x-3) i = x-3;
    i = (x-3)-i;
    old_ch = getbkgd(win);
    wbkgdset(win, COLOR_PAIR(2) | A_BOLD);
    mvwaddnstr(win, y1-2, (i>>1)+2, description, x-3);
    wbkgdset(win, old_ch);
  }
}

/* fill in names
   win = window;
   ys  = first useable y-position
   ye  = last useable y-position
   x   = last useable x-position
   list= content-list
 */
void req_dir_fill(WINDOW *win, int ys, int ye, int x, content *list)
{
  content *curr;
  int y, len;

  y    = ys;
  curr = list;
  while((y <= ye) && curr) {
    wmove(win, y, 1);
    len = strlen(curr->name);
    if(len > x) len = x;
    waddnstr(win, curr->name, len);
    for(;len<x;len++) waddch(win, ' ');
    y++;
    curr = curr->next;
  }
  while(y <= ye) {
    wmove(win, y, 1);
    whline(win, ' ', x);
    y++;
  }
}

void req_refresh(WINDOW *win, int arg1, int arg2)
{
  req_refresh_needed = TRUE;
}

char *req_finish_path(char *path)
{
  path = req_correct_dir(file_glob_in_name(path));
  if(! path) {
    if(! getenv("HOME")) return NULL;
    path = copy_char_str(getenv("HOME"));
  }

  return path;
}

void req_draw_dirname(WINDOW *win, int y, int len, char *path)
{
  chtype old_ch;
  char *own_path;

  wmove(win, y, 1);
  whline(win, ACS_HLINE, len);
  if(! path) return;
  own_path = file_glob_out_name(copy_char_str(path));
  old_ch = getbkgd(win);
  wbkgdset(win, COLOR_PAIR(7));
  mvwaddch(win, y, 1, '<');
  mvwaddnstr(win, y, 3, own_path, len-2);
  wbkgdset(win, old_ch);
  free(own_path);
}

void req_toggle_hidden(WINDOW *win, int y1, int y2)
{
  if(config_req_hidden) config_req_hidden = FALSE;
  else                  config_req_hidden = TRUE;
  
  req_draw_help(win, y1, y2, config_req_hidden);
  wrefresh(win);
}

void req_handle_key_up(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
		       int ye, int xe, int tot_files, content **cl_abs,
		       content **cl_rel)
{
  int i;
  
  if((*cl_rel)->prev) {
    /* going up is allowed */
    if(*pos_rel > 1) {
      /* not on top of the screen -> just move cursor */
      req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
      *cl_rel = (*cl_rel)->prev;       /* switch to previous entry */
      (*pos_rel)--;
    } else {
      /* first move absolutly up */
      (*pos_abs)--;
      *cl_abs = (*cl_abs)->prev;
      *cl_rel = *cl_abs;
      /* now make a jump over the half screen for a smarter scrolling */
      for(i=0;i<=((ye-ys)>>1);i++) {
	if(! (*cl_abs)->prev) break;
	(*pos_abs)--;
	*cl_abs = (*cl_abs)->prev;
	(*pos_rel)++;
      }
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

void req_handle_key_down(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
			 int ye, int xe, int tot_files, content **cl_abs,
			 content **cl_rel)
{
  int i;
  content *counter;

  if((*cl_rel)->next) {
    /* going down is allowed */
    if(*pos_rel <= (ye-ys)) {
      /* not on bottom of the screen -> just move cursor */
      req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
      *cl_rel = (*cl_rel)->next;       /* switch to next entry */
      (*pos_rel)++;
    } else {
      /* first move absolutly down */
      (*pos_abs)++;
      *cl_abs = (*cl_abs)->next;
      *cl_rel = (*cl_rel)->next;
      /* now make a jump over the half screen for a smarter scrolling */
      counter = *cl_rel;
      for(i=0;i<=((ye-ys)>>1);i++) {
	if(! counter->next) break;
	(*pos_abs)++;
	*cl_abs = (*cl_abs)->next;
	(*pos_rel)--;
	counter = counter->next;
      }
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

void req_handle_key_home(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
			 int ye, int xe, int tot_files, content **cl_abs,
			 content **cl_rel, content *cl_first)
{
  if((*cl_rel)->prev) {
    /* going up is allowed */
    if(*pos_rel > 1) {
      /* not on top of the screen -> just move cursor */
      req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
      *cl_rel  = *cl_abs;    /* switch to top */
      *pos_rel = 1;
    } else {
      /* go to first entry */
      *cl_abs  = cl_first;
      *cl_rel  = cl_first;
      *pos_rel = 1;
      *pos_abs = 1;
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

void req_handle_key_end(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
		      	int ye, int xe, int tot_files, content **cl_abs,
		     	content **cl_rel)
{
  if((*cl_rel)->next) {
    /* going down is allowed */
    if(*pos_rel <= (ye-ys)) {
      /* not on bottom of the screen -> just move cursor */
      req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
      while((*cl_rel)->next && *pos_rel <= (ye-ys)) {
	(*pos_rel)++;
	*cl_rel = (*cl_rel)->next;
      }
    } else {
      /* go to last entry */
      while((*cl_rel)->next) {
	(*pos_abs)++;
	*cl_abs = (*cl_abs)->next;
	*cl_rel = (*cl_rel)->next;
      }
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

void req_handle_key_ppage(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
			  int ye, int xe, int tot_files, content **cl_abs,
			  content **cl_rel)
{
  int i;
  BOOL refill_flag;
  
  if((*cl_rel)->prev) {
    /* going up is allowed */
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
    refill_flag = FALSE;
    for(i=0;i<(ye-ys);i++) {
      if((*cl_abs)->prev) {
	/* absolutly upgoing */
	refill_flag = TRUE;
	(*pos_abs)--;
	*cl_abs = (*cl_abs)->prev;
	*cl_rel = (*cl_rel)->prev;
      } else if((*cl_rel)->prev) {
	/* relative upgoing */
	(*pos_rel)--;
	*cl_rel = (*cl_rel)->prev;
      } else {
	break;
      }
    }
    if(refill_flag) {
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

void req_handle_key_npage(WINDOW *win, int *pos_abs, int *pos_rel, int ys,
		     	  int ye, int xe, int tot_files, content **cl_abs,
		     	  content **cl_rel)
{
  int i;
  BOOL refill_flag;
  
  if((*cl_rel)->next) {
    /* going down is allowed */
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_NORMAL);
    refill_flag = FALSE;
    for(i=0;i<(ye-ys);i++) {
      if(*pos_abs + (ye-ys) < tot_files) {
	/* absolutly downgoing */
	refill_flag = TRUE;
	(*pos_abs)++;
	*cl_abs = (*cl_abs)->next;
	*cl_rel = (*cl_rel)->next;
      } else if((*cl_rel)->next) {
	/* relative downgoing */
	(*pos_rel)++;
	*cl_rel = (*cl_rel)->next;
      } else {
	break;
      }
    }
    if(refill_flag) {
      req_dir_fill(win, ys, ye, xe, *cl_abs);
    }
    req_draw_mark(win, (*pos_rel)+ys-1, xe, (*cl_rel)->name, A_REVERSE);
    req_draw_stat(win, ys, ye, xe+1, *pos_abs, *pos_rel, tot_files);
    wrefresh(win);
  }
}

char *req_curs_dir(char *path, const char *description)
{
  int maxy, maxx;
  WINDOW *requester;
  content *dir_list, *dir_abs, *dir_rel, *dir_old;
  int tot_dirs, pos_abs, pos_rel;
  BOOL dir_found;
  char *old_name, *new_path;
  int dummy;
  DIR *direc;
  int inp_char;
  int exit_code;

  direc = opendir(path);
  if(! direc) {
    if(path) free(path);
    path = req_finish_path(NULL);
  } else {
    closedir(direc);
  }

  getmaxyx(stdscr, maxy, maxx);
  requester = c_newwin(maxy, maxx, 0, 0, req_refresh, 0, 0);
  req_refresh_needed = TRUE;

  dir_list = NULL;
  old_name = NULL;
  if(req_read_dir_data(path, &dir_list, &tot_dirs, NULL, &dummy, config_req_hidden,
		       NULL, &dir_old, &dir_found)) {
    popup_error_win(_("reading dir failed"));
    return NULL;
  }
  dir_abs   = dir_list;
  dir_rel   = dir_list;
  pos_abs   = 1;
  pos_rel   = 1;
  exit_code = 0;

  while(1) {
    halfdelay(10);
    if(req_refresh_needed) {
      getmaxyx(stdscr, maxy, maxx);
      wresize(requester, maxy, maxx);
      mvwin(requester, 0, 0);
      if(pos_rel > maxy-4) {
	pos_rel = 1;
	dir_rel = dir_abs;
      }
      scrollok(requester, FALSE);
      keypad(requester, TRUE);
      wbkgd(requester, COLOR_PAIR(1));
      req_draw_layout(requester, maxy-1, 0, maxx-1, config_req_hidden,
		      description);
      req_dir_fill(requester, 1, maxy-4, maxx-2, dir_abs);
      req_draw_mark(requester, pos_rel, maxx-2, dir_rel->name, A_REVERSE);
      req_draw_stat(requester, 1, maxy-4, maxx-1, pos_abs, pos_rel, tot_dirs);
      req_draw_dirname(requester, 0, maxx-2, path);
      wrefresh(requester);
      req_refresh_needed = FALSE;
    }

    inp_char = wgetch(requester);
    dummy    = 0;
    new_path = NULL;
    switch(inp_char) {
      case KEY_ENTER:
      case '\n':
      case '\r':
	/* change directory */
	if(strcmp(dir_rel->name, "..") != 0) {
  	  new_path = (char *) malloc(sizeof(char) * (strlen(path) + strlen(dir_rel->name) + 2));
	  if(! new_path) {
	    wuuush(1);
	  }
	  if(strcmp(path, "/") != 0) {
	    sprintf(new_path, "%s/%s", path, dir_rel->name);
	  } else {
	    sprintf(new_path, "/%s", dir_rel->name);
	  }
	} else {
	  old_name = strrchr(path, '/');
	  if(! old_name) {
	    new_path = NULL;        /* CANNOT occur! */
	  } else {
	    *old_name = 0;
	    if(! *path) {
	      new_path = copy_char_str("/");
	    } else {
	      new_path = copy_char_str(path);
	    }
	    *old_name = '/';
	    old_name = copy_char_str(old_name+1);
	  }
	}
	dummy = 1;        /* reread directory */
	break;

      case KEY_F(1):
      case '1':
	/* accept directory */
	exit_code = 2;
	break;

      case KEY_F(3):
      case '3':
	/* switch to homedirectory */
	new_path = copy_char_str(getenv("HOME"));
	dummy = 1;
	break;

      case KEY_F(4):
      case '4':
	/* toggle hiddenflag */
	req_toggle_hidden(requester, maxy-1, 0);
	new_path = copy_char_str(path);
	dummy = 1;
	break;

      case KEY_F(7):
      case '7':
	/* Make directory */
	old_name = input_line(10, NULL, _("enter new directory name"), FALSE);
	if(old_name) {
	  new_path = (char *) malloc(sizeof(char) * (strlen(path) + strlen(old_name) + 2));
	  if(! new_path) {
	    wuuush(1);
	  }
	  if(strcmp(path, "/") != 0) {
	    sprintf(new_path, "%s/%s", path, old_name);
	  } else {
	    sprintf(new_path, "/%s", old_name);
	  }
	  free(old_name);
	  old_name = NULL;
	  if(mkdir(new_path, 0777)) {
	    popup_error_win(_("could not create directory"));
	    free(new_path);
	    new_path = NULL;
	  } else {
	    dummy = 1;
	  }
	}
	req_refresh_needed = TRUE;
	break;

      case KEY_F(12):
      case 27:
      case 'q':
      case 'Q':
      case '\'':
	/* Abort */
	exit_code = 1;
	break;

      case '':
	/* Refresh screen */
	req_refresh_needed = TRUE;
	break;

      case KEY_UP:
	/* Key up */
	req_handle_key_up(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			  tot_dirs, &dir_abs, &dir_rel);
	break;

      case KEY_DOWN:
	/* Key down */
	req_handle_key_down(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			    tot_dirs, &dir_abs, &dir_rel);
	break;

      case KEY_HOME:
      case 362:
	/* Key home */
	req_handle_key_home(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			    tot_dirs, &dir_abs, &dir_rel, dir_list);
	break;

      case KEY_END:
      case 385:
	/* Key end */
	req_handle_key_end(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			   tot_dirs, &dir_abs, &dir_rel);
	break;

      case KEY_PPAGE:
	/* Key previous page */
	req_handle_key_ppage(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			     tot_dirs, &dir_abs, &dir_rel);
	break;
	
      case KEY_NPAGE:
	/* Key next page */
	req_handle_key_npage(requester, &pos_abs, &pos_rel, 1, maxy-4, maxx-2,
			     tot_dirs, &dir_abs, &dir_rel);
	break;
    }
    if(dummy) {
      /* directory was changed */
      if(new_path && (! access(new_path, X_OK))) {
	/* access granted */
	free(path);
	path     = new_path;
	new_path = NULL;
	req_free_content_list(&dir_list);
	dir_list = NULL;
	if(req_read_dir_data(path, &dir_list, &tot_dirs, NULL, &dummy,
			     config_req_hidden, old_name, &dir_old,
			     &dir_found)) {
	  popup_error_win(_("reading dir failed"));
	  exit_code = 1;
	}
	dir_abs = dir_list;
	dir_rel = dir_list;
	pos_abs = 1;
	pos_rel = 1;
	if(dir_found) {
	  while(dir_abs && dir_abs != dir_old) {
	    pos_abs++;
	    dir_abs = dir_abs->next;
	  }
	  if(! dir_abs) {
	    dir_abs = dir_list;
	    pos_abs = 1;
	  } else {
	    dir_rel = dir_abs;
	    for(dummy=0;dummy<=((maxy-5)>>1);dummy++) {
	      if(! dir_abs->prev) break;
	      pos_abs--;
	      dir_abs = dir_abs->prev;
	      pos_rel++;
	    }
	  }
	}
	req_dir_fill(requester, 1, maxy-4, maxx-2, dir_abs);
	req_draw_mark(requester, pos_rel, maxx-2, dir_rel->name, A_REVERSE);
	req_draw_stat(requester, 1, maxy-4, maxx-1, pos_abs, pos_rel, tot_dirs);
	req_draw_dirname(requester, 0, maxx-2, path);
	wrefresh(requester);
      } else {
	popup_error_win(_("accessing directory failed"));
	req_refresh_needed = TRUE;
      }
      if(old_name) {
	free(old_name);
	old_name = NULL;
      }
      dummy=0;
    }
    
    if(exit_code) break;
  }

  switch(exit_code) {
    case 1:	/* ABORT */
      if(path) free(path);
      path = NULL;
      break;
    case 2:	/* OK */
      break;
  }

  req_free_content_list(&dir_list);
  if(old_name) {
    free(old_name);
    old_name = NULL;
  }

  c_delwin(requester);
  
  return path;
}

char *req_curs_file(char *path, const char *description)
{
  int maxy, maxx;
  WINDOW *requester;
  content *dir_list[2], *dir_abs[2], *dir_rel[2], *dir_old;
  int tot_dirs[2], pos_abs[2], pos_rel[2];
  BOOL dir_found;
  char *old_name, *new_path;
  int dummy;
  DIR *direc;
  int inp_char;
  int exit_code;
  int act_side;
  int ys[2], ye[2];

  direc = opendir(path);
  if((! direc) && path) {
    old_name = strrchr(path, '/');
    if(old_name) {
      *old_name = 0;
      direc = opendir(path);
    }
  }
  if(! direc) {
    if(path) free(path);
    path = req_finish_path(NULL);
  } else {
    closedir(direc);
  }

  getmaxyx(stdscr, maxy, maxx);
  requester = c_newwin(maxy, maxx, 0, 0, req_refresh, 0, 0);
  req_refresh_needed = TRUE;
  
  dir_list[0] = NULL;
  dir_list[1] = NULL;
  old_name    = NULL;
  if(req_read_dir_data(path, &(dir_list[0]), &(tot_dirs[0]), &(dir_list[1]),
		       &(tot_dirs[1]), config_req_hidden, NULL, &dir_old,
		       &dir_found)) {
    popup_error_win(_("reading dir failed"));
    return NULL;
  }

  act_side   = 0;
  dir_abs[0] = dir_list[0];
  dir_abs[1] = dir_list[1];
  dir_rel[0] = dir_list[0];
  dir_rel[1] = dir_list[1];
  pos_abs[0] = 1;
  pos_abs[1] = 1;
  pos_rel[0] = 1;
  pos_rel[1] = 1;
  exit_code  = 0;

  while(1) {
    halfdelay(10);
    if(req_refresh_needed) {
      getmaxyx(stdscr, maxy, maxx);
      wresize(requester, maxy, maxx);
      mvwin(requester, 0, 0);
      ys[0] = 1;
      ys[1] = ((maxy-2)>>1);
      ye[0] = ys[1]-2;
      ye[1] = maxy-4;
      scrollok(requester, FALSE);
      if(pos_rel[0] > ye[0]) {
	pos_rel[0] = 1;
	dir_rel[0] = dir_abs[0];
      }
      if(pos_rel[1] > (ye[1]-ys[1])+1) {
	pos_rel[1] = 1;
	dir_rel[1] = dir_abs[1];
      }
      keypad(requester, TRUE);
      wbkgd(requester, COLOR_PAIR(1));
      req_draw_layout(requester, maxy-1, ye[0]+1, maxx-1, config_req_hidden,
		      description);
      req_dir_fill(requester, ys[0], ye[0], maxx-2, dir_abs[0]);
      req_dir_fill(requester, ys[1], ye[1], maxx-2, dir_abs[1]);
      req_draw_mark(requester, pos_rel[act_side] + (ys[act_side]-1), maxx-2,
		    (dir_rel[act_side])->name, A_REVERSE);
      req_draw_stat(requester, ys[0], ye[0], maxx-1, pos_abs[0], pos_rel[0],
		    tot_dirs[0]);
      req_draw_stat(requester, ys[1], ye[1], maxx-1, pos_abs[1], pos_rel[1],
		    tot_dirs[1]);
      req_draw_dirname(requester, 0, maxx-2, path);
      wrefresh(requester);
      req_refresh_needed = FALSE;
    }

    inp_char = wgetch(requester);
    dummy    = 0;
    new_path = NULL;
    switch(inp_char) {
      case KEY_ENTER:
      case '\n':
      case '\r':
	/* change directory / accept file */
	if(! act_side) {
	  /* change directory */
	  if(strcmp((dir_rel[0])->name, "..") != 0) {
	    new_path = (char *) malloc(sizeof(char) * (strlen(path) +
						       strlen((dir_rel[0])->name) + 2));
	    if(! new_path) {
	      wuuush(1);
	    }
	    if(strcmp(path, "/") != 0) {
	      sprintf(new_path, "%s/%s", path, (dir_rel[0])->name);
	    } else {
	      sprintf(new_path, "/%s", (dir_rel[0])->name);
	    }
	  } else {
	    old_name = strrchr(path, '/');
	    if(! old_name) {
	      new_path = NULL;      /* CANNOT occur! */
	    } else {
	      *old_name = 0;
	      if(! *path) {
		new_path = copy_char_str("/");
	      } else {
		new_path = copy_char_str(path);
	      }
	      *old_name = '/';
	      old_name = copy_char_str(old_name + 1);
	    }
	  }
	  dummy = 1;      /* reread directory */
	} else {
	  /* accept file */
	  old_name = copy_char_str((dir_rel[1])->name);
	  exit_code = 2;
	}
	break;
	
      case KEY_F(1):
      case '1':
	/* create new file */
	if(act_side) {
	  old_name = input_line(10, (dir_rel[1])->name, _("enter new filename"), TRUE);
	} else {
	  old_name = input_line(10, NULL, _("enter new filename"), FALSE);
	}
	if(old_name) {
	  exit_code = 2;
	}
	req_refresh_needed = TRUE;
	break;
	
      case KEY_F(3):
      case '3':
	/* switch to homedirectory */
	new_path = copy_char_str(getenv("HOME"));
	dummy = 1;
	break;
	
      case KEY_F(4):
      case '4':
	/* toggle hiddenflag */
	req_toggle_hidden(requester, maxy-1, ye[0]+1);
	new_path = copy_char_str(path);
	dummy = 1;
	break;
	
      case KEY_F(7):
      case '7':
	/* Make directory */
	old_name = input_line(10, NULL, _("enter new directory name"), FALSE);
	if(old_name) {
	  new_path = (char *) malloc(sizeof(char) * (strlen(path) +
						     strlen(old_name) + 2));
	  if(! new_path) {
	    wuuush(1);
	  }
	  if(strcmp(path, "/") != 0) {
	    sprintf(new_path, "%s/%s", path, old_name);
	  } else {
	    sprintf(new_path, "/%s", old_name);
	  }
	  free(old_name);
	  old_name = NULL;
	  if(mkdir(new_path, 0777)) {
	    popup_error_win(_("could not create directory"));
	    free(new_path);
	    new_path = NULL;
	  } else {
	    dummy = 1;
	  }
	}
	req_refresh_needed = TRUE;
	break;
	
      case KEY_F(12):
      case '\'':
      case 27:
      case 'q':
      case 'Q':
	/* Abort */
	exit_code = 1;
	break;
	
      case '':
	/* Refresh screen */
	req_refresh_needed = TRUE;
	break;
	
      case KEY_STAB:
      case KEY_CTAB:
      case '\t':
	/* change window */
	if(dir_rel[1]) {
	  /* switching allowed */
	  req_draw_mark(requester, pos_rel[act_side] + (ys[act_side]-1), maxx-2,    
			(dir_rel[act_side])->name, A_NORMAL);
	  act_side = 1-act_side;
	  req_draw_mark(requester, pos_rel[act_side] + (ys[act_side]-1), maxx-2,    
			(dir_rel[act_side])->name, A_REVERSE);
	  wrefresh(requester);
	}
       	break;
	
      case KEY_UP:
	/* Key up */
	req_handle_key_up(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			  ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			  &(dir_abs[act_side]), &(dir_rel[act_side]));
	break;
	
      case KEY_DOWN:
	/* Key down */
	req_handle_key_down(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			    ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			    &(dir_abs[act_side]), &(dir_rel[act_side]));
	break;
	
      case KEY_HOME:
      case 362:
	/* Key home */
	req_handle_key_home(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			    ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			    &(dir_abs[act_side]), &(dir_rel[act_side]),
			    dir_list[act_side]);
	break;

      case KEY_END:
      case 385:
	/* Key end */
	req_handle_key_end(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			   ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			   &(dir_abs[act_side]), &(dir_rel[act_side]));
	break;

      case KEY_PPAGE:
	/* Key previous page */
	req_handle_key_ppage(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			     ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			     &(dir_abs[act_side]), &(dir_rel[act_side]));
	break;

      case KEY_NPAGE:
	/* Key next page */
	req_handle_key_npage(requester, &(pos_abs[act_side]), &(pos_rel[act_side]),
			     ys[act_side], ye[act_side], maxx-2, tot_dirs[act_side],
			     &(dir_abs[act_side]), &(dir_rel[act_side]));
	break;
    }
    if(dummy) {
      /* directory was changed */
      if(new_path && (! access(new_path, X_OK))) {
	/* access granted */
	free(path);
	path     = new_path;
	new_path = NULL;
	req_free_content_list(&(dir_list[0]));
	req_free_content_list(&(dir_list[1]));
	dir_list[0] = NULL;
	dir_list[1] = NULL;
	if(req_read_dir_data(path, &(dir_list[0]), &(tot_dirs[0]),
			     &(dir_list[1]), &(tot_dirs[1]), config_req_hidden,
			     old_name, &dir_old, &dir_found)) {
	  popup_error_win(_("reading dir failed"));
	  exit_code = 1;
	}
	dir_abs[0] = dir_list[0];
	dir_abs[1] = dir_list[1];
	dir_rel[0] = dir_list[0];
	dir_rel[1] = dir_list[1];
	pos_abs[0] = 1;
	pos_abs[1] = 1;
	pos_rel[0] = 1;
	pos_rel[1] = 1;
	if(dir_found) {
	  while(dir_abs[0] && dir_abs[0] != dir_old) {
	    (pos_abs[0])++;
	    dir_abs[0] = (dir_abs[0])->next;
	  }
	  if(! dir_abs[0]) {
	    dir_abs[0] = dir_list[0];
	    pos_abs[0] = 1;
	  } else {
	    dir_rel[0] = dir_abs[0];
	    for(dummy=0;dummy<=((ye[0]-ys[0])>>1);dummy++) {
	      if(! (dir_abs[0])->prev) break;
	      (pos_abs[0])--;
	      dir_abs[0] = (dir_abs[0])->prev;
	      (pos_rel[0])++;
	    }
	  }
	}
	act_side = 0;
	req_dir_fill(requester, ys[0], ye[0], maxx-2, dir_abs[0]);
	req_dir_fill(requester, ys[1], ye[1], maxx-2, dir_abs[1]);
	req_draw_mark(requester, pos_rel[0] + (ys[0]-1), maxx-2,
		      (dir_rel[0])->name, A_REVERSE);
	req_draw_stat(requester, ys[0], ye[0], maxx-1, pos_abs[0], pos_rel[0],
		      tot_dirs[0]);
	req_draw_stat(requester, ys[1], ye[1], maxx-1, pos_abs[1], pos_rel[1],
		      tot_dirs[1]);
	req_draw_dirname(requester, 0, maxx-2, path);
	wrefresh(requester);
      } else {
	popup_error_win(_("accessing directory failed"));
	req_refresh_needed = TRUE;
      }
      if(old_name) {
	free(old_name);
	old_name = NULL;
      }
      dummy = 0;
    }

    if(exit_code) break;
  }

  switch(exit_code) {
    case 1:        /* ABORT */
      if(path) free(path);
      path = NULL;
      break;
    case 2:        /* file selected, name in old_name */
      path = (char *) realloc(path, sizeof(char) * (strlen(path) +
						    strlen(old_name) + 2));
      if(! path) {
	wuuush(1);
      }
      if(strcmp(path, "/") != 0) strcat(path, "/");
      strcat(path, old_name);
      path = req_finish_path(path);
      break;
  }

  req_free_content_list(&(dir_list[0]));
  req_free_content_list(&(dir_list[1]));
  if(old_name) {
    free(old_name);
    old_name = NULL;
  }

  c_delwin(requester);

  return path;
}

char *req_get_dir(const char *path, const char *description)
{
  char *dir;

  dir = copy_char_str((char *) path);
  dir = req_finish_path(dir);
  if(config_curs_dir) return req_curs_dir(dir, description);
  
  dir = input_line2(5, dir, (char *) description,
		    _("please input path of directory"),
		    _("(you can use a directory-requester by changing value in preferences)"),
		    NULL,
		    _("currently TAB-expansion does not work :-("),
		    NULL, NULL, TRUE);
  return req_finish_path(dir);
}

char *req_get_file(const char *path, const char *description)
{
  char *file;

  file = copy_char_str((char *) path);
  file = req_finish_path(file);
  if(config_curs_dir) return req_curs_file(file, description);

  file = input_line2(5, file, (char *) description,
		     _("please input path of file"),
		     _("(you can use a file-requester by changing value in preferences)"),
		     NULL,
		     _("currently TAB-expansion does not work :-("),
		     NULL, NULL, TRUE);
  return req_finish_path(file);
}

