#include <stdio.h>
#include <locale.h>
#include <MRI.h>

#define	EPS	1.0e-5

int
LoadToneFile (const char *filename, double input[], double output[], int MAX)
{
	FILE *f;
	int  nEntries = 0;
	char	buffer[512];
	int	lineNo = 0;
	int	messageOutput = FALSE;
	int	nErrors = 0;
	int	i;
	char	*oldlocale;

	if ((f = fopen (filename, "r")) == NULL) {
		fprintf (stderr, "Unable to open tone curve file %s\n", filename);
		return -1;
	}
	oldlocale = setlocale (LC_NUMERIC, "C");
	while (fgets (buffer, sizeof (buffer), f)) {
		double idx, val;
		char *s = strchr (buffer, ';');
		if (s)
			*s = '\0';
		lineNo++;
		for (s = buffer; *s; s++)
			if (*s != ' ' && *s != '\t' && *s != '\r' && *s != '\n')
				break;
		if (*s) {
			if (sscanf (s, "%lg,%lg", &idx, &val) != 2) {
				fprintf (stderr, "LoadToneFile: Error loading tone curve %s\n", filename);
				fprintf (stderr, "LoadToneFile: Cannot parse line %d: %s\n", lineNo, buffer);
				nErrors++;
				continue;
			}
			if (nEntries > 0 && idx < input[nEntries-1]) {
				fprintf (stderr, "LoadToneFile: Line %d in tone file %s is out of order.\n", lineNo, filename);
				nErrors++;
				continue;
			}
			if (nEntries == MAX) {
				if (!messageOutput) {
					fprintf (stderr, "LoadToneFile: Tone file %s contains more than %d entries.\n", filename, MAX);
					fprintf (stderr, "LoadToneFile: dropping excess entries into the bit bucket\n");
					messageOutput = TRUE;
				}
				nEntries--;
			}
			input[nEntries] = idx;
			output[nEntries] = val;
			nEntries++;
		}
	}
	if (oldlocale)
		setlocale (LC_NUMERIC, oldlocale);
	fclose (f);

	if (nEntries == 0) {
		fprintf (stderr, "LoadToneFile: tone file doesn't contain any data points\n");
		nErrors++;
	}
	else {
		if (fabs (input[0]) > EPS || fabs(output[0]) > EPS) {
			fprintf (stderr, "LoadToneFile: first data point (%g,%g) in tone file must be 0,0.\n", input[0], output[0]);
			nErrors++;
		}
		if (fabs (input[nEntries-1]) <= EPS) {
			fprintf (stderr, "LoadToneFile: last data point in tone file must be larger than 0,0.\n");
			nErrors++;
		}
		if (fabs(input[nEntries-1]-output[nEntries-1]) > EPS) {
			fprintf (stderr, "LoadToneFile: input and output of last data point in tone file must be equal.\n");
			nErrors++;
		}
	}
	if (nErrors > 0)
		return -1;
	for (i = 0; i < nEntries; i++) {
		output[i] /= input[nEntries-1];
		input[i] /= input[nEntries-1];
	}
	return nEntries;
}

MRI_SampledCurve *
LoadSampledToneCurve (const char *filename)
{
	int i, n;
	double index[512];
	double value[512];
	MRI_SampledCurve *crv;

	n = LoadToneFile (filename, index, value, 512);
	if (n <= 0) return NULL;

	for (i = 0; i < n; i++) {
		index[i] = pow (index[i], 1.0/3.0);
		value[i] = pow (value[i], 1.0/3.0);
	}
	crv = MRI_NewSampledCurve (1024, n, index, value);
	return crv;
}
