#include <stdio.h>
#include <stdlib.h>
#include "mrwtiff.h"

int
mrw_val (const unsigned char *data, int offset, int length)
{
	int val;
	data += offset;

	val = (*data & 0x80) ? -1 : 0;
	while (length-- > 0)
		val = (val << 8) | *data++;
	return val;
}

void
mrw_setval (unsigned char *data, int offset, int length, int val)
{
	data += offset + length;

	while (length-- > 0) {
		*--data = (val & 0xFF);
		val = val >> 8;
	}
}

const char *
IFD_GetString (IFD *ifd, int tag)
{
	int i;

	if (ifd == (IFD *)0)
		return (const char *)0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_STRING)
			return &ifd->ifd_data[ifd->ifd_entry[i].ifd_offset];
	return (const char *)0;
}

int
IFD_GetShort (IFD *ifd, int tag)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_SHORT)
			return ifd->ifd_entry[i].ifd_offset >> 16;
	return 0;
}

int
IFD_GetLong (IFD *ifd, int tag)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_LONG)
			return ifd->ifd_entry[i].ifd_offset;
	return 0;
}

int
IFD_GetUndefined (IFD *ifd, int tag)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_UNDEFINED)
			return ifd->ifd_entry[i].ifd_offset;
	return 0;
}

int
IFD_GetUndefinedLength (IFD *ifd, int tag)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_UNDEFINED)
			return ifd->ifd_entry[i].ifd_count;
	return 0;
}

int
IFD_SetUndefinedLength (IFD *ifd, int tag, int count)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_UNDEFINED) {
			mrw_setval (ifd->ifd_data, ifd->ifd_offset+i*12+6, 4, count);
			return 1;
		}
	return 0;
}

int
IFD_GetRational (IFD *ifd, int tag, int* num, int *denom)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_RATIONAL) {
			*num = mrw_val (ifd->ifd_data, ifd->ifd_entry[i].ifd_offset, 4);
			*denom = mrw_val (ifd->ifd_data, ifd->ifd_entry[i].ifd_offset + 4, 4);
			return 1;
		}
	return 0;
}

int
IFD_GetSRational (IFD *ifd, int tag, int* num, int *denom)
{
	int i;

	if (ifd == (IFD *)0)
		return 0;
	for (i = 0 ; i < ifd->ifd_count; i++)
		if (ifd->ifd_entry[i].ifd_tag == tag &&
		    ifd->ifd_entry[i].ifd_type == IFDTYPE_SRATIONAL) {
			*num = mrw_val (ifd->ifd_data, ifd->ifd_entry[i].ifd_offset, 4);
			*denom = mrw_val (ifd->ifd_data, ifd->ifd_entry[i].ifd_offset + 4, 4);
			return 1;
		}
	return 0;
}

IFD *
ParseIFD (const unsigned char *data, int offset)
{
	IFD *ifd;
	int i;

	ifd = (IFD *)malloc(sizeof(*ifd));
	if (ifd == (IFD *)0) {
#ifdef DEBUG
		fprintf (stderr, "ParseIFD: Error allocating memory for header\n");
#endif
		return ifd;
	}
	ifd->ifd_data = data;
	ifd->ifd_offset = offset;
#ifdef DEBUG
	fprintf (stderr, "Loading IFD at offset %d\n", offset);
#endif
	ifd->ifd_count = mrw_val (data, offset, 2);
#ifdef DEBUG
	fprintf (stderr, "    %d entries:\n", ifd->ifd_count);
#endif
	ifd->ifd_entry = (IFD_entry *)malloc(sizeof(IFD_entry) * ifd->ifd_count);
	if (ifd->ifd_entry == (IFD_entry *)0) {
#ifdef DEBUG
		fprintf (stderr, "ParseIFD: Error allocating memory for %d entries\n", ifd->ifd_count);
#endif
		free (ifd);
		return (IFD *)0;
	}
	for (i = 0; i < ifd->ifd_count; i++) {
		ifd->ifd_entry[i].ifd_tag = mrw_val (data, offset + i*12 +2, 2) & 0xFFFF;
		ifd->ifd_entry[i].ifd_type = mrw_val (data, offset + i*12 +4, 2);
		ifd->ifd_entry[i].ifd_count = mrw_val (data, offset + i*12 +6, 4);
		ifd->ifd_entry[i].ifd_offset = mrw_val (data, offset + i*12 +10, 4);
#ifdef DEBUG
		fprintf (stderr, "    %2d (@0x%04x): tag=0x%04x, type=%d, tcount=%d, toffset=%d\n", i, offset+i*12+10, ifd->ifd_entry[i].ifd_tag, ifd->ifd_entry[i].ifd_type, ifd->ifd_entry[i].ifd_count, ifd->ifd_entry[i].ifd_offset);
#endif
	}
	ifd->ifd_next = mrw_val (data, offset+ifd->ifd_count*12+2, 4);
	return ifd;
}

