/* File rw.c
 *
 * Due to some problems with the 2.6 kernels, I had to find a way in
 * order to make reading and writing from/to the device file /dev/lp0 and
 * /dev/usb/lp0 working in the same way.
 * With the 2.6 kernel, the read() call will no more be interrupted by
 * an alarm timer. Read on /dev/lp0 is a blocking call, read on 
 * /dev/usb/lp0 will never block.
 * Both device has different behaviour and the kernel people seem not to
 * be able to understood the problem and fix it.
 *
 * Solution: If we open the device file with the O_NDELAY flag set,
 * both interface will return immediatly with the value -1 and errno set
 * to EAGAIN if nothing is to be read. 
 *
 * Writing may be working for now without changes.
 */
 
#ifndef MACOS
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <signal.h>

ssize_t devRead(int fd, unsigned char *buf, size_t sz, int to)
{
   int rd = 0;
   to /= 10;
   errno = 0;
   while(to > 0 && rd != -1 )
   {
      /* wait a little bit */
      poll(NULL, 0, 10);
      to--;
      rd = read(fd, buf, sz);
      if ( rd > 0 )
      {
         return rd;
      }
   }
   /* at this point we are probably interrupted */
   if ( rd == 0 )
   {
      errno = EINTR;
   }
   return -1;
}

ssize_t devWrite(int fd, unsigned char *buf, size_t sz, int to)
{
   int wr = 0;
   to /= 10;
   errno = 0;
   while(to > 0 && wr != -1 )
   {
      /* wait a little bit */
      poll(NULL, 0, 10);
      to--;
      wr = write(fd, buf, sz);
      if ( wr > 0 )
      {
         return wr;
      }
   }
   /* at this point we are probably interrupted */
   if ( wr == 0 )
   {
      errno = EINTR;
   }
   return -1;
}
#endif
