//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: prcanvas.cpp,v 1.20.2.6 2006/01/05 18:14:39 spamatica Exp $
//  (C) Copyright 1999-2004 Werner Schweer (ws@seh.de)
//=========================================================

#include <qapplication.h>
#include <qclipboard.h>
#include <qpainter.h>
#include <qdragobject.h>

#include <values.h>
#include <stdio.h>
#include <math.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include "xml.h"
#include "prcanvas.h"
#include "midiport.h"
#include "event.h"
#include "mpevent.h"
#include "globals.h"
#include "cmd.h"
#include "gatetime.h"
#include "velocity.h"
#include "song.h"
#include "audio.h"

//---------------------------------------------------------
//   NEvent
//---------------------------------------------------------

NEvent::NEvent(Event& e, Part* p, int y) : CItem(e, p)
      {
      y = y - KH/4;
      unsigned tick = e.tick() + p->tick();
      setPos(QPoint(tick, y));
      setBBox(QRect(tick, y, e.lenTick(), KH/2));
      }

//---------------------------------------------------------
//   addItem
//---------------------------------------------------------

void PianoCanvas::addItem(Part* part, Event& event)
      {
      if (signed(event.tick())<0) {
            printf("ERROR: trying to add event before current part!\n");
            return;
      }
      
      NEvent* ev = new NEvent(event, part, pitch2y(event.pitch()));
      items.add(ev);
      
      int diff = event.endTick()-part->lenTick();
      if (diff > 0)  {// too short part? extend it
            //printf("addItem - this code should not be run!\n");
            //Part* newPart = part->clone();
            //newPart->setLenTick(newPart->lenTick()+diff);
            //audio->msgChangePart(part, newPart,false);
            //part = newPart;
            part->setLenTick(part->lenTick()+diff);
            }
      }

//---------------------------------------------------------
//   PianoCanvas
//---------------------------------------------------------

PianoCanvas::PianoCanvas(MidiEditor* pr, QWidget* parent, int sx, int sy)
   : EventCanvas(pr, parent, sx, sy)
      {
      colorMode = 0;
      cmdRange  = 0;     // all Events
      playedPitch = -1;

      songChanged(SC_TRACK_INSERTED);
      connect(song, SIGNAL(midiNote(int, int)), SLOT(midiNote(int,int)));
      }

//---------------------------------------------------------
//   pitch2y
//---------------------------------------------------------

int PianoCanvas::pitch2y(int pitch) const
      {
      int tt[] = {
            5, 12, 19, 26, 33, 44, 51, 58, 64, 71, 78, 85
            };
      int y = (75 * KH) - (tt[pitch%12] + (7 * KH) * (pitch/12));
      if (y < 0)
            y = 0;
      return y;
      }

//---------------------------------------------------------
//   y2pitch
//---------------------------------------------------------

int PianoCanvas::y2pitch(int y) const
      {
      const int total = (10 * 7 + 5) * KH;       // 75 Ganztonschritte
      y = total - y;
      int oct = (y / (7 * KH)) * 12;
      char kt[] = {
            0, 0, 0, 0, 0,  0,   0, 0, 0, 0,          // 5
            1, 1, 1,      1,   1, 1, 1,               // 13
            2, 2,         2,   2, 2, 2,               // 19
            3, 3, 3,      3,   3, 3, 3,               // 26
            4, 4, 4, 4,   4,   4, 4, 4, 4,            // 34
            5, 5, 5, 5,   5,   5, 5, 5, 5, 5,         // 43
            6, 6, 6,      6,   6, 6, 6,               // 52
            7, 7,         7,   7, 7, 7,               // 58
            8, 8, 8,      8,   8, 8, 8,               // 65
            9, 9,         9,   9, 9, 9,               // 71
            10, 10, 10,  10,   10, 10, 10,            // 78
            11, 11, 11, 11, 11,   11, 11, 11, 11, 11  // 87
            };
      return kt[y % 91] + oct;
      }

//---------------------------------------------------------
//   drawEvent
//    draws a note
//---------------------------------------------------------

void PianoCanvas::drawItem(QPainter& p, const CItem* item,
   const QRect&)
      {
      p.setPen(black);
      QRect r = item->bbox();

      struct Triple {
            int r, g, b;
            };
      static Triple color1[12] = {
            { 0xff, 0x3d, 0x39 },
            { 0x39, 0xff, 0x39 },
            { 0x39, 0x3d, 0xff },
            { 0xff, 0xff, 0x39 },
            { 0xff, 0x3d, 0xff },
            { 0x39, 0xff, 0xff },
            { 0xff, 0x7e, 0x7a },
            { 0x7a, 0x7e, 0xff },
            { 0x7a, 0xff, 0x7a },
            { 0xff, 0x7e, 0xbf },
            { 0x7a, 0xbf, 0xff },
            { 0xff, 0xbf, 0x7a }
            };

      NEvent* nevent   = (NEvent*) item;
      Event event = nevent->event();

      QColor color;
      if (nevent->part() != curPart)
            p.setBrush(lightGray);
      else {
            if (item->isMoving()) {
                  p.setBrush(gray);
                  p.drawRect(r);
                  p.setBrush(NoBrush);
                  p.drawRect(item->mp().x(), item->mp().y() - item->height()/2, item->width(), item->height());
                  return;
                  }
            else if (item->isSelected()) {
                  p.setBrush(black);
                  }
            else {
                  color.setRgb(0, 0, 255);
                  switch(colorMode) {
                        case 0:
                              break;
                        case 1:     // pitch
                              {
                              Triple* c = &color1[event.pitch() % 12];
                              color.setRgb(c->r, c->g, c->b);
                              }
                              break;
                        case 2:     // velocity
                              {
                              int velo = event.velo();
                              if (velo < 64)
                                    color.setRgb(velo*4, 0, 0xff);
                              else
                                    color.setRgb(0xff, 0, (127-velo) * 4);
                              }
                              break;
                        }
                  p.setBrush(color);
                  }
            }
      p.drawRect(r);
      }

//---------------------------------------------------------
//   viewMouseDoubleClickEvent
//---------------------------------------------------------

void PianoCanvas::viewMouseDoubleClickEvent(QMouseEvent* event)
      {
      if ((_tool != PointerTool) && (event->button() != QMouseEvent::LeftButton)) {
            mousePress(event);
            return;
            }
      }

//---------------------------------------------------------
//   moveItem
//    called after moving an object
//---------------------------------------------------------

bool PianoCanvas::moveItem(CItem* item, const QPoint& pos, DragType dtype)
      {
      NEvent* nevent = (NEvent*) item;
      Event event    = nevent->event();
      int npitch     = y2pitch(pos.y());
      Event newEvent = event.clone();
      int x          = pos.x();
      if (x < 0)
            x = 0;

      if (event.pitch() != npitch && _playEvents) {
            int port    = track()->outPort();
            int channel = track()->outChannel();
            // release note:
            MidiPlayEvent ev1(0, port, channel, 0x90, event.pitch() + track()->transposition, 0);
            audio->msgPlayMidiEvent(&ev1);
            MidiPlayEvent ev2(0, port, channel, 0x90, npitch + track()->transposition, event.velo());
            audio->msgPlayMidiEvent(&ev2);
            }
      //Part* part = nevent->part(); //
      Part * part = Canvas::part();  // part can be dynamically recreated, ask the authority
      newEvent.setPitch(npitch);
      int ntick = editor->rasterVal(x) - part->tick();
      if (ntick < 0)
            ntick = 0;
      newEvent.setTick(ntick);
      newEvent.setLenTick(event.lenTick());



      int modified=SC_EVENT_MODIFIED;
      song->startUndo();
      int diff = newEvent.endTick()-part->lenTick();
      if (diff > 0)  {// too short part? extend it
            // if there are several events that are moved outside the part, it will be recreated for each
            // so the part _in_ the event will not be valid, ask the authority.
            //Part* newPart = part->clone();
            Part* newPart = Canvas::part()->clone();
            
            newPart->setLenTick(newPart->lenTick()+diff);
            audio->msgChangePart(Canvas::part(), newPart,false);
            modified=modified|SC_PART_MODIFIED;
            part = newPart; // reassign
            }

      if (dtype == MOVE_COPY || dtype == MOVE_CLONE)
            audio->msgAddEvent(newEvent, part, false); 
      else
            audio->msgChangeEvent(event, newEvent, part, false); 
      song->endUndo(modified);

      return true;
      }

//---------------------------------------------------------
//   newItem(p, state)
//---------------------------------------------------------

CItem* PianoCanvas::newItem(const QPoint& p, int)
      {
      //printf("newItem point\n");
      int pitch = y2pitch(p.y());
      int tick  = editor->rasterVal1(p.x());
      int len   = p.x() - tick;
      tick     -= curPart->tick();
      if (tick < 0) 
            tick=0;
      Event e =  Event(Note);
      e.setTick(tick);
      e.setPitch(pitch);
      e.setVelo(curVelo);
      e.setLenTick(len);
      return new NEvent(e, curPart, pitch2y(pitch));
      }

void PianoCanvas::newItem(CItem* item, bool noSnap)
      {
      //printf("newItem citem\n");
      NEvent* nevent = (NEvent*) item;
      Event event    = nevent->event();
      int x = item->x();
      if (x<0)
            x=0;
      int w = item->width();

      if (!noSnap) {
            x = editor->rasterVal1(x); //round down
            w = editor->rasterVal(x + w) - x;
            if (w == 0)
                  w = editor->raster();
            }
      Part* part = nevent->part();
      event.setTick(x - part->tick());
      event.setLenTick(w);
      event.setPitch(y2pitch(item->y()));
      
      song->startUndo();
      int modified=SC_EVENT_MODIFIED;
      int diff = event.endTick()-part->lenTick();
      if (diff > 0)  {// too short part? extend it
            //printf("extend Part!\n");
            Part* newPart = part->clone();
            newPart->setLenTick(newPart->lenTick()+diff);
            audio->msgChangePart(part, newPart,false);
            modified=modified|SC_PART_MODIFIED;
            part = newPart; // reassign
            }
      audio->msgAddEvent(event, part,false); 
      song->endUndo(modified);
      }

//---------------------------------------------------------
//   resizeItem
//---------------------------------------------------------

void PianoCanvas::resizeItem(CItem* item, bool noSnap)         // experimental changes to try dynamically extending parts
      {
      //printf("resizeItem!\n");
      NEvent* nevent = (NEvent*) item;
      Event event    = nevent->event();
      Event newEvent = event.clone();
      int len;

      Part* part = nevent->part();

      if (noSnap)
            len = nevent->width();
      else {
            //Part* part = nevent->part();
            unsigned tick = event.tick() + part->tick();
            len = editor->rasterVal(tick + nevent->width()) - tick;
            if (len <= 0)
                  len = editor->raster();
      }
      song->startUndo();
      int modified=SC_EVENT_MODIFIED;
      //printf("event.tick()=%d len=%d part->lenTick()=%d\n",event.endTick(),len,part->lenTick());
      int diff = event.tick()+len-part->lenTick();
      if (diff > 0)  {// too short part? extend it
            //printf("extend Part!\n");
            Part* newPart = part->clone();
            newPart->setLenTick(newPart->lenTick()+diff);
            audio->msgChangePart(part, newPart,false);
            modified=modified|SC_PART_MODIFIED;
            part = newPart; // reassign
            }

      newEvent.setLenTick(len);
      audio->msgChangeEvent(event, newEvent, nevent->part(),false); 
      song->endUndo(modified);
      }

//---------------------------------------------------------
//   deleteItem
//---------------------------------------------------------

bool PianoCanvas::deleteItem(CItem* item)
      {
      NEvent* nevent = (NEvent*) item;
      if (nevent->part() == curPart) {
            Event ev = nevent->event();
            audio->msgDeleteEvent(ev, curPart);
            return true;
            }
      return false;
      }

//---------------------------------------------------------
//   pianoCmd
//---------------------------------------------------------

void PianoCanvas::pianoCmd(int cmd)
      {
      switch(cmd) {
            case CMD_LEFT:
                  {
                  int frames = pos[0] - editor->rasterStep(pos[0]);
                  if (frames < 0)
                        frames = 0;
                  Pos p(frames,true);
                  song->setPos(0, p, true, true, true); //CDW
                  }
                  break;
            case CMD_RIGHT:
                  {
                  Pos p(pos[0] + editor->rasterStep(pos[0]), true);
                  //if (p > part->tick())
                  //      p = part->tick();
                  song->setPos(0, p, true, true, true); //CDW
                  }
                  break;
            case CMD_INSERT:
                  {
                  if (pos[0] < start() || pos[0] >= end())
                        break;
                  MidiPart* part = (MidiPart*)curPart;

                  if (part == 0)
                        break;
                  song->startUndo();
                  EventList* el = part->events();

                  std::list <Event> elist;
                  for (iEvent e = el->lower_bound(pos[0] - part->tick()); e != el->end(); ++e)
                        elist.push_back((Event)e->second);
                  for (std::list<Event>::iterator i = elist.begin(); i != elist.end(); ++i) {
                        Event event = *i;
                        Event newEvent = event.clone();
                        newEvent.setTick(event.tick() + editor->raster());// - part->tick());
                        audio->msgChangeEvent(event, newEvent, part, false);
                        }
                  song->endUndo(SC_EVENT_MODIFIED);
                  Pos p(editor->rasterVal(pos[0] + editor->rasterStep(pos[0])), true);
                  song->setPos(0, p, true, false, true);
                  }
                  return;
            case CMD_DELETE:
                  if (pos[0] < start() || pos[0] >= end())
                        break;
                  {
                  MidiPart* part = (MidiPart*)curPart;
                  if (part == 0)
                        break;
                  song->startUndo();
                  EventList* el = part->events();

                  std::list<Event> elist;
                  for (iEvent e = el->lower_bound(pos[0]); e != el->end(); ++e)
                        elist.push_back((Event)e->second);
                  for (std::list<Event>::iterator i = elist.begin(); i != elist.end(); ++i) {
                        Event event = *i;
                        Event newEvent = event.clone();
                        newEvent.setTick(event.tick() - editor->raster() - part->tick());
                        audio->msgChangeEvent(event, newEvent, part, false);
                        }
                  song->endUndo(SC_EVENT_MODIFIED);
                  Pos p(editor->rasterVal(pos[0] - editor->rasterStep(pos[0])), true);
                  song->setPos(0, p, true, false, true);
                  }
                  break;
            }
      }

//---------------------------------------------------------
//   pianoPressed
//---------------------------------------------------------

void PianoCanvas::pianoPressed(int pitch, int velocity, bool shift)
      {
      int port      = track()->outPort();
      int channel   = track()->outChannel();
      pitch        += track()->transposition;

      // play note:
      //MidiPlayEvent e(0, port, channel, 0x90, pitch, 127);
      MidiPlayEvent e(0, port, channel, 0x90, pitch, velocity);
      audio->msgPlayMidiEvent(&e);

      if (_steprec && pos[0] >= start_tick && pos[0] < end_tick) {
            if (curPart == 0)
                  return;
            int len  = editor->raster();
            unsigned tick = pos[0] - curPart->tick(); //CDW
            if (shift)
                  tick -= editor->rasterStep(tick);
            Event e(Note);
            e.setTick(tick);
            e.setPitch(pitch);
            e.setVelo(127);
            e.setLenTick(len);
            audio->msgAddEvent(e, curPart);
            tick += editor->rasterStep(tick) + curPart->tick();
            if (tick != song->cpos()) {
                  Pos p(tick, true);
                  song->setPos(0, p, true, false, true);
                  }
            }
      }

//---------------------------------------------------------
//   pianoReleased
//---------------------------------------------------------

void PianoCanvas::pianoReleased(int pitch, bool)
      {
      int port    = track()->outPort();
      int channel = track()->outChannel();
      pitch      += track()->transposition;

      // release key:
      MidiPlayEvent e(0, port, channel, 0x90, pitch, 0);
      audio->msgPlayMidiEvent(&e);
      }

//---------------------------------------------------------
//   drawTickRaster
//---------------------------------------------------------

void drawTickRaster(QPainter& p, int x, int y, int w, int h, int raster)
      {
      int bar1, bar2, beat;
      unsigned tick;
      sigmap.tickValues(x, &bar1, &beat, &tick);
      sigmap.tickValues(x+w, &bar2, &beat, &tick);
      ++bar2;
      int y2 = y + h;
      for (int bar = bar1; bar < bar2; ++bar) {
            unsigned x = sigmap.bar2tick(bar, 0, 0);
            p.setPen(Qt::black);
            p.drawLine(x, y, x, y2);
            int z, n;
            sigmap.timesig(x, z, n);
            int q = p.xForm(QPoint(raster, 0)).x() - p.xForm(QPoint(0, 0)).x();
            int qq = raster;
            if (q < 8)        // grid too dense
                  qq *= 2;
            //switch (quant) {
            //      case 32:
            //      case 48:
            //      case 64:
            //      case 96:
            //      case 192:         // 8tel
            //      case 128:         // 8tel Triolen
            //      case 288:
            p.setPen(Qt::lightGray);
            if (raster>=4) {
                        int xx = x + qq;
                        int xxx = sigmap.bar2tick(bar, z, 0);
                        while (xx <= xxx) {
                               p.drawLine(xx, y, xx, y2);
                               xx += qq;
                               }
                        xx = xxx;
                        }
            //            break;
            //      default:
            //            break;
            // }
            p.setPen(Qt::gray);
            for (int beat = 1; beat < z; beat++) {
                        int xx = sigmap.bar2tick(bar, beat, 0);
                        p.drawLine(xx, y, xx, y2);
                        }

            }
      }

//---------------------------------------------------------
//   draw
//---------------------------------------------------------

void PianoCanvas::drawCanvas(QPainter& p, const QRect& rect)
      {
      int x = rect.x();
      int y = rect.y();
      int w = rect.width();
      int h = rect.height();

      //---------------------------------------------------
      //  horizontal lines
      //---------------------------------------------------

      int yy  = ((y-1) / KH) * KH + KH;
      int key = 75 - (yy / KH);
      for (; yy < y + h; yy += KH) {
            switch (key % 7) {
                  case 0:
                  case 3:
                        p.setPen(black);
                        break;
                  default:
                        p.setPen(gray);
                        break;
                  }
            p.drawLine(x, yy, x + w, yy);
            --key;
            }

      //---------------------------------------------------
      // vertical lines
      //---------------------------------------------------

      drawTickRaster(p, x, y, w, h, editor->raster());
      }

//---------------------------------------------------------
//   cmd
//    pulldown menu commands
//---------------------------------------------------------

void PianoCanvas::cmd(int cmd, int quantStrength,
   int quantLimit, bool quantLen, int range)
      {
      cmdRange = range;
      switch (cmd) {
            case CMD_CUT:
                  copy();
                  song->startUndo();
                  for (iCItem i = items.begin(); i != items.end(); ++i) {
                        if (!(i->second->isSelected()))
                              continue;
                        NEvent* e = (NEvent*)(i->second);
                        Event ev  = e->event();
                        audio->msgDeleteEvent(ev, e->part(), false);
                        }
                  song->endUndo(SC_EVENT_REMOVED);
                  break;
            case CMD_COPY:
                  copy();
                  break;
            case CMD_PASTE:
                  paste();
                  break;
            case CMD_DEL:
                  if (selectionSize()) {
                        song->startUndo();
                        for (iCItem i = items.begin(); i != items.end(); ++i) {
                              if (!i->second->isSelected())
                                    continue;
                              Event ev = i->second->event();
                              audio->msgDeleteEvent(ev, i->second->part(), false);
                              }
                        song->endUndo(SC_EVENT_REMOVED);
                        }
                  return;
            case CMD_OVER_QUANTIZE:     // over quantize
                  quantize(100, 1, quantLen);
                  break;
            case CMD_ON_QUANTIZE:     // note on quantize
                  quantize(50, 1, false);
                  break;
            case CMD_ONOFF_QUANTIZE:     // note on/off quantize
                  quantize(50, 1, true);
                  break;
            case CMD_ITERATIVE_QUANTIZE:     // Iterative Quantize
                  quantize(quantStrength, quantLimit, quantLen);
                  break;
            case CMD_SELECT_ALL:     // select all
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        if (!k->second->isSelected())
                              selectItem(k->second, true);
                        }
                  break;
            case CMD_SELECT_NONE:     // select none
                  deselectAll();
                  break;
            case CMD_SELECT_INVERT:     // invert selection
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        selectItem(k->second, !k->second->isSelected());
                        }
                  break;
            case CMD_SELECT_ILOOP:     // select inside loop
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        NEvent* nevent = (NEvent*)(k->second);
                        Event event    = nevent->event();
                        unsigned tick  = event.tick();
                        if (tick < song->lpos() || tick >= song->rpos())
                              selectItem(k->second, false);
                        else
                              selectItem(k->second, true);
                        }
                  break;
            case CMD_SELECT_OLOOP:     // select outside loop
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        NEvent* nevent = (NEvent*)(k->second);
                        Event event    = nevent->event();
                        unsigned tick  = event.tick();
                        if (tick < song->lpos() || tick >= song->rpos())
                              selectItem(k->second, true);
                        else
                              selectItem(k->second, false);
                        }
                  break;
            case CMD_MODIFY_GATE_TIME:
                  {
                  GateTime w(this);
                  w.setRange(range);
                  if (!w.exec())
                        break;
                  int range  = w.range();        // all, selected, looped, sel+loop
                  int rate   = w.rateVal();
                  int offset = w.offsetVal();

                  song->startUndo();
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        NEvent* nevent =(NEvent*)(k->second);
                        Event event    = nevent->event();
                        if (event.type() != Note)
                              continue;
                        unsigned tick = event.tick();
                        bool selected = k->second->isSelected();
                        bool inLoop   = (tick >= song->lpos()) && (tick < song->rpos());

                        if ((range == 0)
                           || (range == 1 && selected)
                           || (range == 2 && inLoop)
                           || (range == 3 && selected && inLoop)) {
                              int len   = event.lenTick();

                              len = rate ? (len * 100) / rate : 1;
                              len += offset;
                              if (len <= 1)
                                    len = 1;

                              if (event.lenTick() != len) {
                                    Event newEvent = event.clone();
                                    newEvent.setLenTick(len);
                                    audio->msgChangeEvent(event, newEvent, nevent->part(), false);
                                    }
                              }
                        }
                  song->endUndo(SC_EVENT_MODIFIED);
                  }
                  break;

            case CMD_MODIFY_VELOCITY:
                  {
                  Velocity w(this);
                  w.setRange(range);
                  if (!w.exec())
                        break;
                  int range  = w.range();        // all, selected, looped, sel+loop
                  int rate   = w.rateVal();
                  int offset = w.offsetVal();

                  song->startUndo();
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        NEvent* nevent = (NEvent*)(k->second);
                        Event event    = nevent->event();
                        if (event.type() != Note)
                              continue;
                        unsigned tick      = event.tick();
                        bool selected = k->second->isSelected();
                        bool inLoop   = (tick >= song->lpos()) && (tick < song->rpos());

                        if ((range == 0)
                           || (range == 1 && selected)
                           || (range == 2 && inLoop)
                           || (range == 3 && selected && inLoop)) {
                              int velo = event.velo();

                              //velo = rate ? (velo * 100) / rate : 64;
                              velo = (velo * rate) / 100;
                              velo += offset;

                              if (velo <= 0)
                                    velo = 1;
                              if (velo > 127)
                                    velo = 127;
                              if (event.velo() != velo) {
                                    Event newEvent = event.clone();
                                    newEvent.setVelo(velo);
                                    audio->msgChangeEvent(event, newEvent, nevent->part(), false);
                                    }
                              }
                        }
                  song->endUndo(SC_EVENT_MODIFIED);
                  }
                  break;

            case CMD_FIXED_LEN: //Set notes to the length specified in the drummap
                  if (!selectionSize())
                        break;
                  song->startUndo();
                  for (iCItem k = items.begin(); k != items.end(); ++k) {
                        if (k->second->isSelected()) {
                              NEvent* nevent = (NEvent*)(k->second);
                              Event event = nevent->event();
                              Event newEvent = event.clone();
                              newEvent.setLenTick(editor->raster());
                              audio->msgChangeEvent(event, newEvent, nevent->part() , false);
                              }
                        }
                  song->endUndo(SC_EVENT_MODIFIED);
                  break;

            case CMD_CRESCENDO:
            case CMD_TRANSPOSE:
            case CMD_THIN_OUT:
            case CMD_ERASE_EVENT:
            case CMD_NOTE_SHIFT:
            case CMD_MOVE_CLOCK:
            case CMD_COPY_MEASURE:
            case CMD_ERASE_MEASURE:
            case CMD_DELETE_MEASURE:
            case CMD_CREATE_MEASURE:
                  break;
            default:
//                  printf("unknown ecanvas cmd %d\n", cmd);
                  break;
            }
      updateSelection();
      redraw();
      }

//---------------------------------------------------------
//   quantize
//---------------------------------------------------------

void PianoCanvas::quantize(int strength, int limit, bool quantLen)
      {
      song->startUndo();
      for (iCItem k = items.begin(); k != items.end(); ++k) {
            NEvent* nevent = (NEvent*)(k->second);
            Event event    = nevent->event();
            Part* part     = nevent->part();
            if (event.type() != Note)
                  continue;

            if ((cmdRange & CMD_RANGE_SELECTED) && !k->second->isSelected())
                  continue;

            unsigned tick = event.tick() + part->tick();

            if ((cmdRange & CMD_RANGE_LOOP)
               && ((tick < song->lpos() || tick >= song->rpos())))
                  continue;

            int len   = event.lenTick();
            int tick2 = tick + len;

            // quant start position
            int diff  = sigmap.raster(tick, editor->quant()) - tick;
            if (abs(diff) > limit)
                  tick += ((diff * strength) / 100);

            // quant len
            diff = sigmap.raster(tick2, editor->quant()) - tick2;
            if (quantLen && (abs(diff) > limit))
                  len += ((diff * strength) / 100);

            // something changed?
            if (((event.tick() + part->tick()) != tick) || (event.lenTick() != len)) {
                  Event newEvent = event.clone();
                  newEvent.setTick(tick - part->tick());
                  newEvent.setLenTick(len);
                  audio->msgChangeEvent(event, newEvent, part, false);
                  }
            }
      song->endUndo(SC_EVENT_MODIFIED);
      }

//---------------------------------------------------------
//   midiNote
//---------------------------------------------------------

void PianoCanvas::midiNote(int pitch, int velo)
      {
      if (_midiin && _steprec && curPart
         && !audio->isPlaying() && velo && pos[0] >= start_tick
         && pos[0] < end_tick
         && !(globalKeyState & AltButton)) {
            int len            = editor->quant();
            unsigned tick      = pos[0]; //CDW
            unsigned starttick = tick;
            if (globalKeyState & ShiftButton)
                  tick -= editor->rasterStep(tick);

            //
            // extend len of last note?
            //
            EventList* events = curPart->events();
            if (globalKeyState & ControlButton) {
                  for (iEvent i = events->begin(); i != events->end(); ++i) {
                        Event ev = i->second;
                        if (!ev.isNote())
                              continue;
                        if (ev.pitch() == pitch && ((ev.tick() + ev.lenTick()) == (int)starttick)) {
                              Event e = ev.clone();
                              e.setLenTick(ev.lenTick() + editor->rasterStep(starttick));
                              audio->msgChangeEvent(ev, e, curPart);
                              tick += editor->rasterStep(tick);
                              if (tick != song->cpos()) {
                                    Pos p(tick, true);
                                    song->setPos(0, p, true, false, true);
                                    }
                              return;
                              }
                        }
                  }

            //
            // if we already entered the note, delete it
            //
            EventRange range = events->equal_range(tick);
            for (iEvent i = range.first; i != range.second; ++i) {
                  Event ev = i->second;
                  if (ev.isNote() && ev.pitch() == pitch) {
                        audio->msgDeleteEvent(ev, curPart);
                        if (globalKeyState & ShiftButton)
                              tick += editor->rasterStep(tick);
                        return;
                        }
                  }
            Event e(Note);
            e.setTick(tick - curPart->tick());
            e.setPitch(pitch);
            e.setVelo(velo);
            e.setLenTick(len);
            audio->msgAddEvent(e, curPart);
            tick += editor->rasterStep(tick);
            if (tick != song->cpos()) {
                  Pos p(tick, true);
                  song->setPos(0, p, true, false, true);
                  }
            }
      }

//---------------------------------------------------------
//   getTextDrag
//---------------------------------------------------------

QTextDrag* PianoCanvas::getTextDrag(QWidget* parent)
      {
      //---------------------------------------------------
      //   generate event list from selected events
      //---------------------------------------------------

      EventList el;
      unsigned startTick = MAXINT;
      for (iCItem i = items.begin(); i != items.end(); ++i) {
            if (!i->second->isSelected())
                  continue;
            NEvent* ne = (NEvent*)(i->second);
            Event e   = ne->event();
            if (startTick == MAXINT)
                  startTick = e.tick();
            el.add(e);
            }

      //---------------------------------------------------
      //    write events as XML into tmp file
      //---------------------------------------------------

      FILE* tmp = tmpfile();
      if (tmp == 0) {
            fprintf(stderr, "PianoCanvas::copy() fopen failed: %s\n",
               strerror(errno));
            return 0;
            }
      Xml xml(tmp);

      int level = 0;
      xml.tag(level++, "eventlist");
      for (ciEvent e = el.begin(); e != el.end(); ++e)
            e->second.write(level, xml, -startTick);
      xml.etag(--level, "eventlist");

      //---------------------------------------------------
      //    read tmp file into QTextDrag Object
      //---------------------------------------------------

      fflush(tmp);
      struct stat f_stat;
      if (fstat(fileno(tmp), &f_stat) == -1) {
            fprintf(stderr, "PianoCanvas::copy() fstat failes:<%s>\n",
               strerror(errno));
            fclose(tmp);
            return 0;
            }
      int n = f_stat.st_size;
      char* fbuf  = (char*)mmap(0, n+1, PROT_READ|PROT_WRITE,
         MAP_PRIVATE, fileno(tmp), 0);
      fbuf[n] = 0;
      QTextDrag* drag = new QTextDrag(QString(fbuf), parent);
      drag->setSubtype("eventlist");
      munmap(fbuf, n);
      fclose(tmp);
      return drag;
      }

//---------------------------------------------------------
//   copy
//    cut copy paste
//---------------------------------------------------------

void PianoCanvas::copy()
      {
      QTextDrag* drag = getTextDrag(0);
      if (drag)
            QApplication::clipboard()->setData(drag, QClipboard::Clipboard);
      }

//---------------------------------------------------------
//   paste
//---------------------------------------------------------

void PianoCanvas::pasteAt(const QString& pt, int pos)
      {
      const char* p = pt.latin1();
      Xml xml(p);
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "eventlist") {
                              song->startUndo();
                              EventList* el = new EventList();
                              el->read(xml, "eventlist", true);
                              int modified = SC_EVENT_INSERTED;
                              for (iEvent i = el->begin(); i != el->end(); ++i) {
                                    Event e = i->second;
                                    int tick = e.tick() + pos - curPart->tick();
                                    if (tick<0) { 
                                            printf("ERROR: trying to add event before current part!\n");
                                            song->endUndo(SC_EVENT_INSERTED);
                                            delete el;
                                            return;
                                            }
                                    int diff = e.endTick()-curPart->lenTick();
                                    if (diff > 0)  {// too short part? extend it
                                            Part* newPart = curPart->clone();
                                            newPart->setLenTick(newPart->lenTick()+diff);
                                            audio->msgChangePart(curPart, newPart,false);
                                            modified=modified|SC_PART_MODIFIED;
                                            curPart = newPart; // reassign
                                            }
                                    e.setTick(tick);
                                    audio->msgAddEvent(e, curPart, false);
                                    }
                              song->endUndo(modified);
                              delete el;
                              return;
                              }
                        else
                              xml.unknown("pasteAt");
                        break;
                  case Xml::Attribut:
                  case Xml::TagEnd:
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   paste
//    paste events
//---------------------------------------------------------

void PianoCanvas::paste()
      {
      QCString subtype("eventlist");
      QMimeSource* ms = QApplication::clipboard()->data(QClipboard::Clipboard);
      QString pt;
      if (!QTextDrag::decode(ms, pt, subtype)) {
            printf("cannot paste: bad data type\n");
            return;
            }
      pasteAt(pt, song->cpos());
      }

//---------------------------------------------------------
//   startDrag
//---------------------------------------------------------

void PianoCanvas::startDrag(CItem* /* item*/, bool copymode)
      {
      QTextDrag* drag = getTextDrag(this);
      if (drag) {
//            QApplication::clipboard()->setData(drag, QClipboard::Clipboard);

            if (copymode)
                  drag->dragCopy();
            else
                  drag->dragMove();
            }
      }

//---------------------------------------------------------
//   dragEnterEvent
//---------------------------------------------------------

void PianoCanvas::dragEnterEvent(QDragEnterEvent* event)
      {
      event->accept(QTextDrag::canDecode(event));
      }

//---------------------------------------------------------
//   dragMoveEvent
//---------------------------------------------------------

void PianoCanvas::dragMoveEvent(QDragMoveEvent*)
      {
//      printf("drag move %x\n", this);
      }

//---------------------------------------------------------
//   dragLeaveEvent
//---------------------------------------------------------

void PianoCanvas::dragLeaveEvent(QDragLeaveEvent*)
      {
//      printf("drag leave\n");
      }

//---------------------------------------------------------
//   dropEvent
//---------------------------------------------------------

void PianoCanvas::viewDropEvent(QDropEvent* event)
      {
      QString text;
      if (event->source() == this) {
            printf("local DROP\n");
            return;
            }
      if (QTextDrag::decode(event, text)) {
            int x = editor->rasterVal(event->pos().x());
            if (x < 0)
                  x = 0;
            pasteAt(text, x);
            }
      else {
            printf("cannot decode drop\n");
            }
      }

//---------------------------------------------------------
//   itemPressed
//---------------------------------------------------------

void PianoCanvas::itemPressed(const CItem* item)
      {
      if (!_playEvents)
            return;

      int port         = track()->outPort();
      int channel      = track()->outChannel();
      NEvent* nevent   = (NEvent*) item;
      Event event      = nevent->event();
      playedPitch      = event.pitch() + track()->transposition;
      int velo         = event.velo();

      // play note:
      MidiPlayEvent e(0, port, channel, 0x90, playedPitch, velo);
      audio->msgPlayMidiEvent(&e);
      }

//---------------------------------------------------------
//   itemReleased
//---------------------------------------------------------

void PianoCanvas::itemReleased(const CItem*, const QPoint&)
      {
      if (!_playEvents)
            return;
      int port    = track()->outPort();
      int channel = track()->outChannel();

      // release note:
      MidiPlayEvent ev(0, port, channel, 0x90, playedPitch, 0);
      audio->msgPlayMidiEvent(&ev);
      playedPitch = -1;
      }

//---------------------------------------------------------
//   itemMoved
//---------------------------------------------------------

void PianoCanvas::itemMoved(const CItem* item, const QPoint& pos)
      {
      int npitch = y2pitch(pos.y());
      if ((playedPitch != -1) && (playedPitch != npitch)) {
            int port         = track()->outPort();
            int channel      = track()->outChannel();
            NEvent* nevent   = (NEvent*) item;
            Event event      = nevent->event();

            // release note:
            MidiPlayEvent ev1(0, port, channel, 0x90, playedPitch, 0);
            audio->msgPlayMidiEvent(&ev1);
            // play note:
            MidiPlayEvent e2(0, port, channel, 0x90, npitch + track()->transposition, event.velo());
            audio->msgPlayMidiEvent(&e2);
            playedPitch = npitch + track()->transposition;
            }
      }

//---------------------------------------------------------
//   curPartChanged
//---------------------------------------------------------

void PianoCanvas::curPartChanged()
      {
      editor->setCaption(getCaption());
      }

//---------------------------------------------------------
//   modifySelected
//---------------------------------------------------------

void PianoCanvas::modifySelected(NoteInfo::ValType type, int delta)
      {
      audio->msgIdle(true);
      song->startUndo();
      for (iCItem i = items.begin(); i != items.end(); ++i) {
            if (!(i->second->isSelected()))
                  continue;
            NEvent* e   = (NEvent*)(i->second);
            Event event = e->event();
            if (event.type() != Note)
                  continue;

            MidiPart* part = (MidiPart*)(e->part());
            Event newEvent = event.clone();

            switch (type) {
                  case NoteInfo::VAL_TIME:
                        {
                        int newTime = event.tick() + delta;
                        if (newTime < 0)
                           newTime = 0;
                        newEvent.setTick(newTime);
                        }
                        break;
                  case NoteInfo::VAL_LEN:
                        {
                        int len = event.lenTick() + delta;
                        if (len < 1)
                              len = 1;
                        newEvent.setLenTick(len);
                        }
                        break;
                  case NoteInfo::VAL_VELON:
                        {
                        int velo = event.velo() + delta;
                        if (velo > 127)
                              velo = 127;
                        else if (velo < 0)
                              velo = 0;
                        newEvent.setVelo(velo);
                        }
                        break;
                  case NoteInfo::VAL_VELOFF:
                        {
                        int velo = event.veloOff() + delta;
                        if (velo > 127)
                              velo = 127;
                        else if (velo < 0)
                              velo = 0;
                        newEvent.setVeloOff(velo);
                        }
                        break;
                  case NoteInfo::VAL_PITCH:
                        {
                        int pitch = event.pitch() + delta;
                        if (pitch > 127)
                              pitch = 127;
                        else if (pitch < 0)
                              pitch = 0;
                        newEvent.setPitch(pitch);
                        }
                        break;
                  }
            song->changeEvent(event, newEvent, part);
            song->undoOp(UndoOp::ModifyEvent, newEvent, event, part);
            }
      song->endUndo(SC_EVENT_MODIFIED);
      audio->msgIdle(false);
      }

