/*
 * Implementation of the proxy that uses the best available proxy.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include <config.h>

#include "ma-any-proxy.h"
#include "ma-rhythmbox-proxy.h"

#ifdef WITH_XMMS2
#  include "ma-xmms2-proxy.h"
#endif

#ifdef WITH_DBUS
#  include "ma-banshee-proxy.h"
#  include "ma-muine-proxy.h"
#endif

#include <gtk/gtkcelllayout.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkliststore.h>

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_ANY_PROXY, MaAnyProxyPrivate))


typedef struct _MaAnyProxyPrivate	MaAnyProxyPrivate;


struct _MaAnyProxyPrivate
{
	GSList *proxies;
	MaProxy *active;
};

static MaProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void	ma_any_proxy_class_init (MaAnyProxyClass *klass);
static void	ma_any_proxy_init (MaAnyProxy *bproxy);

static void	ma_any_proxy_dispose (GObject *object);
static void	ma_any_proxy_finalize (GObject *object);

static void	ma_any_proxy_toggle_playback (MaProxy *proxy);
static void	ma_any_proxy_previous (MaProxy *proxy);
static void	ma_any_proxy_next (MaProxy *proxy);

static void	ma_any_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void	ma_any_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void	ma_any_proxy_enable (MaProxy *proxy);
static void	ma_any_proxy_disable (MaProxy *proxy);

static void	ma_any_proxy_launch (MaProxy *proxy);

static MaProxy *get_preferred_proxy (MaAnyProxy *aproxy);

static void	choose_active_proxy (MaAnyProxy *aproxy);

static void	notify_cb (MaProxy *proxy, GParamSpec *pspec, MaAnyProxy *aproxy);

static void	preferred_player_changed_cb (GConfClient *client,
					     guint cnxn_id,
					     GConfEntry *entry,
					     gpointer user_data);

static void	error_reported_cb (MaProxy *proxy,
				   const gchar *primary,
				   const gchar *secondary,
				   gboolean important,
				   MaAnyProxy *aproxy);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_any_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaAnyProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_any_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaAnyProxy),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_any_proxy_init,		/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_PROXY, "MaAnyProxy", &info, 0);
	}

	return type;
}

static void
ma_any_proxy_class_init (MaAnyProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_any_proxy_dispose;
	object_class->finalize = ma_any_proxy_finalize;

	proxy_class->toggle_playback = ma_any_proxy_toggle_playback;
	proxy_class->previous = ma_any_proxy_previous;
	proxy_class->next = ma_any_proxy_next;

	proxy_class->rate_song = ma_any_proxy_rate_song;

	proxy_class->prepare_prefs = ma_any_proxy_prepare_prefs;

	proxy_class->enable = ma_any_proxy_enable;
	proxy_class->disable = ma_any_proxy_disable;

	proxy_class->launch = ma_any_proxy_launch;

	g_type_class_add_private (klass, sizeof (MaAnyProxyPrivate));
}

static void
ma_any_proxy_init (MaAnyProxy *aproxy)
{
	MaAnyProxyPrivate *priv = GET_PRIVATE (aproxy);

	priv->proxies = NULL;
	priv->active = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_any_proxy_new (MaConf *conf)
{
	MaAnyProxy *aproxy;
	MaAnyProxyPrivate *priv;

	GSList *elem;
	MaProxy *preferred;

	aproxy = g_object_new (MA_TYPE_ANY_PROXY,
			       "name", "(any)",
			       "conf", conf,
			       NULL);

	priv = GET_PRIVATE (aproxy);

	/* Proxies are listed in decreasing order of "priority", which is
	 * fairly arbitrary.  Since the list is created from back to front,
	 * the lowest-priority proxies appear first below.
	 */

#ifdef WITH_XMMS2
	priv->proxies = g_slist_prepend (priv->proxies,
					 ma_xmms2_proxy_new (conf));
#endif

#ifdef WITH_DBUS
	priv->proxies = g_slist_prepend (priv->proxies,
					 ma_muine_proxy_new (conf));

	priv->proxies = g_slist_prepend (priv->proxies,
					 ma_banshee_proxy_new (conf));
#endif

	priv->proxies = g_slist_prepend (priv->proxies,
					 ma_rhythmbox_proxy_new (conf));

	for (elem = priv->proxies; elem != NULL; elem = elem->next)
		g_signal_connect (elem->data, "error-reported", G_CALLBACK (error_reported_cb), aproxy);

	preferred = get_preferred_proxy (aproxy);
	_ma_proxy_set_icon_name (MA_PROXY (aproxy), ma_proxy_get_icon_name (preferred));

	ma_conf_add_notify (conf, "preferred_player", preferred_player_changed_cb, aproxy);

	return MA_PROXY (aproxy);
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_any_proxy_dispose (GObject *object)
{
	MaAnyProxy *aproxy = MA_ANY_PROXY (object);
	MaAnyProxyPrivate *priv = GET_PRIVATE (aproxy);

	GSList *elem;

	priv->active = NULL;

	for (elem = priv->proxies; elem != NULL; elem = elem->next)
	{
		/* Prevent a nasty feedback if notify::connected gets triggered. */

		g_signal_handlers_disconnect_by_func (elem->data, notify_cb, aproxy);
		g_signal_handlers_disconnect_by_func (elem->data, error_reported_cb, aproxy);
		g_object_unref (elem->data);
	}

	g_slist_free (priv->proxies);
	priv->proxies = NULL;

	(G_OBJECT_CLASS (parent_class))->dispose (object);
}

static void
ma_any_proxy_finalize (GObject *object)
{
	(G_OBJECT_CLASS (parent_class))->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_any_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	ma_proxy_toggle_playback (GET_PRIVATE (proxy)->active);
}

static void
ma_any_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	ma_proxy_previous (GET_PRIVATE (proxy)->active);
}

static void
ma_any_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	ma_proxy_next (GET_PRIVATE (proxy)->active);
}

static void
ma_any_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	ma_proxy_set_rating (GET_PRIVATE (proxy)->active, rating);
}

static void
ma_any_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	MaConf *conf;

	GtkWidget *preferred;
	GtkCellRenderer *renderer;
	GtkListStore *store;
	GtkTreeIter iter;

	GSList *elem;
	gchar *name;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);
	preferred = glade_xml_get_widget (xml, "preferred");
	store = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_combo_box_set_model (GTK_COMBO_BOX (preferred), GTK_TREE_MODEL (store));

	renderer = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (preferred), renderer, FALSE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (preferred), renderer, "text", 0, NULL);

	for (elem = GET_PRIVATE (proxy)->proxies; elem != NULL; elem = elem->next)
	{
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				    0, ma_proxy_get_name (MA_PROXY (elem->data)),
				    -1);

		ma_proxy_prepare_prefs (MA_PROXY (elem->data), xml);
	}

	ma_conf_bind_string_combo_box (conf, "preferred_player", GTK_COMBO_BOX (preferred), 0);
}

static void
ma_any_proxy_enable (MaProxy *proxy)
{
	GSList *elem;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	for (elem = GET_PRIVATE (proxy)->proxies; elem != NULL; elem = elem->next)
	{
		g_signal_connect (elem->data, "notify", G_CALLBACK (notify_cb), proxy);
		ma_proxy_enable (MA_PROXY (elem->data));
	}
}

static void
ma_any_proxy_disable (MaProxy *proxy)
{
	GSList *elem;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	for (elem = GET_PRIVATE (proxy)->proxies; elem != NULL; elem = elem->next)
	{
		ma_proxy_disable (MA_PROXY (elem->data));
		g_signal_handlers_disconnect_by_func (elem->data, notify_cb, proxy);
	}
}

static void
ma_any_proxy_launch (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_ANY_PROXY (proxy));

	ma_proxy_launch (get_preferred_proxy (MA_ANY_PROXY (proxy)));
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static MaProxy *
get_preferred_proxy (MaAnyProxy *aproxy)
{
	MaAnyProxyPrivate *priv = GET_PRIVATE (aproxy);

	MaConf *conf = _ma_proxy_get_conf (MA_PROXY (aproxy));
	gchar *preferred = ma_conf_get_string (conf, "preferred_player");
	GSList *elem;
	MaProxy *result = NULL;

	if (preferred != NULL)
	{
		for (elem = priv->proxies; elem != NULL; elem = elem->next)
		{
			if (strcmp (ma_proxy_get_name (MA_PROXY (elem->data)), preferred) == 0)
			{
				result = MA_PROXY (elem->data);
				break;
			}
		}
	}

	if (result == NULL)
	{
		g_warning ("Could not find a music player named '%s'", preferred);
		result = MA_PROXY (priv->proxies->data);
	}

	g_free (preferred);
	return result;
}

static void
choose_active_proxy (MaAnyProxy *aproxy)
{
	MaAnyProxyPrivate *priv = GET_PRIVATE (aproxy);

	GSList *elem;

	for (elem = priv->proxies; elem != NULL; elem = elem->next)
	{
		if (ma_proxy_get_state (MA_PROXY (elem->data)) == MA_PROXY_STATE_CONNECTED)
		{
			priv->active = elem->data;

			g_debug ("Now connected to: %s", ma_proxy_get_name (priv->active));

			_ma_proxy_set_connected (MA_PROXY (aproxy), TRUE);

			_ma_proxy_set_title (MA_PROXY (aproxy), ma_proxy_get_title (priv->active));
			_ma_proxy_set_artist (MA_PROXY (aproxy), ma_proxy_get_artist (priv->active));
			_ma_proxy_set_album (MA_PROXY (aproxy), ma_proxy_get_album (priv->active));
			_ma_proxy_set_duration (MA_PROXY (aproxy), ma_proxy_get_duration (priv->active));
			_ma_proxy_set_rating (MA_PROXY (aproxy), ma_proxy_get_rating (priv->active));
			_ma_proxy_set_playing (MA_PROXY (aproxy), ma_proxy_get_playing (priv->active));
			_ma_proxy_set_elapsed (MA_PROXY (aproxy), ma_proxy_get_elapsed (priv->active));

			return;
		}
	}

	priv->active = NULL;
	g_debug ("Now connected to: (nothing)");

	_ma_proxy_set_connected (MA_PROXY (aproxy), FALSE);
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
notify_cb (MaProxy *proxy, GParamSpec *pspec, MaAnyProxy *aproxy)
{
	MaAnyProxyPrivate *priv = GET_PRIVATE (aproxy);

	if (strcmp (pspec->name, "state") == 0)
	{
		choose_active_proxy (aproxy);
	}
	else if (proxy == priv->active)
	{
		if (strcmp (pspec->name, "elapsed") == 0)
		{
			_ma_proxy_set_elapsed (MA_PROXY (aproxy), ma_proxy_get_elapsed (proxy));
		}
		else if (strcmp (pspec->name, "playing") == 0)
		{
			_ma_proxy_set_playing (MA_PROXY (aproxy), ma_proxy_get_playing (proxy));
		}
		else if (strcmp (pspec->name, "rating") == 0)
		{
			_ma_proxy_set_rating (MA_PROXY (aproxy), ma_proxy_get_rating (proxy));
		}
		else if (strcmp (pspec->name, "title") == 0)
		{
			_ma_proxy_set_title (MA_PROXY (aproxy), ma_proxy_get_title (proxy));
		}
		else if (strcmp (pspec->name, "artist") == 0)
		{
			_ma_proxy_set_artist (MA_PROXY (aproxy), ma_proxy_get_artist (proxy));
		}
		else if (strcmp (pspec->name, "album") == 0)
		{
			_ma_proxy_set_album (MA_PROXY (aproxy), ma_proxy_get_album (proxy));
		}
		else if (strcmp (pspec->name, "duration") == 0)
		{
			_ma_proxy_set_duration (MA_PROXY (aproxy), ma_proxy_get_duration (proxy));
		}
	}
}

static void
preferred_player_changed_cb (GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     gpointer user_data)
{
	MaAnyProxy *aproxy = MA_ANY_PROXY (user_data);
	MaProxy *preferred;

	/*
	 * If it changed *to* NULL, the applet is being removed from the
	 * panel, so don't bother updating things.
	 */

	if (gconf_entry_get_value (entry) != NULL)
	{
		preferred = get_preferred_proxy (aproxy);
		_ma_proxy_set_icon_name (MA_PROXY (aproxy), ma_proxy_get_icon_name (preferred));
	}
}

static void
error_reported_cb (MaProxy *proxy,
		   const gchar *primary,
		   const gchar *secondary,
		   gboolean important,
		   MaAnyProxy *aproxy)
{
	const gchar *name = (important) ? "error-reported::important" : "error-reported::normal";
	g_signal_emit_by_name (aproxy, name, primary, secondary, important);
}
