/*
 *  arch-tag: Implementation of rating renderer object
 *
 *  Copyright (C) 2002 Olivier Martin <oleevye@wanadoo.fr>
 *  Various changes to adapt code to Music Applet (C) 2006
 *      Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include <string.h>
#include <eel/eel-graphic-effects.h>
#include <gdk/gdkdrawable.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkiconfactory.h>

#include "ma-rating.h"
#include "ma-stock-icons.h"

/* Offset at the beggining of the widget */
#define X_OFFSET 4

/* Vertical offset */
#define Y_OFFSET 2

/* Number of stars */
#define MAX_SCORE 5

#define COLOR_OFFSET 120

static void ma_rating_class_init (MaRatingClass *class);
static void ma_rating_init (MaRating *label);
static void ma_rating_finalize (GObject *object);
static void ma_rating_get_property (GObject *object,
				    guint param_id,
				    GValue *value,
				    GParamSpec *pspec);
static void ma_rating_set_property (GObject *object,
				    guint param_id,
				    const GValue *value,
				    GParamSpec *pspec);
static void ma_rating_size_request (GtkWidget *widget,
				    GtkRequisition *requisition);
static gboolean ma_rating_expose(GtkWidget *widget, 
				 GdkEventExpose *event);
static double ma_rating_mouse_to_score (GtkWidget *widget);
static gboolean ma_rating_button_press_cb (GtkWidget *widget,
					   GdkEventButton *event,
					   MaRating *rating);
static gboolean ma_rating_enter_notify_cb (GtkWidget *widget,
					   GdkEventCrossing *event,
					   MaRating *rating);
static gboolean ma_rating_motion_notify_cb (GtkWidget *widget,
					    GdkEventMotion *event,
					    MaRating *rating);
static gboolean ma_rating_leave_notify_cb (GtkWidget *widget,
					   GdkEventCrossing *event,
					   MaRating *rating);

struct _MaRatingPrivate
{
	double score;
	double mouseover;

	gboolean inert;

	GdkPixbuf *pix_star;
	GdkPixbuf *pix_dot;
	GdkPixbuf *pix_blank;
};

enum
{
	PROP_0,
	PROP_SCORE,
	PROP_INERT,
};

enum
{
	RATED,
	LAST_SIGNAL
};

static GObjectClass *parent_class = NULL;

static guint ma_rating_signals[LAST_SIGNAL] = { 0 };

GType 
ma_rating_get_type (void)
{
        static GType ma_rating_type = 0;

        if (ma_rating_type == 0) {
                static const GTypeInfo our_info = {
                        sizeof (MaRatingClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) ma_rating_class_init,
                        NULL,
                        NULL, /* class_data */
                        sizeof (MaRating),
                        0, /* n_preallocs */
                        (GInstanceInitFunc) ma_rating_init,
			NULL
                };

                ma_rating_type = g_type_register_static (GTK_TYPE_EVENT_BOX,
							 "MaRating",
							 &our_info, 0);
        }

        return ma_rating_type;
}

static void
ma_rating_class_init (MaRatingClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class;

	widget_class = (GtkWidgetClass*) klass;
	parent_class = g_type_class_peek_parent (klass);
	
	object_class->finalize = ma_rating_finalize;
	object_class->get_property = ma_rating_get_property;
	object_class->set_property = ma_rating_set_property;

	widget_class->expose_event = ma_rating_expose;
	widget_class->size_request = ma_rating_size_request;

	g_object_class_install_property (object_class,
					 PROP_SCORE,
					 g_param_spec_double ("score",
							      "Rating score",
							      "Rating score",
							      0.0, 5.0, 2.5,
							      G_PARAM_READWRITE));

	/* "sensitive" isn't good enough -- we still need right clicks */
	g_object_class_install_property (object_class,
					 PROP_INERT,
					 g_param_spec_boolean ("inert",
						 	       "inert",
							       "Whether the widget should allow user interaction",
							       FALSE,
							       G_PARAM_READWRITE));

	ma_rating_signals[RATED] = 
		g_signal_new ("rated",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (MaRatingClass, rated),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE,
			      1,
			      G_TYPE_DOUBLE);
}

static void
ma_rating_init (MaRating *rating)
{
	GtkWidget *dummy;
	GdkPixbuf *pixbuf;

	rating->priv = g_new0 (MaRatingPrivate, 1);

	/* create the needed icons */
	dummy = gtk_label_new (NULL);
	pixbuf = gtk_widget_render_icon (dummy,
					 MA_STOCK_SET_STAR,
					 GTK_ICON_SIZE_MENU,
					 NULL);
	rating->priv->pix_star = eel_create_colorized_pixbuf 
					(pixbuf,
					 (dummy->style->text[GTK_STATE_NORMAL].red + COLOR_OFFSET) >> 8,
					 (dummy->style->text[GTK_STATE_NORMAL].green + COLOR_OFFSET) >> 8,
					 (dummy->style->text[GTK_STATE_NORMAL].blue + COLOR_OFFSET) >> 8);
	g_object_unref (G_OBJECT (pixbuf));

	pixbuf = gtk_widget_render_icon (dummy,
					 MA_STOCK_NO_STAR,
					 GTK_ICON_SIZE_MENU,
					 NULL);
	rating->priv->pix_blank = eel_create_colorized_pixbuf 
					(pixbuf,
					 dummy->style->text[GTK_STATE_NORMAL].red >> 8,
					 dummy->style->text[GTK_STATE_NORMAL].green >> 8,
					 dummy->style->text[GTK_STATE_NORMAL].blue >> 8);
	g_object_unref (G_OBJECT (pixbuf));

	pixbuf = gtk_widget_render_icon (dummy,
					 MA_STOCK_UNSET_STAR,
					 GTK_ICON_SIZE_MENU,
					 NULL);
	rating->priv->pix_dot = eel_create_colorized_pixbuf (pixbuf,
							     dummy->style->text[GTK_STATE_NORMAL].red >> 8,
							     dummy->style->text[GTK_STATE_NORMAL].green >> 8,
							     dummy->style->text[GTK_STATE_NORMAL].blue >> 8);
	g_object_unref (G_OBJECT (pixbuf));

	gtk_widget_destroy (dummy);

	rating->priv->mouseover = -1.0;

	rating->priv->inert = FALSE;

	/* register some signals */
	gtk_widget_add_events (GTK_WIDGET (rating), GDK_POINTER_MOTION_MASK);
	g_signal_connect (rating,
			  "button_press_event",
			  G_CALLBACK (ma_rating_button_press_cb),
			  rating);
	g_signal_connect (rating,
			  "enter_notify_event",
			  G_CALLBACK (ma_rating_enter_notify_cb),
			  rating);
	g_signal_connect (rating,
			  "motion_notify_event",
			  G_CALLBACK (ma_rating_motion_notify_cb),
			  rating);
	g_signal_connect (rating,
			  "leave_notify_event",
			  G_CALLBACK (ma_rating_leave_notify_cb),
			  rating);
}

static void
ma_rating_finalize (GObject *object)
{
	MaRating *rating;

	rating = MA_RATING (object);

	g_object_unref (G_OBJECT (rating->priv->pix_star));
	g_object_unref (G_OBJECT (rating->priv->pix_dot));
	g_object_unref (G_OBJECT (rating->priv->pix_blank));
	g_free (rating->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
ma_rating_get_property (GObject *object,
			guint param_id,
			GValue *value,
			GParamSpec *pspec)
{
	MaRating *rating = MA_RATING (object);
  
	switch (param_id) {
	case PROP_SCORE:
		g_value_set_double (value, rating->priv->score);
		break;
	case PROP_INERT:
		g_value_set_boolean (value, rating->priv->inert);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}


static void
ma_rating_set_property (GObject *object,
			guint param_id,
			const GValue *value,
			GParamSpec *pspec)
{
	MaRating *rating= MA_RATING (object);
  
	switch (param_id) {
	case PROP_SCORE:
		rating->priv->score = g_value_get_double (value);
		gtk_widget_queue_draw (GTK_WIDGET (rating));
		break;
	case PROP_INERT:
		rating->priv->inert = g_value_get_boolean (value);
		gtk_widget_queue_draw (GTK_WIDGET (rating));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

MaRating *
ma_rating_new ()
{
	MaRating *rating;
  
	rating = g_object_new (MA_TYPE_RATING, NULL);

	g_return_val_if_fail (rating->priv != NULL, NULL);
  
	return rating;
}

static void
ma_rating_size_request (GtkWidget *widget,
			GtkRequisition *requisition)
{
	int icon_size;

	g_return_if_fail (requisition != NULL);

	gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &icon_size, NULL);

	requisition->width = 5 * icon_size + X_OFFSET * 2;
	requisition->height = icon_size + Y_OFFSET * 2;
}

static gboolean
ma_rating_expose (GtkWidget *widget, 
		  GdkEventExpose *event)
{
	int icon_size;
	gboolean rtl;

	g_return_val_if_fail (MA_IS_RATING (widget), FALSE);

	rtl = (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL);

	gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &icon_size, NULL);

	if (GTK_WIDGET_DRAWABLE (widget) == TRUE) {
		int i;
		MaRating *rating = MA_RATING (widget);
		double displayed_score;

		if (rating->priv->mouseover >= 0.0 && !rating->priv->inert)
			displayed_score = rating->priv->mouseover;
		else
			displayed_score = rating->priv->score;

		/* make the widget prettier */
		/* Disabled to reduce clutter */
#if 0
		gtk_paint_flat_box (widget->style, widget->window,
				  GTK_STATE_NORMAL, GTK_SHADOW_IN,
				  NULL, widget, "entry_bg", 0, 0,
				  widget->allocation.width,
				  widget->allocation.height);

		gtk_paint_shadow (widget->style, widget->window,
				  GTK_STATE_NORMAL, GTK_SHADOW_IN,
				  NULL, widget, "text", 0, 0,
				  widget->allocation.width,
				  widget->allocation.height);
#endif

		/* draw a blank area at the beggining, this lets the user click
		 * in this area to unset the rating */
		gdk_draw_pixbuf (widget->window,
				 NULL,
				 rating->priv->pix_blank,
				 0, 0,
				 0, Y_OFFSET,
				 X_OFFSET, icon_size,
				 GDK_RGB_DITHER_NORMAL, 0, 0);


		/* draw the stars */
		for (i = 0; i < MAX_SCORE; i++) {
			GdkPixbuf *pixbuf;
			gint icon_x_offset;

			if (i < displayed_score)
				pixbuf = rating->priv->pix_star;
			else
				pixbuf = rating->priv->pix_dot;

			if (pixbuf == NULL)
				return FALSE;

			if (rtl) {
				icon_x_offset = X_OFFSET + (MAX_SCORE-i-1) * icon_size;
			} else {
				icon_x_offset = X_OFFSET + i * icon_size;
			}
			gdk_draw_pixbuf (widget->window,
					 NULL,
					 pixbuf,
					 0, 0,
					 icon_x_offset, Y_OFFSET,
					 icon_size, icon_size,
					 GDK_RGB_DITHER_NORMAL, 0, 0);
		}


		return TRUE;
	}

	return FALSE;
}

static double
ma_rating_mouse_to_score (GtkWidget *widget)
{
	int mouse_x, mouse_y, icon_size;
	double score = -1.0;

	g_return_val_if_fail (widget != NULL, FALSE);

	gtk_icon_size_lookup (GTK_ICON_SIZE_MENU, &icon_size, NULL);
	gtk_widget_get_pointer (widget, &mouse_x, &mouse_y);

	/* ensure the user left-clicks within the good area */
	if (mouse_x >= 0 && mouse_x <= widget->allocation.width) {
		gboolean rtl;
		
		if (mouse_x <= X_OFFSET)
			score = 0.0;
		else
			score = ((mouse_x - X_OFFSET) / icon_size) + 1.0;

		rtl = (gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL);
		if (rtl) {
			score = MAX_SCORE - score + 1.0;
		}

		if (score > 5.0) score = 5.0;
		if (score < 0.0) score = 0.0;
	}

	return score;
}

static gboolean
ma_rating_button_press_cb (GtkWidget *widget,
			   GdkEventButton *event,
			   MaRating *rating)
{
	double score = -1.0;
	
	if (event->button == 1 && !rating->priv->inert)
		score = ma_rating_mouse_to_score (widget);

	if (score >= 0.0)
	{
		g_signal_emit (G_OBJECT (rating),
			       ma_rating_signals[RATED],
			       0, score);

		return TRUE;
	}

	return FALSE;
}

static gboolean
ma_rating_enter_notify_cb (GtkWidget *widget,
			   GdkEventCrossing *event,
			   MaRating *rating)
{
	double mouseover = ma_rating_mouse_to_score (widget);

	if (mouseover >= 0.0)
	{
		rating->priv->mouseover = mouseover;
		gtk_widget_queue_draw (GTK_WIDGET (rating));
	}

	return FALSE;
}

static gboolean
ma_rating_motion_notify_cb (GtkWidget *widget,
			    GdkEventMotion *event,
			    MaRating *rating)
{
	double mouseover = ma_rating_mouse_to_score (widget);

	if (mouseover >= 0.0)
	{
		double delta = mouseover - rating->priv->mouseover;
		rating->priv->mouseover = mouseover;
		if (delta > 0.001 || delta < -0.001)
			gtk_widget_queue_draw (GTK_WIDGET (rating));
	}

	return FALSE;
}

static gboolean
ma_rating_leave_notify_cb (GtkWidget *widget,
			   GdkEventCrossing *event,
			   MaRating *rating)
{
	rating->priv->mouseover = -1.0;
	gtk_widget_queue_draw (GTK_WIDGET (rating));

	return FALSE;
}
