/*
 * Implementation of the proxy around Rhythmbox's Bonobo interface.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include <config.h>

#include "ma-rhythmbox-bonobo-proxy.h"

#include <libbonobo.h>
#include <string.h>

#include "Rhythmbox.h"


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_RHYTHMBOX_BONOBO_PROXY, MaRhythmboxBonoboProxyPrivate))


typedef struct _MaRhythmboxBonoboProxyPrivate	MaRhythmboxBonoboProxyPrivate;


struct _MaRhythmboxBonoboProxyPrivate
{
	GNOME_Rhythmbox rb;
	Bonobo_PropertyBag pb;
	CORBA_Environment ev;

	Bonobo_Listener song_listener;
	Bonobo_Listener playing_listener;
};

static MaProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void		ma_rhythmbox_bonobo_proxy_class_init (MaRhythmboxBonoboProxyClass *klass);
static void		ma_rhythmbox_bonobo_proxy_init (MaRhythmboxBonoboProxy *bproxy);

static void		ma_rhythmbox_bonobo_proxy_dispose (GObject *object);
static void		ma_rhythmbox_bonobo_proxy_finalize (GObject *object);

static void		ma_rhythmbox_bonobo_proxy_toggle_playback (MaProxy *proxy);
static void		ma_rhythmbox_bonobo_proxy_previous (MaProxy *proxy);
static void		ma_rhythmbox_bonobo_proxy_next (MaProxy *proxy);

static void		ma_rhythmbox_bonobo_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void		ma_rhythmbox_bonobo_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void		ma_rhythmbox_bonobo_proxy_enable (MaProxy *proxy);
static void		ma_rhythmbox_bonobo_proxy_disable (MaProxy *proxy);

static void		ma_rhythmbox_bonobo_proxy_launch (MaProxy *proxy);

static void		ma_rhythmbox_bonobo_proxy_poll_disconnected (MaProxy *proxy);
static void		ma_rhythmbox_bonobo_proxy_poll_connected (MaProxy *proxy);

static Bonobo_Listener	add_pb_listener (MaRhythmboxBonoboProxy *bproxy,
					 const gchar *mask,
					 gpointer callback);

static void		fetch_status (MaRhythmboxBonoboProxy *bproxy);

static void		cache_song_metadata (MaRhythmboxBonoboProxy *bproxy,
					     const GNOME_Rhythmbox_SongInfo *info);

static void		disconnect (MaRhythmboxBonoboProxy *bproxy);
static gboolean		handle_error (MaRhythmboxBonoboProxy *bproxy, const gchar *message);

static void		song_change_cb (BonoboListener *listener,
					const char *event_name,
					const CORBA_any *any,
					CORBA_Environment *ev,
					MaRhythmboxBonoboProxy *bproxy);

static void		playing_change_cb (BonoboListener *listener,
					   const char *event_name,
					   const CORBA_any *any,
					   CORBA_Environment *ev,
					   MaRhythmboxBonoboProxy *bproxy);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_rhythmbox_bonobo_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaRhythmboxBonoboProxyClass),			/* class_size */
			NULL,							/* base_init */
			NULL,							/* base_finalize */
			(GClassInitFunc) ma_rhythmbox_bonobo_proxy_class_init,	/* class_init */
			NULL,							/* class_finalize */
			NULL,							/* class_data */
			sizeof (MaRhythmboxBonoboProxy),			/* instance_size */
			0,							/* n_preallocs */
			(GInstanceInitFunc) ma_rhythmbox_bonobo_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_PROXY, "MaRhythmboxBonoboProxy", &info, 0);
	}

	return type;
}

static void
ma_rhythmbox_bonobo_proxy_class_init (MaRhythmboxBonoboProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_rhythmbox_bonobo_proxy_dispose;
	object_class->finalize = ma_rhythmbox_bonobo_proxy_finalize;

	proxy_class->toggle_playback = ma_rhythmbox_bonobo_proxy_toggle_playback;
	proxy_class->previous = ma_rhythmbox_bonobo_proxy_previous;
	proxy_class->next = ma_rhythmbox_bonobo_proxy_next;

	proxy_class->rate_song = ma_rhythmbox_bonobo_proxy_rate_song;

	proxy_class->prepare_prefs = ma_rhythmbox_bonobo_proxy_prepare_prefs;

	proxy_class->enable = ma_rhythmbox_bonobo_proxy_enable;
	proxy_class->disable = ma_rhythmbox_bonobo_proxy_disable;

	proxy_class->launch = ma_rhythmbox_bonobo_proxy_launch;

	proxy_class->poll_disconnected = ma_rhythmbox_bonobo_proxy_poll_disconnected;
	proxy_class->poll_connected = ma_rhythmbox_bonobo_proxy_poll_connected;

	g_type_class_add_private (klass, sizeof (MaRhythmboxBonoboProxyPrivate));
}

static void
ma_rhythmbox_bonobo_proxy_init (MaRhythmboxBonoboProxy *bproxy)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	if (!bonobo_is_initialized ())
	{
		gint fake_argc = 0;
		bonobo_init (&fake_argc, NULL);
		bonobo_activate ();
	}

	priv->rb = CORBA_OBJECT_NIL;
	priv->pb = CORBA_OBJECT_NIL;

	priv->song_listener = CORBA_OBJECT_NIL;
	priv->playing_listener = CORBA_OBJECT_NIL;

	CORBA_exception_init (&priv->ev);
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_rhythmbox_bonobo_proxy_new (MaConf *conf)
{
	MaProxy *proxy;

	g_return_val_if_fail (conf != NULL, NULL);

	proxy = g_object_new (MA_TYPE_RHYTHMBOX_BONOBO_PROXY,
			      "name", "Rhythmbox",
			      "icon-name", "rhythmbox",
			      "conf", conf,
			      NULL);

	return proxy;
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_bonobo_proxy_dispose (GObject *object)
{
	MaRhythmboxBonoboProxy *bproxy = MA_RHYTHMBOX_BONOBO_PROXY (object);

	disconnect (bproxy);

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_rhythmbox_bonobo_proxy_finalize (GObject *object)
{
	MaRhythmboxBonoboProxy *bproxy = MA_RHYTHMBOX_BONOBO_PROXY (object);
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	CORBA_exception_free (&priv->ev);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_bonobo_proxy_toggle_playback (MaProxy *proxy)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	GNOME_Rhythmbox_playPause (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_playPause failed");
}

static void
ma_rhythmbox_bonobo_proxy_previous (MaProxy *proxy)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	GNOME_Rhythmbox_previous (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_previous failed");
}

static void
ma_rhythmbox_bonobo_proxy_next (MaProxy *proxy)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	GNOME_Rhythmbox_next (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_next failed");
}

static void
ma_rhythmbox_bonobo_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	GNOME_Rhythmbox_setRating (priv->rb, rating, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_setRating failed");
}

static void
ma_rhythmbox_bonobo_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	/* Nothing to do? */
}

static void
ma_rhythmbox_bonobo_proxy_enable (MaProxy *proxy)
{
	/* Nothing to do? */
}

static void
ma_rhythmbox_bonobo_proxy_disable (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	disconnect (MA_RHYTHMBOX_BONOBO_PROXY (proxy));
}

static void
ma_rhythmbox_bonobo_proxy_launch (MaProxy *proxy)
{
	_ma_proxy_launch_command (proxy, "rhythmbox_command");
}

static void
ma_rhythmbox_bonobo_proxy_poll_disconnected (MaProxy *proxy)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	priv->rb = bonobo_activation_activate (
		"((repo_ids.has('IDL:GNOME/Rhythmbox:1.0')) AND (_active == TRUE))",
		NULL, 0, NULL, &priv->ev);
	if (priv->rb == CORBA_OBJECT_NIL)
		return;

	priv->pb = GNOME_Rhythmbox_getPlayerProperties (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
	{
		if (handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_getPlayerProperties failed"))
			return;
	}

	/*
	 * If bonobo_event_source_client_add_listener gave some way to remove
	 * a listener, we'd use that.  But since it doesn't, we get to set up
	 * event listeners the (relatively) hard way.
	 */

	priv->song_listener = add_pb_listener (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "Bonobo/Property:change:song", song_change_cb);
	if (priv->song_listener == CORBA_OBJECT_NIL)
		return;

	priv->playing_listener = add_pb_listener (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "Bonobo/Property:change:playing", playing_change_cb);
	if (priv->playing_listener == CORBA_OBJECT_NIL)
		return;

	_ma_proxy_set_connected (proxy, TRUE);

	fetch_status (MA_RHYTHMBOX_BONOBO_PROXY (proxy));
}

static void
ma_rhythmbox_bonobo_proxy_poll_connected (MaProxy *proxy)
{
	MaRhythmboxBonoboProxyPrivate *priv;

	glong elapsed;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_BONOBO_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	elapsed = GNOME_Rhythmbox_getPlayingTime (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (MA_RHYTHMBOX_BONOBO_PROXY (proxy), "GNOME_Rhythmbox_getPlayingTime failed");
	else
		_ma_proxy_set_elapsed (proxy, elapsed);
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static Bonobo_Listener
add_pb_listener (MaRhythmboxBonoboProxy *bproxy,
		 const gchar *mask,
		 gpointer callback)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	GClosure *closure;
	Bonobo_Listener listener;

	closure = g_cclosure_new (G_CALLBACK (callback), bproxy, NULL);
	listener = bonobo_event_source_client_add_listener_full (priv->pb, closure, mask, &priv->ev);

	if (BONOBO_EX (&priv->ev))
	{
		if (handle_error (bproxy, "bonobo_event_source_client_add_listener_full failed"))
			return CORBA_OBJECT_NIL;
	}

	return listener;
}

static void
fetch_status (MaRhythmboxBonoboProxy *bproxy)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	gboolean playing;
	glong elapsed;
	CORBA_any *any;

	any = bonobo_pbclient_get_value (priv->pb, "song", TC_GNOME_Rhythmbox_SongInfo, &priv->ev);
	if (BONOBO_EX (&priv->ev))
	{
		if (handle_error (bproxy, "bonobo_pbclient_get_value failed"))
			return;
	}
	else if (any != NULL)
	{
		cache_song_metadata (bproxy, (const GNOME_Rhythmbox_SongInfo *) any->_value);
		CORBA_free (any);
	}

	playing = bonobo_pbclient_get_boolean (priv->pb, "playing", &priv->ev);
	if (BONOBO_EX (&priv->ev))
	{
		if (handle_error (bproxy, "bonobo_pbclient_get_boolean failed"))
			return;
	}
	else
		_ma_proxy_set_playing (MA_PROXY (bproxy), playing);

	elapsed = GNOME_Rhythmbox_getPlayingTime (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
	{
		if (handle_error (bproxy, "GNOME_Rhythmbox_getPlayingTime failed"))
			return;
	}
	else
		_ma_proxy_set_elapsed (MA_PROXY (bproxy), elapsed);
}

static void
cache_song_metadata (MaRhythmboxBonoboProxy *bproxy,
		     const GNOME_Rhythmbox_SongInfo *info)
{
	if (info != NULL)
	{
		_ma_proxy_set_title (MA_PROXY (bproxy), info->title);
		_ma_proxy_set_artist (MA_PROXY (bproxy), info->artist);
		_ma_proxy_set_album (MA_PROXY (bproxy), info->album);
		_ma_proxy_set_duration (MA_PROXY (bproxy), info->duration);
		_ma_proxy_set_rating (MA_PROXY (bproxy), info->rating);
	}
	else
		_ma_proxy_set_no_song (MA_PROXY (bproxy));
}

static void
disconnect (MaRhythmboxBonoboProxy *bproxy)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	if (priv->song_listener != CORBA_OBJECT_NIL)
	{
		bonobo_event_source_client_remove_listener (priv->pb, priv->song_listener, NULL);
		priv->song_listener = CORBA_OBJECT_NIL;
	}

	if (priv->playing_listener != CORBA_OBJECT_NIL)
	{
		bonobo_event_source_client_remove_listener (priv->pb, priv->playing_listener, NULL);
		priv->playing_listener = CORBA_OBJECT_NIL;
	}

	if (priv->pb != CORBA_OBJECT_NIL)
	{
		bonobo_object_release_unref ((Bonobo_Unknown) priv->pb, NULL);
		priv->pb = CORBA_OBJECT_NIL;
	}

	if (priv->rb != CORBA_OBJECT_NIL)
	{
		/* An unref would shut down Rhythmbox! */
		priv->rb = CORBA_OBJECT_NIL;
	}

	_ma_proxy_set_connected (MA_PROXY (bproxy), FALSE);
}

static gboolean
handle_error (MaRhythmboxBonoboProxy *bproxy, const gchar *message)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	gboolean fatal = FALSE;
	gchar *what = bonobo_exception_get_text (&priv->ev);

	g_warning ("%s: %s", message, what);
	if (strstr (what, "'IDL:omg.org/CORBA/COMM_FAILURE:1.0'"))
	{
		disconnect (bproxy);
		fatal = TRUE;
	}

	g_free (what);
	return fatal;
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
song_change_cb (BonoboListener *listener,
		const char *event_name,
		const CORBA_any *any,
		CORBA_Environment *ev,
		MaRhythmboxBonoboProxy *bproxy)
{
	const GNOME_Rhythmbox_SongInfo *info;

	g_return_if_fail (CORBA_TypeCode_equivalent (any->_type,
						     TC_GNOME_Rhythmbox_SongInfo,
						     NULL));
	info = (const GNOME_Rhythmbox_SongInfo *) any->_value;

	cache_song_metadata (bproxy, info);
}

static void
playing_change_cb (BonoboListener *listener,
		   const char *event_name,
		   const CORBA_any *any,
		   CORBA_Environment *ev,
		   MaRhythmboxBonoboProxy *bproxy)
{
	MaRhythmboxBonoboProxyPrivate *priv = GET_PRIVATE (bproxy);

	glong elapsed;

	g_return_if_fail (CORBA_TypeCode_equivalent (any->_type,
						     TC_CORBA_boolean,
						     NULL));

	/* Rhythmbox doesn't explicitly send a song-changed-to-NULL notification, so... */

	elapsed = GNOME_Rhythmbox_getPlayingTime (priv->rb, &priv->ev);
	if (BONOBO_EX (&priv->ev))
		handle_error (bproxy, "GNOME_Rhythmbox_getPlayingTime failed");
	else if (elapsed < 0)
		_ma_proxy_set_no_song (MA_PROXY (bproxy));

	_ma_proxy_set_playing (MA_PROXY (bproxy), *((gboolean *) any->_value));
}
