/* Terminal control based on terminfo capabilities.

	Copyright (C) 1985, 1986, 1987 Free Software Foundation, Inc.
	Originally part of GNU Emacs.
	Vastly edited and modified for use within ne.

	This file is part of ne, the nice editor.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2, or (at your option) any
	later version.
	
	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.
	
	You should have received a copy of the GNU General Public License along
	with this program; see the file COPYING.  If not, write to the Free
	Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA.  */


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifndef _AMIGA
#include <termios.h>
#include <sys/ioctl.h>
#endif

#ifndef TERMCAP
#include <curses.h>
#include <term.h>
#else
#include "info2cap.h"
#endif

#include "termchar.h"
#include "cm.h"
#include "utf8.h"


/* When displaying errors about the terminal database, we try to print the
correct name. */

#ifdef TERMCAP
#define DATABASE_NAME "termcap"
#else
#define DATABASE_NAME "terminfo"
#endif

/* If true, we want the use the built-in ANSI terminal, not a real one. */

#ifdef ANSI
int ansi = TRUE;
#else
int ansi = FALSE;
#endif

/* We use internal copies of the terminfo capabilities because we want to be
	able to use a hardwired set. */

int ne_generic_type;

int ne_lines;
int ne_columns;

char *ne_column_address;
char *ne_row_address;

char *ne_cursor_address;

char *ne_carriage_return;

char *ne_cursor_home;
char *ne_cursor_to_ll;

char *ne_cursor_right;
char *ne_cursor_down;
char *ne_cursor_left;
char *ne_cursor_up;

int ne_auto_right_margin;
int ne_eat_newline_glitch;

char *ne_clr_eos;
char *ne_clear_screen;

char *ne_bell;
char *ne_flash_screen;

char *ne_scroll_forward;
char *ne_scroll_reverse;

char *ne_enter_delete_mode;
char *ne_exit_delete_mode;
char *ne_enter_insert_mode;
char *ne_exit_insert_mode;

char *ne_enter_standout_mode;
char *ne_exit_standout_mode;
int ne_magic_cookie_glitch;
int ne_move_standout_mode;

char *ne_change_scroll_region;

char *ne_insert_line;
char *ne_parm_insert_line;
char *ne_delete_line;
char *ne_parm_delete_line;

char *ne_insert_character;
char *ne_insert_padding;
char *ne_parm_ich;

char *ne_delete_character;
char *ne_parm_dch;

int ne_move_insert_mode;

char *ne_cursor_invisible;
char *ne_cursor_normal;

char *ne_init_1string;
char *ne_init_2string;
char *ne_init_3string;
char *ne_enter_ca_mode;
char *ne_exit_ca_mode;

char *ne_exit_attribute_mode;
char *ne_exit_alt_charset_mode;

char *ne_repeat_char;

int ne_tilde_glitch;
int ne_memory_below;

int ne_has_meta_key;
char *ne_meta_on;
char *ne_meta_off;

char *ne_set_window;

char *ne_keypad_local;
char *ne_keypad_xmit;

char *ne_clr_eol;
int ne_transparent_underline;







/* This is the real instantiation of the cm structure used by cm.c to hold
the cursor motion strings. */

struct cm Wcm;


#define OUTPUT(a) tputs (a, ne_lines - curY, cmputc)
#define OUTPUT1(a) tputs (a, 1, cmputc)
#define OUTPUTL(a, lines) tputs (a, lines, cmputc)
#define OUTPUT_IF(a) { if (a) tputs (a, ne_lines - curY, cmputc); }
#define OUTPUT1_IF(a) { if (a) tputs (a, 1, cmputc); }



/* Terminal charateristics that higher levels want to look at.
   These are all extern'd in termchar.h */

int	line_ins_del_ok;		/* Terminal can insert and delete lines */
int	char_ins_del_ok;		/* Terminal can insert and delete chars */
int	scroll_region_ok;		/* Terminal supports setting the scroll window */
int	standout_ok;			/* Terminal supports standout without magic cookies */
int	cursor_on_off_ok;		/* Terminal can make the cursor visible or invisible */


static int	RPov;		/* Least number of chars to start a TS_repeat.
								Less wouldn't be worth. */

static int	delete_in_insert_mode;	/* delete mode == insert mode */
static int	se_is_so;					/* 1 if same string both enters and leaves standout mode */

static int	standout_requested;	/* Nonzero when supposed to write text in standout mode. */
static int	insert_mode;			/* Nonzero when in insert mode. */
static int	standout_mode;			/* Nonzero when in standout mode. */


/* Size of window specified by higher levels. This is the number of lines,
starting from top of screen, to participate in ins/del line operations.
Effectively it excludes the bottom lines - specified_window_size lines from
those operations.  */

int	specified_window;

/* If TRUE, then all I/O is to be performed in UTF-8. */

int	io_utf8;


/* PORTABILITY PROBLEM: this function is responsible for filtering nonprintable
	characters. On systems with a wider system character set, it could be
	redefined, for instance, in order to allow characters between 128 and 160 to
	be printed. Currently, it returns '?' on all control characters (and
	non-ISO-8859-1 characters, if io_utf8 is false), space on 160, and the
	obvious capital letter for control characters below 32. */

int de_control(const int c) {
	if (c >= 127 && c < 160) return '?';
	if (c == 160) return ' ';
	if (c < ' ') return '@'+c;
	if (c > 0xFF && !io_utf8) return '?';

	/* If io_utf8 is off, we consider all characters in the range of ISO-8859-x
	encoding schemes as printable. */

	return io_utf8 && wcwidth(c) <= 0 ? '?' : c;
}

/* Returns the output width of the given character. It is maximised with 1
 w.r.t. wcwidth(), so its result is equivalent to the width of the character
 returned by de_control(). */

int output_width(const int c) {
	const int width = wcwidth(c);
	return width > 0 ? width : 1;
}

/* Returns the output width of the given string. If s is NULL, returns len.  If
the width of the string exceeds maxWidth, modifies len so that it contains the
longest prefix of s whose width is not greater than maxWidth, and returns the
corresponding width. */

static int string_output_width(const unsigned char *s, int *len, int maxWidth, int utf8) {
	if (s == NULL) {
		if (*len > maxWidth) *len = maxWidth;
		return *len;
	}
	else {
		int width = 0, char_width, l = *len;
		if (utf8) {
			while(l-- != 0) {
				char_width = output_width(utf8char(s));
				if (width + char_width > maxWidth) {
					*len -= l + 1;
					break;
				}
				width += char_width;
				s += utf8len(*s);
			}
		}
		else while(l-- != 0) {
			char_width = output_width(*(s++));
			if (width + char_width > maxWidth) {
				*len -= l + 1;
				break;
			}
			width += char_width;
		}
		return width;
	}
}

/* Depending on the value of io_utf8, this function will do a simple putchar(),
	or a series of putchar() that expand the given character in UTF-8 encoding. */

void out(const int c) {
	if (io_utf8) {
		if (c < 0x80) putchar( c ); /* ASCII */
		else if (c < 0x800) {
			putchar(0xC0 | (c >> 6));
			putchar(0x80 | (c >> 0) & 0x3F);
		}
		else if (c < 0x10000) {
			putchar( 0xE0 | (c >> 12));
			putchar( 0x80 | (c >> 6) & 0x3F );
			putchar( 0x80 | (c >> 0) & 0x3F );
		}
		else if (c < 0x200000) {
			putchar( 0xF0 | (c >> 18));
			putchar( 0x80 | (c >> 12) & 0x3F );
			putchar( 0x80 | (c >> 6) & 0x3F );
			putchar( 0x80 | (c >> 0) & 0x3F );
		}
		else if (c < 0x4000000) {
			putchar( 0xF8 | (c >> 24));
			putchar( 0x80 | (c >> 18) & 0x3F );
			putchar( 0x80 | (c >> 12) & 0x3F );
			putchar( 0x80 | (c >> 6) & 0x3F );
			putchar( 0x80 | (c >> 0) & 0x3F );
		}
		else {
			putchar( 0xFC | (c >> 30));
			putchar( 0x80 | (c >> 24) & 0x3F );
			putchar( 0x80 | (c >> 18) & 0x3F );
			putchar( 0x80 | (c >> 12) & 0x3F );
			putchar( 0x80 | (c >> 6) & 0x3F );
			putchar( 0x80 | (c >> 0) & 0x3F );
		}
	}
	else putchar(c);
}




/* Rings a bell or flashes the screen. If the service is not available, the
	other one is tried. */


void ring_bell(void) {
	OUTPUT1_IF (ne_bell ? ne_bell : ne_flash_screen);
}


void do_flash(void) {
	OUTPUT1_IF (ne_flash_screen ? ne_flash_screen : ne_bell);
}



/* Sets correctly the scroll region (first line is line 0). This function
	assumes scroll_region_ok == TRUE.  The cursor position is lost, as from the
	terminfo specs. */

static void set_scroll_region (const int start, const int stop) {

	char *buf;

	assert(scroll_region_ok);

	/* Both control string have line range 0 to lines-1 */

	if (ne_change_scroll_region) buf = tparm (ne_change_scroll_region, start, stop);
	else buf = tparm (ne_set_window, start, stop, 0, ne_columns - 1);

	OUTPUT1(buf);
	losecursor();
}


static void turn_on_insert (void) {
	if (!insert_mode) OUTPUT1(ne_enter_insert_mode);
	insert_mode = TRUE;
}


static void turn_off_insert (void) {
	if (insert_mode) OUTPUT1(ne_exit_insert_mode);
	insert_mode = FALSE;
}


/* These functions are called on all terminals in order to handle highlighting,
but do nothing on terminals with a magic cookie (or without standout).  */

static void turn_off_highlight (void) {
	if (standout_ok) {
		if (standout_mode) OUTPUT1(ne_exit_standout_mode);
		standout_mode = FALSE;
	}
}


static void turn_on_highlight (void) {
	if (standout_ok) {
		if (!standout_mode) OUTPUT1(ne_enter_standout_mode);
		standout_mode = TRUE;
	}
}


/* Prepares the terminal for interactive I/O. It
initializes the terminal, prepares the cursor address mode, and
activates the keypad and the meta key. */

void set_terminal_modes(void) {

	/* Note that presently we do not support if and iprog, the program
	and the file which should be used, if present, to initialize the
	terminal. */

	/*OUTPUT1_IF(ne_init_1string);
	OUTPUT1_IF(ne_init_2string);
	OUTPUT1_IF(ne_init_3string);*/

	OUTPUT1_IF(ne_exit_attribute_mode);
	OUTPUT1_IF(ne_exit_alt_charset_mode);
	OUTPUT1_IF(ne_exit_standout_mode);

	OUTPUT1_IF(ne_enter_ca_mode);
	OUTPUT1_IF(ne_keypad_xmit);

	if (ne_has_meta_key) OUTPUT1_IF(ne_meta_on);

	losecursor();
}


/* Puts again the terminal in its normal state. */

void reset_terminal_modes (void) {

	OUTPUT1_IF(ne_exit_attribute_mode);
	OUTPUT1_IF(ne_exit_alt_charset_mode);
	OUTPUT1_IF(ne_exit_standout_mode);
	OUTPUT1_IF(ne_keypad_local);
	OUTPUT1_IF(ne_exit_ca_mode);
}


/* Sets the variable specified_window. Following line insert/delete operations
	will be limited to lines 0 to (size-1). */

void set_terminal_window(const int size) {

	specified_window = size ? size : ne_lines;

}




/* These functions are the external interface to standout activation. */

void standout_on (void) {
	standout_requested = TRUE;
}


void standout_off (void) {
	standout_requested = FALSE;
	turn_off_highlight ();
}


/* These functions are the external interface to cursor on/off strings. */

void cursor_on (void) {
	if (cursor_on_off_ok) OUTPUT1(ne_cursor_normal);
}


void cursor_off (void) {
	if (cursor_on_off_ok) OUTPUT1(ne_cursor_invisible);
}


/* Set standout mode to the mode specified for the text to be output.  */

static void highlight_if_desired (void) {
	if (standout_requested) turn_on_highlight ();
	else turn_off_highlight ();
}


/* Move to absolute position, specified origin 0 */

void move_cursor (const int row, const int col) {
	if (curY == row && curX == col) return;
	if (!ne_move_standout_mode) turn_off_highlight ();
	if (!ne_move_insert_mode) turn_off_insert ();
	cmgoto (row, col);
}


/* Clears from the cursor position to the end of line. It assumes that the line
	is already clear starting at column first_unused_hpos. Note that the cursor
	may be moved, on terminals lacking a `ce' string.  */

void clear_end_of_line(const int first_unused_hpos) {

	int i;

	if (curX >= first_unused_hpos) return;

	if (ne_clr_eol) OUTPUT1 (ne_clr_eol);
	else {
		/* We have to do it the hard way. */
		turn_off_insert ();
		for (i = curX; i < first_unused_hpos; i++) putchar (' ');
		cmplus (first_unused_hpos - curX);
	}
}


/* Shorthand; use this if you don't know anything about the state
of the line. */

void clear_to_eol(void) {
	clear_end_of_line(ne_columns);
}


/* Clears from the cursor position to the end of screen */

void clear_to_end (void) {

	if (ne_clr_eos) OUTPUT(ne_clr_eos);
	else {
		int i;

		for (i = curY; i < ne_lines; i++) {
			move_cursor (i, 0);
			clear_to_eol();
		}
	}
}


/* Clears the entire screen */

void clear_entire_screen (void) {

	if (ne_clear_screen) {
		OUTPUTL(ne_clear_screen, ne_lines);
		cmat (0, 0);
	}
	else {
		move_cursor (0, 0);
		clear_to_end();
	}
}


/* Outputs len characters pointed at by string. The characters will be
	truncated to the end of the current line. Passing a NULL for string results
	in outputting spaces. A len of 0 causes no action. If utf8 is TRUE, the
	string is UTF-8 encoded. */

void output_chars (const unsigned char *string, const int raw_len, const int utf8) {

	const unsigned char *p;
	unsigned char *buf;
	int	c, len;
	const unsigned char *first_check;

	if (raw_len == 0) return;

	highlight_if_desired ();
	turn_off_insert ();

	/* If the string is UTF-8 encoded, compute its real length. */
	if (utf8 && string != NULL) len = utf8strlen(string, raw_len);
	else len = raw_len;

	/* If the width of the string exceeds the remaining columns, we reduce
		len. Moreovero, we don't dare write in last column of bottom line, if
		AutoWrap, since that would scroll the whole screen on some terminals. */

	cmplus(string_output_width(string, &len, ne_columns - curX - (AutoWrap && curY == ne_lines - 1), utf8));

	if (string == NULL) {
		if (len > RPov) {
			buf = tparm (ne_repeat_char, ' ', len);
			OUTPUT1(buf);
		}
		else {
			while(--len >= 0) putchar(' ');
		}
		return;
	}

	first_check = string;

	if (RPov > len && !ne_transparent_underline && !ne_tilde_glitch) {
		while(--len >= 0) {
			if (utf8) {
				c = utf8char(string);
				string += utf8len(*string);
				out(de_control(c));
			}
			else {
				c = *string++;
				out(de_control(c));
			}
		}
	} else
		while (--len >= 0) {
			c = utf8 ? *string : utf8char(string);
			if (!io_utf8 && RPov + 1 < len && string >= first_check) {
				int	repeat_count;

				p = string;

				/* Now, len is number of chars left starting at p */
				while (++p - string <= len && *p == c);

				repeat_count = p - string;

				if (repeat_count > RPov) {
					buf = tparm (ne_repeat_char, de_control(c), repeat_count);
					OUTPUT1(buf);
					string = p;
					len -= repeat_count - 1;
					continue;
				}
				else {
					/* If all N identical chars are too few,
		 			don't even consider the last N-1, the last N-2,...  */

					first_check = p;
				}
			}

			if (c == '_' && ne_transparent_underline) {
				putchar (' ');
				OUTPUT1(Left);
			}

			if (ne_tilde_glitch && c == '~')
				c = '`';

			out(de_control(c));
			string += utf8 ? utf8len(*string) : 1;
		}
}



/* Outputs a NULL-terminated string. */

void output_string(const char * const s, const int utf8) {
	assert(s != NULL);
	output_chars(s, strlen(s), utf8);
}


/* Outputs a single ISO 10646 character. */

void output_char(const int c, const int utf8) {
	static unsigned char t[8];

	if (utf8) {
		memset(t, 0, sizeof t);
		utf8str(c, t);
	}
	else {
		t[0] = c;
		t[1] = 0;
	}

	assert(c != 0);

	output_string(t, utf8);
}


/* Outputs spaces. */

void output_spaces(const int n) {
   output_chars(NULL, n, FALSE);
}

/* Same as output_chars(), but inserts instead. */

void insert_chars (const unsigned char *start, const int raw_len, const int utf8) {

	const unsigned char *buf;
	int c, len, width;

	if (raw_len == 0) return;

	highlight_if_desired();

	/* If the string is non-NULL and UTF-8 encoded, compute its real length. */
	if (utf8 && start != NULL) len = utf8strlen(start, raw_len);
	else len = raw_len;

	if (ne_parm_ich) {
		int i, width = 0;

		if (start != NULL) {
			if (utf8) for(i = 0; i < raw_len; i += utf8len(start[i])) width += output_width(utf8char(start + i));
			else for(i = 0; i < raw_len; i++) width += output_width(start[i]);
		}
		else width = len;

		buf = tparm (ne_parm_ich, width);
		OUTPUT1 (buf);

		if (start) output_chars (start, raw_len, utf8);

		return;
	}

	turn_on_insert ();

	/* If the width of the string exceeds the remaining columns, we reduce
		len. Moreovero, we don't dare write in last column of bottom line, if
		AutoWrap, since that would scroll the whole screen on some terminals. */

	cmplus(string_output_width(start, &len, ne_columns - curX - (AutoWrap && curY == ne_lines - 1), utf8));

	if (!ne_transparent_underline && !ne_tilde_glitch && start
	     && ne_insert_padding == NULL && ne_insert_character == NULL) {
		while(--len >= 0) {
			if (utf8) {
				c = utf8char(start);
				start += utf8len(*start);
			}
			else c = *start++;
			out(de_control(c));
		}
	}
	else
		while (--len >= 0) {

			OUTPUT1_IF (ne_insert_character);

			if (!start) c = ' ';
			else {
				if (utf8) {
					c = utf8char(start);
					start += utf8len(*start);
				}
				else c = *start++;
				
				if (ne_tilde_glitch && c == '~') c = '`';
			}

			out(de_control(c));
			OUTPUT1_IF(ne_insert_padding);
		}
}


/* Inserts a single ISO 10646 character. */

void insert_char(const int c, const int utf8) {
	static unsigned char t[8];

	if (utf8) {
		memset(t, 0, sizeof t);
		utf8str(c, t);
	}
	else {
		t[0] = c;
		t[1] = 0;
	}

	assert(c != 0);

	insert_chars(t, strlen(t), utf8);
}

/* Deletes n characters at the current cursor position. */

void delete_chars (int n) {

	char *buf;

	if ( n == 0 ) return;

	if (delete_in_insert_mode) turn_on_insert();
	else {
		turn_off_insert();
		OUTPUT1_IF(ne_enter_delete_mode);
	}

	if (ne_parm_dch) {
		buf = tparm(ne_parm_dch, n);
		OUTPUT1(buf);
	}
	else while(--n >= 0) OUTPUT1(ne_delete_character);

	if (!delete_in_insert_mode) OUTPUT_IF(ne_exit_delete_mode);
}


/* This internal function will do an insertion or deletion
for n lines, given a parametrized and/or a one-line capability
for that purpose. */

static void do_multi_ins_del(char * const multi, const char * const single, int n) {

	if (multi) {
		char *const buf = tparm(multi, n);
		OUTPUT(buf);
	}
	else while(--n >= 0) OUTPUT(single);
}


/* Inserts n lines at vertical position vpos. If n is negative, it deletes -n
	lines. specified_window is taken into account. This function assumes
	line_ins_del_ok == TRUE. */

void ins_del_lines (const int vpos, const int n) {

	int i = n > 0 ? n : -n;

	assert(line_ins_del_ok);
	assert(i != 0);
	assert(vpos < specified_window);

	if (scroll_region_ok && vpos + i >= specified_window) return;

	if (!ne_memory_below && vpos + i >= ne_lines) return;

	if (scroll_region_ok) {
		if (specified_window != ne_lines) set_scroll_region(vpos, specified_window - 1);

		if (n < 0) {
			move_cursor(specified_window - 1, 0);
			while (--i >= 0) OUTPUTL(ne_scroll_forward, specified_window - vpos + 1);
		}
		else {
			move_cursor(vpos, 0);
			while (--i >= 0) OUTPUTL(ne_scroll_reverse, specified_window - vpos + 1);
		}

		if (specified_window != ne_lines) set_scroll_region(0, ne_lines - 1);
	}
	else {
		if (n > 0) {
			if (specified_window != ne_lines) {
				move_cursor(specified_window - i, 0);
				do_multi_ins_del(ne_parm_delete_line, ne_delete_line, i);
			}

			move_cursor(vpos, 0);
			do_multi_ins_del(ne_parm_insert_line, ne_insert_line, i);
		}
		else {
			move_cursor(vpos, 0);
			do_multi_ins_del(ne_parm_delete_line, ne_delete_line, i);

			if (specified_window != ne_lines) {
				move_cursor(specified_window - i, 0);
				do_multi_ins_del(ne_parm_insert_line, ne_insert_line, i);
			}
			else if (ne_memory_below) {
				move_cursor(ne_lines + n, 0);
				clear_to_end ();
			}
		}
	}
}


extern int cost;		/* In cm.c */
extern int evalcost(int);


/* Performs the cursor motion cost setup, and sets the variable RPov to the
	number of characters (with padding) which are really output when repeating
	one character. RPov is disable using UTF-8 I/O. */

static void calculate_costs (void) {

	if (ne_repeat_char) {

		char *const buf = tparm(ne_repeat_char, ' ', 1);

		cost = 0;
		tputs(buf, 1, evalcost);

		RPov = cost + 1;
	}
	else RPov = ne_columns * 2;

	cmcostinit();
}



/* Gets the window size using TIOCGWINSZ, or does nothing if TIOCGWINSZ is not
	available. It is called by the signal handler for SIGWINCH on systems that
	support it. Return 1 if the window size has changed. */

int ttysize(void) {
#ifdef TIOCGWINSZ
	struct winsize size;
	
	/* try using the TIOCGWINSZ call, if defined */
	D(fprintf(stderr,"ttysize: CHECKING...\n");)
	
	if (ioctl(0, TIOCGWINSZ, &size) >= 0) {
		D(fprintf(stderr,"ttysize:...size is (%d,%d)\n",size.ws_row,size.ws_col);)
		if ( (ne_lines != size.ws_row) || ( ne_columns != size.ws_col ) ) {
			ScreenRows = ne_lines    = size.ws_row;
			ScreenCols = ne_columns  = size.ws_col;
			set_terminal_window(ne_lines - 1);
			if (scroll_region_ok) set_scroll_region(0, ne_lines - 1);
			D(fprintf(stderr,"ttysize: size changed.\n");)
			return 1;
		}
	}
#endif
	/* no special way to detect screen size */
	return 0;
}


#ifndef TERMCAP

/* If we get capabilities from the database, then we copy them into our
   internal counterparts. */

void copy_caps(void) {

	ne_generic_type = generic_type;

	ne_lines = lines;
	ne_columns = columns;

	ne_column_address = column_address;
	ne_row_address = row_address;

	ne_cursor_address = cursor_address;

	ne_carriage_return = carriage_return;

	ne_cursor_home = cursor_home;
	ne_cursor_to_ll = cursor_to_ll;

	ne_cursor_right = cursor_right;
	ne_cursor_down = cursor_down;
	ne_cursor_left = cursor_left;
	ne_cursor_up = cursor_up;

	ne_auto_right_margin = auto_right_margin;
	ne_eat_newline_glitch = eat_newline_glitch;

	ne_clr_eos = clr_eos;
	ne_clear_screen = clear_screen;

	ne_bell = bell;
	ne_flash_screen = flash_screen;

	ne_scroll_forward = scroll_forward;
	ne_scroll_reverse = scroll_reverse;

	ne_enter_delete_mode = enter_delete_mode;
	ne_exit_delete_mode = exit_delete_mode;
	ne_enter_insert_mode = enter_insert_mode;
	ne_exit_insert_mode = exit_insert_mode;

	ne_enter_standout_mode = enter_standout_mode;
	ne_exit_standout_mode = exit_standout_mode;
	ne_magic_cookie_glitch = magic_cookie_glitch;
	ne_move_standout_mode = move_standout_mode;

	ne_change_scroll_region = change_scroll_region;

	ne_insert_line = insert_line;
	ne_parm_insert_line = parm_insert_line;
	ne_delete_line = delete_line;
	ne_parm_delete_line = parm_delete_line;

	ne_insert_character = insert_character;
	ne_insert_padding = insert_padding;
	ne_parm_ich = parm_ich;

	ne_delete_character = delete_character;
	ne_parm_dch = parm_dch;

	ne_move_insert_mode = move_insert_mode;

	ne_cursor_invisible = cursor_invisible;
	ne_cursor_normal = cursor_normal;

	ne_init_1string = init_1string;
	ne_init_2string = init_2string;
	ne_init_3string = init_3string;
	ne_enter_ca_mode = enter_ca_mode;
	ne_exit_ca_mode = exit_ca_mode;

	ne_exit_attribute_mode = exit_attribute_mode;
	ne_exit_alt_charset_mode = exit_alt_charset_mode;

	ne_repeat_char = repeat_char;

	ne_tilde_glitch = tilde_glitch;
	ne_memory_below = memory_below;

	ne_has_meta_key = has_meta_key;
	ne_meta_on = meta_on;
	ne_meta_off = meta_off;

	ne_set_window = set_window;

	ne_keypad_local = keypad_local;
	ne_keypad_xmit = keypad_xmit;

	ne_clr_eol = clr_eol;
	ne_transparent_underline = transparent_underline;
}


#endif



void setup_ansi_term(void) {

#ifdef TERMCAP
	ne_cursor_address = "\x1b[%i%d;%dH";
#else
	ne_cursor_address = "\x1b[%i%p1%d;%p2%dH";
#endif
	ne_lines = 25;
	ne_columns = 80;
	ne_carriage_return = "\xd";
	ne_cursor_home = "\x1b[H";
	ne_cursor_right = "\x1b[C";
	ne_cursor_down = "\x1b[B";
	ne_cursor_left = "\x1b[D";
	ne_cursor_up = "\x1b[A";
	ne_auto_right_margin = 1;
	ne_eat_newline_glitch = 0;
	ne_clr_eos = "\x1b[J";
	ne_clear_screen = "\x1b[H\x1b[J";
	ne_bell = "\x7";
	ne_scroll_forward = "\xa";
	ne_enter_standout_mode = "\x1b[7m";
	ne_exit_standout_mode = "\x1b[m";
	ne_magic_cookie_glitch = -1;
	ne_move_standout_mode = 0;
	ne_insert_line = "\x1b[L";
	ne_delete_line = "\x1b[M";
	ne_delete_character = "\x1b[P";
	ne_move_insert_mode = 1;
	ne_exit_attribute_mode = "\x1b[0;10m";
	ne_exit_alt_charset_mode = "\x1b[10m";
	ne_tilde_glitch = 0;
	ne_memory_below = 0;
	ne_has_meta_key = 0;
	ne_clr_eol = "\x1b[K";
	ne_transparent_underline = 0;
}



/* This is the main terminal initialization function. It sets up Wcm,
patches here and there the terminfo database, calculates the costs, and
initializes the terminal characteristics variables. Note that this function
can exit(). */

void term_init (void) {

	int errret;

	/* First of all we initialize the terminfo database. */

	if (ansi) setup_ansi_term();
	else if (setupterm(0, 1, &errret) == ERR) {
		printf("There are problems in finding your terminal in the database.\n"
				 "Please check that the variable TERM is set correctly, and that\n"
				 "your " DATABASE_NAME " database is up to date.\n"
				 "If your terminal is ANSI-compatible, you can also try to use\n"
				 "the --ansi switch.\n");
		exit(1);
	}
#ifndef TERMCAP
	else copy_caps();
#endif

	ColPosition = ne_column_address;
	RowPosition = ne_row_address;
	AbsPosition = ne_cursor_address;
	CR = ne_carriage_return;
	Home = ne_cursor_home;
	LastLine = ne_cursor_to_ll;
	Right = ne_cursor_right;
	Down = ne_cursor_down;
	Left = ne_cursor_left;
	Up = ne_cursor_up;
	AutoWrap = ne_auto_right_margin;
	MagicWrap = ne_eat_newline_glitch;
	ScreenRows = ne_lines;
	ScreenCols = ne_columns;

	ttysize();

	if (!ne_bell) ne_bell = "\07";

	if (!ne_scroll_forward) ne_scroll_forward = Down;
	if (!ne_scroll_reverse) ne_scroll_reverse = Up;

	if (!ansi && key_backspace && key_left && !strcmp(key_backspace, key_left)) {
		/* In case left and backspace produce the same sequence,
		we want to get key_left. */

		key_backspace = NULL;
	}

	specified_window = ne_lines;

	if (Wcm_init ())	{

		/* We can't do cursor motion */

		if (ne_generic_type) {
			printf("Your terminal type is a generic terminal, not a real\n"
			       "terminal, and it lacks the ability to position the cursor.\n"
			       "Please check that the variable TERM is set correctly, and that\n"
			       "your " DATABASE_NAME " database is up to date.\n");
		}
		else {
			printf("Your terminal type is not powerful enough to run ne:\n"
			       "it lacks the ability to position the cursor.\n"
			       "Please check that the variable TERM is set correctly, and that\n"
			       "your " DATABASE_NAME "database is up to date.\n");
		}

		printf("If your terminal is ANSI-compatible, you can also try to use\n"
				 "the --ansi switch.\n");

		exit(1);
	}

	calculate_costs();

	delete_in_insert_mode
	     = ne_enter_delete_mode && ne_enter_insert_mode
	     && !strcmp (ne_enter_delete_mode, ne_enter_insert_mode);

	se_is_so = ne_enter_standout_mode && ne_exit_standout_mode
	     && !strcmp (ne_enter_standout_mode, ne_exit_standout_mode);

	scroll_region_ok = ne_set_window || ne_change_scroll_region;

	line_ins_del_ok = (((ne_insert_line || ne_parm_insert_line)
	     && (ne_delete_line || ne_parm_delete_line))
	     || (scroll_region_ok
	     && ne_scroll_forward
	     && ne_scroll_reverse));

	char_ins_del_ok = ((ne_insert_character || ne_enter_insert_mode ||
	    ne_insert_padding || ne_parm_ich)
	     && (ne_delete_character || ne_parm_dch));

	standout_ok = (ne_enter_standout_mode && ne_exit_standout_mode && ne_magic_cookie_glitch < 0);

	cursor_on_off_ok = (ne_cursor_invisible && ne_cursor_normal);
}
