/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;
using SCG = System.Collections.Generic;

namespace Nemerle.Compiler
{
  // For indentation syntax
  public class PreParserIndent : PreParser {
    // The number of unmatched { ( [ tokens found in the user code at this point
    mutable explicit_groups : int;
    mutable indentation_syntax_active : bool = true;
    mutable insertLocation : Location;

    mutable had_some_real_input : bool;
    mutable last_token_was_semicolon : bool;
    
    mutable force_brace_after_newline : bool;
    indent_strings : Stack [string] = Stack (20);
    tokens_pending : Queue [Token] = Queue ();

    // 'set' directives handling
    mutable set_namespace : bool;
    mutable set_class : bool;
    
    public this (lex : LexerBase) {
      base (lex);
      
      indent_strings.Push ("");
    }

    push_end_brace () : void
    {
      tokens_pending.Push (Token.EndBrace (insertLocation, true));
    }

    CurrentIndent : string
    {
      get { indent_strings.Top }
    }

    handle_real_token (lexer_tok : Token) : void
    {
      unless (lexer_tok is Token.EndOfFile)
        had_some_real_input = true;

      last_token_was_semicolon = lexer_tok is Operator (";");

      match (lexer_tok) {
        | Token.EndOfFile =>
          insertLocation = lexer_tok.Location;

          while (indent_strings.Count > 1) {
            _ = indent_strings.Pop ();
            push_end_brace ();
          }
          when (set_class)
            push_end_brace ();
          when (set_namespace)
            push_end_brace ();

          tokens_pending.Push (lexer_tok);

        | Token.Operator ("\\") when indentation_syntax_active =>
          match (lexer.GetToken ()) {
            | Token.Indent => {}
            | _ =>
              throw PreParserException (lexer_tok.Location, 
                                        "unexpected `\\' in middle of a line");
          }
          
        | BeginBrace
        | BeginRound
        | BeginSquare
        | BeginQuote =>
          explicit_groups++;
          indentation_syntax_active = false;
          tokens_pending.Push (lexer_tok)
          
        | EndBrace
        | EndRound
        | EndSquare
        | EndQuote =>
          when (explicit_groups == 0)
            // TODO: better error message
            throw PreParserException (lexer_tok.Location, "unmatched group end");
          explicit_groups--;
          when (explicit_groups == 0)
            indentation_syntax_active = true;
          tokens_pending.Push (lexer_tok)
          
        // If this is the 'set' keyword in the first column of a new line:
        | Identifier ("set") when indentation_syntax_active && CurrentIndent == ""
        | Keyword ("set") when indentation_syntax_active && CurrentIndent == "" =>
          def next = lexer.GetToken ();
          match (next) {
            | Token.Keyword ("namespace") =>
              when (set_namespace) 
                throw PreParserException (next.Location, 
                        "the 'set namespace' directive can only be used once per file");
              set_namespace = true;
              force_brace_after_newline = true;
            
            | Token.Keyword ("class") =>
              when (set_class) 
                throw PreParserException (next.Location,
                        "the 'set class' directive can only be used once per file");
              set_class = true;
              force_brace_after_newline = true;
            
            | _ =>
              throw PreParserException (next.Location, $"unrecognized 'set' directive: '$next'");
          }

          tokens_pending.Push (next);
        
        | _ =>
          tokens_pending.Push (lexer_tok)
      }
    }

    /** Fetch next token (from one token buffer or lexer if it's empty */
    protected override get_token () : Token
    {
      // last_token has priority over tokens_pending
      if (last_token != null) {
        def result = last_token;
        last_token = null;
        result;
      } else if (! tokens_pending.IsEmpty) {
        def t = tokens_pending.Take ();
        //Message.Debug ($ "tok: $t");
        t
      }
      else {
        try {
          match (lexer.GetToken ()) {
            | Token.Indent as t =>
              if (indentation_syntax_active) {
                //Message.Debug (insertLocation, "Calling get_token_after_indent ()");
                get_token_after_indent (t)
              }
              else
                get_token ()
                
           | tok =>
             handle_real_token (tok);
             get_token ()
         }
        }
        catch {
          | e is LexerBase.Error =>
            Message.Error (lexer.Location, e.name);
            get_token ()
        }
      }
    }

    #pragma warning disable 10003
    static display_space (s : string) : string
    {
      s.Replace (" ", "S").Replace("\t", "T")
    }
    #pragma warning restore 10003

    get_token_after_indent (tok : Token, new_indent : string = "") : Token
    {
      match (tok) {
        | Token.Indent (value) =>
          //Message.Debug ($"tok indent: $(display_space(value))");
          insertLocation = tok.Location;
          get_token_after_indent (lexer.GetToken (), value)
          
        | Token.Comment =>
          get_token_after_indent (lexer.GetToken (), new_indent)

        | _ when force_brace_after_newline =>
          force_brace_after_newline = false;
          tokens_pending.Push (Token.BeginBrace (insertLocation, true));
          get_token_after_indent (tok, new_indent)

        | _ =>
          // If we have not unindented
          if (new_indent.Length >= CurrentIndent.Length) {
            //Message.Debug ($ "ni=$(display_space(new_indent)) ci=$(display_space(CurrentIndent))");
            // Make sure that the beginning of the new indent string
            //  starts with the contents of the current indent string.
            when (!new_indent.StartsWith (CurrentIndent)) {
              // TODO: make a better error message -- checkout python error message
              Message.Error (tok.Location, "inconsistent indentation");
            }
            
            // If we have remained at the same indentation level
            if (new_indent == CurrentIndent) {
              when (had_some_real_input && 
                    !last_token_was_semicolon && 
                    !(tok is Token.BeginBrace)) {
                //Message.Debug (tok.Location, $"Generate ';'");
                tokens_pending.Push (Token.Semicolon (insertLocation, true));
              }

              handle_real_token (tok);
            }
            // If we've indented further than the previous line
            else {
              indent_strings.Push (new_indent);
              tokens_pending.Push (Token.BeginBrace (insertLocation, true));
              handle_real_token (tok);
              //Message.Debug (tok.Location, "Generate '{'");
            }
          }
          // Otherwise, we've unintented:
          else {
            if (indent_strings.Contains (new_indent)) {
              while (indent_strings.Top != new_indent) {
                push_end_brace ();
                _ = indent_strings.Pop ();
              }
              handle_real_token (tok)
            } else {
              // TODO: make a better error message -- checkout python error message
              throw PreParserException (tok.Location, "inconsistent indentation");
            }
          }

          get_token ()
      }
    }
  }
}
