/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle;
using System;
using System.Reflection;
using Nemerle.Collections;
using SRE = System.Reflection.Emit;

namespace Nemerle.Compiler 
{
  /** Represents an explicitly bound type variable, like the one from
      polymorphic type or method definition. It is also used in
      generalizations of inferred types. */
  public class StaticTyVar
  {
    public override ToString () : string
    {
      // XXX include constraints?
      $ "$(Name).$(Id)"
    }


    /** Return lower constraints on given type variable (that is the
        [where] clauses). The constraints should be already normalized (that 
        is they should follow MType.Intersection invariants).
      */
    public Constraints : list [MType]
    {
      get {
        assert (constraints != null);
        constraints
      }

      set {
        assert (constraints == null);
        constraints = value;
        unless (value.IsEmpty)
          LowerBound.Validate ();
      }
    }


    public LowerBound : MType
    {
      get {
        assert (constraints != null);
        match (constraints) {
          | [] =>
            assert (InternalType.Object != null);
            InternalType.Object
          | [x] => x
          | lst => MType.Intersection (lst)
        }
      }
    }
    

    public Name : string
    {
      get { name }
    }


    public Id : int
    {
       get { id }
    }
    

    public this (name : string)
    {
      this.name = name;
      id = current_id;
      ++current_id;
    }

    private this (other : StaticTyVar)
    {
      this.name = other.name;
      this.special = other.special;
      this.id = current_id;
      
      ++current_id;
    }
      
    public this (name : string, gtype : System.Type) {
      this (name);
      this.gtype = gtype;
    }
    
    public SystemType : System.Type
    {
      get {
        assert (gtype != null, ToString ());
        gtype
      }
    }

    internal SpecialConstraints : GenericParameterAttributes
    {
      get { special }
    }
    
    internal SetConstraints (special : GenericParameterAttributes, subtype : list [MType]) : void
    {
      when ((special & GenericParameterAttributes.NotNullableValueTypeConstraint) %&&
            (special & ~GenericParameterAttributes.NotNullableValueTypeConstraint))
        Message.Error ("`struct' generic constraint cannot be used together with `class' or `new ()'");

      this.special = special;
      Constraints = subtype;
    }

    public IsValueType : bool {
      get {
        special %&& GenericParameterAttributes.NotNullableValueTypeConstraint
      }
    }
    
    public HasDefaultConstructor : bool {
      get {
        special %&& GenericParameterAttributes.DefaultConstructorConstraint
      }
    }
    
    
    [Nemerle.OverrideObjectEquals]
    public Equals (o : StaticTyVar) : bool
    {
      id == o.id
    }


    /** Copy a list of type parameters for use in a new type.
        Also return a substitution from the old to the new list.  */
    static public CopyList (typarms : list [StaticTyVar]) : Subst * list [StaticTyVar]
    {
      def sub = Subst ();
      def newtp =
        typarms.Map (fun (tp) {
          def ntp = StaticTyVar (tp);
          sub.Add (tp, MType.TyVarRef (ntp));
          (ntp, tp.Constraints)
        }).Map (fun (tv : StaticTyVar, constr : list [_]) {
          tv.Constraints = constr.Map (sub.MonoApply);
          tv
        });
      (sub, newtp)
    }


    internal SetGenericBuilder (x : SRE.GenericTypeParameterBuilder) : void
    {
      Util.cassert (gtype == null);
      Util.cassert (x != null);
      x.SetGenericParameterAttributes (special);
      gtype = x;
    }

    internal UpdateConstraints () : void
    {
      def gbuil = gtype :> SRE.GenericTypeParameterBuilder;
      def ifaces = match (constraints) {
        | (MType.Class as x) :: _ when x.tycon.IsInterface => constraints
        | x :: xs => gbuil.SetBaseTypeConstraint (x.SystemType); xs
        | [] => []
      }
      unless (ifaces.IsEmpty) {
        def arr = array (ifaces.Length);
        _ = ifaces.FoldLeft (0, fun (x, i) {
          arr [i] = x.SystemType;
          i + 1
        });
      
        gbuil.SetInterfaceConstraints (arr);
      }
    }


    /// Check if [tv] fullfills [this]'s constraints.
    public CheckConstraints (tv : TyVar, thing : object) : void
    {
      def has_default_ctor (t) {
        | MType.Class (ti, _) =>
          ! t.CanBeNull || // value types always have this ctor
          ti.LookupMember (".ctor").Exists (fun (_) {
            | meth is IMethod =>
              meth.Attributes %&& NemerleAttributes.Public &&
              meth.GetHeader ().parms.IsEmpty
            | _ => false
          })
        | MType.TyVarRef (s) =>
          s.SpecialConstraints %&& 
          GenericParameterAttributes.DefaultConstructorConstraint
        | _ => false
      }
      
      def t = tv.Fix ();
      
      when (SpecialConstraints %&& 
            GenericParameterAttributes.NotNullableValueTypeConstraint &&
            t.CanBeNull)
        // cs453
        Message.Error ($ "the type `$t' must be a value type "
                         "in order to use it as type parameter "
                         "`$(Name)' in $thing");

      when (SpecialConstraints %&& 
            GenericParameterAttributes.ReferenceTypeConstraint &&
            ! t.CanBeNull)
        // cs452
        Message.Error ($ "the type `$t' must be a reference type "
                         "in order to use it as type parameter "
                         "`$(Name)' in $thing");
      
      when (SpecialConstraints %&& 
            GenericParameterAttributes.DefaultConstructorConstraint &&
            ! has_default_ctor (t))
        // cs310
        Message.Error ($ "the type `$t' must have a public parameterless "
                         "constructor in order to use it as type parameter "
                         "`$(Name)' in $thing");
    }
    
    mutable gtype : System.Type; 

    internal id : int;
    name : string;
    mutable special : GenericParameterAttributes;
    mutable constraints : list [MType];

    internal mutable current_type : TypeInfo = null;
    internal mutable current_method : IMethod = null;

    static mutable current_id : int;    
  }
}
