/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;
using Nemerle.Logging;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

using PT = Nemerle.Compiler.Parsetree;

[assembly: LogFunction (Message.Debug),
           LogFlag (TYPING, 0),
           LogFlag (MACRO_EXPANSIONS, 0)
]

namespace Nemerle.Compiler
{
  [Record]
  public class CompletionResult : System.Exception
  {
    public Overloads : list [OverloadPossibility];
    public NamePrefix : string;
  }

  /** A class responsible for typing a single (global or local) function.  */
  public partial class Typer
  {
    is_instance_ctor : bool;
    is_method : bool;
    env : GlobalEnv;
    tenv : TyVarEnv;
    current_fun : Fun_header;
    [Accessor]
    current_type : TypeBuilder;
    parent_typer : option [Typer];
    mutable local_context : LocalContext;
    messenger : Messenger;
    solver : Solver;
    locals_stack : Stack [LocalContext];
    current_method_builder : MethodBuilder;
    mutable skip_n_return : bool;
    
    mutable inside_yielding_function : bool;
    mutable yield_labels : list [int];

    #region Toplevel typing
    class SwitchToYielding : System.Exception { }
    
    // FIXME: copy doc from TypeMethod
    /** Walk through method body typing it. */
    public this (m : MethodBuilder)
    {
      current_method_builder = m;
      solver = Passes.Solver;
      current_type = m.DeclaringType;
      parent_typer = None ();
      env = current_type.GlobalEnv;
      current_fun = m.GetHeader ();
      tenv = current_fun.tenv;
      messenger = solver.CurrentMessenger;
      locals_stack = Stack ();

      local_context = LocalContext.Empty;

      assert (solver.IsTopLevel);
      solver.dt_store = Map ();

      match (m.GetFunKind ()) {
        | FunKind.Constructor =>
          is_instance_ctor = true;
          is_method = true;
        | FunKind.Method | FunKind.BoundMethod =>
          is_method = true;
        | _ => ()
      }

      try {
        Util.locate (m.Location, {
          MaybeAddBaseCall (m);

          mutable typer_done = false;
          assert (messenger.IsTopLevel);

          def put_in_error_mode () {
            solver.Unwind ();
            messenger.Cleanup ();
            assert (!messenger.InErrorMode);
            assert (messenger.IsTopLevel);
            messenger.InErrorMode = true;
            local_context = LocalContext.Empty;
            assert (!typer_done);
          }

          try {
            messenger.Cleanup ();
            messenger.InErrorMode = false;
            def errcnt = Message.ErrorCount;
            
            RunTyper ();
            MaybeDumpTypedTree ();
            typer_done = true;
            
            assert (!messenger.SeenError);
            messenger.InErrorMode = true;
            RunDelayedTypings ();
            MaybeDumpTypedTree ();
            
            when (errcnt == Message.ErrorCount) {
              RunSecondPass (m);
              MaybeDumpTypedTree ();
            }
          } catch {
            | _ is RestartInErrorMode =>
              def errcnt = Message.ErrorCount;
              put_in_error_mode ();
              RunTyper ();

              when (errcnt == Message.ErrorCount)
                Util.ice ($"hidden errors compiling $m");
                
            | _ is SwitchToYielding =>
              log (TYPING, "got STY, trying again");
              current_fun.yield_type = GetYieldType ();
              def errcnt = Message.ErrorCount;
              // just in case
              put_in_error_mode ();
              m.Body = WrapYieldingFunction (m.Body);
              inside_yielding_function = true;
              when (errcnt == Message.ErrorCount) {
                assert (solver.IsTopLevel);
                solver.dt_store = Map ();
                RunTyper ();
              }
              when (errcnt == Message.ErrorCount)
                RunDelayedTypings ();
              when (errcnt == Message.ErrorCount) {
                RunSecondPass (m);
                MaybeDumpTypedTree ();
              }
          }
        })
      } finally {
        messenger.InErrorMode = true;
      }
    }
    

    internal static Fake (tb : TypeBuilder, tenv : TyVarEnv, fn : Fun_header, env : GlobalEnv) : Typer
    {
      Typer (tb, tenv, fn, env)
    }
    

    this (tb : TypeBuilder, tenv : TyVarEnv, fn : Fun_header, env : GlobalEnv)
    {
      solver = Passes.Solver;
      current_type = tb;
      parent_typer = None ();
      this.env = env;
      current_fun = fn;
      this.tenv = tenv;
      messenger = solver.CurrentMessenger;

      locals_stack = Stack ();
      local_context = LocalContext.Empty;
      assert (solver.IsTopLevel);
    }

    /** Walk through a local function body typing it. */
    public this (parent : Typer, fn : Fun_header)
    {
      parent_typer = Some (parent);

      // copy from parent
      solver = parent.solver;
      current_type = parent.current_type;
      local_context = parent.local_context;
      is_method = parent.is_method;
      is_instance_ctor = parent.is_instance_ctor;
      env = parent.env;
      inside_yielding_function = parent.inside_yielding_function;
      messenger = parent.messenger;
      locals_stack = Stack ();
      
      current_fun = fn;
      tenv = current_fun.tenv;

      RunTyper ();
    }

    public GetLocals () : LocalContext
    {
      local_context
    }

    public GetMethodBuilder () : MethodBuilder
    {
      current_method_builder
    }

    MaybeAddBaseCall (m : MethodBuilder) : void
    {
      when (is_instance_ctor && !m.DeclaringType.IsValueType)
        match (current_fun.body) {
          | FunBody.Parsed (Sequence (exprs) as seq) =>
            def exists_ctor_call (_) {
              | [] => false 
              | <[ base (.. $_) ]> :: _ 
              | <[ this (.. $_) ]> :: _ => true
              | _ :: xs => exists_ctor_call (xs)
            }
            if (exists_ctor_call (exprs))
              ()
            else
              Util.locate (seq.loc,
                current_fun.body = FunBody.Parsed (<[ { ..$(<[ base (); ]> :: exprs) } ]>)
              )
          | _ => ()
        }
    }


    MaybeDumpTypedTree () : void
    {
      when (Options.ShouldDump (current_fun)) {
        match (current_fun.body) {
          | FunBody.Typed (expr) =>
            Message.Debug (expr.loc, $ "$current_type.$(current_fun.name) -> "
                                       "$(current_fun.ret_type) : " +
                                     PrettyPrint.SprintTyExpr (None (), expr) + "\n")
          | _ => ()
        }
      }
    }


    RunTyper () : void
    {
      foreach (p : Fun_parm in current_fun.parms) {
        Util.locate (p.loc, {
          when (p.kind != ParmKind.Normal && !IsTopLevel)
            ReportError (messenger, "ref/out parameters are not supported in local methods");
          def is_mutable =
            p.modifiers.mods %&& NemerleAttributes.Mutable ||
            p.kind != ParmKind.Normal;
          def l = LocalValue (current_fun, p.name, StripRefOut (p.ty),
                              LocalValue.Kind.FunParm (p.kind),
                              is_mutable = is_mutable);
          AddLocal (PT.Name (p.name, p.color, null), l);
          p.decl = l;
        })
      }

      when (inside_yielding_function &&
            current_fun.name == "_N_yielding_function") {
        Util.cassert (yield_labels == null); 
        yield_labels = [];
      }

      match (current_fun.body) {
        | FunBody.Parsed (e) =>
          def skip_n_return =
            match (parent_typer) {
              | Some (t) =>
                if (t.skip_n_return) {
                  t.skip_n_return = false;
                  true
                } else false
              | None => false
            }
          def local = 
            if (skip_n_return) 
              null
            else 
              MakeImplicitBlockJumpOut ("_N_return", current_fun.ret_type);
          def e' = TypeExpr (e);
          messenger.CleanLocalError ();
          def e' = AddCastTo (e', current_fun.ret_type, "function return type");
          def e' =
            if (skip_n_return) e' 
            else TExpr.Block (e'.Type, local, e');
          current_fun.body = FunBody.Typed (e');

        | FunBody.Abstract => ()
          
        | FunBody.ILed
        | FunBody.Typed => 
          Util.ice ($"$(current_fun.body)")
      }

      when (inside_yielding_function &&
            current_fun.name == "_N_yielding_function")
        match (current_fun.body) {
          | FunBody.Typed (t) => AddYieldStateMachine (t)
          | _ => Util.ice ()
        }
    }


    RunSecondPass (meth : MethodBuilder) : void
    {
      def t2 = Typer2 (current_type, meth);
      t2.Run ();
      def t3 = Typer3 (meth);
      t3.Run ();
    }
    #endregion


    #region Delayed typing queues
    RunDelayedTypings () : void
    {
      mutable seen_unresolved = false;
      mutable did_something = false;
      
      solver.dt_store.Iter (fun (dt : DelayedTyping, _) {
        unless (dt.IsResolved) {
          dt.Resolve ();
          if (dt.IsResolved)
            did_something = true;
          else
            seen_unresolved = true;
        }
      });

      when (seen_unresolved)
        if (did_something)
          RunDelayedTypings ();
        else {
          mutable error_dt = null;
          
          solver.dt_store.Iter (fun (dt : DelayedTyping, _) {
            unless (dt.IsResolved)
              when (error_dt == null)
                error_dt = dt;
          });
          assert (error_dt != null);

          Util.locate (error_dt.loc,
             ReportError (messenger, 
                          $ "typing fails on "
                            "$(error_dt.GetDescriptionForError ())"))
        }
    }
    #endregion


    #region Utilities
    internal CurrentFunction : Fun_header
    {
      get { current_fun }
    }


    public Env : GlobalEnv
    {
      get { env }
    }


    public CurrentTypeBuilder : TypeBuilder
    {
      get { current_type }
    }


    public CurrentMethodBuilder : MethodBuilder
    {
      get {
        match (parent_typer) {
          | Some (t) => t.CurrentMethodBuilder
          | None => current_method_builder
        }
      }
    }


    IsTopLevel : bool
    {
      get { ! (parent_typer.IsSome) }
    }

    AddLocal (name : PT.Name, local : LocalValue) : void
    {
      local_context = local_context.WithLocal (name, local);
    }


    static TypeOf (expr : TExpr) : TyVar
    {
      expr.Type
    }


    DefineLocal (name : PT.Name, ty : TyVar,
                 kind : LocalValue.Kind, is_mutable : bool) : LocalValue
    {
       LocalValue (current_fun, name.Id, ty, kind, is_mutable)
    }

    public AddRedirection (name : PT.Name, subst : PT.PExpr) : void
    {
      def kind = LocalValue.Kind.MacroRedirection (subst);
      def loc = LocalValue (current_fun, name.Id, null, kind, false);
      local_context = local_context.WithLocal (name, loc);
    }
    

    public BindType (t : PT.PExpr) : TyVar
    {
      tenv.Bind (env, current_type, t, 
                 allow_tyvars = true, 
                 check_parms = true)
    }


    public MonoBindType (t : PT.PExpr) : MType
    {
      tenv.MonoBind (env, current_type, t, 
                     check_parms = true)
    }


    Delay (kind : DelayedTyping.Kind, expected : TyVar) : TExpr
    {
      TExpr.Delayed (expected, DelayedTyping (this, kind, expected))
    }


    /** Delay execution of the current macro. You need to provide a function
        that will resolve it at a later time, when more info is needed. It shall
        take one boolean parameter, when it is true, you should fail with a
        descriptive error message stating that more typing information is needed.
        If it is false, you should just return None ().

        For example:

        macro foo (x) {
          def x' = Macros.ImplicitCTX ().TypeExpr (x);
          
          Macros.ImplicitCTX ().DelayMacro (fun (fail_loudly) {
            match (x'.Type.Hint) {
              | Some (t) =>
                // do something with the type
                Some (...)
              | None =>
                when (fail_loudly)
                  Message.Error (x.loc, $ "cannot deduce type of $x");
                None ()
            }
          })
        }

        This function will first try to call your resolution function and only if
        it fails, it will get queued for later.
    */
    public DelayMacro (resolve : bool -> option [PT.PExpr],
                       mutable expected : TyVar = null) : PT.PExpr
    {
      when (expected == null)
        expected = FreshTyVar ();
      <[ $(DelayAction (expected, DelayedMacro (this, expected, resolve)) : typed) ]>
    }

    class DelayedMacro : DelayedAction
    {
      typer : Typer;
      expected : TyVar;
      fn : bool -> option [PT.PExpr];

      context : int * int * GlobalEnv;

      public this (typer : Typer, expected : TyVar, fn : bool -> option [PT.PExpr])
      {
        this.typer = typer;
        this.expected = expected;
        this.fn = fn;

        context = 
          (MacroColorizer.Color, 
           MacroColorizer.UseColor, 
           MacroColorizer.UseContext);
      }
      
      public override Resolve (fail_loudly : bool) : option [TExpr]
      {
        def backup = 
          (MacroColorizer.Color, 
           MacroColorizer.UseColor, 
           MacroColorizer.UseContext);
           
        (MacroColorizer.Color, 
         MacroColorizer.UseColor, 
         MacroColorizer.UseContext) = context;

        try {
          match (fn (fail_loudly)) {
            | Some (e) => Some (typer.TypeExpr (e, expected))
            | None => None ()
          }
        } finally {
          (MacroColorizer.Color, 
           MacroColorizer.UseColor, 
           MacroColorizer.UseContext) = backup;
        }
      }
    }

    internal DelayAction (expected : TyVar, action : DelayedAction) : TExpr
    {
      def kind = DelayedTyping.Kind.Macro (action);
      def dt = DelayedTyping (this, kind, expected);
      dt.Resolve ();
      TExpr.Delayed (expected, dt)
    }


    /** Filter out wrong nodes, and if more then one left -- make the
        [Overloaded] node, and register it.  Removes duplicates.

        Wrong node is not [StillPossible] node, or one type of which
        cannot unify with [expected].  
      */
    MakeOverloadedNode (overloads : list [OverloadPossibility], expected : TyVar) : TExpr
    {
      def res = OverloadPossibility.Unique (overloads);
      match (OverloadPossibility.OnlyPossible (res, expected)) {
        | [] => TExpr.Error ()

        | [o] when o.ExtensionMethodObject == null =>
          def expr = o.Compile ();
          _ = Expect (expected, expr.Type, "overloaded symbol");
          expr
        
        | lst =>
          Delay (DelayedTyping.Kind.Overloaded (lst), expected)
      }
    }


    /** Run given typing action, without possiblity of any error messages. */
    TryTyping (fn : void -> TExpr) : TExpr
    {
      def is_wrong =
        try {
          solver.PushState ();
          _ = fn ();
          messenger.SeenError
        } finally {
          solver.PopState ()
        }
      if (is_wrong) TExpr.Error ()
      else
        // we cannot reuse the previous result, since it's has been
        // popped off the stack
        fn ()
    }


    static IsError (expr : TExpr) : bool
    {
      expr is TExpr.Error
    }

    IsLValue (e : TExpr, need_ref : bool) : bool
    {
      match (e) {
        | TExpr.LocalRef (d) => d.IsMutable
        | TExpr.StaticRef (_, m, _) =>
          match (m.GetKind ()) {
            | MemberKind.Field (f) =>
              f.IsMutable || (current_fun.name == ".cctor" &&
                              f.DeclaringType.Equals (current_type))
            | MemberKind.Property (p) when !need_ref => p.IsMutable
            | _ => false
          }
        | TExpr.StaticPropertyRef (_, p) when !need_ref => p.IsMutable
        | TExpr.FieldMember (TExpr.This, _) when is_instance_ctor => true
        | TExpr.FieldMember (_, mem) => mem.IsMutable || e.SkipWriteCheck
        | TExpr.PropertyMember (_, p) when !need_ref => p.IsMutable
        | TExpr.Call (TExpr.PropertyMember (_, p), _, _) when !need_ref => p.IsMutable
        | TExpr.Call (TExpr.Delayed (dt), _, _) when !need_ref => dt.IsMutableIndexer
        | TExpr.This when current_type.IsValueType
        | TExpr.ArrayIndexer => true
        | TExpr.Delayed (dt) => dt.ExpectLValue (need_ref)
        | TExpr.EventMember
        | TExpr.StaticEventRef =>
          // this is for add/remove
          // should we do some stricter checking here?
          true
        | _ => false
      }
    }
   

    /** Check if [e] is an l-value, and if not display an error message
        mentioning [desc]. Additionally is [need_ref] require location of
        the expression to be known (i.e. no properties and indexers).
      */
    CheckLValue (e : TExpr, need_ref : bool, desc : string) : void
    {
      unless (IsLValue (e, need_ref) || e is TExpr.Error) {
        ReportError (messenger, $ "needed a writable location for $desc, "
                                  "got $(DescribeExpression (e)), which "
                                  "is read-only");
        messenger.CleanLocalError ();
      }
    }

    static LookupStaticMethod (ti : TypeInfo, name : string) : list [IMethod]
    {
      mutable res = [];
      foreach (mem : IMember in ti.LookupMember (name)) { 
        match (mem) {
          | meth is IMethod when meth.IsStatic =>
            res = meth :: res;
          | _ => {}
        }
      }
      match (ti.SuperClass ()) {
        | Some (ti) =>
          res + LookupStaticMethod (ti, name)
        | None => res
      }
    }

    
    static LookupStaticMethod (ty : MType, name : string) : list [IMethod]
    {
      match (ty) {
        | MType.Class (ti, _) =>
          LookupStaticMethod (ti, name)
        | _ => []
      }
    }


    static internal SquashDuplicates['a] (lst : list ['a]) : list ['a]
      where 'a : IMember
    {
      def seen = Hashtable ();
      mutable res = [];
      foreach (elem in lst) {
        if (seen.Contains (elem.GetHashCode ())) {}
        else {
          seen [elem.GetHashCode ()] = null;
          res = elem :: res;
        }
      }
      res
    }

    This () : TExpr
    {
      TExpr.This (current_type.GetMemType ())
    }


    static internal VoidLiteral () : TExpr
    {
      TExpr.Literal (InternalType.Void, Literal.Void ())
    }


    static StripRefOut (t : TyVar) : TyVar
    {
      if (t.IsFixed)
        match (t.FixedValue) {
          | MType.Ref (t)
          | MType.Out (t) => t
          | _ => t
        }
      else t
    }
    

    public JustTry['a] (f : void -> 'a) : 'a
    {
      solver.PushState ();
      PushLocals ();
      try {
        f ()
      } finally {
        PopLocals ();
        solver.PopState ()
      }
    }
    

    public static ImplicitCast (expr : TExpr, ty : TyVar) : TExpr
    {
      TExpr.TypeConversion (expr.loc, ty, expr, ty, ConversionKind.Implicit ())
    }


    static CanBeTypeName (expr : PT.PExpr) : bool
    {
      match (expr) {
        | PT.PExpr.Ref => true
        | <[ $expr . $_ ]>
        | <[ $expr . [ .. $_ ] ]>
        | <[ $expr [ .. $_ ] ]> =>
          CanBeTypeName (expr)
        | _ => false
      }
    }
    

    // [true] if expr will result in a functional type
    // this is just a heuristic
    static IsFunctional (expr : TExpr) : bool
    {
      match (expr) {
        | TExpr.StaticRef (_, _ is IMethod, _) => true
        | TExpr.Delayed (dt) => dt.IsFunctional
        | _ => expr.Type.Hint is Some (MType.Fun)
      }
    }
    #endregion


    #region Implicit conversions
    static LookupConversions (tv : TyVar) : list [IMember]
    {
      def lookup (acc, tc) {
        def acc = tc.LookupMember ("op_Implicit") + acc;
        match (tc.SuperClass ()) {
          | Some (tc) => lookup (acc, tc)
          | None => acc
        }
      }

      match (tv.Hint) {
        | Some (Class (tc, _)) => lookup ([], tc)
        | _ => []
      }
    }

    SubtypingOrImplicitConv (from : TyVar, to : TyVar, method : out TExpr.StaticRef) : bool
    {
      method = null;
      if (from.TryRequire (to)) {
        from.Require (to)
      } else {
        def methods = LookupConversions (from) + LookupConversions (to);
        def methods = SquashDuplicates (methods);
        
        if (methods.IsEmpty) false
        else {
          def needed = MType.ConstructFunctionType ([from], to);
          def fresh_type (meth) {
            meth.DeclaringType.GetFreshType ().TypeOfMethodWithTyparms (meth) [0]
          }
          def res = $[ meth | meth is IMethod in methods, 
                              meth.IsStatic,
                              fresh_type (meth).TryRequire (needed) ];
          def res = GetBestOverloads1 (res);
          
          match (res) {
            | [one] =>
              def overload = ConstructMethodOverload (one).Head;
              method = overload.Compile () :> TExpr.StaticRef;
              def ok = method.Type.Require (needed);
              Util.cassert (ok);
              true
            | [] => false
            | lst =>
              ReportError (messenger, 
                           $"ambiguity between implicit conversions $lst");
              false
          }
        }
      }
    }


    AddCastTo (expr : TExpr, target : TyVar, place : string) : TExpr
    {
      def expr' = TryAddCastTo (expr, target);
      if (expr' != null) expr'
      else {
        _ = ExpectSubtyping (target, expr.Type, place);
        TExpr.Error (target)
      }
    }


    static internal LiteralConversionPossible (lit : Literal, target : MType) : bool
    {
      match (lit) {
        | Literal.Integer (res, is_negative, _) =>
          if (target.Equals (InternalType.UInt64))
            ! is_negative
          // there is one more negative number
          else if (res <= System.Int64.MaxValue :> ulong ||
                   (is_negative && res == 0x8000000000000000UL)) {
            def res' = 
              if (res == 0x8000000000000000UL)
                System.Int64.MinValue
              else
                if (is_negative) 
                  -(res :> long) 
                else 
                  res :> long;

            if (target.Equals (InternalType.Int32))
              res' >= System.Int32.MinValue && res' <= System.Int32.MaxValue
            else if (target.Equals (InternalType.UInt32))
              res' >= System.UInt32.MinValue && res' <= System.UInt32.MaxValue
            else if (target.Equals (InternalType.Byte))
              res' >= System.Byte.MinValue && res' <= System.Byte.MaxValue
            else if (target.Equals (InternalType.Int64))
              res' >= System.Int64.MinValue && res' <= System.Int64.MaxValue
            else if (target.Equals (InternalType.SByte))
              res' >= System.SByte.MinValue && res' <= System.SByte.MaxValue
            else if (target.Equals (InternalType.Int16))
              res' >= System.Int16.MinValue && res' <= System.Int16.MaxValue
            else if (target.Equals (InternalType.UInt16))
              res' >= System.UInt16.MinValue && res' <= System.UInt16.MaxValue
            else false
          }
          else false
        
        | Literal.Null =>
          target.CanBeNull &&
          ! (target is MType.TyVarRef)

        | _ =>
          target.TryUnify (lit.GetInternalType ())
      }
    }

    TryAddCastTo (expr : TExpr, target : TyVar) : TExpr
    {
      match (expr) {
        | TExpr.Literal (lit)
          when target.Hint.IsSome && 
               LiteralConversionPossible (lit, Option.UnSome (target.Hint)) =>
          target.Fixate ();
          if (target.Equals (InternalType.Int64) ||
              target.Equals (InternalType.UInt64))
            TExpr.TypeConversion (target, expr, target,
                                  ConversionKind.IL (local_context.IsChecked))
          else if (lit is Literal.Null) {
            _ = expr.Type.Unify (target);
            expr
          } else
            expr

        // 0 -> any enum
        | TExpr.Literal (Literal.Integer (0, _, _))
          when target.IsFixed && target.FixedValue.IsEnum =>
          TExpr.TypeConversion (target, expr, target,
                                ConversionKind.IL (local_context.IsChecked))

        | _ =>
          mutable meth = null;

          def del_tc =
            if (IsFunctional (expr))
              match (target.Hint) {
                | Some (MType.Class (del, _)) when del.IsDelegate => del
                | _ => null
              }
            else null;

          if (del_tc != null) {
            log (TYPING, $ "delegate conversion, $(expr.Type) ---> $del_tc");
            // we don't want any delegate named del_tc, we want this specific delegate
            def con = MakeOverloadedNode (TypeToConstructor (null, del_tc), FreshTyVar ());
            def expr = TryTyping (fun () {
              TypeExpr (<[ $(con : typed) ($(expr : typed)) ]>)
            });
            if (IsError (expr) || ! expr.Type.TryRequire (target))
              null
            else {
              _ = expr.Type.Require (target);
              expr
            }
          } else if (SubtypingOrImplicitConv (expr.Type, target, out meth)) {
            log (TYPING, $ "got subtyping, $(expr.Type) ---> $target, meth=$meth");
            if (meth == null)
              expr
            else
              match (meth) {
                | StaticRef (_, meth is IMethod, _) 
                  when meth.BuiltinKind is ValueTypeConversion () =>
                  TExpr.TypeConversion (target, expr, target,
                                        ConversionKind.IL (local_context.IsChecked))
                | _ =>
                  TExpr.TypeConversion (target, expr, target,
                                        ConversionKind.MethodCall (meth))
              }
          } else if (target.Hint is Some (MType.Void)) {
            _ = target.Unify (InternalType.Void);
            assert (expr != null);
            TExpr.TypeConversion (target, expr, target,
                                  ConversionKind.IgnoreValue ())
          } else if (target.Hint is Some (Fun) && 
                     expr.Type.Hint is Some (Fun) &&
                     expr.Type.Fix ().TrySigRequire (target.Fix ())) {
            log (TYPING, $ "function subtyping, $(expr.Type) ---> $target, meth=$meth");
            def target = target.Fix ();
            _ = expr.Type.FixedValue.SigRequire (target);
            def srctypes = Option.UnSome (target.FunReturnTypeAndParms ()) [0];
            def (parms, refs) =
              List.Split (srctypes.Map (fun (ty) {
                def name = Util.tmpname ("p");
                ( <[ parameter : $(name : dyn) : $(ty : typed) ]>,
                  <[ $(name : dyn) ]> )
              }));
              TypeExpr (<[ fun (.. $parms) { $(expr : typed) (.. $refs) } ]>, target)
          } else null
      }
    }
    #endregion


    #region Solver interaction
    static FreshTyVar () : TyVar
    {
      Solver.FreshTyVar ()
    }


    static ConstructFunctionType (parms : list [TyVar], res : TyVar) : TyVar
    {
      MType.ConstructFunctionType (parms, res)
    }

   
    static internal AtLeast (m : TyVar) : TyVar
    {
      def tv = FreshTyVar ();
      tv.ForceRequire (m);
      tv
    }


    PushLocals () : void
    {
      //Message.Debug ("push locals{ ");
      locals_stack.Push (local_context)
    }


    PopLocals () : void
    {
      //Message.Debug ("pop locals }");
      local_context = locals_stack.Pop ()
    }
    #endregion


    #region Expressions in general
    public TypeExpr (e : PT.PExpr) : TExpr
    {
      TypeExpr (e, FreshTyVar ())
    }

    
    public TypeExpr (e : PT.PExpr, expected : TyVar) : TExpr
    {
      TypeExpr (e, expected, false)
    }


    public TypeExpr (e : PT.PExpr, expected : TyVar, is_toplevel_in_seq : bool) : TExpr
    {
      log (TYPING, e.loc, $ "(typing expression: $e, exp=$expected");
      Util.locate (e.loc, {
        def e' = InterceptSpecialMacros (e, expected);
        def e' = 
          try {
            if (e' != null) {
              log (TYPING, e.loc, $ "intercepted: $e'");
              e'
            } else {
              log (MACRO_EXPANSIONS, e.loc, $ "running expand with: $e");
              def e = MacroRegistry.expand_macro (this, e);
              log (MACRO_EXPANSIONS, e.loc, $ "after expansion: $e");

              def res = DoType (e, expected, is_toplevel_in_seq);
              
              log (TYPING, e.loc, 
                   $ "done typing: $e --> $res : $(if (res.ty != null) res.Type else null))");
              res
            }
          } catch {
            | _ is Recovery =>
              messenger.SeenError = true;
              TExpr.Error ()
          }

        if (e'.ty == null)
          e'.ty = expected;
        else
          _ = Expect (expected, e'.ty, "previously typed expression");
          
        e'
      })
    }

    
    public InErrorMode : bool {
      get { messenger.InErrorMode } 
    }

    DoExpect (expected : TyVar, actual : TyVar, place : string, is_subtyping : bool) : bool
    {
      if (messenger.InErrorMode) {
        def exp_s = expected.ToString ();
        def act_s = actual.ToString ();
        if (if (is_subtyping) actual.Require (expected) else expected.Unify (actual))
          true
        else {
          ReportError (messenger, $ "expected $exp_s, got $act_s in $place");
          false
        }
      } else {
        if (if (is_subtyping) actual.Require (expected) else expected.Unify (actual))
          true
        else {
          // the message is not going to be used anyway
          ReportError (messenger, null);
          false
        }
      }
    }

   
    Expect (expected : TyVar, actual : TyVar, place : string) : bool
    {
      DoExpect (expected, actual, place, false)
    }

   
    ExpectSubtyping (expected : TyVar, actual : TyVar, place : string) : bool
    {
      DoExpect (expected, actual, place, true)
    }

   
    DoType (expression : PT.PExpr, expected : TyVar, is_toplevel_in_seq : bool) : TExpr
    {
      match (expression) {
        | PT.PExpr.ParmByRef
        | PT.PExpr.ParmOut =>
          ReportFatal (messenger, 
                       "ref and out parameters are only allowed in function calls")

        | PT.PExpr.DefMutable (x, null) => 
          DoType (PT.PExpr.DefMutable (x, <[ $(TExpr.DefaultValue (FreshTyVar ()) : typed) ]>), expected, is_toplevel_in_seq)
                       
        | PT.PExpr.DefMutable (PT.PExpr.Ref (name), val)          
        | PT.PExpr.Define (PT.PExpr.Ref (name), val) =>
          def is_mutable = expression is PT.PExpr.DefMutable;
          if (Expect (expected, InternalType.Void, "definition ``result''"))
            if (is_toplevel_in_seq)
              TypeLocalDefinition (is_mutable, name, val)
            else
              try {
                PushLocals ();
                TypeLocalDefinition (is_mutable, name, val)
              } finally {
                PopLocals ()
              }
          else
            TExpr.Error ()

        | DefMutable (Tuple (Ref (first_n) :: names), Tuple (first_v :: vals)) =>
          if (Expect (expected, InternalType.Void, "definition ``result''"))
            try {
              unless (is_toplevel_in_seq)
                PushLocals ();

              def first_v = if (first_v == null) <[ $(TExpr.DefaultValue (FreshTyVar ()) : typed) ]> else first_v;
              def top = TypeLocalDefinition (true, first_n, first_v);
              
              def create_single (n, mutable v, acc) {
                when (v == null) 
                  v = <[ $(TExpr.DefaultValue (FreshTyVar ()) : typed) ]>;
                match (n) {
                  | PT.PExpr.Ref (name) =>
                    def def_val_in = TypeLocalDefinition (true, name, v);
                    acc.body = def_val_in;
                    def_val_in

                  | PT.PExpr.TypeEnforcement (nm, ty) =>
                    create_single (nm, PT.PExpr.TypeEnforcement (v, ty), acc)
                    
                  | _ =>
                    Message.FatalError (n.Location, "expected simple variable name in mutable definition")
                }
              }
              _ = List.FoldLeft2 (names, vals, top, create_single);
              top
                
            } finally {
              unless (is_toplevel_in_seq)
                PopLocals ()
            }
          else
            TExpr.Error ()

        | DefMutable (TypeEnforcement (nm, ty), val) =>
          DoType (PT.PExpr.DefMutable (nm, PT.PExpr.TypeEnforcement (val, ty)),
                  expected, is_toplevel_in_seq)
        
        | DefMutable => Message.FatalError ($"incorrect mutable variables definition: $expression")
        
        | PT.PExpr.Define (pat, expr) =>
          if (Expect (expected, InternalType.Void, "definition ``result''"))
            if (is_toplevel_in_seq)
              TypeDefPattern (pat, expr);
            else
              try {
                PushLocals ();
                TypeDefPattern (pat, expr);
              } finally {
                PopLocals ()
              }
          else
            TExpr.Error ()
          

        | PT.PExpr.DefFunctions (functions) =>
          if (Expect (expected, InternalType.Void, "definition ``result''"))
            if (is_toplevel_in_seq)
              TypeLocalFunction (functions)
            else
              try {
                PushLocals ();
                TypeLocalFunction (functions)
              } finally {
                PopLocals ()
              }
          else
            TExpr.Error ()


        // FIXME: this should be a macro
        | PT.PExpr.Lambda (d) =>
          def newname = d.header.ParsedName.NewName (Util.tmpname ("l"));
          d.header.name = PT.Splicable.Name (newname);
          def rf = PT.PExpr.Ref (newname);
          def sq = PT.PExpr.Sequence ([PT.PExpr.DefFunctions ([d]), rf]);
          TypeExpr (sq, expected)

        | PT.PExpr.Array (<[ $(r : int) ]>, PT.PExpr.ListLiteral ([])) =>
          def list_of_zeros (r) {
            | 1 => [ <[ 0 ]> ]
            | _ => <[ 0 ]> :: list_of_zeros (r - 1)
          }
          TypeExpr (<[ array (..$(list_of_zeros (r))) ]>, expected)


        | PT.PExpr.Array (<[ $(rank : int) ]>, args) =>
          TypeArray (rank, args, expected)

        // array [...]
        | PT.PExpr.Array (PT.PExpr.Literal (Literal.Integer (r, false, _)), PT.PExpr.ListLiteral ([])) =>
          def list_of_zeros (r) {
            | 1UL => [ <[ 0 ]> ]
            | _ => <[ 0 ]> :: list_of_zeros (r - 1)
          }
          TypeExpr (<[ array (..$(list_of_zeros (r))) ]>, expected)


        | PT.PExpr.Array (PT.PExpr.Literal (Literal.Integer (rank, false, _)), args) =>
          TypeArray (rank :> int, args, expected)


        | PT.PExpr.Array (PT.PExpr.Spliced, _) =>
          ReportFatal (messenger, "$ splicing outside of <[ quotation ]>")


        | PT.PExpr.Array =>
          Message.Debug (expression.ToString ());
          Util.ice ("wrongly parsed array initializer")


        | PT.PExpr.EmptyArray (sizes) =>
          mutable size = 0;
          def tsizes = List.Map (sizes, fun (x) {
            ++size;
            TypeExpr (x, InternalType.Int32)
          });
          def arty = MType.Array (FreshTyVar (), size);
          _ = Expect (expected, arty, "empty array initializer");
          TExpr.Array ([], tsizes)


        | PT.PExpr.Indexer (obj, args) =>
          TypeIndexer (obj, args, expected)


        | (PT.PExpr.TryWith) as x =>
          def body = TypeExpr (x.body);
          if (ExpectSubtyping (expected, body.Type, "try body"))
            match (x.exn) {
              | PT.Splicable.Name (exn) =>
                def decl = DefineLocal (exn, BindType (x.exn_ty),
                                        LocalValue.Kind.ExceptionValue (), false);
                PushLocals ();
                def handler = 
                  try {
                    AddLocal (exn, decl);
                    TypeExpr (x.handler);
                  } finally {
                    PopLocals ();
                  }

                _ = ExpectSubtyping (expected, handler.Type, "catch body");
                TExpr.TryWith (ImplicitCast (body, expected), decl,
                               ImplicitCast (handler, expected))

              | _ =>
                ReportFatal (messenger, 
                             "$ operator used outside quotation <[ ... ]> context")
            }
          else TExpr.Error ()


        | PT.PExpr.TryFinally (e1, e2) =>
          def body = TypeExpr (e1, expected);
          def tv = FreshTyVar ();
          def handler = TypeExpr (e2, tv);
          TExpr.TryFinally (body, handler)


        | PT.PExpr.Typeof (t) =>
          _ = Expect (expected, InternalType.Type, "typeof result");
          TExpr.TypeOf (BindType (t))


        | PT.PExpr.TypeConversion (e, t) =>
          def t = BindType (t).Fix ();
          if (Expect (expected, t, "type-conversion result"))
            TypeConversion (TypeExpr (e), t)
          else
            TExpr.Error ()

        | PT.PExpr.TypeEnforcement (label, PT.PExpr.Sequence as body) =>
          match (label) {
            | PT.PExpr.Ref (name) =>
              TypeBlock (name, body, expected)
            | _ =>
              ReportFatal (messenger,
                           "block expects a simple identifier as a label")
          }
          
        | PT.PExpr.TypeEnforcement (e, t) =>
          def t = BindType (t);
          if (Expect (expected, t, "type-enforcement body")) {
            def e = AddCastTo (TypeExpr (e), t, "type-enforced expression");
            TExpr.TypeConversion (e, t, ConversionKind.UpCast ())
          } else
            TExpr.Error ()


        | PT.PExpr.Ref (name) =>
          match (TypeLocal (name, expected)) {
            | null => TypeName (expression, expected)
            | e => e
          }
          

        | PT.PExpr.Member (obj, PT.Splicable.Name (name)) =>
          def mem_name = name.Id;
          if (obj is <[ _ ]>)
            TypeExpr (PartialApplication (expression), expected)
          else if (CanBeTypeName (obj))
            match (TryTyping (fun () { TypeName (expression, expected) })) {
              | TExpr.Error =>
                // XXX this is a proper place to employ some heuristic
                // for example see if obj is uppercase identifier
                def obj =
                  if (messenger.InErrorMode)
                    TryTyping (fun () { TypeExpr (obj) })
                  else
                    TypeExpr (obj);
                  
                if (IsError (obj))
                  TypeName (expression, expected)
                else if (IsFunctional (obj)) {
                  if (InternalType.Object_tc.LookupMember (mem_name) is [])
                    TypeName (expression, expected)
                  else
                    if (messenger.InErrorMode) {
                      def e = TryTyping (fun () { TypeMemberExpr (obj, name, expected) });
                      if (IsError (e))
                        TypeName (expression, expected)
                      else
                        TypeMemberExpr (obj, name, expected)
                    } else
                      TypeMemberExpr (obj, name, expected)
                  } else
                    TypeMemberExpr (obj, name, expected)

              | e => e
            }
          else
            TypeMemberExpr (TypeExpr (obj), name, expected)
            

        | PT.PExpr.Member (obj, PT.Splicable.HalfId (name)) =>
          def skip_name =
            ! CanBeTypeName (obj) ||
            try {
              solver.PushState ();
              TypeNameFull (expression, expected, for_completion = true).IsEmpty
            } finally {
              solver.PopState ()
            }
          
          def name_overloads =
            if (skip_name) []
            else
              TypeNameFull (expression, expected, for_completion = true);

          def member_overloads = 
            match (TypeMember (TypeExpr (obj), name, expected,
                               for_completion = true)) {
              | Some (lst)
              | None with lst = [] => lst
            }

          throw CompletionResult (member_overloads + name_overloads, name.Id)


        | PT.PExpr.ToComplete (name) =>
          // FIXME: check locals first
          def overloads =
            TypeNameFull (expression, expected, for_completion = true);
          throw CompletionResult (overloads, name.Id)

          
        | PT.PExpr.Member (_, _) =>
          ReportFatal (messenger,
                       "$ operator outside quotation <[ ... ]> context")


        // 2 * 3 * 4 is Call(*,[2,3,4]), we need to transform it
        | PT.PExpr.Call (PT.PExpr.Ref (PT.Name where (idl = "*")) as star_ref, parms)
          when parms.Length > 2 =>
          def loop (_) {
            | [p1, p2] =>
              PT.PExpr.Call (star_ref, [p2, p1])
            | p :: ps =>
              PT.PExpr.Call (star_ref, [loop (ps), p])
            | [] => assert (false)
          }
          TypeExpr (loop (List.Rev (parms)), expected)
          
        | <[ $("_N_ref_cache" : dyn) ($(name : name), $val) ]> =>
            if (Expect (expected, InternalType.Void, "definition ``result''")) {
              def res = TypeLocalDefinition (false, name, val);
              res.name.IsCache = true;
              res
            } else
              TExpr.Error ()

        | PT.PExpr.Call (Ref (name = (idl = "_N_skip_write_check")), [parm]) =>
          def e = TypeExpr (parm, expected);
          e.SkipWriteCheck = true;
          e

        | PT.PExpr.Call (fnc, parms) =>
          if (parms.Exists (fun (_) {
                  | <[ _ ]> => true
                  | _ => false
                }) || fnc is <[ _ . $_ ]>)
            TypeExpr (PartialApplication (expression), expected)
          else
            TypeCall (fnc, parms, expected, is_property = false)


        | PT.PExpr.Assign (PT.PExpr.Tuple (vars), e2) =>
          if (Expect (expected, InternalType.Void, "assignment ``result''")) {
            def tempnames = List.RevMap (vars, fun (_) { <[ $(Macros.NewSymbol () : name) ]> });
            def assigns = List.Map2 (vars, tempnames, fun (x, y) { <[ $x = $y ]> });
            def expr = <[
              def (..$tempnames) = $e2;
              {..$assigns }
            ]>;
            TypeExpr (expr, expected)
          } else 
            TExpr.Error ()


        | PT.PExpr.Assign (PT.PExpr.Wildcard, e2) =>
          if (Expect (expected, InternalType.Void, "assignment ``result''"))
            TypeExpr (<[ def _ = $e2 ]>, expected)
          else
            TExpr.Error ()


        | PT.PExpr.Assign (e1, e2) =>
          if (Expect (expected, InternalType.Void, "assignment ``result''")) {
            def e1 = TypeExpr (e1);
            CheckLValue (e1, need_ref = false, desc = "assignment target");
            def e2 = TypeExpr (e2);
            TExpr.Assign (e1, AddCastTo (e2, e1.Type, "assigned value"))
          } else
            TExpr.Error ()
            

        | PT.PExpr.GenericSpecifier (fnc, gen_params) =>
          def fnc = TypeExpr (fnc, expected);
          ApplyGenericSpecifier (fnc, gen_params.Map (BindType));
          fnc


        | <[ _ is $pat ]> =>
          def sym = Macros.NewSymbol ();
          TypeExpr (<[ fun ($(sym : name)) { $(sym : name) is $pat } ]>,
                    expected)

        | PT.PExpr.Is (e1, e2) =>
          def e =
            if (CanBeTypeName (e2))
              <[ match ($e1) { 
                   | _ is $e2 => true 
                   | _ => false 
                 } ]>
            else
              <[ match ($e1) { 
                   | $e2 => true 
                   | _ => false 
                 } ]>;

          TypeExpr (e, expected)


        | PT.PExpr.ListLiteral (elems) =>
          TypeExpr (Macros.Lift (elems), expected)


        | PT.PExpr.Throw (null) => TExpr.Throw (null)
        | PT.PExpr.Throw (e) =>
          def e = TypeExpr (e);
          _ = ExpectSubtyping (InternalType.Exception, e.Type, "thrown value");
          TExpr.Throw (e)


        | PT.PExpr.Literal (l) =>
          _ = Expect (expected, TypeOfLiteral (l), "literal");
          TExpr.Literal (expected, l)


        | PT.PExpr.This =>
          if (is_method) {
            _ = Expect (expected, current_type.GetMemType (), "`this'");
            This ()
          } else
            ReportFatal (messenger, "`this' used outside method");


        | PT.PExpr.Base =>
          if (is_method) {
            def baseti = Option.UnSome (current_type.SuperClass ());
            def parms = Option.UnSome (current_type.SuperType (baseti));
            def parms = Solver.MonoTypes (parms);
            def ty = MType.Class (baseti, parms);
            _ = Expect (expected, ty, "`base'");
            TExpr.This (ty)
          } else
            ReportFatal (messenger, "`base' used outside method");


        | PT.PExpr.Sequence ([]) =>
          _ = Expect (expected, InternalType.Void, "empty sequence");
          VoidLiteral ()


        | PT.PExpr.Sequence (l) =>
          def loop (res, lst) {
            match (lst) {
              | [e] =>
                def e' = TypeExpr (e, expected);
                FixupSequence (e' :: res)
              | x :: xs =>
                def x' = TypeExpr (x, FreshTyVar (), is_toplevel_in_seq = true);
                messenger.CleanLocalError ();
                loop (x' :: res, xs)
              | [] => assert (false)
            }
          }

          //Message.Debug ("push seq");
          PushLocals ();
          def res = loop ([], l);
          //Message.Debug ("pop seq");
          PopLocals ();
          res


        | PT.PExpr.Tuple (l) =>
          mutable res_exprs = [];
          mutable res_types = [];
          foreach (e in l) {
            def tv1 = FreshTyVar ();
            def tv2 = FreshTyVar ();
            tv1.ForceRequire (tv2);
            def e = TypeExpr (e, tv1);
            res_exprs = ImplicitCast (e, tv2) :: res_exprs;
            res_types = tv2 :: res_types;
          }
          res_exprs = List.Rev (res_exprs);
          res_types = List.Rev (res_types);
          _ = Expect (expected, MType.Tuple (res_types), "tuple constructor");
          TExpr.Tuple (res_exprs)


        | (PT.PExpr.Match) as m =>
          TypeMatch (m, expected)


        | PT.PExpr.Typed (tytree) =>
          _ = Expect (expected, TypeOf (tytree), "typed expression");
          tytree


        | PT.PExpr.Quoted (parse_element) =>
          def lifted =
            match (parse_element) {
              | PT.SyntaxElement.Expression (e) => Macros.quoted_expr (e)
              | PT.SyntaxElement.MatchCase (e) => Macros.QuotedMatchCase (e)
              | PT.SyntaxElement.Function (e) => Macros.quoted_fundecl (e)
              | PT.SyntaxElement.Parameter (e) => Macros.quoted_fparam (e)
              | PT.SyntaxElement.ClassMember (e) => Macros.quoted_member (e)
              | PT.SyntaxElement.TType (e) => Macros.quoted_ttype (e)

              | PT.SyntaxElement.RawToken
              | PT.SyntaxElement.TypeBuilder
              | PT.SyntaxElement.MethodBuilder
              | PT.SyntaxElement.FieldBuilder
              | PT.SyntaxElement.EventBuilder
              | PT.SyntaxElement.PropertyBuilder
              | PT.SyntaxElement.ParameterBuilder =>
                Util.ice ("strange syntax element appeared in quotation")
            };
          TypeExpr (lifted, expected)


        | PT.PExpr.Spliced =>
          ReportFatal (messenger, "$ macro keyword used in regular expression")
          

        | PT.PExpr.TypedType
        | PT.PExpr.Wildcard
        | PT.PExpr.Where
        | PT.PExpr.As =>
          ReportFatal (messenger, $ "found pattern expression ($expression) inside a raw expression")

          
        | PT.PExpr.Void =>
          ReportFatal (messenger, "found type expression inside a raw expression")


        | PT.PExpr.Error =>
          ReportFatal (messenger, "parse error")
          

        | PT.PExpr.Ellipsis =>
          Util.ice ("List of expression parameters out of any construct")
          

        | PT.PExpr.MacroCall  =>
          Util.ice ("Macrocalls should have been expanded already")


        | PT.PExpr.TypedPattern  =>
          Util.ice ("typed pattern in raw expr")
      }
    }
    #endregion

       
    #region ,,def''
    TypeLocalDefinition (is_mutable : bool, name : PT.Name, val : PT.PExpr) : TExpr.DefValIn
    {
      // check for mutable symbol redefinitions
      match (local_context.FindLocal (name)) {
        | Some (l) =>
          def warn (was_of_kind, is_of_kind) {
              Message.Warning ($ "redefinition of a local $was_of_kind "
                                 "value `$(l.Name)' $is_of_kind");
              Message.Warning (l.loc, "  <-- previously seen here")
          }

          match ((is_mutable, l.IsMutable)) {
            | (true, true) =>
              warn ("mutable", "")
            | (false, true) =>
              warn ("mutable", "as non-mutable")
            | (true, false) =>
              warn ("non-mutable", "as mutable")
            | _ => {}
          }
        | _ => {}
      }
      
      def tv = FreshTyVar ();
      def val = TypeExpr (val, tv);
      def (val, decl_ty) =
        if (is_mutable) {
          def tv' = FreshTyVar ();
          _ = tv'.Provide (tv);
          (ImplicitCast (val, tv'), tv')
        } else (val, tv);
      def decl = DefineLocal (name, decl_ty, 
                              LocalValue.Kind.Plain (), 
                              is_mutable);
      AddLocal (name, decl);
      TExpr.DefValIn (decl, val, null)
    }

   
    TypeDefPattern (pattern : PT.PExpr, value : PT.PExpr) : TExpr
    {
      def res = FreshTyVar ();
      def matched_value = TypeExpr (value, res);
      def pats = TypePatterns (res, [pattern]);
      def case = Match_case (pats, null, true);

      when (messenger.SeenError) {
        when (messenger.NeedMessage)
          DecisionTreeBuilder.CheckMatching ([case]);
        FixupMatchCase (case)
      }
      TExpr.Match (matched_value, [case])
    }


    TypeLocalFunction (functions : list [PT.Function_decl]) : TExpr
    {
      mutable headers = [];

      foreach (fn in functions) {
        def (tenv, typarms) =
          current_type.BindTyparms (tenv, fn.header.typarms);
        def parms = List.Map (fn.header.parms,
          fun (p : PT.Fun_parm) {
            def name = p.name.GetName ();
            def fp =
              Fun_parm (loc = p.loc,
                        name = name.Id,
                        color = name.color,
                        ty = current_type.BindType (tenv, p.ty),
                        kind = ParmKind.Normal,
                        modifiers = p.modifiers);
            fp.GetLocalDefaultValueFromModifiers (this);
            fp
          });
        def name_obj = fn.header.name.GetName ();
        def header = Fun_header (
           ret_type = current_type.BindType (tenv, fn.header.ret_type),
           typarms = typarms,
           name = name_obj.Id,
           parms = parms,
           tenv = tenv,
           loc = fn.header.loc);

        def parm_types = List.Map (parms, fun (p : Fun_parm) { p.ty });
        def fun_type = ConstructFunctionType (parm_types, header.ret_type);
        def parents = current_fun :: current_fun.GetParents ();
        def local = 
          DefineLocal (name_obj, 
                       fun_type, 
                       LocalValue.Kind.Function (header, parents),
                       is_mutable = false);
        header.decl = local;
        header.body = FunBody.Parsed (fn.body);
        AddLocal (name_obj, local);
        headers = header :: headers;
      }

      headers = List.Rev (headers);

      foreach (hd in headers) {
        _ = Typer (this, hd);
      }

      TExpr.DefFunctionsIn (headers, null)
    }


    /** Given a sequence of expression in a sequence (in reverse order),
        reverse them and nest properly (def ... => let ... in ...).  */
    static FixupSequence (exprs : list [TExpr]) : TExpr
    {
      def loop (acc, lst) {
        match (lst) {
          | x :: xs =>
            def acc =
              match (x) {
                | TExpr.DefValIn (_, _, body) as dv when body == null =>
                  if (acc == null) dv.body = VoidLiteral ();
                  else dv.body = acc;
                  dv.ty = dv.body.Type;
                  x

                | TExpr.DefFunctionsIn (_, body) as dv when body == null =>
                  if (acc == null) dv.body = VoidLiteral ();
                  else dv.body = acc;
                  dv.ty = dv.body.Type;
                  x
                  
                | TExpr.Match (_, [case]) when case.body == null =>
                  if (acc == null) case.body = VoidLiteral ();
                  else case.body = acc;
                  x.ty = case.body.Type;
                  x

                | x =>
                  if (acc == null) x
                  else TExpr.Sequence (acc.Type, x, acc);
              }
            loop (acc, xs)

          | [] => acc
        }
      }

      assert (!exprs.IsEmpty);

      loop (null, exprs)
    }
    #endregion


    #region Arrays, literals
    // FIXME: make this tail rec
    TypeArray (rank : int, args : PT.PExpr, expected : TyVar) : TExpr
    {
      def expr_list_flatten_and_count (e, remaining) {
        match (e) {
          | PT.PExpr.ListLiteral (elems) =>
            when (List.Length (elems) != remaining)
              ReportError (messenger, 
                           "incorrectly structured array initializer");
            elems
          | _ =>
            ReportError (messenger,
                         "array initializer must be of form "
                         "`array .[rank] [ [..], .., [..] ]'");
            []
        }
      }

      def dimensions (args, remaining_rank) {
        if (remaining_rank == 0) []
        else
          match (args) {
            | PT.PExpr.ListLiteral ((head :: _) as elems) =>
              List.Length (elems) :: dimensions (head, remaining_rank - 1)
            | _ =>
              ReportError (messenger, 
                           "array initializer must be of form "
                           "`array [rank] [ [..], .., [..] ]'");
              []
          }
      }
      
      /* Flattens the initializers and checks that there are the correct
         number of them */
      def flatten (args, dims) {
        | (elem, []) => [elem]
        | (l, h::t) => 
          def flat = expr_list_flatten_and_count (l, h);
          List.FoldLeft (flat, [], 
                         fun (i, a) {
                           List.Concat ([a, flatten (i, t)])
                         })
      }

      def dimensions = dimensions (args, rank);
      def lst = flatten (args, dimensions);

      def dimensions =
        List.Map (dimensions,
                  fun (x : int) {
                    TExpr.Literal (InternalType.Int32, Literal.FromInt (x)) 
                  });
      def element_type = FreshTyVar ();

      _ = Expect (expected, MType.Array (element_type, rank), 
                  "array initializer");

      def initializers =
        List.Map (lst,
                  fun (expr) {
                    AddCastTo (TypeExpr (expr), element_type, 
                               "array initializer")
                  });

      TExpr.Array (initializers, dimensions)
    }


    static internal TypeOfLiteral (l : Literal) : TyVar
    {
      def mono_type =
        match (l) {
          | Literal.Null => null
          | Literal.Void => InternalType.Void
          | Literal.Integer (_, _, t) => t
          | Literal.Char => InternalType.Char
          | Literal.String => InternalType.String
          | Literal.Float => InternalType.Single
          | Literal.Double => InternalType.Double
          | Literal.Decimal => InternalType.Decimal
          | Literal.Bool => InternalType.Boolean
          | Literal.Enum (_, tc) => MType.Class (tc, [])
        }

      if (mono_type == null) {
        def tv = FreshTyVar ();
        tv.ForceRequire (InternalType.Object);
        tv.IsFromNull = true;
        tv
      } else {
        mono_type
      }
    }

    #endregion


    #region Indexers
    TypeIndexer (obj : PT.PExpr, args : list [PT.PExpr], expected : TyVar) : TExpr
    {
      def obj = TypeExpr (obj);

      DelayAction (expected, 
        DelayedLambdaAction (desc = "", fn = fun (fail_loudly) {
        match (obj) {
          | TExpr.PropertyMember (_, prop) when prop.IsIndexer
          | TExpr.StaticPropertyRef (_, prop) when prop.IsIndexer
          | TExpr.Delayed (DelayedTyping where (DtKind = Overloaded)) =>
            Some (TypeCall (<[ $(obj : typed) ]>, args, expected, is_property = true))
          
          | _ =>
            match (obj.Type.Hint) {
              | Some (MType.Class (ti, _))
                when Option.IsSome (ti.DefaultIndexerName) =>
                // Message.Debug ($ "using $(ti.DefaultIndexerName) for $ti");
                def idx = Option.UnSome (ti.DefaultIndexerName);
                Some (TypeIndexer (<[ $(obj : typed) . $(idx : dyn) ]>, args, expected))

              | Some (MType.Tuple (types)) =>
                match (args) {
                  | [expr] =>
                    match (TypeExpr (expr)) {
                      | TExpr.Literal (Literal where (AsInt = Some (k))) =>
                        if (k < 0 || k >= types.Length) {
                          ReportError (messenger,
                                       $ "the indexed tuple has only "
                                         "$(types.Length) members, cannot "
                                         "index at argument $k");
                          None ()
                        } else {
                          _ = Expect (expected, types.Nth (k), "tuple indexer");
                          Some (TExpr.TupleIndexer (expected, obj, k, types.Length))
                        }
                      | _ =>
                        ReportError (messenger,
                                     "argument to tuple indexer must be a "
                                     "constant integer");
                        None ()
                    }
                  | _ =>
                    ReportError (messenger, 
                                 "indexer over tuples expects a single "
                                 "integer argument");
                    None ()
                }
                
              | Some (MType.Array (indexer_over_type, rank) as arty) =>
                // indexers over arrays
                def typed_args = List.Map (args, TypeExpr);

                _ = TypeOf (obj).Unify (arty);
                
                mutable args_amount = rank;
                foreach (index in typed_args) {
                  _ = Expect (TypeOf (index), InternalType.Int32, "array index");
                  --args_amount;
                }

                _ = Expect (expected, indexer_over_type, "array indexer");

                if (args_amount != 0) {
                  ReportError (messenger, "wrong number of indexes");
                  None ()
                } else {
                  Some (TExpr.ArrayIndexer (indexer_over_type, obj, typed_args))
                }
                
              | hint =>
                when (hint.IsSome || fail_loudly)
                  ReportError (messenger, 
                               $ "cannot find any suitable indexer in "
                                 "$(TypeOf (obj))");
                None ()
            }
        }
      }))
    }
    #endregion


    #region Member access
    ConstrainLeadingTyparms (pt_from : PT.PExpr, from : MType.Class) : void
    {
      if (pt_from == null) {}
      else
        match (current_type.BindType (pt_from)) {
          | MType.Class (tc, args) =>
            if (tc.Equals (from.tycon.DeclaringType)) {
              _ = args.FoldLeft (from.args, fun (pa, fas) {
                    match (fas) {
                      | fa :: fas =>
                        fa.ForceUnify (pa);
                        fas
                      | [] => Util.ice ()
                    }
                  })
            } else
              ReportError (messenger, $ "$pt_from is not declaring type of $from");
          | _ =>
            ReportError (messenger, $ "$pt_from is not a class type");
        }
    }

    
    TypeToConstructor (pt_from : PT.PExpr, t : TypeInfo) : list [OverloadPossibility]
    {
      match (t.GetTydecl ()) {
        | TypeDeclaration.Alias (MType.Class (t, _)) => 
          TypeToConstructor (pt_from, t)
        | _ =>
          if (t.Attributes %&& NemerleAttributes.Abstract) {
            ReportError (messenger, 
                         $ "  the type `$(t)' is abstract and cannot "
                           "be constructed");
            []
          } else
            match (t.GetConstantObject ()) {
              | null =>
                mutable seen_empty_ctor = false;
                mutable res = [];

                foreach (meth is IMethod in t.LookupMember (".ctor")) {
                  match (meth.GetFunKind ()) {
                    | FunKind.Constructor 
                      when meth.CanAccess (current_type) =>
                      when (meth.GetParameters ().IsEmpty)
                        seen_empty_ctor = true;
                      def from =
                        match (meth.GetMemType ()) {
                          | MType.Fun (from, _) => from
                          | _ => assert (false)
                        }
                      def mt = MType.Fun (from, t.GetMemType ());
                      def ty = t.FreshSubst ().Apply (mt);
                      def ret_type = (ty.FixedValue :> MType.Fun).to :> MType.Class;
                      ConstrainLeadingTyparms (pt_from, ret_type);
                      res = OverloadPossibility (this, ty, null, ret_type, meth) :: res;
                      when (meth.IsVarArgs) {
                        def op = OverloadPossibility (this, ty, null, ret_type, meth);
                        op.VarArgs = true;
                        res = op :: res;
                      }
                    | _ => {}
                  }
                }

                when (t.IsValueType && !seen_empty_ctor) {
                  def res_type = t.GetFreshType ();
                  def ty = ConstructFunctionType ([], res_type);
                  ConstrainLeadingTyparms (pt_from, res_type);
                  def op = OverloadPossibility (this, ty, null, res_type, t);
                  res = op :: res;
                }

                when (res.IsEmpty)
                  ReportError (messenger, 
                               $ "  the type `$(t)' has no accessible constructors");

                res

              | f =>
                def res = t.FreshSubst ().Apply (f.GetMemType ()) :> MType.Class;
                def ty = ConstructFunctionType ([], res);
                ConstrainLeadingTyparms (pt_from, res);
                def op = OverloadPossibility (this, ty, null, res, f);
                op.IsConstantObject = true;
                [op]
            }
       }
    }


    InCurrentType (mem : IMember) : bool
    {
      def loop (mem, t) {
        if (t == null) false
        else if (mem.Equals (t)) true
        else loop (mem, t.DeclaringType)
      }
      loop (current_type, mem.DeclaringType) ||
      loop (mem.DeclaringType, current_type)
    }


    ConstructEnclosingType (pt_from : PT.PExpr, symbol : IMember, 
                            additional_sig_types : list [MType] = null) : MType.Class
    {
      if (pt_from != null) {
        match (current_type.BindType (pt_from)) {
          | MType.Class as c => c
          | _ =>
            ReportError (messenger, $ "$pt_from is not a class type");
            InternalType.Object
        }
      } else if (InCurrentType (symbol) &&
                 symbol.DeclaringType.TyparmsCount <= current_type.TyparmsCount) {
        def typarms_count = symbol.DeclaringType.TyparmsCount;
        def typarms = current_type.GetMemType ().args.FirstN (typarms_count);
        MType.Class (symbol.DeclaringType, typarms)        
      } else {
        def declaring = symbol.DeclaringType;
        def from = declaring.GetFreshType ();

        unless (declaring.Typarms.IsEmpty) {
          def type_tyvars = Hashtable ();
          foreach (tv in declaring.Typarms)
            type_tyvars [tv.Id] = tv;

          def clear_ht (_) {
            | MType.TyVarRef (tv) when type_tyvars.ContainsKey (tv.Id) =>
              type_tyvars [tv.Id] = null;
            | _ => {}
          }
          
          symbol.GetMemType ().Iter (clear_ht);
          
          when (additional_sig_types != null)
            foreach (t in additional_sig_types)
              t.Iter (clear_ht);

          if (symbol.DeclaringType.Equals (current_type))
            List.Iter2 (from.args, declaring.Typarms, fun (d, s) {
              when (type_tyvars [s.Id] != null) {
                def res = d.Unify (MType.TyVarRef (s));
                Util.cassert (res);
              }
            })

          else
            List.Iter2 (from.args, declaring.Typarms, fun (d, s) {
              when (type_tyvars [s.Id] != null)
                match (s.Constraints) {
                  | [MType.Class (tc, [_])] when tc.Name == "IComparable" =>
                    d.ForceUnify (NamespaceTree.LookupInternalType (
                        ["Nemerle", "Hacks", "IComparableClass"]).GetMemType ());
                  | _ => {}
                }
            })
        }

        from
      }
    }

    
    ConstructMethodOverload (m : IMethod, pt_from : PT.PExpr = null, 
                             maybe_varargs = false) : list [OverloadPossibility]
    {
      def bounds = m.GetHeader ().typarms.Map (fun (tv) { tv.LowerBound });

      def make_overload () {
        def from = ConstructEnclosingType (pt_from, m, bounds);

        def (ty, vars) = from.TypeOfMethodWithTyparms (m);
        OverloadPossibility (this, ty, null, from, m,
                             method_typarms = vars)
      }

      if (maybe_varargs && m.IsVarArgs) {
        def o1 = make_overload ();
        def o2 = make_overload ();
        o2.VarArgs = true;
        [o1, o2]
      } else {
        [make_overload ()]
      }
    }


    InterpretGlobal (pt_from : PT.PExpr, symbol : IMember) : list [OverloadPossibility]
    {
      def single (mem) {
        //Message.Debug ($"access of $mem");
        def from = ConstructEnclosingType (pt_from, mem);
        [OverloadPossibility (this, from.TypeOfMember (mem), null, from, mem)]
      }
      
      match (symbol) {
        | t is TypeInfo =>
          if (!t.CanAccess (current_type)) {
            ReportError (messenger, $ "  cannot access type `$(t)'");
            []
          } else {
            TypeToConstructor (pt_from, t)
          }
        
        | mem when !mem.IsStatic =>
          if (!is_method && mem.DeclaringType.Equals (current_type))
            ReportError (messenger,
                         $ "  trying to access an instance member "
                         "$(mem.DeclaringType).$(mem.Name) from a static "
                         "method");
          else
            ReportError (messenger,
                         $ "  trying to access an instance member "
                         "$(mem.DeclaringType).$(mem.Name) without an object");
          []

        | mem when !mem.CanAccess (current_type) =>
          ReportError (messenger,
                       $ "  cannot access `$(mem.DeclaringType).$(mem.Name)'");
          []

        | mem =>
          match (mem.GetKind ()) {
            | MemberKind.Type (t) =>
              TypeToConstructor (pt_from, t)

            | MemberKind.Method (m) =>
              ConstructMethodOverload (m, pt_from, maybe_varargs = true)

            | MemberKind.Property
            | MemberKind.Field =>
              single (mem)

            | MemberKind.Event (e) =>
              if (e.DeclaringType.Equals (current_type)) {
                def field = (e :> EventBuilder).storage_field;
                if (field == null)
                  [] // XXX how?
                else
                  single (field)
              } else
                single (e)
          }
      }
    }


    InterpretGlobals (pt_from : PT.PExpr,
                      symbols : list [IMember], 
                      expected : TyVar, 
                      dump_errors : bool) 
                              : list [OverloadPossibility]
    {
      mutable res = [];

      foreach (symbol in symbols) {
        if (dump_errors) {
          foreach (op in InterpretGlobal (pt_from, symbol)) {
            try {
              solver.PushState ();
              messenger.NeedMessage = true;
              _ = Expect (op.Type, expected, op.ToString ());
            } finally {
              solver.PopState ();
            }
          }
        } else {
          def not_ok =
            try {
              solver.PushState ();
              InterpretGlobal (pt_from, symbol).IsEmpty;
            } finally {
              solver.PopState ();
            }
          unless (not_ok)
            foreach (op in InterpretGlobal (pt_from, symbol)) {
              def ok =
                try {
                  solver.PushState ();
                  op.Type.Unify (expected);
                } finally {
                  solver.PopState ();
                }
              when (ok)
                res = op :: res;
            }
        }
      }

      res
    }


    TypeMemberExpr (obj : TExpr, name : PT.Name, expected : TyVar) : TExpr
    {
      match (TypeMember (obj, name, expected)) {
        | Some ([]) =>
            if (Completion.Engine.IsInCompletionMode) {
              TExpr.Error()
            }
            else
            {
              ReportError (messenger, $ "there is no member named `$(name.Id)' "
                                    "in $(TypeOf (obj)) with type $expected");
              TExpr.Error ()
            }

        | Some (lst) =>
          MakeOverloadedNode (lst, expected)
        | None =>
          Delay (DelayedTyping.Kind.MemberAccess (obj, name), expected)
      }
    }

                    
    TypeMember (obj : TExpr, name : PT.Name, expected : TyVar, for_completion = false) 
                    : option [list [OverloadPossibility]]
    {
      def mem_name = name.Id;
      // XXX we should add a bound here, that the type should have the
      // field mem_name
      def possible_type =
        match (obj.Type.Hint) {
          | Some (t) => t
          | None => null
        }
      if (possible_type != null) {
        def mems = Hashtable ();
        mutable visited = []; // this set is in practice very small
        mutable res = [];
        mutable print_errors = false;

        def lookup (t) {
          | MType.Class (tc, _) when !List.ContainsRef (visited, tc) =>
            visited = tc :: visited;
            foreach (mem in tc.LookupMember (mem_name, for_completion))
              when (!mems.Contains (mem)) {
                mems [mem] = null;
                if (mem.IsStatic)
                  when (print_errors)
                    ReportError (messenger, 
                                 $ "trying to access static member ($mem) "
                                   "through an instance")
                else if (!mem.CanAccess (current_type))
                  when (print_errors)
                    ReportError (messenger,
                                 $ "$mem is not accessible")
                else {
                  def (ty, vars) =
                    match (mem) {
                      | meth is IMethod => t.TypeOfMethodWithTyparms (meth)
                      | _ => (t.TypeOfMember (mem), [])
                    }
                  def allowed = expected.TryUnify (ty);
                  if (!allowed)
                    when (print_errors) {
                      solver.PushState ();
                      messenger.NeedMessage = true;
                      def before = expected.ToString ();
                      def allowed = expected.Unify (ty);
                      assert (!allowed);
                      ReportError (messenger, 
                                   $ "$mem has invalid type $ty (expected $before)")
                    }
                  else {
                    res = OverloadPossibility (this, ty, obj, null, mem,
                                               method_typarms = vars) :: res;
                    match (mem) {
                      | meth is IMethod when meth.IsVarArgs =>
                        def op = OverloadPossibility (this, ty, obj, null, mem,
                                                      method_typarms = vars);
                        op.VarArgs = true;
                        res = op :: res;
                      | _ => {}
                    }
                  }
                }
              }
              
          | MType.TyVarRef (tv) =>
            lookup (InternalType.Object);
            foreach (t in tv.Constraints) lookup (t)
            
          | MType.Array (t, n) => 
            lookup (MType.Class (InternalType.GetArrayType (n), [t]))

          | MType.Intersection (lst) => foreach (t in lst) lookup (t)
          | MType.Tuple
          | MType.Fun => lookup (InternalType.Object)
          | _ => ()
        }

        lookup (possible_type);
        
        when (res.IsEmpty) {
          print_errors = true;
          visited = [];
          mems.Clear ();
          lookup (possible_type);
        }

        foreach (meth in LibraryReferenceManager.GetExtensionMethods (name.Id, name.GetEnv (env)))
          foreach (over in ConstructMethodOverload (meth, maybe_varargs = true)) {
            over.ExtensionMethodObject = obj;
            res ::= over;
          }

        Some (res)
      }
      else if (IsError (obj))
        Some ([])
      else {
        // FIXME: here we should traverse all type paths from lower to
        // upper bound and look how this field is defined there, and finally
        // add maybe-more-specific lower bound
        None ()
      }
    }


    IsBlockReturn (name : PT.Name) : bool
    {
      local_context.FindLocal (name)
        is Some (LocalValue where (ValKind = BlockReturn))
    }


    TypeLocal (name : PT.Name, expected : TyVar,
               allow_block_return : bool = false) : TExpr
    {
      match (local_context.FindLocal (name)) {
        | Some (local) =>
          mutable result = null;
          mutable ty = null;
          
          match (local.ValKind) {
            | LocalValue.Kind.MacroRedirection (expr) =>
              result = TypeExpr (expr, expected);
                
            | LocalValue.Kind.Function (fun_header, _) =>
              mutable sub = null;
              mutable typarms = [];

              def fnty =
                if (fun_header.typarms.IsEmpty)
                  local.Type
                else {
                  // XXX this fixes entire type, but we cannot do any better
                  // I guess
                  (sub, typarms) = Subst.Fresh (fun_header.typarms);
                  sub.Apply (local.Type.DeepFix ())
                }
                
              _ = Expect (expected, fnty, "local function reference");
              //Message.Debug ($"local fun ref: $typarms $fnty");
              result = TExpr.LocalFunRef (expected, local, typarms);
            
            // for block of type 'a, block exit is forall 'b. 'a -> 'b
            | LocalValue.Kind.BlockReturn =>
              when (!allow_block_return)
                ReportError (messenger,
                             $ "block-return expression `$(local.Name)' "
                               "needs to be followed by (...)");
              local.EverUsed = true;
              ty = MType.Fun (local.Type, FreshTyVar ());
              
            | _ => ty = local.Type
          }
          if (ty != null) {
            // Message.Debug ($ "type local: $name : $ty");
            _ = Expect (expected, ty, "value reference");
            TExpr.LocalRef (expected, local)
          }
          else
            result
          
        | _ => null
      }
    }


    TypeNameFull (expr : PT.PExpr, expected : TyVar, for_completion = false) : list [OverloadPossibility]
    {
      mutable mem_name = null;
      
      def pt_from = 
        match (expr) {
          | <[ $expr . $(mem : dyn) ]> when CanBeTypeName (expr) =>
            mem_name = mem;
            def was_error =
              try {
                solver.PushState ();
                ! (current_type.BindType (expr) is MType.Class (_, _ :: _)) ||
                messenger.LocalError
              } finally {
                solver.PopState ()
              }
            if (was_error) null
            else expr
          | _ => null
        }
      
      // Message.Debug ($ "type name: $idl");
      def from_this =
        match (expr) {
          // FIXME | PT.PExpr.ToComplete 
          | PT.PExpr.Ref (name) when is_method =>
            def res =
              try {
                solver.PushState ();
                Option.UnSome (TypeMember (This (), name, expected));
              } finally {
                solver.PopState ();
              }
            if (res.IsEmpty) []
            else Option.UnSome (TypeMember (This (), name, expected))
          | _ => []
        };

      def globals =
        if (pt_from == null)
          match (Util.QidOfExpr (expr)) {
            | Some ((idl, name)) =>
              name.GetEnv (env).LookupSymbol (idl, current_type,
                                              for_completion)
            | None =>
              // oops
              match (expr) {
                | <[ $broken_type . $_ ]>
                | broken_type // XXX this shouldn't happen
                  =>
                  ReportError (messenger, $ "unbound type `$broken_type'");
                  []
              }
          }
        else
          match (current_type.BindType (pt_from)) {
            | MType.Class (tc, _) =>
              tc.LookupMember (mem_name, for_completion)
            | _ =>
              ReportError (messenger, $ "$pt_from is not a class type");
              []
          }
        
      def from_global = InterpretGlobals (pt_from, globals, expected, dump_errors = false);
      //Message.Debug ($ "type name g: $globals $from_global");

      match (from_this + from_global) {
        | [] =>
          if (globals.IsEmpty) {
            ReportError (messenger,
                         $ "unbound name `$expr'");
          } else {
            ReportError (messenger, 
                         $ "none of the meanings of "
                           "`$expr' meets the type "
                           "$expected:");
            _ = InterpretGlobals (pt_from, globals, expected, dump_errors = true);
          }
          []
        | overloads => overloads
      }
    }

    
    TypeName (expr : PT.PExpr, expected : TyVar) : TExpr
    {
      match (TypeNameFull (expr, expected)) {
        | [] => TExpr.Error ()
        | overloads => MakeOverloadedNode (overloads, expected)
      }
    }
    #endregion

    
    #region Generic specifier
    ApplyGenericSpecifier (meth : IMember, 
                           inferred : list [TyVar],
                           supplied : list [TyVar]) : void
    {
      def constrain (inferred, supplied) {
        _ = Expect (inferred, supplied, "generic specifier")
      }
      
      if (inferred.Length != supplied.Length)
        match (meth) {
          | meth is IMethod when 
            meth.DeclaringType.DeclaringType != null &&
            meth.GetFunKind () is FunKind.Constructor && 
            supplied.Length == 
              meth.DeclaringType.TyparmsCount - 
              meth.DeclaringType.DeclaringType.TyparmsCount =>
            List.Iter2 (inferred.LastN (supplied.Length), supplied, constrain)
          | _ =>
            ReportError (Passes.Solver.CurrentMessenger,
                         $ "$meth takes $(inferred.Length) generic parameters, "
                           "while $(supplied.Length) was supplied")
        }
      else 
        List.Iter2 (inferred, supplied, constrain)
    }
    
    ApplyGenericSpecifier (expr : TExpr, gen_parms : list [TyVar]) : void
    {
      match (expr) {
        | TExpr.ImplicitValueTypeCtor =>
          def c = (expr.Type.Fix () :> MType.Fun).to :> MType.Class;
          ApplyGenericSpecifier (c.tycon, c.args, gen_parms);
          
        | TExpr.StaticRef (MType.Class (_, tp), meth is IMethod, [])
          when meth.GetFunKind () is FunKind.Constructor =>
          ApplyGenericSpecifier (meth, tp, gen_parms);
        
        | TExpr.MethodRef (_, meth, tp, _)
        | TExpr.StaticRef (_, meth is IMethod, tp) =>
          ApplyGenericSpecifier (meth, tp, gen_parms);

        | TExpr.Delayed (susp) =>
          susp.ApplyGenericSpecifier (gen_parms);

        | _ =>
          ReportError (messenger,
                       $ "don't know how to apply generic specifier to $expr");
      }
    }
    #endregion
    

    #region Static caches -- operators and conversions
    static binary_operators : Hashtable [string, string] = Hashtable ();
    static unary_operators : Hashtable [string, string] = Hashtable ();
    // a cache
    
    static this ()
    {
      def binary = [
        ("+", "op_Addition"),
        ("&", "op_BitwiseAnd"),
        ("%&", "op_BitwiseAnd"),
        ("|", "op_BitwiseOr"),
        ("%|", "op_BitwiseOr"),
        ("/", "op_Division"),
        ("==", "op_Equality"),
        ("^", "op_ExclusiveOr"),
        ("%^", "op_ExclusiveOr"),
        (">", "op_GreaterThan"),
        (">=", "op_GreaterThanOrEqual"),
        ("!=", "op_Inequality"),
        ("<<", "op_LeftShift"),
        ("<", "op_LessThan"),
        ("<=", "op_LessThanOrEqual"),
        ("%", "op_Modulus"),
        ("*", "op_Multiply"),
        (">>", "op_RightShift"),
        ("-", "op_Subtraction")
      ];

      def unary = [
      /*
        ("--", "op_Decrement"),
        ("++", "op_Increment"),
       */
        ("!", "op_LogicalNot"),
        ("~", "op_OnesComplement"),
        ("-", "op_UnaryNegation"),
        ("+", "op_UnaryPlus")
      ];

      List.Iter (binary, binary_operators.Add);
      List.Iter (unary, unary_operators.Add);

      Passes.OnInit += Clear;
    }

    static Clear () : void
    {
      better_type_cache.Clear ();
      checked_macro = null;
      unchecked_macro = null;
    }

    static IsOperator (name : string) : bool
    {
      name != "" && LexerBase.IsOperatorChar (name [0])
    }
    #endregion
    

    #region Function calls
    TryTypeCall (fnc : TExpr, parms : list [Parm], expected : TyVar,
                 var_args : bool, final : bool) : TExpr
    {
      log (TYPING, $ "try type call to $fnc");
      def ct = CallTyper (this, fnc, parms, expected);
      ct.is_final = final;
      ct.is_var_args = var_args;
      ct.Run ();
      if (ct.result == null || (var_args && !ct.used_var_args))
        TExpr.Error ()
      else
        ct.result
    }


    GetBaseOverloads (is_base : bool) : TExpr
    {
      def name = if (is_base) "base" else "this";
      if (!is_instance_ctor)
        ReportFatal (messenger, $ "$name (...) used outside constructor")
      else {
        def lookup_in =
          if (is_base)
            Option.UnSome (current_type.SuperClass ())
          else current_type;
        
        def sub = current_type.SubtypingSubst (lookup_in);
        
        mutable res = [];
        foreach (meth is IMethod in lookup_in.LookupMember (".ctor"))
          when (meth.CanAccess (current_type) && 
                meth.GetFunKind () is FunKind.Constructor) {
            def from = sub.Apply (lookup_in.GetMemType ()) :> MType.Class;
            def ty = sub.Apply (meth.GetMemType ());
            def o = OverloadPossibility (this, ty, null, from, meth);
            o.IsBaseCall = true;
            res = o :: res;
            when (meth.IsVarArgs) {
              def op = OverloadPossibility (this, ty, null, from, meth);
              op.IsBaseCall = true;
              op.VarArgs = true;
              res = op :: res;
            }
          }

        match (res) {
          | [] =>
            ReportFatal (messenger, 
                         $ "the type `$(lookup_in)' provides no accessible "
                           "constructor for $name (...)")
          | lst =>
            MakeOverloadedNode (lst, FreshTyVar ())
        }
      }
    }


    TypeCall (fnc : PT.PExpr,
              parms : list [PT.PExpr], 
              expected : TyVar, 
              is_property : bool) : TExpr
    {
      log (TYPING, $ "type call to $fnc");
      def refout (name, expr) {
        def tv = FreshTyVar ();
        match (expr) {
          | PT.PExpr.ParmByRef (ex)
          | PT.PExpr.ParmOut (ex) =>
            def ex = TypeExpr (ex, tv);
            CheckLValue (ex, need_ref = true, desc = "ref/out parameter");
            def kind =
              match (expr) {
                | PT.PExpr.ParmByRef => ParmKind.Ref
                | _ => ParmKind.Out
              };
            Parm (kind, ex, name, null)
          | ex =>
            def ex = TypeExpr (ex, tv);
            Parm (ParmKind.Normal, ex, name, null)
        }
      }

      def compile_parm (p : PT.PExpr) {
        | <[ $(name : name) = $e ]> => refout (name.Id, e)
        | e => refout ("", e)
      };

      mutable parameters = List.Map (parms, compile_parm);

      def fnc =
        match ((fnc, parameters)) {
          | (PT.PExpr.Ref (name), _) when IsBlockReturn (name) =>
            TypeLocal (name, FreshTyVar (), allow_block_return = true)

          | (PT.PExpr.Ref (name), [parm1, parm2]) 
            when IsOperator (name.Id) =>

            def operator_name =
              if (binary_operators.Contains (name.Id))
                binary_operators [name.Id]
              else name.Id;
                
            def kind =
              DelayedTyping.Kind.Operator (TypeOf (parm1.expr), 
                                           TypeOf (parm2.expr), 
                                           operator_name, 
                                           name.GetEnv (env));
            when (parm1.kind != ParmKind.Normal ||
                  parm2.kind != ParmKind.Normal)
              ReportError (messenger, 
                           $ "ref/out parameters not supported in "
                             "`$(name)' operands");

            def folded = ConstantFolder.FoldTyped (name.Id, parm1.expr,
                                                   parm2.expr, local_context.IsChecked);
            if (folded == null)
              Delay (kind, FreshTyVar ())
            else {
              parameters = null;
              folded
            }
          
          | (PT.PExpr.Ref (name), [parm1]) when IsOperator (name.Id) =>
            def operator_name =
              if (unary_operators.Contains (name.Id))
                unary_operators [name.Id]
              else name.Id;
                
            def kind =
              DelayedTyping.Kind.Operator (TypeOf (parm1.expr), null,
                                           operator_name,
                                           name.GetEnv (env));
            when (parm1.kind != ParmKind.Normal)
              ReportError (messenger, 
                           $ "ref/out parameters not supported in "
                             "`$(name)' operand");
            
            def folded = ConstantFolder.FoldTyped (name.Id, parm1.expr, local_context.IsChecked);
            if (folded == null)
              Delay (kind, FreshTyVar ())
            else {
              parameters = null;
              _ = Expect (expected, folded.Type, "literal");
              folded
            }

          | (Ref (name), []) when tenv.Find (name) is Some (_) =>
            def var = Option.UnSome (tenv.Find (name));
            parameters = null;
            def defaultv = TExpr.DefaultValue (MType.TyVarRef (var));
            
            if (var.IsValueType)
              defaultv
            else if (var.HasDefaultConstructor) {
              TypeExpr (<[
                {
                  def _N_def = $(defaultv : typed);
                  if (_N_def : object == null)
                    System.Activator.CreateInstance ()
                  else _N_def
                }
              ]>, MType.TyVarRef (var))
            }
            else {
              Message.FatalError ($"generic parameter $name cannot be used to create new instance, "
                                  "because it doesn't have `struct' or `new ()' constraint");
            }
          
          | (PT.PExpr.Ref (PT.Name where (Id = "_N_op_Increment") as name), [parm1])
          | (PT.PExpr.Ref (PT.Name where (Id = "_N_op_Decrement") as name), [parm1]) =>
            def operator_name = name.Id.Substring (3);
            def kind =
              DelayedTyping.Kind.Operator (TypeOf (parm1.expr), null,
                                           operator_name,
                                           name.GetEnv (env));
            when (parm1.kind != ParmKind.Normal)
              ReportError (messenger, 
                           $ "ref/out parameters not supported with ++/--");
            
            Delay (kind, FreshTyVar ())
            
              
          | (<[ this ]>, _) => GetBaseOverloads (is_base = false)

          | (<[ base ]>, _) => GetBaseOverloads (is_base = true)

          | _ =>
            def ex = TypeExpr (fnc, FreshTyVar ());
            match (ex.Type.Hint) {
              | Some (MType.Class (tc, _)) when tc.IsDelegate =>
                TypeExpr (<[ $(ex : typed).Invoke ]>)
              | _ => ex
            }
        }
       
      def res =
        match (fnc) {
          | TExpr.Delayed ((CanSetCallExpr = true) as dt) as expr =>
            log (TYPING, $"result is delayed: $dt");
            def expr = TExpr.Call (expected, expr, parameters, false);
            dt.SetCallExpr (expr);
            dt.Resolve (); // just try
            expr

          // special cases when call has been folded to something else
          // (literal, 'a(), etc.)
          | _ when parameters == null =>
            fnc
            
          | fnc =>
            def res = TryTypeCall (fnc, parameters, expected,
                                   var_args = false, 
                                   final = true);
            res
        }
      
      match (res) {
        | TExpr.Call (TExpr.PropertyMember (_, prop), _, _) when prop.IsIndexer =>
          if (is_property) {}
          else
            ReportError (messenger, 
                         $ "attempt to call an indexer property `$(prop.Name)'");
        // it fails for overloaded indexers
        // | expr when is_property =>
        //  ReportError (messenger, 
        //               $ "the thing indexed isn't property, it is $(expr)");
        | _ => ()
      }

      res
    }
    #endregion


    #region Type conversions
    TypeConversion (expr : TExpr, to : MType) : TExpr
    {
      def from = 
        match (expr.Type.Hint) {
          | Some (t) => t
          | None => expr.Type
        }

      if (to.TryRequire (from)) {
        to.ForceRequire (from);
        TExpr.TypeConversion (expr, to, ConversionKind.DownCast ());
      } else if (from.TryRequire (to)) {
        from.ForceRequire (to);
        Message.Warning (10001, $"there is no check needed to cast $from to $to");
        Message.HintOnce (10001, "consider using : instead of :>");

        TExpr.TypeConversion (expr, to, ConversionKind.UpCast ())
      } else {
        def operators =  
          LookupStaticMethod (to, "op_Explicit") +
          LookupStaticMethod (to, "op_Implicit");
        def operators =
          match (from.Hint) {
            | Some (t) =>
              LookupStaticMethod (t, "op_Explicit") +
              LookupStaticMethod (t, "op_Implicit") +
              operators
            | None => operators
          }
        def operators = SquashDuplicates (operators);

        def needed = ConstructFunctionType ([from], to);
        def operators = List.RevFilter (operators, fun (op : IMethod) {
          op.GetFreshType () [0].TryRequire (needed);
        });

        def operators = GetBestOverloads1 (operators);

        def is_strange_type (t) {
          match (t.Hint) {
            | Some (MType.TyVarRef)
            | Some (t) when t.IsInterface => true
            | _ => false
          }
        }

        def is_enum (t) {
          match (t.Hint) {
            | Some (t : MType) => t.IsEnum
            | _ => false
          }
        }

        match (operators) {
          | [single] =>
            if (single.BuiltinKind is BuiltinMethodKind.ValueTypeConversion)
              TExpr.TypeConversion (expr, to, ConversionKind.IL (local_context.IsChecked))
            else {
              def (ty, vars) = single.GetFreshType ();
              ty.ForceRequire (needed);
              def from = single.DeclaringType.GetMemType ();
              def sr = TExpr.StaticRef (ty, from, single, vars);
              TExpr.TypeConversion (expr, to, 
                                    ConversionKind.MethodCall (sr))
            }
          | [] =>
            if (is_strange_type (from) || is_strange_type (to)) {
              TExpr.TypeConversion (expr, to, ConversionKind.DownCast ())
            } else if (is_enum (from) && is_enum (to)) {
              TExpr.TypeConversion (expr, to, ConversionKind.IL (local_context.IsChecked))
            } else {
              ReportError (messenger, $ "cannot convert $from to $to");
              TExpr.TypeConversion (expr, to, ConversionKind.DownCast ())
            }
          | lst =>
            ReportError (messenger, $"overload ambiguity in type conversion $lst");
            TExpr.Error ()
        }
      }
    }
    #endregion


    #region Matching typing
    TypePatterns (matched_value_type : TyVar, 
                  patterns : list [PT.PExpr]) 
                  : list [Pattern * TExpr * list [LocalValue * TExpr]]
    {
      def decode_when (e, a) {
        match (e) {
          | <[ @when ($pat, $expr) ]> => (pat, Some (expr), a)
          | <[ $pat ]> => (pat, None (), a)
        }
      }

      def decode_assigns (pat, assigns) {
        decode_when (pat, assigns.Map (fun (_) {
          | <[ $(name : name) = $expr ]> =>
            (name, expr)
          | e =>
            ReportError (messenger, $ "invalid with-assignment: $e");
            (PT.Name ("dummy"), e)
        }))
      }
        
      def patterns = List.Map (patterns, fun (e) {
        match (e) {
          | <[ @with ($pat, ( .. $assigns )) ]> =>
            decode_assigns (pat, assigns)
          | <[ @with ($pat, $expr) ]> =>
            decode_assigns (pat, [expr])
          | pat => decode_when (pat, [])
        }
      });

      def typer = PatternTyper (this, matched_value_type, patterns);
      typer.Run ()
    }


    static internal FixupMatchCase (case : Match_case) : void
    {
      case.patterns = 
        case.patterns.Map (fun (p, e, a) { (p.StripEnums (), e, a) });
    }

    
    TypeMatch (mtch : PT.PExpr.Match, expected : TyVar) : TExpr
    {
      def matched_value_type = FreshTyVar ();

      // check if there is some expression to match,
      // it could be null for foo () { | _ => () }
      when (mtch.expr == null)
        Message.Error (mtch.Location, "there is nothing to match on, forgot function parameter?");
      
      def matched_value = TypeExpr (mtch.expr, matched_value_type);
      mutable disable_warnings = false;
      mutable res = [];

      foreach (case in mtch.cases) {
        when (case.disable_warnings)
          disable_warnings = true;

        PushLocals ();
        log (TYPING, $ "{  typing patterns: $(case.patterns), t=$matched_value_type");
        def pats = TypePatterns (matched_value_type, case.patterns);
        log (TYPING, $ "done typing patterns: $(case.patterns), t=$matched_value_type  }");
        messenger.CleanLocalError ();
        def body = TypeExpr (case.body);
        
        // conversions are added in Typer2 (for some reason?)
        unless (IsError (body))
          Util.locate (body.loc,
            unless (ExpectSubtyping (expected, body.Type, "computation branch"))
              Message.HintOnce ("this means two branches of ``if'' or "
                                "``match'' have different types"));

        messenger.CleanLocalError ();
        PopLocals ();

        res = Match_case (pats, body, disable_warnings) :: res;
      }

      if (res.IsEmpty) {
        ReportError (messenger, "cannot have an empty match construct");
        TExpr.Error ()
      } else {
        res = List.Rev (res);

        when (messenger.SeenError) {
          when (! disable_warnings && messenger.NeedMessage)
            DecisionTreeBuilder.CheckMatching (res);
          List.Iter (res, FixupMatchCase)
        }

        TExpr.Match (matched_value, res)
      }
    }
    #endregion


    #region Special macros
    static mutable checked_macro : NamespaceTree.Node;
    static mutable unchecked_macro : NamespaceTree.Node;
    static mutable yield_macro : NamespaceTree.Node;
    

    TypeBlock (name : PT.Name, body : PT.PExpr, expected : TyVar) : TExpr
    {
      PushLocals ();
      try {
        // the type is bogus here, it is handled specially in TypeLocal
        def jumpout = 
          DefineLocal (name, expected, 
                       LocalValue.Kind.BlockReturn (null, -1), 
                       is_mutable = false);
        when (name.Equals (PT.Name.Global ("_N_break")))
          skip_n_return = true;
        AddLocal (name, jumpout);
        def body = TypeExpr (body, expected);
        TExpr.Block (expected, jumpout, body)
      } finally {
        skip_n_return = false;
        PopLocals ();
      }
    }


    MakeImplicitBlockJumpOut (name : string, expected : TyVar) : LocalValue
    {
      def name = PT.Name.Global (name);
      def local =
        DefineLocal (name, expected, 
                     LocalValue.Kind.BlockReturn (null, -1), 
                     is_mutable = false);
      AddLocal (name, local);
      local
    }


    InterceptSpecialMacros (expr : PT.PExpr, expected : TyVar) : TExpr
    {
      when (checked_macro == null) {
        checked_macro = GlobalEnv.ExactPath (["Nemerle", "Core", "checked"]);
        unchecked_macro = GlobalEnv.ExactPath (["Nemerle", "Core", "unchecked"]);
        yield_macro = GlobalEnv.ExactPath (["Nemerle", "Core", "yield"]);
      }
      
      match (expr) {
        | PT.PExpr.MacroCall (_, ns, [PT.SyntaxElement.Expression (expr)]) =>
          if (ns.Equals (checked_macro) ||
              ns.Equals (unchecked_macro)) {
            PushLocals ();
            try {
              local_context = local_context.WithChecked (ns.Equals (checked_macro));
              TypeExpr (expr, expected);
            } finally {
              PopLocals ();
            }
          } else if (ns.Equals (yield_macro)) {
            when (! inside_yielding_function) {
              _ = GetYieldType ();
              log (TYPING, "throwing STY");
              throw SwitchToYielding ();
            }
            _ = Expect (expected, InternalType.Void, "yield ``result''");
            HandleYield (expr)
          } else null
        
        | _ => null
      }
    }
    #endregion
    

    #region yield handling
    WrapYieldingFunction (body : PT.PExpr) : PT.PExpr
    {
      def yield_type = GetYieldType ();

      def new_body =
        <[
          mutable _N_current = $(TExpr.DefaultValue (yield_type) : typed);
          mutable _N_state = 0;

          def _N_yielding_function () : bool
          {
            _N_MoveNext_exit : {
              // AddYieldStateMachine will add something here
              match (_N_state) {
                | 0 => {}
                | _ => _N_MoveNext_exit (false)
              }
              {
                $body;
                false
              }
            }
          }

          _ = _N_yielding_function;
          null
        ]>;

      new_body
    }


    GetYieldType () : TyVar
    {
      match (parent_typer) {
        | Some (t) => t.GetYieldType ()
        | None =>
          match (current_fun.ret_type.Fix ()) {
            | Class (tc, [t])
              when tc.Equals (InternalType.Generic_IEnumerator_tc) || 
                   tc.Equals (InternalType.Generic_IEnumerable_tc) => t
            | Class (tc, [])
              when tc.Equals (InternalType.IEnumerator_tc) || 
                   tc.Equals (InternalType.IEnumerable_tc) => InternalType.Object
            | t =>
              ReportError (messenger,
                           $ "yield used in a `$(current_fun.name)' function, which is returning "
                             "$t (it should be System.Collections.Generic."
                             "IEnumerator[T] or System.Collections.Generic."
                             "IEnumerable[T])");
              InternalType.Object
         }
      }
    }

    HandleYield (expr : PT.PExpr) : TExpr
    {
      Util.cassert (solver.IsTopLevel);
      def label_no = Util.next_id ();
      def state_no = AddYieldLabel (label_no);
      def label =
        TExpr.Label (InternalType.Void, label_no, VoidLiteral ());

      def expr =
        MacroColorizer.InGlobalColor (fun () {
          <[ 
            {
              _N_current = $expr;
              _N_state = $(state_no : int);
              _N_MoveNext_exit (true);
              $(label : typed)
            }
          ]>
        });
      
      TypeExpr (expr)
    }
    

    AddYieldLabel (no : int) : int
    {
      if (yield_labels == null)
        Option.UnSome (parent_typer).AddYieldLabel (no)
      else {
        yield_labels ::= no;
        yield_labels.Length
      }
    }


    AddYieldStateMachine (e : TExpr) : void
    {
      match (e) {
        | Block (_, Block (_, Sequence (Match as m, _) as s)) =>
          mutable cnt = 1;
          def cases = yield_labels.Rev ().Map (fun (id) {
            def cntpat = Pattern.Literal (InternalType.Int32, Literal.FromInt (cnt));
            cnt++;
            def yes = TExpr.Literal (InternalType.Boolean, Literal.Bool (true));
            Match_case ([(cntpat, yes, [])], TExpr.Goto (m.Type, id, 1), false)
          });
          s.e1 = TExpr.Match (m.Type, m.expr, cases + m.cases)
        | _ => Util.ice ($ "e = $e")
      }
    }
    #endregion


    #region Partial application
    /** Transform [_.foo] to [fun (x) { x.foo }] and [foo (_, 3)] to
        [fun (x) { foo (x, 3) }].  */
    static PartialApplication (expr : PT.PExpr) : PT.PExpr
    { 
      def transform_parms (parms) {
        mutable lambda_parms = [];
        def res = parms.Map (fun (_) {
          | <[ _ ]> =>
            def sym = Macros.NewSymbol ();
            lambda_parms ::= <[ parameter: $(sym : name) ]>;
            <[ $(sym : name) ]>
          | e => e
        });
        (lambda_parms.Rev (), res)
      }
      
      match (expr) {
        | <[ _ . $mem ]> =>
          def sym = Macros.NewSymbol ();
          <[ fun ($(sym : name)) { $(sym : name) . $mem } ]>
        | <[ _ . $mem ( .. $parms ) ]> =>
          def (lambda_parms, parms) = transform_parms (parms);
          def sym = Macros.NewSymbol ();
          def parm = <[ parameter: $(sym : name) ]>;
          <[ fun (.. $(parm :: lambda_parms)) { $(sym : name) . $mem (.. $parms) } ]>
        | <[ $fn ( .. $parms ) ]> =>
          def (lambda_parms, parms) = transform_parms (parms);
          <[ fun (.. $lambda_parms) { $fn (.. $parms) } ]>
        | _ => Util.ice ($ "$expr")
      }
    }
    #endregion


    #region Pretty printing
    DescribeExpression (expr : TExpr) : string
    {
      match (expr) {
        | TExpr.LocalRef (d) =>
          $ "a reference to local symbol `$(d)'"
        | TExpr.LocalFunRef (d, _) =>
          $ "a reference to `$(d)'"
        | TExpr.StaticRef (from, m, _) =>
          def tyname = from.ToString () + ".";
          "a reference to global symbol `" + tyname + m.Name + "'"
        | TExpr.FieldMember (_, f) =>
          "a reference to field `" + f.Name + "'"
        | TExpr.ConstantObjectRef (from, _) =>
          "a reference to constant constructor of `" +
          from.ToString () + "'"
        | TExpr.ImplicitValueTypeCtor =>
          $"a reference to implicit constructor of `$(expr.Type)'"
        | TExpr.StaticPropertyRef (_, p)
        | TExpr.PropertyMember (_, p) =>
          "a reference to property `" + p.Name + "'"
        | TExpr.StaticEventRef (_, p)
        | TExpr.EventMember (_, p) =>
          "a reference to event `" + p.Name + "'"
        | TExpr.MethodRef (_, m, _, _) =>
          "a reference to method `" + m.Name + "'"
        | TExpr.Call (f, _, _) =>
          "a function call to " + DescribeExpression (f)
        | TExpr.SelfTailCall =>
          "self-recursive function call"
        | TExpr.Assign => "an assignment"
        | TExpr.DefValIn => "a value binding"
        | TExpr.DefFunctionsIn => "a function binding"
        | TExpr.Match => "a `match' expression"
        | TExpr.If => "a `if' expression"
        | TExpr.HasType => "a `is' expression"
        | TExpr.Throw => "a throw expression"
        | TExpr.TryWith => "a try...with expression"
        | TExpr.TryFinally => "a try...finally expression"
        | TExpr.Literal => "a literal value"
        | TExpr.This => "a this pointer reference"
        | TExpr.Base => "a base class reference"
        | TExpr.TypeConversion => "a type conversion"
        | TExpr.Sequence => "a sequence"
        | TExpr.Tuple => "a tuple constructor"
        | TExpr.Array => "an array constructor"
        | TExpr.ArrayIndexer => "an array indexer reference"
        | TExpr.TupleIndexer => "an tuple indexer reference"
        | TExpr.TypeOf => "a typeof expression"
        | TExpr.OpCode => "an operator reference"
        
        | TExpr.Error => "an erroneous expression"
        | TExpr.Delayed (dt) =>
          match (dt.DtKind) {
            | Resolved (e) => DescribeExpression (e)
            | _ => "a yet-untyped expression"
          }

        | TExpr.TryFault
        | TExpr.Switch
        | TExpr.DefaultValue
        | TExpr.Goto
        | TExpr.Label
        | TExpr.Block
        | TExpr.MultipleAssign
        | TExpr.MethodAddress => 
          $ "!!!shouldn't happen: $(expr.GetType())!!!"
      }
    }
    #endregion
  }
}
