/* ------------------------------------------------------------------------- */
/*   Rachunki za FIT                                                         */
/* ------------------------------------------------------------------------- */

using Nemerle.IO;
using System;

class Odbiorca
{
  private mutable _nazwa_i_adres : string;
  private mutable _nip : string;

  public this (nazwa : string, adres : string, nip : string) 
  {
    _nazwa_i_adres = (nazwa + " " + adres).Trim ();
    _nip = nip;
  }
  
  public NazwaOrazAdres () : string { _nazwa_i_adres }
  public Nip () : string { _nip }
}


class Rachunek
{
  private static mutable _id : int;
  private mutable _odbiorca : Odbiorca;
  private mutable _uczestnik : string;
  private mutable _cena : string;
  private mutable _cena_slownie : string;
  
  public static this () 
  {
    _id = 0;
  }
  
  public this (uczestnik : string, cena : string, odbiorca : Odbiorca)
  {
    _uczestnik = uczestnik;
    _cena = cena;
    _odbiorca = odbiorca;
    
    _cena_slownie = 
      match (cena) {
        | "185" => "sto osiemdziesiąt pieć / 00"
        | "260" => "dwieście sześćdziesiąt / 00"
        | "270" => "dwieście siedemdziesiąt / 00"
        | "325" => "trzysta dwadzieścia pięć / 00"
        | "330" => "trzysta trzydzieści / 00"
        | "335" => "trzysta trzydzieści pięć / 00"
        | "340" => "trzysta czterdzieści / 00"
        | "395" => "trzysta dziewięćdziesiąt pięć / 00"
        | "405" => "czterysta pięć / 00"
        | "1220" => "tysiąc dwieście dwadzieścia / 00"
        | _ => 
          printf ("bomba i panika, jak zapisać słownie %s?\n", cena); 
          "bomba/panika"
      }
  }
    
  public ToLaTeX (kopia : bool) : string
  {
    unless (kopia) { _id = _id + 1 };

    def space = if (_uczestnik.Length > 150) @"\vspace{3mm}" else @"\vspace{5mm}";

    def kopia_czy_oryginal =
      if (kopia) @"{\bf \sout{ORYGINAŁ} / KOPIA}" else @"{\bf ORYGINAŁ / \sout{KOPIA}}";


    @"    
      \begin{multicols}{2}
        \begin{tabular}{c}
          \hline
          Polskie Stowarzyszenie \\
          dla Maszyn Liczących we Wrocławiu \\
          ul. Przesmyckiego 20 \\
          51-151 Wrocław \\
          REGON 930724005 \\
          \hline 
        \end{tabular}

        {\Large \bf RACHUNEK NR " + _id.ToString () + @"/FIT/04} \\
        {\small Sprzedawca nie jest płatnikiem podatku VAT.}
        
        \vspace{5mm}
        
        " + kopia_czy_oryginal + @"
                
      \end{multicols}
      
      " + space + @"
            
      Rachunek wystawiono we Wrocławiu dnia 13 XII 2003 r. dla: " + 
      _odbiorca.NazwaOrazAdres () + @", NIP: \mbox{" + _odbiorca.Nip () + @"}
      
      " + space + @"
      " + space + @"

      \begin{tabular}{|p{7cm}|l|c|r|r|}
        \hline
        {\bf Nazwa towaru lub usługi} & {\bf J.m.} & {\bf Ilość} & {\bf Cena} & {\bf Wartość} \\
        \hline \hline
        Opłata konferencyjna za udział w~XVIII Forum Informatyki Teoretycznej: " +
        _uczestnik + @" & & 1 & " + _cena + @" zł 00 gr & " + 
        _cena + @" zł 00 gr \\ \hline \hline
        & & & RAZEM & " + _cena + @" zł 00 gr \\ \hline
      \end{tabular}
      
      " + space + @"
        
      Forma płatności: {\it przelew}
      
      Słownie zł/gr: {\it " + _cena_slownie + @"}
      
      " + space + @"
      " + space + @"
      " + space + @"
      " + space + @"

      \begin{multicols}{2}
        \begin{tabular}{c}
          \hline
          podpis kupującego \\
          \hspace{7 cm} \\
        \end{tabular}
              
        \begin{tabular}{c}
          \hline
          podpis i pieczęć sprzedawcy \\
          \hspace{7 cm} \\
        \end{tabular}
      \end{multicols}
            
      \newpage
    "
  }
}


class RachunkiZaFit
{
  public static LaTeXHeader () : string 
  {
    @"\documentclass[draft,12pt]{article}
\usepackage[latin2]{inputenc}
\usepackage{fancyhdr}
\usepackage{hcolor}
\usepackage{color}
\usepackage{polski}
\usepackage{lscape}
\usepackage{multicol}
\usepackage{ulem}
\usepackage[a4paper]{geometry}

\begin{document}
\begin{landscape}

\pagestyle{empty}

    "
  }

  public static LaTeXFooter () : string
  {
    @"
      \end{landscape}
      \end{document}
    "
  }
  
  public static Main () : void
  {
    printf ("Rachunki za FIT...\n");
    
    def sr = IO.StreamReader ("rachunki-za-fit.csv");
    def sw = IO.StreamWriter ("rachunki-za-fit.tex", false, 
                                Text.Encoding.GetEncoding("iso-8859-2"), 2048);

    mutable rachunki = [];

    // read all the bill information from the CSV file...
    while (sr.Peek () != -1) {

      // read a line and split it along the tab characters
      def line = sr.ReadLine ();
      def split = line.Split (";".ToCharArray ());
  
      when (split.Length != 7) { throw ArgumentException ("line") };
      
      // the format of the bills file is: [NAME; INSTITUTION; ADRESS; NIP; junk; junk; PRICE]
      rachunki = Rachunek (split [0], split [6], Odbiorca (split [1], split [2], split [3])) 
        :: rachunki;
    };
        
    // process the bills and save them in the output format
    def iter (rachunki : list [Rachunek]) : void {    
      match (rachunki) {
        | [] => ()
        | rachunek :: reszta =>
          sw.Write (rachunek.ToLaTeX (false));
          sw.Write (rachunek.ToLaTeX (true));
          iter (reszta)
      }
    }; 

    sw.Write (LaTeXHeader ());
    iter (rachunki);   
    sw.Write (LaTeXFooter ());
    
    sw.Close ();
    sr.Close ()    
  }
}
