/*

Copyright (C) 2000, 2001 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <net/ethernet.h>

#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd.h>
#include <protocols/snap/nd_proto_snap.h>

#include <nd_ether.h>
#include <nd_ether_callbacks.h>

static ND_ProtoField ether_fields[] = {
  { ND_VAL_FIELD, N_("Dst. addr.\n(%s)"),   N_("Destination address"),   48,  nd_ether_dest_addr_cb  },
  { ND_VAL_FIELD, N_("Src. addr.\n(%s)"),   N_("Source address"),        48,  nd_ether_src_addr_cb   },
  { ND_VAL_FIELD, N_("Length/Type\n(%s)"),  N_("Length or type"),        16,  nd_ether_lengthtype_cb },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData ether_menu_type_data[] = {
  { "IP",         N_("IP (0x0800)"),              ETHERTYPE_IP,     nd_ether_type_value_cb },
  { "ARP",        N_("ARP (0x0806)"),             ETHERTYPE_ARP,    nd_ether_type_value_cb },
  { "RARP",       N_("RARP (0x8035)"),            ETHERTYPE_REVARP, nd_ether_type_value_cb },
  { "PPPoE Dsc.", N_("PPPoE Discovery (0x8863)"), 0x8863,           nd_ether_type_value_cb },
  { "PPPoE Ses.", N_("PPPoE Session (0x8864)"),   0x8864,           nd_ether_type_value_cb },
  { "Custom",     N_("Custom type value"),        -1,               nd_ether_type_custom_cb },
  { NULL, NULL, 0, NULL}
};

static ND_Protocol *ether;


static gboolean
ether_header_complete(const ND_Packet *packet, guchar *data)
{
  return (data + ETHER_HDR_LEN <= nd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("Ethernet Plugin");
}


const char *
description(void)
{
  return ("A plugin providing Ethernet support.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  ether = nd_proto_new("Ethernet", ND_PROTO_LAYER_LINK, DLT_EN10MB);
			     		
  ether->create_gui      = nd_ether_create_gui;
  ether->set_gui         = nd_ether_set_gui;
  ether->init_packet     = nd_ether_init_packet;
  ether->header_complete = nd_ether_header_complete;

  /* This one isn't really necessary, just for illustration */
  ether->is_stateful = FALSE;

  /* We're using a button table to display the protocol content,
     so we need to hook it in here: */
  ether->fields = ether_fields;
  ether->header_width = 112;

  /* That's all -- we don't need checksums or state maintenance
     for simple Ethernet. The other methods got initialized
     to dummy null operations in the constructor call above.
     
     We also don't need a special menu to appear in the Protocols
     menu for this plugin.
  */
	     
  return ether;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_ether_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);
  return table;
}


void       
nd_ether_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct ether_header *eh;

  eh = (struct ether_header*) nd_packet_get_data(packet, ether, pinf->inst.nesting);
  
  nd_ether_set_gui_src(pinf, eh);
  nd_ether_set_gui_dst(pinf, eh);
  nd_ether_set_gui_typelength(pinf, eh);

  return;
  TOUCH(pinf);
}


void       
nd_ether_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  ND_Protocol *payload_proto;
  struct ether_header *eh;

  if (!ether_header_complete(packet, data))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  nd_packet_add_proto_data(packet, ether, data, data_end);  
  
  eh = (struct ether_header *) data;


  /* Is this an Ethernet-II frame or a Ethernet 802.3 SNAP frame?*/
  /* Check ether_type > 1500 --> Ethernet II */
  if (htons(eh->ether_type) <= ETHERMTU)
    {
      /* It's Ethernet 802.3, starting with some form of LLC,
	 pass to SNAP implementation: */

      payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_LINK, ND_PROTO_SNAP);
    }
  else 
    {
      /* It's Ethernet II: */
      payload_proto = nd_proto_registry_find(ND_PROTO_LAYER_LINK | ND_PROTO_LAYER_NET,
					     htons(eh->ether_type));
    }  

  payload_proto->init_packet(packet, data + ETHER_HDR_LEN, data_end);
}


gboolean
nd_ether_header_complete(const ND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = nd_packet_get_data(packet, ether, nesting);

  return ether_header_complete(packet, data);
  TOUCH(nesting);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_ether_set_gui_dst(ND_ProtoInfo *pinf, struct ether_header *eh)
{
  char                 s[MAXPATHLEN];

  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     eh->ether_dhost[0], eh->ether_dhost[1],
	     eh->ether_dhost[2], eh->ether_dhost[3],
	     eh->ether_dhost[4], eh->ether_dhost[5]);

  nd_proto_field_set(pinf, &ether_fields[0], s);
}


void
nd_ether_set_gui_src(ND_ProtoInfo *pinf, struct ether_header *eh)
{
  char                 s[MAXPATHLEN];

  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     eh->ether_shost[0], eh->ether_shost[1],
	     eh->ether_shost[2], eh->ether_shost[3],
	     eh->ether_shost[4], eh->ether_shost[5]);

  nd_proto_field_set(pinf, &ether_fields[1], s);
}


void
nd_ether_set_gui_typelength(ND_ProtoInfo *pinf, struct ether_header *eh)
{
  nd_proto_field_set_for_menu(pinf, &ether_fields[2], DATA_TO_PTR(ntohs(eh->ether_type)),
			      ether_menu_type_data, "0x%.4x");
}

ND_Protocol *
nd_ether_get(void)
{
  return ether;
}

