/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_protocol_registry_h
#define __nd_protocol_registry_h

#include <netdude/nd.h>
#include <netdude/nd_protocol.h>


/**
 * ND_ProtocolFunc - signature of protocol iteration callbacks.
 * @proto: iterated protocol.
 * @user_data: arbitrary user data.
 *
 * This is the signature of functions that can be passed to
 * nd_proto_registry_foreach_proto().
 */
typedef void (*ND_ProtocolFunc)(ND_Protocol *proto, void *user_data);


/**
 * nd_proto_registry_init - protocol registry initialization.
 *
 * The function initializes the registry, sets up hashes etc.
 */
void           nd_proto_registry_init(void);

/**
 * nd_proto_registry_register - protocol registration.
 * @proto: protocol to register.
 * 
 * The function registers a protocol in the  registry.
 * For normal protocol plugins, this gets called automatically when
 * the plugin is loaded and initialized, so you don't have to do this
 * manually.
 *
 * Returns: %TRUE when protocol was registered, %FALSE on error or
 * when the protocol is already found in the registry.
 */
gboolean       nd_proto_registry_register(ND_Protocol *proto);

/**
 * nd_proto_registry_unregister - protocol unregistration.
 * @proto: protocol to unregister.
 *
 * The given protcol is removed from the registry. The protocol is
 * not touched otherwise.
 */
void           nd_proto_registry_unregister(ND_Protocol *proto);

/**
 * nd_proto_registry_find - protocol lookup.
 * @layer: layer where to look for protocol
 * @magic: magic of protocol.
 *
 * The function looks up a protocol plugin for a given layer and magic.
 * If the protocol isn't found, the raw data display plugin is
 * returned, which you can discern by a protocol ID value of 1. The @magic
 * is the usual magic for the protocol you're looking for that is used
 * at the layer below the protocol to identify it, e.g. one of the %DLT_xxx
 * values for a link layer protocol, or one of the %ETHERTYPE_xxx values
 * for a network layer protocol.
 *
 * Returns: retrieved protocol - either the one you were looking for, or
 * the raw data protocol otherwise.
 */
ND_Protocol   *nd_proto_registry_find(ND_ProtocolLayer layer, guint64 magic);

/**
 * nd_proto_registry_size - returns size of registry.
 *
 * The function returns the number of protocols in the registry.
 *
 * Returns: number of protocols.
 */
guint          nd_proto_registry_size(void);

/**
 * nd_proto_registry_foreach_proto - protocol iteration.
 * @callback: callback to call for each protocol.
 * @user_data: arbitrary user data passed through to @callback.
 *
 * The function iterates over the registered protocols and applies
 * a callback function to each of them.
 */
void           nd_proto_registry_foreach_proto(ND_ProtocolFunc callback,
					       void *user_data);

#endif
