/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgPkiAdmin.cpp: implementation of the DlgPkiAdmin class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgImportSignReq.h"
#include "DlgManagePkiUsers.h"
#include "DlgManageACL.h"
#include "DlgManageRepInfo.h"
#include "DlgConfigureRep.h"
#include "DlgMailConf.h"
#include "DlgSendMail.h"
#include "DlgManageAudits.h"
#include "DlgManageEntities.h"
#include "DlgCertProperties.h"
#include "DlgConfigureCA.h"
#include "DlgManageGroups.h"
#include "DlgConfigureRA.h"
#include "DlgConfigurePub.h"
#include "DlgConfigureEE.h"

#include "DlgPkiAdmin.h"
#include "dlgs_wdr.h"
#include "clintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BEGIN_EVENT_TABLE(DlgPkiAdminListCtrl, wxListCtrl)
    EVT_LIST_ITEM_RIGHT_CLICK(IDC_LIST_ENTITIES, DlgPkiAdminListCtrl::OnItemRightClick)
    EVT_LIST_ITEM_ACTIVATED(IDC_LIST_ENTITIES, DlgPkiAdminListCtrl::OnItemClick)
END_EVENT_TABLE()



DlgPkiAdmin * m_DlgPkiAdminParent;


#ifdef _WIN32

wxBitmap hPic = Backgrounds_GetBitmap(0);

long oldProc;
int centerX;
int centerY;
bool inProc = false;

LRESULT CALLBACK lstvProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	RECT rRect;
	PAINTSTRUCT ps;
	HDC hDC;
	HDC hDCParent;
	HBITMAP bmp;
	HBITMAP bmpOld;
				
	switch (message) 
	{
		case WM_PAINT:
			GetClientRect(hwnd,&rRect);

			if(inProc)
			{
				inProc = false;
				InvalidateRect(hwnd, &rRect, TRUE);
				return TRUE;
			}
			inProc = true;


			CallWindowProcA((WNDPROC)oldProc, hwnd, message, wParam, lParam);		

			BeginPaint(hwnd, &ps);
			
			hDC=GetDC(hwnd);
			hDCParent=CreateCompatibleDC(hDC);
			bmp = (HBITMAP)hPic.GetHBITMAP();
			
			
			bmpOld = (HBITMAP)SelectObject(hDCParent, bmp);
			
			
			centerX = ((rRect.right - rRect.left) - hPic.GetWidth()) / 2;
			centerY = ((rRect.bottom - rRect.top) - hPic.GetHeight())  / 2;
			
			
			BitBlt(hDC,centerX, centerY, hPic.GetWidth(),hPic.GetHeight(), hDCParent, 0, 0,SRCAND );
			
			ReleaseDC(hwnd,hDC);
			SelectObject(hDCParent, bmpOld);
			DeleteDC(hDCParent);

			EndPaint(hwnd, &ps);
			return TRUE;
			
		default:
			return CallWindowProcA((WNDPROC)oldProc, hwnd, message, wParam, lParam);
	}
	return FALSE;
}

#endif

DlgPkiAdminListCtrl::DlgPkiAdminListCtrl(wxWindow *parent, const wxWindowID id, const wxPoint& pos,	const wxSize& size, long style):wxListCtrl(parent, id, pos, size, style)
{
	#ifdef _WIN32
		oldProc = SetWindowLong((HWND)GetHWND(), GWL_WNDPROC, (long)lstvProc);
	#endif
}

DlgPkiAdminListCtrl::~DlgPkiAdminListCtrl()
{
}

void DlgPkiAdminListCtrl::OnItemRightClick(wxListEvent& event)
{
	m_DlgPkiAdminParent->OnItemRightClick(event);
}

void DlgPkiAdminListCtrl::OnItemClick(wxListEvent& event)
{
	m_DlgPkiAdminParent->OnItemClick(event);
}



BEGIN_EVENT_TABLE(DlgPkiAdmin, wxFrame)
	EVT_MENU(IDM_QUIT, DlgGUI::OnQuit)
    EVT_MENU(IDM_RELOAD, DlgPkiAdmin::OnReload)
    EVT_MENU(IDM_SHOW_LOGS, DlgGUI::OnShowLogs)
    EVT_MENU(IDM_RECONNECT, DlgGUI::OnReconnect)
    EVT_MENU(IDM_SHOW_SSL_CERT, DlgGUI::ShowSSL_Certificate)
    EVT_MENU(IDM_SHOW_ENTITY_CERT, DlgGUI::ShowEntity_Certificate)
    EVT_MENU(IDM_ABOUT, DlgGUI::OnAbout)
    EVT_MENU(IDC_SET_OFFLINE, DlgPkiAdmin::OnSetOffline)
    EVT_MENU(IDC_IMPORT_REQ, DlgPkiAdmin::OnImportReq)
    EVT_MENU(IDC_MANAGE_USER, DlgPkiAdmin::OnManageUsers)
    EVT_MENU(IDM_REVOKE_ENTITY, DlgPkiAdmin::OnRevokeEntity)
    EVT_MENU(IDM_CONFIGURE_ENTITY, DlgPkiAdmin::OnConfigureEntity)
    EVT_MENU(IDM_MANAGE_ACL, DlgPkiAdmin::OnManageEntityACL)
    EVT_MENU(IDC_EXPORT_CONF, DlgPkiAdmin::OnExportConf)
    EVT_MENU(IDC_MANAGE_REP, DlgPkiAdmin::OnManageRepInfo)
    EVT_MENU(IDM_SHOW_CERT, DlgPkiAdmin::OnShowEntityCert)
    EVT_MENU(IDM_MANAGE_ACL, DlgPkiAdmin::OnShowEntityCert)
    EVT_MENU(IDM_SHOW_CERT, DlgPkiAdmin::OnShowEntityCert)
    EVT_MENU(IDM_MANAGE_ENTITY_ACL, DlgPkiAdmin::OnManageEntityACL)
    EVT_MENU(IDM_MANAGE_ENTITY_MAIL_CONF, DlgPkiAdmin::OnManageEntityMailConf)
    EVT_MENU(IDM_MANAGE_MAIL_CONF, DlgPkiAdmin::OnManageEntityMailConf)
    EVT_MENU(IDC_SEND_ADMIN_MAIL, DlgPkiAdmin::OnSendAdminMail)
    EVT_MENU(IDM_MANAGE_ENTITY_AUDITS, DlgPkiAdmin::OnManageEntityAudits)
    EVT_MENU(IDM_MANAGE_AUDITS, DlgPkiAdmin::OnManageEntityAudits)
    EVT_MENU(IDC_MANAGE_ENTITIES, DlgPkiAdmin::OnManageEntities)
    EVT_MENU(IDC_MANAGE_GROUPS, DlgPkiAdmin::OnManageGroups)
END_EVENT_TABLE()


DlgPkiAdmin::DlgPkiAdmin(wxWindow * wParent, char * EntityName, char * UserName, char * Password, PkiClient * ClientPki):DlgGUI(wParent, ClientPki, EntityName, UserName, Password)
{
	m_DlgPkiAdminParent = this;

    wxToolBar * toolBar = new wxToolBar(this, -1);

	wxString mTitle;
	mTitle.sprintf("PKI Admin (%s/%s) [%s:%d]", UserName, EntityName, ClientPki->GetRemoteServer().c_str(), ClientPki->GetRemotePort());
	SetTitle(mTitle);

	DlgPkiAdmin_SetWindow(this);
	SetMenuBar(DlgPkiAdmin_GetMenu());

	toolBar->SetToolBitmapSize(wxSize(32,32));
	DlgPkiAdmin_SetToolbar(toolBar);
    SetToolBar(toolBar);


	CenterOnScreen();

	m_imageListSmall = new wxImageList(16, 16, TRUE);

	
	m_listCtrl = (DlgPkiAdminListCtrl *)FindWindow(IDC_LIST_ENTITIES);
	if(!m_listCtrl) return;

	wxIcon ico;
	
	ico.CopyFromBitmap(Entities_GetBitmap(IDB_CA_SMALL));
	CaIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_RA_SMALL));
	RaIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_REP_SMALL));
	RepositoryIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_PUB_SMALL));
	PublicationIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_KEY_SMALL));
	KeyIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_EE_SMALL));
	EEIconId = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(Entities_GetBitmap(IDB_UNKNOWN_SMALL));
	UnknownIconId = m_imageListSmall->Add( ico );	

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Name"), wxLIST_ALIGN_DEFAULT, 320);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Type"), wxLIST_ALIGN_DEFAULT, 80);

	LoadEntitiesList();
}

DlgPkiAdmin::~DlgPkiAdmin()
{
	if(m_listCtrl) m_listCtrl->ClearAll();
	if(m_imageListSmall) delete m_imageListSmall;
}


bool DlgPkiAdmin::LoadEntitiesList()
{
	size_t i;
	int IconId;
	const char * strType;
	bool offline;
	ASN1_BIT_STRING * my_acl;
	wxMenuBar * menubar;
	wxMenu * menu;
	


	m_listCtrl->DeleteAllItems();	
	m_entities.clear();

	m_DlgMsg->wShow(_("Loading Configuration..."));
	if(!m_ClientPki->GetMyACL(&my_acl))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	menubar = GetMenuBar();
	menu = menubar->GetMenu(1);

	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_VIEW_LOGS))
	{
		menu->Enable(IDM_SHOW_LOGS, false);
	}
	else
	{
		menu->Enable(IDM_SHOW_LOGS, true);
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_SEND_ADMIN_MAIL))
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, FALSE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, TRUE );
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_MANAGE_ACLS))
	{

		menu->Enable(IDM_MANAGE_ACL, false);
	}
	else
	{
		menu->Enable(IDM_MANAGE_ACL, true);
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_MANAGE_USERS))
	{
		GetToolBar()->EnableTool( IDC_MANAGE_USER, FALSE );
		GetToolBar()->EnableTool( IDC_MANAGE_GROUPS, FALSE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_MANAGE_USER, TRUE );
		GetToolBar()->EnableTool( IDC_MANAGE_GROUPS, TRUE );
	}

	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_MANAGE_ENTITIES))
	{
		GetToolBar()->EnableTool( IDC_IMPORT_REQ, FALSE );
		GetToolBar()->EnableTool( IDC_MANAGE_REP, FALSE );
		m_listCtrl->Enable(FALSE);
	}
	else
	{
		GetToolBar()->EnableTool( IDC_IMPORT_REQ, TRUE );
		GetToolBar()->EnableTool( IDC_MANAGE_REP, TRUE );
		m_listCtrl->Enable(TRUE);

		if(!m_ClientPki->EnumEntities(m_entities))
		{
			m_DlgMsg->wHide();
			ASN1_BIT_STRING_free(my_acl);
			HandleError(m_ClientPki->GetError(), this);
			return false;
		}

		for(i=0; i < m_entities.size(); i++)
		{
			switch(m_entities[i].get_type())
			{
				case ENTITY_TYPE_RA:
					IconId = RaIconId;
					strType = _("RA");
					break;

				case ENTITY_TYPE_CA:
					IconId = CaIconId;
					strType = _("CA");
					break;

				case ENTITY_TYPE_REPOSITORY:
					IconId = RepositoryIconId;
					strType = _("Repository");
					break;

				case ENTITY_TYPE_PUBLICATION:
					IconId = PublicationIconId;
					strType = _("Publication");
					break;

				case ENTITY_TYPE_KEY_STORE:
					IconId = KeyIconId;
					strType = _("Key Store");
					break;

				case ENTITY_TYPE_EE:
					IconId = EEIconId;
					strType = _("EE");
					break;				

				default:
					strType = _("Unknown");
					IconId = UnknownIconId;
					break;
			}

			m_listCtrl->InsertItem(i, m_entities[i].get_name().c_str(), IconId);
			m_listCtrl->SetItemData(i, (long)i);
			m_listCtrl->SetItem(i, 1, strType);
		}
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_MODIFY_PKI_STATUS))
	{
		GetToolBar()->EnableTool( IDC_SET_OFFLINE, FALSE );
		GetToolBar()->EnableTool( IDC_MANAGE_ENTITIES, FALSE );
		menu->Enable(IDM_MANAGE_MAIL_CONF, false);
		menu->Enable(IDM_MANAGE_AUDITS, false);
	}
	else
	{
		GetToolBar()->EnableTool( IDC_SET_OFFLINE, TRUE );
		GetToolBar()->EnableTool( IDC_MANAGE_ENTITIES, TRUE );
		menu->Enable(IDM_MANAGE_MAIL_CONF, true);
		menu->Enable(IDM_MANAGE_AUDITS, true);


		if(!m_ClientPki->GetOfflineState(offline))
		{
			m_DlgMsg->wHide();
			ASN1_BIT_STRING_free(my_acl);
			HandleError(m_ClientPki->GetError(), this);
			return false;
		}
		GetToolBar()->DeleteTool(IDC_SET_OFFLINE);
		if(offline)
		{
			GetToolBar()->InsertTool(0, IDC_SET_OFFLINE, DlgPkiAdmin_GetBitmap( 1 ), wxNullBitmap, TRUE, NULL, _("Toggle PKI entity online") );
		}
		else
		{
			GetToolBar()->InsertTool(0, IDC_SET_OFFLINE, DlgPkiAdmin_GetBitmap( 2 ), wxNullBitmap, TRUE, NULL, _("Toggle PKI entity offline") );
		}
		GetToolBar()->ToggleTool(IDC_SET_OFFLINE, offline);
	    GetToolBar()->Realize();
	}
	m_DlgMsg->wHide();

	ASN1_BIT_STRING_free(my_acl);

	return true;
}


void DlgPkiAdmin::OnReload(wxCommandEvent& event)
{
	ReloadConf();
}

void DlgPkiAdmin::OnItemClick(wxListEvent &event)
{

}

void DlgPkiAdmin::OnItemRightClick(wxListEvent &event)
{
	wxMenu * menu;
	long SelectedItem;
	int Index;
	
	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	Index = (size_t)m_listCtrl->GetItemData(SelectedItem);

	wxMenuBar * menu_b = DlgPkiAdmin_GetPopup();
	if(!menu_b) return;

	menu = menu_b->GetMenu(0);
	if(menu)
	{
		switch(m_entities[Index].get_type())
		{
			case ENTITY_TYPE_KEY_STORE:
			case ENTITY_TYPE_REPOSITORY:
				menu->Enable(IDM_CONFIGURE_ENTITY, false);
				break;
		}
		m_listCtrl->PopupMenu(menu, event.GetPoint().x, event.GetPoint().y);
	}

	delete menu_b;
}

void DlgPkiAdmin::OnImportReq(wxCommandEvent &event)
{
	DlgImportSignReq Dlg(this, m_ClientPki);
	if(Dlg.IsOK())
		LoadEntitiesList();
}

void DlgPkiAdmin::OnManageUsers(wxCommandEvent &event)
{
	DlgManagePkiUsers Dlg(this, m_ClientPki);
}

void DlgPkiAdmin::OnRevokeEntity(wxCommandEvent &event)
{
	size_t Index;
	long SelectedItem;
	
	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	Index = (size_t)m_listCtrl->GetItemData(SelectedItem);

	if(DisplayMessage(this, _("Are you sure, you want to revoke this certificate ?"), wxYES_NO) == wxNO) return;

	m_DlgMsg->wShow(_("Revoking Certificate..."));

	if(!m_ClientPki->RevokeEntityCert(m_entities[Index].get_certificate().GetSerial()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	LoadEntitiesList();
}

void DlgPkiAdmin::OnConfigureEntity(wxCommandEvent &event)
{
	DlgConfigureEntity * Dlg;
	size_t Index;
	long SelectedItem;
	
	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	Index = (size_t)m_listCtrl->GetItemData(SelectedItem);
	
	switch(m_entities[Index].get_type())
	{
		case ENTITY_TYPE_CA:
			Dlg = new DlgConfigureCA(this, m_entities[Index].get_certificate(), m_ClientPki);
			break;
		case ENTITY_TYPE_RA:
			Dlg = new DlgConfigureRA(this, m_entities[Index].get_certificate(), m_ClientPki);
			break;
		case ENTITY_TYPE_PUBLICATION:
			Dlg = new DlgConfigurePub(this, m_entities[Index].get_certificate(), m_ClientPki);
			break;
		case ENTITY_TYPE_KEY_STORE:
			Dlg = NULL;
			break;
		case ENTITY_TYPE_EE:
			Dlg = new DlgConfigureEE(this, m_entities[Index].get_certificate(), m_ClientPki);
			break;
		case ENTITY_TYPE_REPOSITORY:
			Dlg = new DlgConfigureRep(this, m_entities[Index].get_certificate(), m_ClientPki);
			break;
		default:
			Dlg = NULL;
			break;
	}
	if(Dlg)
		delete Dlg;
}

void DlgPkiAdmin::OnSetOffline(wxCommandEvent &event)
{
	bool state = GetToolBar()->GetToolState(IDC_SET_OFFLINE);
	if(state)
	{
		if(DisplayMessage(this, _("Are you sure to want to toggle the PKI offline ?\nBy doing this you will be forced to manually export \nthe global PKI Configuration and to reimport it to a online \"repository\" entity !"), wxYES_NO) == wxNO)
		{
			GetToolBar()->ToggleTool(IDC_SET_OFFLINE, false);
			return;
		}
	}

	m_DlgMsg->wShow(_("Setting PKI's Status..."));
	if(!m_ClientPki->SetOfflineState(state))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
	LoadEntitiesList();
}


void DlgPkiAdmin::OnExportConf(wxCommandEvent &event)
{
	ExportedPkiConf conf;
	FILE * fd;
	mString pem;
	wxFileDialog dialog(this, _("Save PKI configuration"), "", m_EntityName.c_str(), _("Configuration File (*.conf)|*.conf"), wxSAVE|wxOVERWRITE_PROMPT);

	
	m_DlgMsg->wShow(_("Exporting Configuration..."));
	if(!m_ClientPki->GetConfiguration(conf))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();


	if(!conf.to_PEM(pem))
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}


	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}

	fd = fopen(dialog.GetPath() , "wb");
	if(!fd)
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}
	fwrite(pem.c_str(), pem.size(), 1, fd);
	fclose(fd);
}

void DlgPkiAdmin::OnManageRepInfo(wxCommandEvent &event)
{
	if(!m_ClientPki->ResourceLock(PKI_REPOSITORIES_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	DlgManageRepInfo Dlg(this, m_ClientPki);
	if(!m_ClientPki->ResourceUnlock(PKI_REPOSITORIES_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
}

void DlgPkiAdmin::OnShowEntityCert(wxCommandEvent &event)
{
	size_t Index;
	long SelectedItem;
	
	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	Index = (size_t)m_listCtrl->GetItemData(SelectedItem);

	DlgCertProperties Dlg(this, m_entities[Index].get_certificate());
}

bool DlgPkiAdmin::ReloadConf()
{
	return LoadEntitiesList();
}

void DlgPkiAdmin::OnManageEntityACL(wxCommandEvent &event)
{
	X509Acl acls;
	size_t Index;
	long SelectedItem;
	PKI_CERT Cert;
	const char * ResourceName;

	switch(event.GetId())
	{
		case IDM_MANAGE_ACL:
			Cert = m_EntityCert;
			ResourceName = m_EntityCert.GetStringName();
			break;

		case IDM_MANAGE_ENTITY_ACL:
			SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
			if(SelectedItem == -1) return;
			Index = (size_t)m_listCtrl->GetItemData(SelectedItem);
			Cert = m_entities[Index].get_certificate();
			ResourceName = m_entities[Index].get_certificate().GetStringName();
			break;
		default:
			return;
	}
	
	m_DlgMsg->wShow(_("Loading Entity's ACLs..."));

	//We lock the resource
	if(!m_ClientPki->ResourceLock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}

	//We get our acl
	if(!m_ClientPki->GetEntityAcl(Cert, acls))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}
	m_DlgMsg->wHide();

	DlgManageACL Dlg(this, acls);
	if(!Dlg.IsOK())
	{
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	m_DlgMsg->wShow(_("Modifying Entity's ACLs..."));
	if(!m_ClientPki->SetEntityAcl(Cert, acls.get_aclEntries()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	if(!m_ClientPki->ResourceUnlock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
}

void DlgPkiAdmin::OnManageEntityMailConf(wxCommandEvent &event)
{
	EmailConf mail_conf;
	EmailConf pki_mail_conf;
	long SelectedItem;
	const char * ResourceName;
	PKI_CERT EntityCert;
	size_t Index;

	switch(event.GetId())
	{
		case IDM_MANAGE_MAIL_CONF:
			ResourceName = m_EntityCert.GetStringName();
			EntityCert = m_EntityCert;
			break;

		case IDM_MANAGE_ENTITY_MAIL_CONF:
			SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
			if(SelectedItem == -1) return;
			Index = (size_t)m_listCtrl->GetItemData(SelectedItem);
			ResourceName = m_entities[Index].get_certificate().GetStringName();
			EntityCert = m_entities[Index].get_certificate();
			break;
		default:
			return;
	}
	
	m_DlgMsg->wShow(_("Loading Entity's E-Mail Configuration..."));

	//We lock the resource
	if(!m_ClientPki->ResourceLock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}

	//We get then entity mail conf
	if(!m_ClientPki->GetEntityMailConf(EntityCert, mail_conf))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	// If the entity mail conf is empty we try to get the 
	// PKI mail conf as a default value
	if(!mail_conf.get_server().size() && event.GetId() != IDM_MANAGE_MAIL_CONF)
	{
		if(DisplayMessage(this, _("The email configuration is not set, do you want to load the PKI values (values are just loaded not set) ?"), wxYES_NO) == wxYES)
		{
			if(m_ClientPki->ResourceLock(m_EntityCert.GetStringName()))
			{
				if(m_ClientPki->GetEntityMailConf(m_EntityCert, pki_mail_conf))
				{
					mail_conf = pki_mail_conf;
				}
				m_ClientPki->ResourceUnlock(m_EntityCert.GetStringName());
			}
		}
	}
	m_DlgMsg->wHide();

	DlgMailConf Dlg(this, mail_conf, GetEmail(EntityCert));
	if(!Dlg.IsOK())
	{
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	m_DlgMsg->wShow(_("Setting Entity's E-Mail Configuration..."));
	if(!m_ClientPki->SetEntityMailConf(EntityCert, mail_conf))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	if(!m_ClientPki->ResourceUnlock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
}

const char * DlgPkiAdmin::GetEmail(const PKI_CERT & Cert)
{
	const char *EntityMail;
	int Pos;
	Pos = Cert.GetCertDN().SeekEntryName("emailAddress", -1);
	if(Pos == HASHTABLE_NOT_FOUND)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return NULL;
	}

	EntityMail = Cert.GetCertDN().Get(Pos);
	if(!EntityMail)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return NULL;
	}
	return EntityMail;
}

void DlgPkiAdmin::OnSendAdminMail(wxCommandEvent &event)
{
	DlgSendMail Dlg(this, m_ClientPki);
}

void DlgPkiAdmin::OnManageEntityAudits(wxCommandEvent &event)
{
	mVector<EntityAuditEntry> audits;
	EntityEntryInfo entity;
	long SelectedItem;
	mVector<unsigned long> Types;
	int EntityType;
	size_t Index;
	PKI_CERT Cert;
	const char * ResourceName;

	switch(event.GetId())
	{
		case IDM_MANAGE_AUDITS:
			Cert = m_EntityCert;
			ResourceName = m_EntityCert.GetStringName();
			EntityType = ENTITY_TYPE_PKI;
			break;
		case IDM_MANAGE_ENTITY_AUDITS:
			SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
			if(SelectedItem == -1) return;
			Index = (size_t)m_listCtrl->GetItemData(SelectedItem);
			Cert = m_entities[Index].get_certificate();
			ResourceName = m_entities[Index].get_certificate().GetStringName();
			EntityType = m_entities[Index].get_type();
			break;
		default:
			return;
	}

	m_DlgMsg->wShow(_("Loading Entity's Audits Configuration..."));

	//We lock the resource
	if(!m_ClientPki->ResourceLock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}

	switch(event.GetId())
	{
		case IDM_MANAGE_AUDITS:
			//We get the entity logs types
			if(!m_ClientPki->GetLogsType(Types))
			{
				m_DlgMsg->wHide();
				HandleError(m_ClientPki->GetError(), this);
				m_ClientPki->ResourceUnlock(ResourceName);
				return;
			}
			break;
		case IDM_MANAGE_ENTITY_AUDITS:
			//We get the entity logs types
			if(!m_ClientPki->GetEntityLogsType(EntityType, Types))
			{
				m_DlgMsg->wHide();
				HandleError(m_ClientPki->GetError(), this);
				m_ClientPki->ResourceUnlock(ResourceName);
				return;
			}
			break;
	}

	//We get our acl
	if(!m_ClientPki->GetEntityAudits(Cert, audits))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}
	m_DlgMsg->wHide();

	DlgManageAudits Dlg(this, Types, audits);
	if(!Dlg.IsOK())
	{
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	m_DlgMsg->wShow(_("Setting Entity's Audits Configuration..."));
	if(!m_ClientPki->SetEntityAudits(Cert, audits))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		m_ClientPki->ResourceUnlock(ResourceName);
		return;
	}

	if(!m_ClientPki->ResourceUnlock(ResourceName))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
}

void DlgPkiAdmin::OnManageEntities(wxCommandEvent &event)
{
	if(!m_ClientPki->ResourceLock(PKI_ENTITIES_LINKS_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	DlgManageEntities(this, m_EntityName.c_str(), m_ClientPki);
	if(!m_ClientPki->ResourceUnlock(PKI_ENTITIES_LINKS_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
}

void DlgPkiAdmin::OnManageGroups(wxCommandEvent &event)
{
	if(!m_ClientPki->ResourceLock(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	DlgManageGroups(this, m_ClientPki);
	if(!m_ClientPki->ResourceUnlock(PKI_ENTITIES_GROUPS_RESOURCE))
	{
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
}
