/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1Entity.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_CHOICE(GEN_PRIVATE_KEY) = {
	ASN1_EXP(GEN_PRIVATE_KEY, d.keylen, ASN1_INTEGER, GEN_PRIVATE_KEY_TYPE_KEYLEN),
	ASN1_EXP(GEN_PRIVATE_KEY, d.keyid, ASN1_UTF8STRING, GEN_PRIVATE_KEY_TYPE_ENGINE),
} ASN1_CHOICE_END(GEN_PRIVATE_KEY)
ASN1_SEQUENCE(ENTITY_ENTRY_INFO) = {
	ASN1_EXP(ENTITY_ENTRY_INFO, Name, ASN1_UTF8STRING, 0),
	ASN1_EXP(ENTITY_ENTRY_INFO, Type, ASN1_INTEGER, 1),
	ASN1_EXP(ENTITY_ENTRY_INFO, Loaded, ASN1_INTEGER, 2),
	ASN1_EXP_OPT(ENTITY_ENTRY_INFO, certificate, X509, 4),
}ASN1_SEQUENCE_END(ENTITY_ENTRY_INFO)
ASN1_SEQUENCE(PKI_CREATION_REQ) = {
	ASN1_SIMPLE(PKI_CREATION_REQ, Email, ASN1_UTF8STRING),
	ASN1_SIMPLE(PKI_CREATION_REQ, validity, ASN1_INTEGER),
	ASN1_SIMPLE(PKI_CREATION_REQ, AdminCreate, CREATE_PKI_USER_REQUEST),
	ASN1_SIMPLE(PKI_CREATION_REQ, dn, X509_NAME),
	ASN1_SIMPLE(PKI_CREATION_REQ, root_ca, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(PKI_CREATION_REQ, entities_ca, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(PKI_CREATION_REQ, users_ca, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(PKI_CREATION_REQ, ocsp_ca, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(PKI_CREATION_REQ, entity_key, GEN_PRIVATE_KEY),
}ASN1_SEQUENCE_END(PKI_CREATION_REQ)
ASN1_SEQUENCE(PUB_CREATION_REQ) = {
	ASN1_SIMPLE(PUB_CREATION_REQ, entity_key, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(PUB_CREATION_REQ, ocsp_key, GEN_PRIVATE_KEY),
}ASN1_SEQUENCE_END(PUB_CREATION_REQ)
ASN1_CHOICE(ENTITY_CREATION_DATAS) = {
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_RA),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_CA),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_REPOSITORY),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.pub_create, PUB_CREATION_REQ, ENTITY_TYPE_PUBLICATION),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_KEY_STORE),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.pki_create, PKI_CREATION_REQ, ENTITY_TYPE_PKI),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_BACKUP),
	ASN1_EXP(ENTITY_CREATION_DATAS, d.entity_key, GEN_PRIVATE_KEY, ENTITY_TYPE_EE),
} ASN1_CHOICE_END(ENTITY_CREATION_DATAS)
ASN1_SEQUENCE(ENTITY_CREATION_REQ) = {
	ASN1_SIMPLE(ENTITY_CREATION_REQ, Name, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_CREATION_REQ, Datas, ENTITY_CREATION_DATAS),
}ASN1_SEQUENCE_END(ENTITY_CREATION_REQ)
ASN1_SEQUENCE(PUB_ENTITY_CREATION_RESP) = {
	ASN1_SIMPLE(PUB_ENTITY_CREATION_RESP, entity_key, X509_PUBKEY),
	ASN1_SIMPLE(PUB_ENTITY_CREATION_RESP, ocsp_key, X509_PUBKEY),
}ASN1_SEQUENCE_END(PUB_ENTITY_CREATION_RESP)
ASN1_CHOICE(ENTITY_CREATION_RESP) = {
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_RA),
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_CA),
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_REPOSITORY),
	ASN1_EXP(ENTITY_CREATION_RESP, d.pub_create, PUB_ENTITY_CREATION_RESP, ENTITY_TYPE_PUBLICATION),
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_KEY_STORE),
	ASN1_EXP(ENTITY_CREATION_RESP, d.pki_create, CREATE_PKI_USER_RESPONSE, ENTITY_TYPE_PKI),
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_BACKUP),
	ASN1_EXP(ENTITY_CREATION_RESP, d.entity_pub_key, X509_PUBKEY, ENTITY_TYPE_EE),
} ASN1_CHOICE_END(ENTITY_CREATION_RESP)
ASN1_SEQUENCE(ENTITY_SIGNATURE_REQ_PUB) = {
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ_PUB, EntityPubKey, X509_PUBKEY),
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ_PUB, OcspPubKey, X509_PUBKEY),
}ASN1_SEQUENCE_END(ENTITY_SIGNATURE_REQ_PUB)
ASN1_SEQUENCE(ENTITY_SIGNATURE_REQ_REP) = {
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ_REP, EntityPubKey, X509_PUBKEY),
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ_REP, Address, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ_REP, Port, ASN1_INTEGER),
}ASN1_SEQUENCE_END(ENTITY_SIGNATURE_REQ_REP)
ASN1_CHOICE(ENTITY_SIGNATURE_REQ_BODY) = {
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.EntityPubKey, X509_PUBKEY, ENTITY_TYPE_RA),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.EntityPubKey, X509_PUBKEY, ENTITY_TYPE_CA),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.sign_rep, ENTITY_SIGNATURE_REQ_REP, ENTITY_TYPE_REPOSITORY),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.sign_pub, ENTITY_SIGNATURE_REQ_PUB, ENTITY_TYPE_PUBLICATION),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.EntityPubKey, X509_PUBKEY, ENTITY_TYPE_KEY_STORE),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.other, ASN1_NULL, ENTITY_TYPE_PKI),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.EntityPubKey, X509_PUBKEY, ENTITY_TYPE_BACKUP),
	ASN1_EXP(ENTITY_SIGNATURE_REQ_BODY, d.EntityPubKey, X509_PUBKEY, ENTITY_TYPE_EE),
} ASN1_CHOICE_END(ENTITY_SIGNATURE_REQ_BODY)
ASN1_SEQUENCE(ENTITY_SIGNATURE_REQ) = {
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ, Name, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ, Email, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_SIGNATURE_REQ, body, ENTITY_SIGNATURE_REQ_BODY),
}ASN1_SEQUENCE_END(ENTITY_SIGNATURE_REQ)
ASN1_SEQUENCE(ENTITY_SIGNATURE_RESP_PUB) = {
	ASN1_SIMPLE(ENTITY_SIGNATURE_RESP_PUB, EntityCert, X509),
	ASN1_SIMPLE(ENTITY_SIGNATURE_RESP_PUB, OcspCert, X509),
}ASN1_SEQUENCE_END(ENTITY_SIGNATURE_RESP_PUB)
ASN1_CHOICE(ENTITY_SIGNATURE_RESP_BODY) = {
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_RA),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_CA),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_REPOSITORY),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.sign_pub, ENTITY_SIGNATURE_RESP_PUB, ENTITY_TYPE_PUBLICATION),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_KEY_STORE),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.other, ASN1_NULL, ENTITY_TYPE_PKI),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_BACKUP),
	ASN1_EXP(ENTITY_SIGNATURE_RESP_BODY, d.EntityCert, X509, ENTITY_TYPE_EE),
} ASN1_CHOICE_END(ENTITY_SIGNATURE_RESP_BODY)
ASN1_SEQUENCE(ENTITY_SIGNATURE_RESP) = {
	ASN1_SIMPLE(ENTITY_SIGNATURE_RESP, body, ENTITY_SIGNATURE_RESP_BODY),
	ASN1_SIMPLE(ENTITY_SIGNATURE_RESP, conf, ENTITY_CONF_CRYPTED),
	ASN1_SIMPLE(ENTITY_SIGNATURE_RESP, cas, INTERNAL_PKI_CA),
}ASN1_SEQUENCE_END(ENTITY_SIGNATURE_RESP)
ASN1_SEQUENCE(ENTITY_INIT_REQ) = {
	ASN1_SIMPLE(ENTITY_INIT_REQ, name, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_INIT_REQ, sign_resp, ENTITY_SIGNATURE_RESP),
}ASN1_SEQUENCE_END(ENTITY_INIT_REQ)
ASN1_SEQUENCE(ENTITY_LINK_INFO) = {
	ASN1_SIMPLE(ENTITY_LINK_INFO, Name, ASN1_UTF8STRING),
	ASN1_SIMPLE(ENTITY_LINK_INFO, Type, ASN1_INTEGER),
	ASN1_SIMPLE(ENTITY_LINK_INFO, flags, ASN1_BIT_STRING),
}ASN1_SEQUENCE_END(ENTITY_LINK_INFO)
ASN1_SEQUENCE(ENTITY_LINKS) = {
	ASN1_SIMPLE(ENTITY_LINKS, src, ENTITY_LINK_INFO),
	ASN1_SEQUENCE_OF(ENTITY_LINKS, dsts, ENTITY_LINK_INFO),
}ASN1_SEQUENCE_END(ENTITY_LINKS)
ASN1_SEQUENCE(REQ_CREATE_ROOT_CA) = {
	ASN1_SIMPLE(REQ_CREATE_ROOT_CA, privKey, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(REQ_CREATE_ROOT_CA, dn, X509_NAME),
	ASN1_SIMPLE(REQ_CREATE_ROOT_CA, validity, ASN1_INTEGER),
	ASN1_SEQUENCE_OF(REQ_CREATE_ROOT_CA, extensions, EXTENSION_VALUE),
}ASN1_SEQUENCE_END(REQ_CREATE_ROOT_CA)
ASN1_SEQUENCE(REQ_CREATE_CHILD_CA) = {
	ASN1_SIMPLE(REQ_CREATE_CHILD_CA, privKey, GEN_PRIVATE_KEY),
	ASN1_SIMPLE(REQ_CREATE_CHILD_CA, dn, X509_NAME),
}ASN1_SEQUENCE_END(REQ_CREATE_CHILD_CA)
GenPrivateKey GenPrivateKey::EmptyInstance;
bool GenPrivateKey::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int GenPrivateKey::get_type() const
{
	 return m_type;
}

bool GenPrivateKey::set_keyid(const mString & c_keyid)
{
	if(m_type != 1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_keyid) = c_keyid;
	m_isOk=true;
	return true;
}

const mString & GenPrivateKey::get_keyid() const
{
	if((m_type != 1) || !m_keyid)
	{
		return mString::EmptyInstance;
	}
	return (*m_keyid);
}

mString & GenPrivateKey::get_keyid()
{
	if((m_type != 1) || !m_keyid)
	{
		return mString::EmptyInstance;
	}
	return (*m_keyid);
}

bool GenPrivateKey::set_keylen(unsigned long c_keylen)
{
	if(m_type != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_keylen) = c_keylen;
	m_isOk=true;
	return true;
}

unsigned long GenPrivateKey::get_keylen() const
{
	if((m_type != 0) || !m_keylen)
	{
		return 0;
	}
	return (*m_keylen);
}

bool GenPrivateKey::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 1:
			m_keyid = new mString();
			if(!m_keyid)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 0:
			m_keylen = (unsigned long*)malloc(sizeof(unsigned long));
			if(!m_keylen)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

GenPrivateKey::GenPrivateKey():NewPKIObject()
{
	resetAll();
}

GenPrivateKey::GenPrivateKey(const GenPrivateKey & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

GenPrivateKey::~GenPrivateKey()
{
	Clear();
}

void GenPrivateKey::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void GenPrivateKey::freeAll()
{
	if(m_keyid)
	{
		delete m_keyid;
	}
	if(m_keylen)
	{
		free(m_keylen);
	}
}

void GenPrivateKey::resetAll()
{
	m_type = -1;
	m_keyid = NULL;
	m_keylen = NULL;
}

bool GenPrivateKey::load_Datas(const GEN_PRIVATE_KEY * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 1:
			if(Datas->d.keyid)
			{
				(*m_keyid) = Datas->d.keyid;
			}
			break;
		case 0:
			if(Datas->d.keylen)
			{
				(*m_keylen) = ASN1_INTEGER_GET(Datas->d.keylen);
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool GenPrivateKey::give_Datas(GEN_PRIVATE_KEY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (GEN_PRIVATE_KEY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 1:
			if(!((*Datas)->d.keyid = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_keyid).c_ASN1_UTF8STRING(&(*Datas)->d.keyid))
			{
				ASN1_UTF8STRING_free((*Datas)->d.keyid);
				(*Datas)->d.keyid = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 0:
			if(!((*Datas)->d.keylen = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(ASN1_INTEGER_set((*Datas)->d.keylen, (*m_keylen)) <= 0)
			{
				ASN1_INTEGER_free((*Datas)->d.keylen);
				(*Datas)->d.keylen = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
				return false;
			}
			break;
	}
	return true;
}

bool GenPrivateKey::operator=(const GenPrivateKey & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 1:
			if(!other.m_keyid)
				break;
			(*m_keyid) = *(other.m_keyid);
			break;
		case 0:
			if(!other.m_keylen)
				break;
			(*m_keylen) = *(other.m_keylen);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * GenPrivateKey::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(GEN_PRIVATE_KEY);
}
EntityEntryInfo EntityEntryInfo::EmptyInstance;
bool EntityEntryInfo::set_loaded(unsigned long c_loaded)
{
	m_loaded = c_loaded;
	return true;
}

unsigned long EntityEntryInfo::get_loaded() const
{
	return m_loaded;
}

bool EntityEntryInfo::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & EntityEntryInfo::get_name() const
{
	return m_name;
}

mString & EntityEntryInfo::get_name()
{
	return m_name;
}

bool EntityEntryInfo::set_type(unsigned long c_type)
{
	m_type = c_type;
	return true;
}

unsigned long EntityEntryInfo::get_type() const
{
	return m_type;
}

bool EntityEntryInfo::set_certificate(const PKI_CERT & c_certificate)
{
	m_certificate = c_certificate;
	return true;
}

const PKI_CERT & EntityEntryInfo::get_certificate() const
{
	return m_certificate;
}

PKI_CERT & EntityEntryInfo::get_certificate()
{
	return m_certificate;
}

EntityEntryInfo::EntityEntryInfo():NewPKIObject()
{
	resetAll();
}

EntityEntryInfo::EntityEntryInfo(const EntityEntryInfo & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityEntryInfo::~EntityEntryInfo()
{
	Clear();
}

void EntityEntryInfo::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityEntryInfo::freeAll()
{
}

void EntityEntryInfo::resetAll()
{
	m_loaded = 0;
	m_name = "";
	m_type = 0;
	m_certificate.Clear();
}

bool EntityEntryInfo::load_Datas(const ENTITY_ENTRY_INFO * Datas)
{
	Clear();
	if(Datas->Loaded)
	{
		m_loaded = ASN1_INTEGER_GET(Datas->Loaded);
	}
	if(Datas->Name)
	{
		m_name = Datas->Name;
	}
	if(Datas->Type)
	{
		m_type = ASN1_INTEGER_GET(Datas->Type);
	}
	if(Datas->certificate)
	{
		if(!m_certificate.load_Datas(Datas->certificate))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntityEntryInfo::give_Datas(ENTITY_ENTRY_INFO ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_ENTRY_INFO*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Loaded && !((*Datas)->Loaded = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Loaded, m_loaded) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Loaded);
		(*Datas)->Loaded = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->Name && !((*Datas)->Name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->Name))
	{
		ASN1_UTF8STRING_free((*Datas)->Name);
		(*Datas)->Name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Type && !((*Datas)->Type = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Type, m_type) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Type);
		(*Datas)->Type = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(m_certificate)
	{
		if(!(*Datas)->certificate && !((*Datas)->certificate = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
		if(!m_certificate.give_Datas(&(*Datas)->certificate))
		{
			ASN1_item_free((ASN1_VALUE*)(*Datas)->certificate, ASN1_ITEM_rptr(X509));
			(*Datas)->certificate = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	else
	{
		if((*Datas)->certificate)
		{
			ASN1_item_free((ASN1_VALUE*)(*Datas)->certificate, ASN1_ITEM_rptr(X509));
			(*Datas)->certificate = NULL;
		}
	}
	return true;
}

bool EntityEntryInfo::operator=(const EntityEntryInfo & other)
{
	Clear();
	m_loaded = other.m_loaded;
	m_name = other.m_name;
	m_type = other.m_type;
	m_certificate = other.m_certificate;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityEntryInfo::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_ENTRY_INFO);
}
PkiCreationReq PkiCreationReq::EmptyInstance;
bool PkiCreationReq::set_admincreate(const CreatePkiUserRequest & c_admincreate)
{
	m_admincreate = c_admincreate;
	return true;
}

const CreatePkiUserRequest & PkiCreationReq::get_admincreate() const
{
	return m_admincreate;
}

CreatePkiUserRequest & PkiCreationReq::get_admincreate()
{
	return m_admincreate;
}

bool PkiCreationReq::set_email(const mString & c_email)
{
	m_email = c_email;
	return true;
}

const mString & PkiCreationReq::get_email() const
{
	return m_email;
}

mString & PkiCreationReq::get_email()
{
	return m_email;
}

bool PkiCreationReq::set_dn(const X509_NAME * c_dn)
{
	if(m_dn)
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)c_dn);
	if(!m_dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_NAME * PkiCreationReq::get_dn() const
{
	if(!m_dn)
		((PkiCreationReq*)this)->m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

X509_NAME * PkiCreationReq::get_dn()
{
	if(!m_dn)
		m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

bool PkiCreationReq::set_entitiesCa(const GenPrivateKey & c_entitiesCa)
{
	m_entitiesCa = c_entitiesCa;
	return true;
}

const GenPrivateKey & PkiCreationReq::get_entitiesCa() const
{
	return m_entitiesCa;
}

GenPrivateKey & PkiCreationReq::get_entitiesCa()
{
	return m_entitiesCa;
}

bool PkiCreationReq::set_entityKey(const GenPrivateKey & c_entityKey)
{
	m_entityKey = c_entityKey;
	return true;
}

const GenPrivateKey & PkiCreationReq::get_entityKey() const
{
	return m_entityKey;
}

GenPrivateKey & PkiCreationReq::get_entityKey()
{
	return m_entityKey;
}

bool PkiCreationReq::set_ocspCa(const GenPrivateKey & c_ocspCa)
{
	m_ocspCa = c_ocspCa;
	return true;
}

const GenPrivateKey & PkiCreationReq::get_ocspCa() const
{
	return m_ocspCa;
}

GenPrivateKey & PkiCreationReq::get_ocspCa()
{
	return m_ocspCa;
}

bool PkiCreationReq::set_rootCa(const GenPrivateKey & c_rootCa)
{
	m_rootCa = c_rootCa;
	return true;
}

const GenPrivateKey & PkiCreationReq::get_rootCa() const
{
	return m_rootCa;
}

GenPrivateKey & PkiCreationReq::get_rootCa()
{
	return m_rootCa;
}

bool PkiCreationReq::set_usersCa(const GenPrivateKey & c_usersCa)
{
	m_usersCa = c_usersCa;
	return true;
}

const GenPrivateKey & PkiCreationReq::get_usersCa() const
{
	return m_usersCa;
}

GenPrivateKey & PkiCreationReq::get_usersCa()
{
	return m_usersCa;
}

bool PkiCreationReq::set_validity(unsigned long c_validity)
{
	m_validity = c_validity;
	return true;
}

unsigned long PkiCreationReq::get_validity() const
{
	return m_validity;
}

PkiCreationReq::PkiCreationReq():NewPKIObject()
{
	resetAll();
}

PkiCreationReq::PkiCreationReq(const PkiCreationReq & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

PkiCreationReq::~PkiCreationReq()
{
	Clear();
}

void PkiCreationReq::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PkiCreationReq::freeAll()
{
	if(m_dn)
	{
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	}
}

void PkiCreationReq::resetAll()
{
	m_admincreate.Clear();
	m_email = "";
	m_dn = NULL;
	m_entitiesCa.Clear();
	m_entityKey.Clear();
	m_ocspCa.Clear();
	m_rootCa.Clear();
	m_usersCa.Clear();
	m_validity = 0;
}

bool PkiCreationReq::load_Datas(const PKI_CREATION_REQ * Datas)
{
	Clear();
	if(Datas->AdminCreate)
	{
		if(!m_admincreate.load_Datas(Datas->AdminCreate))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->Email)
	{
		m_email = Datas->Email;
	}
	if(Datas->dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), Datas->dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->entities_ca)
	{
		if(!m_entitiesCa.load_Datas(Datas->entities_ca))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->entity_key)
	{
		if(!m_entityKey.load_Datas(Datas->entity_key))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->ocsp_ca)
	{
		if(!m_ocspCa.load_Datas(Datas->ocsp_ca))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->root_ca)
	{
		if(!m_rootCa.load_Datas(Datas->root_ca))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->users_ca)
	{
		if(!m_usersCa.load_Datas(Datas->users_ca))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->validity)
	{
		m_validity = ASN1_INTEGER_GET(Datas->validity);
	}
	m_isOk=true;
	return true;
}

bool PkiCreationReq::give_Datas(PKI_CREATION_REQ ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PKI_CREATION_REQ*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->AdminCreate && !((*Datas)->AdminCreate = (CREATE_PKI_USER_REQUEST*)ASN1_item_new(ASN1_ITEM_rptr(CREATE_PKI_USER_REQUEST))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_admincreate.give_Datas(&(*Datas)->AdminCreate))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->AdminCreate, ASN1_ITEM_rptr(CREATE_PKI_USER_REQUEST));
		(*Datas)->AdminCreate = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Email && !((*Datas)->Email = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_email.c_ASN1_UTF8STRING(&(*Datas)->Email))
	{
		ASN1_UTF8STRING_free((*Datas)->Email);
		(*Datas)->Email = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(m_dn)
	{
		if((*Datas)->dn)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->dn, ASN1_ITEM_rptr(X509_NAME));
		if(!((*Datas)->dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (ASN1_VALUE*)m_dn)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->dn)
		{
			(*Datas)->dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
			if(!(*Datas)->dn)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->entities_ca && !((*Datas)->entities_ca = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entitiesCa.give_Datas(&(*Datas)->entities_ca))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entities_ca, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->entities_ca = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->entity_key && !((*Datas)->entity_key = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityKey.give_Datas(&(*Datas)->entity_key))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_key, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->entity_key = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->ocsp_ca && !((*Datas)->ocsp_ca = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_ocspCa.give_Datas(&(*Datas)->ocsp_ca))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->ocsp_ca, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->ocsp_ca = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->root_ca && !((*Datas)->root_ca = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_rootCa.give_Datas(&(*Datas)->root_ca))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->root_ca, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->root_ca = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->users_ca && !((*Datas)->users_ca = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_usersCa.give_Datas(&(*Datas)->users_ca))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->users_ca, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->users_ca = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->validity && !((*Datas)->validity = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->validity, m_validity) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->validity);
		(*Datas)->validity = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool PkiCreationReq::operator=(const PkiCreationReq & other)
{
	Clear();
	m_admincreate = other.m_admincreate;
	m_email = other.m_email;
	if(other.m_dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)other.m_dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_entitiesCa = other.m_entitiesCa;
	m_entityKey = other.m_entityKey;
	m_ocspCa = other.m_ocspCa;
	m_rootCa = other.m_rootCa;
	m_usersCa = other.m_usersCa;
	m_validity = other.m_validity;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PkiCreationReq::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PKI_CREATION_REQ);
}
PubCreationReq PubCreationReq::EmptyInstance;
bool PubCreationReq::set_entityKey(const GenPrivateKey & c_entityKey)
{
	m_entityKey = c_entityKey;
	return true;
}

const GenPrivateKey & PubCreationReq::get_entityKey() const
{
	return m_entityKey;
}

GenPrivateKey & PubCreationReq::get_entityKey()
{
	return m_entityKey;
}

bool PubCreationReq::set_ocspKey(const GenPrivateKey & c_ocspKey)
{
	m_ocspKey = c_ocspKey;
	return true;
}

const GenPrivateKey & PubCreationReq::get_ocspKey() const
{
	return m_ocspKey;
}

GenPrivateKey & PubCreationReq::get_ocspKey()
{
	return m_ocspKey;
}

PubCreationReq::PubCreationReq():NewPKIObject()
{
	resetAll();
}

PubCreationReq::PubCreationReq(const PubCreationReq & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

PubCreationReq::~PubCreationReq()
{
	Clear();
}

void PubCreationReq::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PubCreationReq::freeAll()
{
}

void PubCreationReq::resetAll()
{
	m_entityKey.Clear();
	m_ocspKey.Clear();
}

bool PubCreationReq::load_Datas(const PUB_CREATION_REQ * Datas)
{
	Clear();
	if(Datas->entity_key)
	{
		if(!m_entityKey.load_Datas(Datas->entity_key))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->ocsp_key)
	{
		if(!m_ocspKey.load_Datas(Datas->ocsp_key))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool PubCreationReq::give_Datas(PUB_CREATION_REQ ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PUB_CREATION_REQ*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_key && !((*Datas)->entity_key = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityKey.give_Datas(&(*Datas)->entity_key))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_key, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->entity_key = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->ocsp_key && !((*Datas)->ocsp_key = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_ocspKey.give_Datas(&(*Datas)->ocsp_key))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->ocsp_key, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->ocsp_key = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool PubCreationReq::operator=(const PubCreationReq & other)
{
	Clear();
	m_entityKey = other.m_entityKey;
	m_ocspKey = other.m_ocspKey;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PubCreationReq::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PUB_CREATION_REQ);
}
EntityCreationDatas EntityCreationDatas::EmptyInstance;
bool EntityCreationDatas::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int EntityCreationDatas::get_type() const
{
	 return m_type;
}

bool EntityCreationDatas::set_entityKey(const GenPrivateKey & c_entityKey)
{
	if(m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_entityKey) = c_entityKey;
	m_isOk=true;
	return true;
}

const GenPrivateKey & EntityCreationDatas::get_entityKey() const
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entityKey)
	{
		return GenPrivateKey::EmptyInstance;
	}
	return (*m_entityKey);
}

GenPrivateKey & EntityCreationDatas::get_entityKey()
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entityKey)
	{
		return GenPrivateKey::EmptyInstance;
	}
	return (*m_entityKey);
}

bool EntityCreationDatas::set_pkiCreate(const PkiCreationReq & c_pkiCreate)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pkiCreate) = c_pkiCreate;
	m_isOk=true;
	return true;
}

const PkiCreationReq & EntityCreationDatas::get_pkiCreate() const
{
	if((m_type != 5) || !m_pkiCreate)
	{
		return PkiCreationReq::EmptyInstance;
	}
	return (*m_pkiCreate);
}

PkiCreationReq & EntityCreationDatas::get_pkiCreate()
{
	if((m_type != 5) || !m_pkiCreate)
	{
		return PkiCreationReq::EmptyInstance;
	}
	return (*m_pkiCreate);
}

bool EntityCreationDatas::set_pubCreate(const PubCreationReq & c_pubCreate)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pubCreate) = c_pubCreate;
	m_isOk=true;
	return true;
}

const PubCreationReq & EntityCreationDatas::get_pubCreate() const
{
	if((m_type != 3) || !m_pubCreate)
	{
		return PubCreationReq::EmptyInstance;
	}
	return (*m_pubCreate);
}

PubCreationReq & EntityCreationDatas::get_pubCreate()
{
	if((m_type != 3) || !m_pubCreate)
	{
		return PubCreationReq::EmptyInstance;
	}
	return (*m_pubCreate);
}

bool EntityCreationDatas::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			m_entityKey = new GenPrivateKey();
			if(!m_entityKey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_pkiCreate = new PkiCreationReq();
			if(!m_pkiCreate)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_pubCreate = new PubCreationReq();
			if(!m_pubCreate)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

EntityCreationDatas::EntityCreationDatas():NewPKIObject()
{
	resetAll();
}

EntityCreationDatas::EntityCreationDatas(const EntityCreationDatas & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityCreationDatas::~EntityCreationDatas()
{
	Clear();
}

void EntityCreationDatas::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityCreationDatas::freeAll()
{
	if(m_entityKey)
	{
		delete m_entityKey;
	}
	if(m_pkiCreate)
	{
		delete m_pkiCreate;
	}
	if(m_pubCreate)
	{
		delete m_pubCreate;
	}
}

void EntityCreationDatas::resetAll()
{
	m_type = -1;
	m_entityKey = NULL;
	m_pkiCreate = NULL;
	m_pubCreate = NULL;
}

bool EntityCreationDatas::load_Datas(const ENTITY_CREATION_DATAS * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(Datas->d.entity_key)
			{
				if(!(*m_entityKey).load_Datas(Datas->d.entity_key))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.pki_create)
			{
				if(!(*m_pkiCreate).load_Datas(Datas->d.pki_create))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.pub_create)
			{
				if(!(*m_pubCreate).load_Datas(Datas->d.pub_create))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool EntityCreationDatas::give_Datas(ENTITY_CREATION_DATAS ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_CREATION_DATAS*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(!((*Datas)->d.entity_key = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_entityKey).give_Datas(&(*Datas)->d.entity_key))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.entity_key, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
				(*Datas)->d.entity_key = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 5:
			if(!((*Datas)->d.pki_create = (PKI_CREATION_REQ*)ASN1_item_new(ASN1_ITEM_rptr(PKI_CREATION_REQ))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pkiCreate).give_Datas(&(*Datas)->d.pki_create))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pki_create, ASN1_ITEM_rptr(PKI_CREATION_REQ));
				(*Datas)->d.pki_create = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 3:
			if(!((*Datas)->d.pub_create = (PUB_CREATION_REQ*)ASN1_item_new(ASN1_ITEM_rptr(PUB_CREATION_REQ))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pubCreate).give_Datas(&(*Datas)->d.pub_create))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pub_create, ASN1_ITEM_rptr(PUB_CREATION_REQ));
				(*Datas)->d.pub_create = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool EntityCreationDatas::operator=(const EntityCreationDatas & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(!other.m_entityKey)
				break;
			(*m_entityKey) = *(other.m_entityKey);
			break;
		case 5:
			if(!other.m_pkiCreate)
				break;
			(*m_pkiCreate) = *(other.m_pkiCreate);
			break;
		case 3:
			if(!other.m_pubCreate)
				break;
			(*m_pubCreate) = *(other.m_pubCreate);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityCreationDatas::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_CREATION_DATAS);
}
EntityCreationReq EntityCreationReq::EmptyInstance;
bool EntityCreationReq::set_datas(const EntityCreationDatas & c_datas)
{
	m_datas = c_datas;
	return true;
}

const EntityCreationDatas & EntityCreationReq::get_datas() const
{
	return m_datas;
}

EntityCreationDatas & EntityCreationReq::get_datas()
{
	return m_datas;
}

bool EntityCreationReq::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & EntityCreationReq::get_name() const
{
	return m_name;
}

mString & EntityCreationReq::get_name()
{
	return m_name;
}

EntityCreationReq::EntityCreationReq():NewPKIObject()
{
	resetAll();
}

EntityCreationReq::EntityCreationReq(const EntityCreationReq & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityCreationReq::~EntityCreationReq()
{
	Clear();
}

void EntityCreationReq::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityCreationReq::freeAll()
{
}

void EntityCreationReq::resetAll()
{
	m_datas.Clear();
	m_name = "";
}

bool EntityCreationReq::load_Datas(const ENTITY_CREATION_REQ * Datas)
{
	Clear();
	if(Datas->Datas)
	{
		if(!m_datas.load_Datas(Datas->Datas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->Name)
	{
		m_name = Datas->Name;
	}
	m_isOk=true;
	return true;
}

bool EntityCreationReq::give_Datas(ENTITY_CREATION_REQ ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_CREATION_REQ*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Datas && !((*Datas)->Datas = (ENTITY_CREATION_DATAS*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CREATION_DATAS))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_datas.give_Datas(&(*Datas)->Datas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->Datas, ASN1_ITEM_rptr(ENTITY_CREATION_DATAS));
		(*Datas)->Datas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Name && !((*Datas)->Name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->Name))
	{
		ASN1_UTF8STRING_free((*Datas)->Name);
		(*Datas)->Name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntityCreationReq::operator=(const EntityCreationReq & other)
{
	Clear();
	m_datas = other.m_datas;
	m_name = other.m_name;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityCreationReq::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_CREATION_REQ);
}
PubEntityCreationResp PubEntityCreationResp::EmptyInstance;
bool PubEntityCreationResp::set_entityKey(const X509_PUBKEY * c_entityKey)
{
	if(m_entityKey)
		ASN1_item_free((ASN1_VALUE*)m_entityKey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_entityKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_entityKey);
	if(!m_entityKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * PubEntityCreationResp::get_entityKey() const
{
	if(!m_entityKey)
		((PubEntityCreationResp*)this)->m_entityKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entityKey;
}

X509_PUBKEY * PubEntityCreationResp::get_entityKey()
{
	if(!m_entityKey)
		m_entityKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entityKey;
}

bool PubEntityCreationResp::set_ocspKey(const X509_PUBKEY * c_ocspKey)
{
	if(m_ocspKey)
		ASN1_item_free((ASN1_VALUE*)m_ocspKey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_ocspKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_ocspKey);
	if(!m_ocspKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * PubEntityCreationResp::get_ocspKey() const
{
	if(!m_ocspKey)
		((PubEntityCreationResp*)this)->m_ocspKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_ocspKey;
}

X509_PUBKEY * PubEntityCreationResp::get_ocspKey()
{
	if(!m_ocspKey)
		m_ocspKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_ocspKey;
}

PubEntityCreationResp::PubEntityCreationResp():NewPKIObject()
{
	resetAll();
}

PubEntityCreationResp::PubEntityCreationResp(const PubEntityCreationResp & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

PubEntityCreationResp::~PubEntityCreationResp()
{
	Clear();
}

void PubEntityCreationResp::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PubEntityCreationResp::freeAll()
{
	if(m_entityKey)
	{
		ASN1_item_free((ASN1_VALUE*)m_entityKey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
	if(m_ocspKey)
	{
		ASN1_item_free((ASN1_VALUE*)m_ocspKey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
}

void PubEntityCreationResp::resetAll()
{
	m_entityKey = NULL;
	m_ocspKey = NULL;
}

bool PubEntityCreationResp::load_Datas(const PUB_ENTITY_CREATION_RESP * Datas)
{
	Clear();
	if(Datas->entity_key)
	{
		if(m_entityKey)
			ASN1_item_free((ASN1_VALUE*)m_entityKey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entityKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->entity_key);
		if(!m_entityKey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->ocsp_key)
	{
		if(m_ocspKey)
			ASN1_item_free((ASN1_VALUE*)m_ocspKey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_ocspKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->ocsp_key);
		if(!m_ocspKey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool PubEntityCreationResp::give_Datas(PUB_ENTITY_CREATION_RESP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PUB_ENTITY_CREATION_RESP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(m_entityKey)
	{
		if((*Datas)->entity_key)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_key, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->entity_key = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_entityKey)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->entity_key)
		{
			(*Datas)->entity_key = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->entity_key)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(m_ocspKey)
	{
		if((*Datas)->ocsp_key)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->ocsp_key, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->ocsp_key = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_ocspKey)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->ocsp_key)
		{
			(*Datas)->ocsp_key = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->ocsp_key)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	return true;
}

bool PubEntityCreationResp::operator=(const PubEntityCreationResp & other)
{
	Clear();
	if(other.m_entityKey)
	{
		if(m_entityKey)
			ASN1_item_free((ASN1_VALUE*)m_entityKey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entityKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_entityKey);
		if(!m_entityKey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(other.m_ocspKey)
	{
		if(m_ocspKey)
			ASN1_item_free((ASN1_VALUE*)m_ocspKey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_ocspKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_ocspKey);
		if(!m_ocspKey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PubEntityCreationResp::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PUB_ENTITY_CREATION_RESP);
}
EntityCreationResp EntityCreationResp::EmptyInstance;
bool EntityCreationResp::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int EntityCreationResp::get_type() const
{
	 return m_type;
}

bool EntityCreationResp::set_entityPubKey(const X509_PUBKEY * c_entityPubKey)
{
	if(m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_entityPubKey)
		ASN1_item_free((ASN1_VALUE*)m_entityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_entityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_entityPubKey);
	if(!m_entityPubKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const X509_PUBKEY * EntityCreationResp::get_entityPubKey() const
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entityPubKey)
	{
		return NULL;
	}
	if(!m_entityPubKey)
		((EntityCreationResp*)this)->m_entityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entityPubKey;
}

X509_PUBKEY * EntityCreationResp::get_entityPubKey()
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entityPubKey)
	{
		return NULL;
	}
	if(!m_entityPubKey)
		m_entityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entityPubKey;
}

bool EntityCreationResp::set_pkiCreate(const CreatePkiUserResponse & c_pkiCreate)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pkiCreate) = c_pkiCreate;
	m_isOk=true;
	return true;
}

const CreatePkiUserResponse & EntityCreationResp::get_pkiCreate() const
{
	if((m_type != 5) || !m_pkiCreate)
	{
		return CreatePkiUserResponse::EmptyInstance;
	}
	return (*m_pkiCreate);
}

CreatePkiUserResponse & EntityCreationResp::get_pkiCreate()
{
	if((m_type != 5) || !m_pkiCreate)
	{
		return CreatePkiUserResponse::EmptyInstance;
	}
	return (*m_pkiCreate);
}

bool EntityCreationResp::set_pubCreate(const PubEntityCreationResp & c_pubCreate)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pubCreate) = c_pubCreate;
	m_isOk=true;
	return true;
}

const PubEntityCreationResp & EntityCreationResp::get_pubCreate() const
{
	if((m_type != 3) || !m_pubCreate)
	{
		return PubEntityCreationResp::EmptyInstance;
	}
	return (*m_pubCreate);
}

PubEntityCreationResp & EntityCreationResp::get_pubCreate()
{
	if((m_type != 3) || !m_pubCreate)
	{
		return PubEntityCreationResp::EmptyInstance;
	}
	return (*m_pubCreate);
}

bool EntityCreationResp::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			m_entityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!m_entityPubKey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_pkiCreate = new CreatePkiUserResponse();
			if(!m_pkiCreate)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_pubCreate = new PubEntityCreationResp();
			if(!m_pubCreate)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

EntityCreationResp::EntityCreationResp():NewPKIObject()
{
	resetAll();
}

EntityCreationResp::EntityCreationResp(const EntityCreationResp & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityCreationResp::~EntityCreationResp()
{
	Clear();
}

void EntityCreationResp::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityCreationResp::freeAll()
{
	if(m_entityPubKey)
	{
		ASN1_item_free((ASN1_VALUE*)m_entityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
	if(m_pkiCreate)
	{
		delete m_pkiCreate;
	}
	if(m_pubCreate)
	{
		delete m_pubCreate;
	}
}

void EntityCreationResp::resetAll()
{
	m_type = -1;
	m_entityPubKey = NULL;
	m_pkiCreate = NULL;
	m_pubCreate = NULL;
}

bool EntityCreationResp::load_Datas(const ENTITY_CREATION_RESP * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(Datas->d.entity_pub_key)
			{
				if(m_entityPubKey)
					ASN1_item_free((ASN1_VALUE*)m_entityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_entityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->d.entity_pub_key);
				if(!m_entityPubKey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.pki_create)
			{
				if(!(*m_pkiCreate).load_Datas(Datas->d.pki_create))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.pub_create)
			{
				if(!(*m_pubCreate).load_Datas(Datas->d.pub_create))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool EntityCreationResp::give_Datas(ENTITY_CREATION_RESP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_CREATION_RESP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(m_entityPubKey)
			{
				if((*Datas)->d.entity_pub_key)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.entity_pub_key, ASN1_ITEM_rptr(X509_PUBKEY));
				if(!((*Datas)->d.entity_pub_key = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_entityPubKey)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.entity_pub_key)
				{
					(*Datas)->d.entity_pub_key = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
					if(!(*Datas)->d.entity_pub_key)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 5:
			if(!((*Datas)->d.pki_create = (CREATE_PKI_USER_RESPONSE*)ASN1_item_new(ASN1_ITEM_rptr(CREATE_PKI_USER_RESPONSE))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pkiCreate).give_Datas(&(*Datas)->d.pki_create))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pki_create, ASN1_ITEM_rptr(CREATE_PKI_USER_RESPONSE));
				(*Datas)->d.pki_create = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 3:
			if(!((*Datas)->d.pub_create = (PUB_ENTITY_CREATION_RESP*)ASN1_item_new(ASN1_ITEM_rptr(PUB_ENTITY_CREATION_RESP))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pubCreate).give_Datas(&(*Datas)->d.pub_create))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pub_create, ASN1_ITEM_rptr(PUB_ENTITY_CREATION_RESP));
				(*Datas)->d.pub_create = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool EntityCreationResp::operator=(const EntityCreationResp & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(other.m_entityPubKey)
			{
				if(m_entityPubKey)
					ASN1_item_free((ASN1_VALUE*)m_entityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_entityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_entityPubKey);
				if(!m_entityPubKey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(!other.m_pkiCreate)
				break;
			(*m_pkiCreate) = *(other.m_pkiCreate);
			break;
		case 3:
			if(!other.m_pubCreate)
				break;
			(*m_pubCreate) = *(other.m_pubCreate);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityCreationResp::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_CREATION_RESP);
}
EntitySignatureReqPub EntitySignatureReqPub::EmptyInstance;
bool EntitySignatureReqPub::set_entitypubkey(const X509_PUBKEY * c_entitypubkey)
{
	if(m_entitypubkey)
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_entitypubkey);
	if(!m_entitypubkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * EntitySignatureReqPub::get_entitypubkey() const
{
	if(!m_entitypubkey)
		((EntitySignatureReqPub*)this)->m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

X509_PUBKEY * EntitySignatureReqPub::get_entitypubkey()
{
	if(!m_entitypubkey)
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

bool EntitySignatureReqPub::set_ocsppubkey(const X509_PUBKEY * c_ocsppubkey)
{
	if(m_ocsppubkey)
		ASN1_item_free((ASN1_VALUE*)m_ocsppubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_ocsppubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_ocsppubkey);
	if(!m_ocsppubkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * EntitySignatureReqPub::get_ocsppubkey() const
{
	if(!m_ocsppubkey)
		((EntitySignatureReqPub*)this)->m_ocsppubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_ocsppubkey;
}

X509_PUBKEY * EntitySignatureReqPub::get_ocsppubkey()
{
	if(!m_ocsppubkey)
		m_ocsppubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_ocsppubkey;
}

EntitySignatureReqPub::EntitySignatureReqPub():NewPKIObject()
{
	resetAll();
}

EntitySignatureReqPub::EntitySignatureReqPub(const EntitySignatureReqPub & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureReqPub::~EntitySignatureReqPub()
{
	Clear();
}

void EntitySignatureReqPub::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureReqPub::freeAll()
{
	if(m_entitypubkey)
	{
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
	if(m_ocsppubkey)
	{
		ASN1_item_free((ASN1_VALUE*)m_ocsppubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
}

void EntitySignatureReqPub::resetAll()
{
	m_entitypubkey = NULL;
	m_ocsppubkey = NULL;
}

bool EntitySignatureReqPub::load_Datas(const ENTITY_SIGNATURE_REQ_PUB * Datas)
{
	Clear();
	if(Datas->EntityPubKey)
	{
		if(m_entitypubkey)
			ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->EntityPubKey);
		if(!m_entitypubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->OcspPubKey)
	{
		if(m_ocsppubkey)
			ASN1_item_free((ASN1_VALUE*)m_ocsppubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_ocsppubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->OcspPubKey);
		if(!m_ocsppubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureReqPub::give_Datas(ENTITY_SIGNATURE_REQ_PUB ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_REQ_PUB*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(m_entitypubkey)
	{
		if((*Datas)->EntityPubKey)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->EntityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->EntityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_entitypubkey)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->EntityPubKey)
		{
			(*Datas)->EntityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->EntityPubKey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(m_ocsppubkey)
	{
		if((*Datas)->OcspPubKey)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->OcspPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->OcspPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_ocsppubkey)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->OcspPubKey)
		{
			(*Datas)->OcspPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->OcspPubKey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	return true;
}

bool EntitySignatureReqPub::operator=(const EntitySignatureReqPub & other)
{
	Clear();
	if(other.m_entitypubkey)
	{
		if(m_entitypubkey)
			ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_entitypubkey);
		if(!m_entitypubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(other.m_ocsppubkey)
	{
		if(m_ocsppubkey)
			ASN1_item_free((ASN1_VALUE*)m_ocsppubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_ocsppubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_ocsppubkey);
		if(!m_ocsppubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureReqPub::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_PUB);
}
EntitySignatureReqRep EntitySignatureReqRep::EmptyInstance;
bool EntitySignatureReqRep::set_address(const mString & c_address)
{
	m_address = c_address;
	return true;
}

const mString & EntitySignatureReqRep::get_address() const
{
	return m_address;
}

mString & EntitySignatureReqRep::get_address()
{
	return m_address;
}

bool EntitySignatureReqRep::set_entitypubkey(const X509_PUBKEY * c_entitypubkey)
{
	if(m_entitypubkey)
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_entitypubkey);
	if(!m_entitypubkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * EntitySignatureReqRep::get_entitypubkey() const
{
	if(!m_entitypubkey)
		((EntitySignatureReqRep*)this)->m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

X509_PUBKEY * EntitySignatureReqRep::get_entitypubkey()
{
	if(!m_entitypubkey)
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

bool EntitySignatureReqRep::set_port(unsigned long c_port)
{
	m_port = c_port;
	return true;
}

unsigned long EntitySignatureReqRep::get_port() const
{
	return m_port;
}

EntitySignatureReqRep::EntitySignatureReqRep():NewPKIObject()
{
	resetAll();
}

EntitySignatureReqRep::EntitySignatureReqRep(const EntitySignatureReqRep & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureReqRep::~EntitySignatureReqRep()
{
	Clear();
}

void EntitySignatureReqRep::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureReqRep::freeAll()
{
	if(m_entitypubkey)
	{
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
}

void EntitySignatureReqRep::resetAll()
{
	m_address = "";
	m_entitypubkey = NULL;
	m_port = 0;
}

bool EntitySignatureReqRep::load_Datas(const ENTITY_SIGNATURE_REQ_REP * Datas)
{
	Clear();
	if(Datas->Address)
	{
		m_address = Datas->Address;
	}
	if(Datas->EntityPubKey)
	{
		if(m_entitypubkey)
			ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->EntityPubKey);
		if(!m_entitypubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->Port)
	{
		m_port = ASN1_INTEGER_GET(Datas->Port);
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureReqRep::give_Datas(ENTITY_SIGNATURE_REQ_REP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_REQ_REP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Address && !((*Datas)->Address = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_address.c_ASN1_UTF8STRING(&(*Datas)->Address))
	{
		ASN1_UTF8STRING_free((*Datas)->Address);
		(*Datas)->Address = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(m_entitypubkey)
	{
		if((*Datas)->EntityPubKey)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->EntityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->EntityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_entitypubkey)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->EntityPubKey)
		{
			(*Datas)->EntityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->EntityPubKey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->Port && !((*Datas)->Port = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Port, m_port) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Port);
		(*Datas)->Port = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool EntitySignatureReqRep::operator=(const EntitySignatureReqRep & other)
{
	Clear();
	m_address = other.m_address;
	if(other.m_entitypubkey)
	{
		if(m_entitypubkey)
			ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_entitypubkey);
		if(!m_entitypubkey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_port = other.m_port;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureReqRep::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_REP);
}
EntitySignatureReqBody EntitySignatureReqBody::EmptyInstance;
bool EntitySignatureReqBody::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int EntitySignatureReqBody::get_type() const
{
	 return m_type;
}

bool EntitySignatureReqBody::set_entitypubkey(const X509_PUBKEY * c_entitypubkey)
{
	if(m_type != 0 && m_type != 1 && m_type != 4 && m_type != 6 && m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_entitypubkey)
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_entitypubkey);
	if(!m_entitypubkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const X509_PUBKEY * EntitySignatureReqBody::get_entitypubkey() const
{
	if((m_type != 0 && m_type != 1 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entitypubkey)
	{
		return NULL;
	}
	if(!m_entitypubkey)
		((EntitySignatureReqBody*)this)->m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

X509_PUBKEY * EntitySignatureReqBody::get_entitypubkey()
{
	if((m_type != 0 && m_type != 1 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entitypubkey)
	{
		return NULL;
	}
	if(!m_entitypubkey)
		m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_entitypubkey;
}

bool EntitySignatureReqBody::set_other(const ASN1_NULL * c_other)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_other)
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)c_other);
	if(!m_other)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const ASN1_NULL * EntitySignatureReqBody::get_other() const
{
	if((m_type != 5) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		((EntitySignatureReqBody*)this)->m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

ASN1_NULL * EntitySignatureReqBody::get_other()
{
	if((m_type != 5) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

bool EntitySignatureReqBody::set_signPub(const EntitySignatureReqPub & c_signPub)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_signPub) = c_signPub;
	m_isOk=true;
	return true;
}

const EntitySignatureReqPub & EntitySignatureReqBody::get_signPub() const
{
	if((m_type != 3) || !m_signPub)
	{
		return EntitySignatureReqPub::EmptyInstance;
	}
	return (*m_signPub);
}

EntitySignatureReqPub & EntitySignatureReqBody::get_signPub()
{
	if((m_type != 3) || !m_signPub)
	{
		return EntitySignatureReqPub::EmptyInstance;
	}
	return (*m_signPub);
}

bool EntitySignatureReqBody::set_signRep(const EntitySignatureReqRep & c_signRep)
{
	if(m_type != 2)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_signRep) = c_signRep;
	m_isOk=true;
	return true;
}

const EntitySignatureReqRep & EntitySignatureReqBody::get_signRep() const
{
	if((m_type != 2) || !m_signRep)
	{
		return EntitySignatureReqRep::EmptyInstance;
	}
	return (*m_signRep);
}

EntitySignatureReqRep & EntitySignatureReqBody::get_signRep()
{
	if((m_type != 2) || !m_signRep)
	{
		return EntitySignatureReqRep::EmptyInstance;
	}
	return (*m_signRep);
}

bool EntitySignatureReqBody::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 0:
		case 1:
		case 4:
		case 6:
		case 7:
			m_entitypubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!m_entitypubkey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
			if(!m_other)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_signPub = new EntitySignatureReqPub();
			if(!m_signPub)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 2:
			m_signRep = new EntitySignatureReqRep();
			if(!m_signRep)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

EntitySignatureReqBody::EntitySignatureReqBody():NewPKIObject()
{
	resetAll();
}

EntitySignatureReqBody::EntitySignatureReqBody(const EntitySignatureReqBody & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureReqBody::~EntitySignatureReqBody()
{
	Clear();
}

void EntitySignatureReqBody::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureReqBody::freeAll()
{
	if(m_entitypubkey)
	{
		ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
	if(m_other)
	{
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	}
	if(m_signPub)
	{
		delete m_signPub;
	}
	if(m_signRep)
	{
		delete m_signRep;
	}
}

void EntitySignatureReqBody::resetAll()
{
	m_type = -1;
	m_entitypubkey = NULL;
	m_other = NULL;
	m_signPub = NULL;
	m_signRep = NULL;
}

bool EntitySignatureReqBody::load_Datas(const ENTITY_SIGNATURE_REQ_BODY * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 0:
		case 1:
		case 4:
		case 6:
		case 7:
			if(Datas->d.EntityPubKey)
			{
				if(m_entitypubkey)
					ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->d.EntityPubKey);
				if(!m_entitypubkey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), Datas->d.other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.sign_pub)
			{
				if(!(*m_signPub).load_Datas(Datas->d.sign_pub))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 2:
			if(Datas->d.sign_rep)
			{
				if(!(*m_signRep).load_Datas(Datas->d.sign_rep))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureReqBody::give_Datas(ENTITY_SIGNATURE_REQ_BODY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_REQ_BODY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 0:
		case 1:
		case 4:
		case 6:
		case 7:
			if(m_entitypubkey)
			{
				if((*Datas)->d.EntityPubKey)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.EntityPubKey, ASN1_ITEM_rptr(X509_PUBKEY));
				if(!((*Datas)->d.EntityPubKey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_entitypubkey)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.EntityPubKey)
				{
					(*Datas)->d.EntityPubKey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
					if(!(*Datas)->d.EntityPubKey)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 5:
			if(m_other)
			{
				if((*Datas)->d.other)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.other, ASN1_ITEM_rptr(ASN1_NULL));
				if(!((*Datas)->d.other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (ASN1_VALUE*)m_other)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.other)
				{
					(*Datas)->d.other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
					if(!(*Datas)->d.other)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 3:
			if(!((*Datas)->d.sign_pub = (ENTITY_SIGNATURE_REQ_PUB*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_PUB))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_signPub).give_Datas(&(*Datas)->d.sign_pub))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.sign_pub, ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_PUB));
				(*Datas)->d.sign_pub = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 2:
			if(!((*Datas)->d.sign_rep = (ENTITY_SIGNATURE_REQ_REP*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_REP))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_signRep).give_Datas(&(*Datas)->d.sign_rep))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.sign_rep, ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_REP));
				(*Datas)->d.sign_rep = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool EntitySignatureReqBody::operator=(const EntitySignatureReqBody & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 0:
		case 1:
		case 4:
		case 6:
		case 7:
			if(other.m_entitypubkey)
			{
				if(m_entitypubkey)
					ASN1_item_free((ASN1_VALUE*)m_entitypubkey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_entitypubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_entitypubkey);
				if(!m_entitypubkey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(other.m_other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)other.m_other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(!other.m_signPub)
				break;
			(*m_signPub) = *(other.m_signPub);
			break;
		case 2:
			if(!other.m_signRep)
				break;
			(*m_signRep) = *(other.m_signRep);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureReqBody::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_BODY);
}
EntitySignatureReq EntitySignatureReq::EmptyInstance;
bool EntitySignatureReq::set_email(const mString & c_email)
{
	m_email = c_email;
	return true;
}

const mString & EntitySignatureReq::get_email() const
{
	return m_email;
}

mString & EntitySignatureReq::get_email()
{
	return m_email;
}

bool EntitySignatureReq::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & EntitySignatureReq::get_name() const
{
	return m_name;
}

mString & EntitySignatureReq::get_name()
{
	return m_name;
}

bool EntitySignatureReq::set_body(const EntitySignatureReqBody & c_body)
{
	m_body = c_body;
	return true;
}

const EntitySignatureReqBody & EntitySignatureReq::get_body() const
{
	return m_body;
}

EntitySignatureReqBody & EntitySignatureReq::get_body()
{
	return m_body;
}

bool EntitySignatureReq::to_PEM(mString & PemDatas) const
{
	ENTITY_SIGNATURE_REQ * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toPEM("NEWPKI ENTITY SIGNATURE REQUEST", get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool EntitySignatureReq::from_PEM(const mString & PemDatas)
{
	ENTITY_SIGNATURE_REQ * c_localvar = NULL;
	if(!Private_fromPEM("NEWPKI ENTITY SIGNATURE REQUEST", get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

EntitySignatureReq::EntitySignatureReq():NewPKIObject()
{
	resetAll();
}

EntitySignatureReq::EntitySignatureReq(const EntitySignatureReq & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureReq::~EntitySignatureReq()
{
	Clear();
}

void EntitySignatureReq::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureReq::freeAll()
{
}

void EntitySignatureReq::resetAll()
{
	m_email = "";
	m_name = "";
	m_body.Clear();
}

bool EntitySignatureReq::load_Datas(const ENTITY_SIGNATURE_REQ * Datas)
{
	Clear();
	if(Datas->Email)
	{
		m_email = Datas->Email;
	}
	if(Datas->Name)
	{
		m_name = Datas->Name;
	}
	if(Datas->body)
	{
		if(!m_body.load_Datas(Datas->body))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureReq::give_Datas(ENTITY_SIGNATURE_REQ ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_REQ*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Email && !((*Datas)->Email = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_email.c_ASN1_UTF8STRING(&(*Datas)->Email))
	{
		ASN1_UTF8STRING_free((*Datas)->Email);
		(*Datas)->Email = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Name && !((*Datas)->Name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->Name))
	{
		ASN1_UTF8STRING_free((*Datas)->Name);
		(*Datas)->Name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->body && !((*Datas)->body = (ENTITY_SIGNATURE_REQ_BODY*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_BODY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_body.give_Datas(&(*Datas)->body))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->body, ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ_BODY));
		(*Datas)->body = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntitySignatureReq::operator=(const EntitySignatureReq & other)
{
	Clear();
	m_email = other.m_email;
	m_name = other.m_name;
	m_body = other.m_body;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureReq::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_REQ);
}
EntitySignatureRespPub EntitySignatureRespPub::EmptyInstance;
bool EntitySignatureRespPub::set_entitycert(const PKI_CERT & c_entitycert)
{
	m_entitycert = c_entitycert;
	return true;
}

const PKI_CERT & EntitySignatureRespPub::get_entitycert() const
{
	return m_entitycert;
}

PKI_CERT & EntitySignatureRespPub::get_entitycert()
{
	return m_entitycert;
}

bool EntitySignatureRespPub::set_ocspcert(const PKI_CERT & c_ocspcert)
{
	m_ocspcert = c_ocspcert;
	return true;
}

const PKI_CERT & EntitySignatureRespPub::get_ocspcert() const
{
	return m_ocspcert;
}

PKI_CERT & EntitySignatureRespPub::get_ocspcert()
{
	return m_ocspcert;
}

EntitySignatureRespPub::EntitySignatureRespPub():NewPKIObject()
{
	resetAll();
}

EntitySignatureRespPub::EntitySignatureRespPub(const EntitySignatureRespPub & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureRespPub::~EntitySignatureRespPub()
{
	Clear();
}

void EntitySignatureRespPub::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureRespPub::freeAll()
{
}

void EntitySignatureRespPub::resetAll()
{
	m_entitycert.Clear();
	m_ocspcert.Clear();
}

bool EntitySignatureRespPub::load_Datas(const ENTITY_SIGNATURE_RESP_PUB * Datas)
{
	Clear();
	if(Datas->EntityCert)
	{
		if(!m_entitycert.load_Datas(Datas->EntityCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->OcspCert)
	{
		if(!m_ocspcert.load_Datas(Datas->OcspCert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureRespPub::give_Datas(ENTITY_SIGNATURE_RESP_PUB ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_RESP_PUB*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->EntityCert && !((*Datas)->EntityCert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entitycert.give_Datas(&(*Datas)->EntityCert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->EntityCert, ASN1_ITEM_rptr(X509));
		(*Datas)->EntityCert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->OcspCert && !((*Datas)->OcspCert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_ocspcert.give_Datas(&(*Datas)->OcspCert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->OcspCert, ASN1_ITEM_rptr(X509));
		(*Datas)->OcspCert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntitySignatureRespPub::operator=(const EntitySignatureRespPub & other)
{
	Clear();
	m_entitycert = other.m_entitycert;
	m_ocspcert = other.m_ocspcert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureRespPub::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_PUB);
}
EntitySignatureRespBody EntitySignatureRespBody::EmptyInstance;
bool EntitySignatureRespBody::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int EntitySignatureRespBody::get_type() const
{
	 return m_type;
}

bool EntitySignatureRespBody::set_entitycert(const PKI_CERT & c_entitycert)
{
	if(m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_entitycert) = c_entitycert;
	m_isOk=true;
	return true;
}

const PKI_CERT & EntitySignatureRespBody::get_entitycert() const
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entitycert)
	{
		return PKI_CERT::EmptyInstance;
	}
	return (*m_entitycert);
}

PKI_CERT & EntitySignatureRespBody::get_entitycert()
{
	if((m_type != 0 && m_type != 1 && m_type != 2 && m_type != 4 && m_type != 6 && m_type != 7) || !m_entitycert)
	{
		return PKI_CERT::EmptyInstance;
	}
	return (*m_entitycert);
}

bool EntitySignatureRespBody::set_other(const ASN1_NULL * c_other)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_other)
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)c_other);
	if(!m_other)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const ASN1_NULL * EntitySignatureRespBody::get_other() const
{
	if((m_type != 5) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		((EntitySignatureRespBody*)this)->m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

ASN1_NULL * EntitySignatureRespBody::get_other()
{
	if((m_type != 5) || !m_other)
	{
		return NULL;
	}
	if(!m_other)
		m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
	return m_other;
}

bool EntitySignatureRespBody::set_signPub(const EntitySignatureRespPub & c_signPub)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_signPub) = c_signPub;
	m_isOk=true;
	return true;
}

const EntitySignatureRespPub & EntitySignatureRespBody::get_signPub() const
{
	if((m_type != 3) || !m_signPub)
	{
		return EntitySignatureRespPub::EmptyInstance;
	}
	return (*m_signPub);
}

EntitySignatureRespPub & EntitySignatureRespBody::get_signPub()
{
	if((m_type != 3) || !m_signPub)
	{
		return EntitySignatureRespPub::EmptyInstance;
	}
	return (*m_signPub);
}

bool EntitySignatureRespBody::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			m_entitycert = new PKI_CERT();
			if(!m_entitycert)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
			if(!m_other)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_signPub = new EntitySignatureRespPub();
			if(!m_signPub)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

EntitySignatureRespBody::EntitySignatureRespBody():NewPKIObject()
{
	resetAll();
}

EntitySignatureRespBody::EntitySignatureRespBody(const EntitySignatureRespBody & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntitySignatureRespBody::~EntitySignatureRespBody()
{
	Clear();
}

void EntitySignatureRespBody::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureRespBody::freeAll()
{
	if(m_entitycert)
	{
		delete m_entitycert;
	}
	if(m_other)
	{
		ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
	}
	if(m_signPub)
	{
		delete m_signPub;
	}
}

void EntitySignatureRespBody::resetAll()
{
	m_type = -1;
	m_entitycert = NULL;
	m_other = NULL;
	m_signPub = NULL;
}

bool EntitySignatureRespBody::load_Datas(const ENTITY_SIGNATURE_RESP_BODY * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(Datas->d.EntityCert)
			{
				if(!(*m_entitycert).load_Datas(Datas->d.EntityCert))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), Datas->d.other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.sign_pub)
			{
				if(!(*m_signPub).load_Datas(Datas->d.sign_pub))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureRespBody::give_Datas(ENTITY_SIGNATURE_RESP_BODY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_RESP_BODY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(!((*Datas)->d.EntityCert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_entitycert).give_Datas(&(*Datas)->d.EntityCert))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.EntityCert, ASN1_ITEM_rptr(X509));
				(*Datas)->d.EntityCert = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 5:
			if(m_other)
			{
				if((*Datas)->d.other)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.other, ASN1_ITEM_rptr(ASN1_NULL));
				if(!((*Datas)->d.other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (ASN1_VALUE*)m_other)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.other)
				{
					(*Datas)->d.other = (ASN1_NULL*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_NULL));
					if(!(*Datas)->d.other)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 3:
			if(!((*Datas)->d.sign_pub = (ENTITY_SIGNATURE_RESP_PUB*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_PUB))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_signPub).give_Datas(&(*Datas)->d.sign_pub))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.sign_pub, ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_PUB));
				(*Datas)->d.sign_pub = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool EntitySignatureRespBody::operator=(const EntitySignatureRespBody & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 0:
		case 1:
		case 2:
		case 4:
		case 6:
		case 7:
			if(!other.m_entitycert)
				break;
			(*m_entitycert) = *(other.m_entitycert);
			break;
		case 5:
			if(other.m_other)
			{
				if(m_other)
					ASN1_item_free((ASN1_VALUE*)m_other, ASN1_ITEM_rptr(ASN1_NULL));
				m_other = (ASN1_NULL*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_NULL), (void*)other.m_other);
				if(!m_other)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(!other.m_signPub)
				break;
			(*m_signPub) = *(other.m_signPub);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureRespBody::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_BODY);
}
EntitySignatureResp EntitySignatureResp::EmptyInstance;
bool EntitySignatureResp::set_body(const EntitySignatureRespBody & c_body)
{
	m_body = c_body;
	return true;
}

const EntitySignatureRespBody & EntitySignatureResp::get_body() const
{
	return m_body;
}

EntitySignatureRespBody & EntitySignatureResp::get_body()
{
	return m_body;
}

bool EntitySignatureResp::set_cas(const InternalPkiCa & c_cas)
{
	m_cas = c_cas;
	return true;
}

const InternalPkiCa & EntitySignatureResp::get_cas() const
{
	return m_cas;
}

InternalPkiCa & EntitySignatureResp::get_cas()
{
	return m_cas;
}

bool EntitySignatureResp::set_conf(const EntityConfCrypted & c_conf)
{
	m_conf = c_conf;
	return true;
}

const EntityConfCrypted & EntitySignatureResp::get_conf() const
{
	return m_conf;
}

EntityConfCrypted & EntitySignatureResp::get_conf()
{
	return m_conf;
}

bool EntitySignatureResp::to_PEM(mString & PemDatas) const
{
	ENTITY_SIGNATURE_RESP * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toPEM("NEWPKI ENTITY SIGNATURE RESPONSE", get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool EntitySignatureResp::from_PEM(const mString & PemDatas)
{
	ENTITY_SIGNATURE_RESP * c_localvar = NULL;
	if(!Private_fromPEM("NEWPKI ENTITY SIGNATURE RESPONSE", get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool EntitySignatureResp::to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const
{
	ENTITY_SIGNATURE_RESP * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, sig_pkey, crypt_pkey, sig_md, crypt_cypher))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool EntitySignatureResp::from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey)
{
	ENTITY_SIGNATURE_RESP * c_localvar = NULL;
	if(!Private_fromSignEncrypt(cryptinfo, get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, sig_pkey, crypt_pkey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

EntitySignatureResp::EntitySignatureResp():NewPKIObject(), NewPKISignCryptObject()
{
	resetAll();
}

EntitySignatureResp::EntitySignatureResp(const EntitySignatureResp & other):NewPKIObject(), NewPKISignCryptObject()
{
	resetAll();
	*this = other;
}

EntitySignatureResp::~EntitySignatureResp()
{
	Clear();
}

void EntitySignatureResp::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntitySignatureResp::freeAll()
{
}

void EntitySignatureResp::resetAll()
{
	m_body.Clear();
	m_cas.Clear();
	m_conf.Clear();
}

bool EntitySignatureResp::load_Datas(const ENTITY_SIGNATURE_RESP * Datas)
{
	Clear();
	if(Datas->body)
	{
		if(!m_body.load_Datas(Datas->body))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->cas)
	{
		if(!m_cas.load_Datas(Datas->cas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->conf)
	{
		if(!m_conf.load_Datas(Datas->conf))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntitySignatureResp::give_Datas(ENTITY_SIGNATURE_RESP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_SIGNATURE_RESP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->body && !((*Datas)->body = (ENTITY_SIGNATURE_RESP_BODY*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_BODY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_body.give_Datas(&(*Datas)->body))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->body, ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP_BODY));
		(*Datas)->body = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->cas && !((*Datas)->cas = (INTERNAL_PKI_CA*)ASN1_item_new(ASN1_ITEM_rptr(INTERNAL_PKI_CA))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cas.give_Datas(&(*Datas)->cas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->cas, ASN1_ITEM_rptr(INTERNAL_PKI_CA));
		(*Datas)->cas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->conf && !((*Datas)->conf = (ENTITY_CONF_CRYPTED*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_CONF_CRYPTED))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_conf.give_Datas(&(*Datas)->conf))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->conf, ASN1_ITEM_rptr(ENTITY_CONF_CRYPTED));
		(*Datas)->conf = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntitySignatureResp::operator=(const EntitySignatureResp & other)
{
	Clear();
	m_body = other.m_body;
	m_cas = other.m_cas;
	m_conf = other.m_conf;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntitySignatureResp::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP);
}
EntityInitReq EntityInitReq::EmptyInstance;
bool EntityInitReq::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & EntityInitReq::get_name() const
{
	return m_name;
}

mString & EntityInitReq::get_name()
{
	return m_name;
}

bool EntityInitReq::set_signResp(const EntitySignatureResp & c_signResp)
{
	m_signResp = c_signResp;
	return true;
}

const EntitySignatureResp & EntityInitReq::get_signResp() const
{
	return m_signResp;
}

EntitySignatureResp & EntityInitReq::get_signResp()
{
	return m_signResp;
}

EntityInitReq::EntityInitReq():NewPKIObject()
{
	resetAll();
}

EntityInitReq::EntityInitReq(const EntityInitReq & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityInitReq::~EntityInitReq()
{
	Clear();
}

void EntityInitReq::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityInitReq::freeAll()
{
}

void EntityInitReq::resetAll()
{
	m_name = "";
	m_signResp.Clear();
}

bool EntityInitReq::load_Datas(const ENTITY_INIT_REQ * Datas)
{
	Clear();
	if(Datas->name)
	{
		m_name = Datas->name;
	}
	if(Datas->sign_resp)
	{
		if(!m_signResp.load_Datas(Datas->sign_resp))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntityInitReq::give_Datas(ENTITY_INIT_REQ ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_INIT_REQ*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->name && !((*Datas)->name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->name))
	{
		ASN1_UTF8STRING_free((*Datas)->name);
		(*Datas)->name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->sign_resp && !((*Datas)->sign_resp = (ENTITY_SIGNATURE_RESP*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_signResp.give_Datas(&(*Datas)->sign_resp))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->sign_resp, ASN1_ITEM_rptr(ENTITY_SIGNATURE_RESP));
		(*Datas)->sign_resp = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntityInitReq::operator=(const EntityInitReq & other)
{
	Clear();
	m_name = other.m_name;
	m_signResp = other.m_signResp;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityInitReq::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_INIT_REQ);
}
EntityLinkInfo EntityLinkInfo::EmptyInstance;
bool EntityLinkInfo::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & EntityLinkInfo::get_name() const
{
	return m_name;
}

mString & EntityLinkInfo::get_name()
{
	return m_name;
}

bool EntityLinkInfo::set_type(unsigned long c_type)
{
	m_type = c_type;
	return true;
}

unsigned long EntityLinkInfo::get_type() const
{
	return m_type;
}

bool EntityLinkInfo::set_flags(const ASN1_BIT_STRING * c_flags)
{
	if(m_flags)
		ASN1_item_free((ASN1_VALUE*)m_flags, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	m_flags = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)c_flags);
	if(!m_flags)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const ASN1_BIT_STRING * EntityLinkInfo::get_flags() const
{
	if(!m_flags)
		((EntityLinkInfo*)this)->m_flags = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_flags;
}

ASN1_BIT_STRING * EntityLinkInfo::get_flags()
{
	if(!m_flags)
		m_flags = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
	return m_flags;
}

EntityLinkInfo::EntityLinkInfo():NewPKIObject()
{
	resetAll();
}

EntityLinkInfo::EntityLinkInfo(const EntityLinkInfo & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityLinkInfo::~EntityLinkInfo()
{
	Clear();
}

void EntityLinkInfo::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityLinkInfo::freeAll()
{
	if(m_flags)
	{
		ASN1_item_free((ASN1_VALUE*)m_flags, ASN1_ITEM_rptr(ASN1_BIT_STRING));
	}
}

void EntityLinkInfo::resetAll()
{
	m_name = "";
	m_type = 0;
	m_flags = NULL;
}

bool EntityLinkInfo::load_Datas(const ENTITY_LINK_INFO * Datas)
{
	Clear();
	if(Datas->Name)
	{
		m_name = Datas->Name;
	}
	if(Datas->Type)
	{
		m_type = ASN1_INTEGER_GET(Datas->Type);
	}
	if(Datas->flags)
	{
		if(m_flags)
			ASN1_item_free((ASN1_VALUE*)m_flags, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		m_flags = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), Datas->flags);
		if(!m_flags)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntityLinkInfo::give_Datas(ENTITY_LINK_INFO ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_LINK_INFO*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Name && !((*Datas)->Name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->Name))
	{
		ASN1_UTF8STRING_free((*Datas)->Name);
		(*Datas)->Name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->Type && !((*Datas)->Type = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Type, m_type) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Type);
		(*Datas)->Type = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(m_flags)
	{
		if((*Datas)->flags)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->flags, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		if(!((*Datas)->flags = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (ASN1_VALUE*)m_flags)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->flags)
		{
			(*Datas)->flags = (ASN1_BIT_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_BIT_STRING));
			if(!(*Datas)->flags)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	return true;
}

bool EntityLinkInfo::operator=(const EntityLinkInfo & other)
{
	Clear();
	m_name = other.m_name;
	m_type = other.m_type;
	if(other.m_flags)
	{
		if(m_flags)
			ASN1_item_free((ASN1_VALUE*)m_flags, ASN1_ITEM_rptr(ASN1_BIT_STRING));
		m_flags = (ASN1_BIT_STRING*)ASN1_item_dup(ASN1_ITEM_rptr(ASN1_BIT_STRING), (void*)other.m_flags);
		if(!m_flags)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityLinkInfo::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_LINK_INFO);
}
EntityLinks EntityLinks::EmptyInstance;
bool EntityLinks::set_dsts(const mVector< EntityLinkInfo > & c_dsts)
{
	m_dsts = c_dsts;
	return true;
}

const mVector< EntityLinkInfo > & EntityLinks::get_dsts() const
{
	return m_dsts;
}

mVector< EntityLinkInfo > & EntityLinks::get_dsts()
{
	return m_dsts;
}

bool EntityLinks::set_src(const EntityLinkInfo & c_src)
{
	m_src = c_src;
	return true;
}

const EntityLinkInfo & EntityLinks::get_src() const
{
	return m_src;
}

EntityLinkInfo & EntityLinks::get_src()
{
	return m_src;
}

EntityLinks::EntityLinks():NewPKIObject()
{
	resetAll();
}

EntityLinks::EntityLinks(const EntityLinks & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

EntityLinks::~EntityLinks()
{
	Clear();
}

void EntityLinks::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void EntityLinks::freeAll()
{
}

void EntityLinks::resetAll()
{
	m_dsts.clear();
	m_src.Clear();
}

bool EntityLinks::load_Datas(const ENTITY_LINKS * Datas)
{
	Clear();
	int i;
	ENTITY_LINK_INFO * currdsts;
	if(Datas->dsts)
	{
		for(i=0; i<SKM_sk_num(ENTITY_LINK_INFO, Datas->dsts); i++)
		{
			currdsts = SKM_sk_value(ENTITY_LINK_INFO, Datas->dsts, i);
			if(!currdsts)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_dsts.insert(m_dsts.begin() + i);
			if(!m_dsts[i].load_Datas(currdsts))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->src)
	{
		if(!m_src.load_Datas(Datas->src))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool EntityLinks::give_Datas(ENTITY_LINKS ** Datas) const
{
	if(!(*Datas) && !(*Datas = (ENTITY_LINKS*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	ENTITY_LINK_INFO * currdsts;
	size_t i;
	if(!(*Datas)->dsts && !((*Datas)->dsts = SKM_sk_new_null(ENTITY_LINK_INFO)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_dsts.size() ; i++)
	{
		currdsts = NULL;
		if(!m_dsts[i].give_Datas(&currdsts))
		{
			ASN1_item_free((ASN1_VALUE*)currdsts, ASN1_ITEM_rptr(ENTITY_LINK_INFO));
			currdsts = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(ENTITY_LINK_INFO, (*Datas)->dsts, currdsts) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currdsts, ASN1_ITEM_rptr(ENTITY_LINK_INFO));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->src && !((*Datas)->src = (ENTITY_LINK_INFO*)ASN1_item_new(ASN1_ITEM_rptr(ENTITY_LINK_INFO))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_src.give_Datas(&(*Datas)->src))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->src, ASN1_ITEM_rptr(ENTITY_LINK_INFO));
		(*Datas)->src = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool EntityLinks::operator=(const EntityLinks & other)
{
	Clear();
	m_dsts = other.m_dsts;
	m_src = other.m_src;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * EntityLinks::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(ENTITY_LINKS);
}
ReqCreateRootCa ReqCreateRootCa::EmptyInstance;
bool ReqCreateRootCa::set_dn(const X509_NAME * c_dn)
{
	if(m_dn)
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)c_dn);
	if(!m_dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_NAME * ReqCreateRootCa::get_dn() const
{
	if(!m_dn)
		((ReqCreateRootCa*)this)->m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

X509_NAME * ReqCreateRootCa::get_dn()
{
	if(!m_dn)
		m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

bool ReqCreateRootCa::set_extensions(const mVector< ExtensionValue > & c_extensions)
{
	m_extensions = c_extensions;
	return true;
}

const mVector< ExtensionValue > & ReqCreateRootCa::get_extensions() const
{
	return m_extensions;
}

mVector< ExtensionValue > & ReqCreateRootCa::get_extensions()
{
	return m_extensions;
}

bool ReqCreateRootCa::set_privkey(const GenPrivateKey & c_privkey)
{
	m_privkey = c_privkey;
	return true;
}

const GenPrivateKey & ReqCreateRootCa::get_privkey() const
{
	return m_privkey;
}

GenPrivateKey & ReqCreateRootCa::get_privkey()
{
	return m_privkey;
}

bool ReqCreateRootCa::set_validity(unsigned long c_validity)
{
	m_validity = c_validity;
	return true;
}

unsigned long ReqCreateRootCa::get_validity() const
{
	return m_validity;
}

ReqCreateRootCa::ReqCreateRootCa():NewPKIObject()
{
	resetAll();
}

ReqCreateRootCa::ReqCreateRootCa(const ReqCreateRootCa & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

ReqCreateRootCa::~ReqCreateRootCa()
{
	Clear();
}

void ReqCreateRootCa::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void ReqCreateRootCa::freeAll()
{
	if(m_dn)
	{
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	}
}

void ReqCreateRootCa::resetAll()
{
	m_dn = NULL;
	m_extensions.clear();
	m_privkey.Clear();
	m_validity = 0;
}

bool ReqCreateRootCa::load_Datas(const REQ_CREATE_ROOT_CA * Datas)
{
	Clear();
	int i;
	EXTENSION_VALUE * currextensions;
	if(Datas->dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), Datas->dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->extensions)
	{
		for(i=0; i<SKM_sk_num(EXTENSION_VALUE, Datas->extensions); i++)
		{
			currextensions = SKM_sk_value(EXTENSION_VALUE, Datas->extensions, i);
			if(!currextensions)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_extensions.insert(m_extensions.begin() + i);
			if(!m_extensions[i].load_Datas(currextensions))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->privKey)
	{
		if(!m_privkey.load_Datas(Datas->privKey))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->validity)
	{
		m_validity = ASN1_INTEGER_GET(Datas->validity);
	}
	m_isOk=true;
	return true;
}

bool ReqCreateRootCa::give_Datas(REQ_CREATE_ROOT_CA ** Datas) const
{
	if(!(*Datas) && !(*Datas = (REQ_CREATE_ROOT_CA*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	EXTENSION_VALUE * currextensions;
	size_t i;
	if(m_dn)
	{
		if((*Datas)->dn)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->dn, ASN1_ITEM_rptr(X509_NAME));
		if(!((*Datas)->dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (ASN1_VALUE*)m_dn)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->dn)
		{
			(*Datas)->dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
			if(!(*Datas)->dn)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->extensions && !((*Datas)->extensions = SKM_sk_new_null(EXTENSION_VALUE)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_extensions.size() ; i++)
	{
		currextensions = NULL;
		if(!m_extensions[i].give_Datas(&currextensions))
		{
			ASN1_item_free((ASN1_VALUE*)currextensions, ASN1_ITEM_rptr(EXTENSION_VALUE));
			currextensions = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(EXTENSION_VALUE, (*Datas)->extensions, currextensions) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currextensions, ASN1_ITEM_rptr(EXTENSION_VALUE));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->privKey && !((*Datas)->privKey = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_privkey.give_Datas(&(*Datas)->privKey))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->privKey, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->privKey = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->validity && !((*Datas)->validity = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->validity, m_validity) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->validity);
		(*Datas)->validity = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool ReqCreateRootCa::operator=(const ReqCreateRootCa & other)
{
	Clear();
	if(other.m_dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)other.m_dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_extensions = other.m_extensions;
	m_privkey = other.m_privkey;
	m_validity = other.m_validity;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * ReqCreateRootCa::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(REQ_CREATE_ROOT_CA);
}
ReqCreateChildCa ReqCreateChildCa::EmptyInstance;
bool ReqCreateChildCa::set_dn(const X509_NAME * c_dn)
{
	if(m_dn)
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)c_dn);
	if(!m_dn)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_NAME * ReqCreateChildCa::get_dn() const
{
	if(!m_dn)
		((ReqCreateChildCa*)this)->m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

X509_NAME * ReqCreateChildCa::get_dn()
{
	if(!m_dn)
		m_dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
	return m_dn;
}

bool ReqCreateChildCa::set_privkey(const GenPrivateKey & c_privkey)
{
	m_privkey = c_privkey;
	return true;
}

const GenPrivateKey & ReqCreateChildCa::get_privkey() const
{
	return m_privkey;
}

GenPrivateKey & ReqCreateChildCa::get_privkey()
{
	return m_privkey;
}

ReqCreateChildCa::ReqCreateChildCa():NewPKIObject()
{
	resetAll();
}

ReqCreateChildCa::ReqCreateChildCa(const ReqCreateChildCa & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

ReqCreateChildCa::~ReqCreateChildCa()
{
	Clear();
}

void ReqCreateChildCa::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void ReqCreateChildCa::freeAll()
{
	if(m_dn)
	{
		ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
	}
}

void ReqCreateChildCa::resetAll()
{
	m_dn = NULL;
	m_privkey.Clear();
}

bool ReqCreateChildCa::load_Datas(const REQ_CREATE_CHILD_CA * Datas)
{
	Clear();
	if(Datas->dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), Datas->dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->privKey)
	{
		if(!m_privkey.load_Datas(Datas->privKey))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool ReqCreateChildCa::give_Datas(REQ_CREATE_CHILD_CA ** Datas) const
{
	if(!(*Datas) && !(*Datas = (REQ_CREATE_CHILD_CA*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(m_dn)
	{
		if((*Datas)->dn)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->dn, ASN1_ITEM_rptr(X509_NAME));
		if(!((*Datas)->dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (ASN1_VALUE*)m_dn)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->dn)
		{
			(*Datas)->dn = (X509_NAME*)ASN1_item_new(ASN1_ITEM_rptr(X509_NAME));
			if(!(*Datas)->dn)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->privKey && !((*Datas)->privKey = (GEN_PRIVATE_KEY*)ASN1_item_new(ASN1_ITEM_rptr(GEN_PRIVATE_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_privkey.give_Datas(&(*Datas)->privKey))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->privKey, ASN1_ITEM_rptr(GEN_PRIVATE_KEY));
		(*Datas)->privKey = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool ReqCreateChildCa::operator=(const ReqCreateChildCa & other)
{
	Clear();
	if(other.m_dn)
	{
		if(m_dn)
			ASN1_item_free((ASN1_VALUE*)m_dn, ASN1_ITEM_rptr(X509_NAME));
		m_dn = (X509_NAME*)ASN1_item_dup(ASN1_ITEM_rptr(X509_NAME), (void*)other.m_dn);
		if(!m_dn)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_privkey = other.m_privkey;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * ReqCreateChildCa::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(REQ_CREATE_CHILD_CA);
}

#define MAX_ENTITY_VALUE ENTITY_TYPE_EE

static int AllowedLinks[MAX_ENTITY_VALUE+1][MAX_ENTITY_VALUE+1] = {
	{0, 1, 1, 0, 1, 0, 1, 1}, // ENTITY_TYPE_RA
	{1, 0, 1, 1, 0, 0, 1, 0}, // ENTITY_TYPE_CA
	{0, 0, 1, 0, 0, 0, 1, 0}, // ENTITY_TYPE_REPOSITORY
	{0, 1, 1, 0, 0, 0, 1, 0}, // ENTITY_TYPE_PUBLICATION
	{1, 0, 1, 0, 0, 0, 1, 0}, // ENTITY_TYPE_KEY_STORE
	{0, 0, 1, 0, 0, 0, 1, 0}, // ENTITY_TYPE_PKI
	{0, 0, 1, 0, 0, 0, 1, 0}, // ENTITY_TYPE_BACKUP
	{1, 0, 1, 0, 0, 0, 1, 0}  // ENTITY_TYPE_EE
};


static int ReciprocalLinks[MAX_ENTITY_VALUE+1][MAX_ENTITY_VALUE+1] = {
	{0, 1, 0, 0, 1, 0, 0, 1}, // ENTITY_TYPE_RA
	{1, 0, 0, 1, 0, 0, 0, 0}, // ENTITY_TYPE_CA
	{0, 0, 1, 0, 0, 0, 0, 0}, // ENTITY_TYPE_REPOSITORY
	{0, 1, 0, 0, 0, 0, 0, 0}, // ENTITY_TYPE_PUBLICATION
	{1, 0, 0, 0, 0, 0, 0, 0}, // ENTITY_TYPE_KEY_STORE
	{0, 0, 0, 0, 0, 0, 0, 0}, // ENTITY_TYPE_PKI
	{0, 0, 0, 0, 0, 0, 0, 0}, // ENTITY_TYPE_BACKUP
	{1, 0, 0, 0, 0, 0, 0, 0}  // ENTITY_TYPE_EE
};

bool IsLinkAllowed(int EntityType, int LinkType)
{
	if(EntityType<0 || EntityType > MAX_ENTITY_VALUE ||
		LinkType<0 || LinkType > MAX_ENTITY_VALUE)
	{
		return false;
	}

	return (AllowedLinks[EntityType][LinkType] == 1);
}

bool IsLinkReciprocal(int EntityType, int LinkType)
{
	if(EntityType<0 || EntityType > MAX_ENTITY_VALUE ||
		LinkType<0 || LinkType > MAX_ENTITY_VALUE)
	{
		return false;
	}

	return (ReciprocalLinks[EntityType][LinkType] == 1);
}

EntityLinks * get_EntityLink(mVector<EntityLinks> & Links, const mString & Name)
{
	size_t i;

	for(i=0; i<Links.size(); i++)
	{
		if(Name == Links[i].get_src().get_name())
		{
			return &Links[i];
		}
	}
	return NULL;
}

EntityLinkInfo * get_LinkToEntity(EntityLinks & EntityLink, const mString & Name)
{
	size_t i;
	
	for(i=0; i<EntityLink.get_dsts().size(); i++)
	{
		if(Name == EntityLink.get_dsts()[i].get_name())
		{
			return &EntityLink.get_dsts()[i];
		}
	}

	return NULL;
}

bool add_LinkToEntity(mVector<EntityLinks> & Links, EntityLinks & SrcEntity, const mString & TargetName)
{
	EntityLinkInfo newEntry;


	// We first get the target information
	EntityLinks * TargetLink = get_EntityLink(Links, TargetName);
	if(!TargetLink)
		return false;
	
	// Is it allowed
	if(!IsLinkAllowed(SrcEntity.get_src().get_type(), TargetLink->get_src().get_type()))
	{
		return false;
	}

	// Do we already have it ?
	if(!get_LinkToEntity(SrcEntity, TargetName))
	{
		// Let's create it
		newEntry = TargetLink->get_src();
		ASN1_BIT_STRING_set((ASN1_BIT_STRING*)newEntry.get_flags(), (unsigned char*)"", 0);
		SrcEntity.get_dsts().push_back(newEntry);
	}

	// Shall we create a reciprocal link ?
	if(IsLinkReciprocal(SrcEntity.get_src().get_type(), TargetLink->get_src().get_type()))
	{
		// Do we already have it ?
		if(!get_LinkToEntity(*TargetLink, SrcEntity.get_src().get_name()))
		{
			// Let's create it
			newEntry = SrcEntity.get_src();
			ASN1_BIT_STRING_set((ASN1_BIT_STRING*)newEntry.get_flags(), (unsigned char*)"", 0);
			TargetLink->get_dsts().push_back(newEntry);
		}
	}

	return true;
}


bool del_LinkToEntity(mVector<EntityLinks> & Links, EntityLinks & SrcEntity, const mString & TargetName)
{
	size_t i;

	// Delete my link to it
	for(i=0; i<SrcEntity.get_dsts().size(); i++)
	{
		if(TargetName == SrcEntity.get_dsts()[i].get_name())
		{
			SrcEntity.get_dsts().erase(SrcEntity.get_dsts().begin() + i);
			i--;
		}
	}

	EntityLinks * TargetLink = get_EntityLink(Links, TargetName);
	if(TargetLink)
	{
		// Delete its link to me	
		for(i=0; i<TargetLink->get_dsts().size(); i++)
		{
			if(SrcEntity.get_src().get_name() == TargetLink->get_dsts()[i].get_name())
			{
				TargetLink->get_dsts().erase(TargetLink->get_dsts().begin() + i);
				i--;
			}
		}
	}

	return true;
}

