/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1ENTITY_H
#define ASN1ENTITY_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include "Asn1Conf.h"
#include "Asn1User.h"

#define GEN_PRIVATE_KEY_TYPE_KEYLEN 0		//!< The key should be generated
#define GEN_PRIVATE_KEY_TYPE_ENGINE 1		//!< The key should be imported from an ENGINE

/*!
	This structure represents the information needed to generate a private key
*/
typedef struct st_GEN_PRIVATE_KEY
{
	int type;
	union
	{
		ASN1_INTEGER * keylen;			//!< The key length
		ASN1_UTF8STRING * keyid;		//!< The key id when imported from an ENGINE
	}d;
}GEN_PRIVATE_KEY;
DECLARE_ASN1_ITEM(GEN_PRIVATE_KEY)

/*!
	This structure represents a PKI creation request
*/
typedef struct st_PKI_CREATION_REQ
{
	ASN1_UTF8STRING * Email;				//!< The email address of the PKI
	ASN1_INTEGER * validity;				//!< The validity in days for the internal CAs
	CREATE_PKI_USER_REQUEST * AdminCreate;	//!< The info for the Admin
	X509_NAME * dn;							//!< The DN for the internal CAs
	GEN_PRIVATE_KEY * root_ca;				//!< The private key info for the internal ROOT CA certificate
	GEN_PRIVATE_KEY * entities_ca;			//!< The private key info for the internal Entities CA certificate
	GEN_PRIVATE_KEY * users_ca;				//!< The private key info for the internal Users CA certificate
	GEN_PRIVATE_KEY * ocsp_ca;				//!< The private key info for the internal OCSP CA certificate
	GEN_PRIVATE_KEY * entity_key;			//!< The private key info for the PKI entity certificate
}PKI_CREATION_REQ;
DECLARE_ASN1_ITEM(PKI_CREATION_REQ)

/*!
	This structure represents a Publication creation request
*/
typedef struct st_PUB_CREATION_REQ
{
	GEN_PRIVATE_KEY * entity_key;	//!< The private key info for the Publication entity certificate
	GEN_PRIVATE_KEY * ocsp_key;		//!< The private key info for the Publication OCSP certificate
}PUB_CREATION_REQ;
DECLARE_ASN1_ITEM(PUB_CREATION_REQ)


/*!
	This structure represents an entity creation request body
*/
typedef struct st_ENTITY_CREATION_DATAS
{
	int type;
	union
	{
		PKI_CREATION_REQ * pki_create;		//!< Creating a PKI
		PUB_CREATION_REQ * pub_create;		//!< Creating a Publication
		GEN_PRIVATE_KEY * entity_key;		//!< Creating a RA, CA, KeyStore, Repository...
	}d;
}ENTITY_CREATION_DATAS;
DECLARE_ASN1_ITEM(ENTITY_CREATION_DATAS)

/*!
	This structure represents an entity creation request
*/
typedef struct st_ENTITY_CREATION_REQ
{
	ASN1_UTF8STRING * Name;				//!< The entity name
	ENTITY_CREATION_DATAS * Datas;		//!< The body
}ENTITY_CREATION_REQ;
DECLARE_ASN1_ITEM(ENTITY_CREATION_REQ)


/*!
	This structure represents a Publication creation response
*/
typedef struct st_PUB_ENTITY_CREATION_RESP
{
	X509_PUBKEY * entity_key;			//!< The Publication entity public key
	X509_PUBKEY * ocsp_key;				//!< The Publication OCSP public key
}PUB_ENTITY_CREATION_RESP;
DECLARE_ASN1_ITEM(PUB_ENTITY_CREATION_RESP)


/*!
	This structure represents an entity creation response
*/
typedef struct st_ENTITY_CREATION_RESP
{
	int type;
	union{
		X509_PUBKEY * entity_pub_key;				//!< The entity public key for a RA, CA, KeyStore, Repository...
		PUB_ENTITY_CREATION_RESP * pub_create;		//!< The creation response for a Publication
		CREATE_PKI_USER_RESPONSE * pki_create;				//!< The PKI admin creation info
	}d;
}ENTITY_CREATION_RESP;
DECLARE_ASN1_ITEM(ENTITY_CREATION_RESP)



/*!
	This structure represents an entity info
*/
typedef struct st_ENTITY_ENTRY_INFO
{
	ASN1_UTF8STRING * Name;			//!< The entity name
	ASN1_INTEGER * Type;			//!< The entity type
	ASN1_INTEGER * Loaded;			//!< Is the entity loaded
	X509 * certificate;				//!< The entity certificate
}ENTITY_ENTRY_INFO;
DECLARE_ASN1_ITEM(ENTITY_ENTRY_INFO)


/*!
	This structure an entity for links
*/
typedef struct st_ENTITY_LINK_INFO
{
	ASN1_UTF8STRING * Name;			//!< The entity name
	ASN1_INTEGER * Type;			//!< The entity type
	ASN1_BIT_STRING * flags;		//!< The flags
}ENTITY_LINK_INFO;
DECLARE_ASN1_ITEM(ENTITY_LINK_INFO)


/*!
	This strucutre represents the links between entities
*/
typedef struct st_ENTITY_LINKS
{
	ENTITY_LINK_INFO * src;				//!< The source of the link
	STACK_OF(ENTITY_LINK_INFO) * dsts;		//!< All the destinations
}ENTITY_LINKS;
DECLARE_ASN1_ITEM(ENTITY_LINKS)



/*!
	This strucutre represents a signature request body for a Publication entity
*/
typedef struct st_ENTITY_SIGNATURE_REQ_PUB
{
	X509_PUBKEY * EntityPubKey;		//!< The Publication public key
	X509_PUBKEY * OcspPubKey;		//!< The Publication OCSP public key
}ENTITY_SIGNATURE_REQ_PUB;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_REQ_PUB)


/*!
	This strucutre represents a signature request body for a Repository entity
*/
typedef struct st_ENTITY_SIGNATURE_REQ_REP
{
	X509_PUBKEY * EntityPubKey;	//!< The Repository public key
	ASN1_UTF8STRING * Address;	//!< The Repository address
	ASN1_INTEGER * Port;		//!< The Repository port
}ENTITY_SIGNATURE_REQ_REP;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_REQ_REP)


/*!
	This strucutre represents an entity signature request body
*/
typedef struct st_ENTITY_SIGNATURE_REQ_BODY
{
	int type;
	union{
		X509_PUBKEY * EntityPubKey;				//!< The entity public key, for RA, CA, KeyStore
		ENTITY_SIGNATURE_REQ_REP * sign_rep;	//!< The body for a repository
		ENTITY_SIGNATURE_REQ_PUB * sign_pub;	//!< The body for a publication
		ASN1_NULL * other;						//!< Nothing
	}d;
}ENTITY_SIGNATURE_REQ_BODY;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_REQ_BODY)


/*!
	This strucutre represents an entity signature request
*/
typedef struct st_ENTITY_SIGNATURE_REQ
{
	ASN1_UTF8STRING * Name;					//!< The name of the entity
	ASN1_UTF8STRING * Email;				//!< The email address of the entity
	ENTITY_SIGNATURE_REQ_BODY * body;		//!< The body
}ENTITY_SIGNATURE_REQ;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_REQ)





/*!
	This strucutre represents a signature response body for a Publication entity
*/
typedef struct st_ENTITY_SIGNATURE_RESP_PUB
{
	X509 * EntityCert;			//!< The entity certificate
	X509 * OcspCert;			//!< The OCSP certificate
}ENTITY_SIGNATURE_RESP_PUB;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_RESP_PUB)


/*!
	This strucutre represents an entity signature response body
*/
typedef struct st_ENTITY_SIGNATURE_RESP_BODY
{
	int type;
	union{
		X509 * EntityCert;						//!< The entity certificate for a RA, CA, Repository, KeyStore
		ENTITY_SIGNATURE_RESP_PUB * sign_pub;	//!< The response for a publication entity
		ASN1_NULL * other;						//!< Nothing
	}d;
}ENTITY_SIGNATURE_RESP_BODY;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_RESP_BODY)


/*!
	This strucutre represents an entity signature response
*/
typedef struct st_ENTITY_SIGNATURE_RESP
{
	ENTITY_SIGNATURE_RESP_BODY * body;		//!< The body
	ENTITY_CONF_CRYPTED * conf;				//!< The entity conf
	INTERNAL_PKI_CA * cas;					//!< The internal CAs info
}ENTITY_SIGNATURE_RESP;
DECLARE_ASN1_ITEM(ENTITY_SIGNATURE_RESP)


/*!
	This strucutre represents an entity initialization response
*/
typedef struct st_ENTITY_INIT_REQ
{
	ASN1_UTF8STRING * name;					//!< The name of the entity to intialize
	ENTITY_SIGNATURE_RESP * sign_resp;		//!< The entity signature response
}ENTITY_INIT_REQ;
DECLARE_ASN1_ITEM(ENTITY_INIT_REQ)


/*!
	This strucutre represents a ROOT CA creation request
*/
typedef struct st_REQ_CREATE_ROOT_CA
{
	GEN_PRIVATE_KEY * privKey;					//!< The private key
	X509_NAME * dn;								//!< The DN
	ASN1_INTEGER * validity;					//!< The validity in days
	STACK_OF(EXTENSION_VALUE) * extensions;		//!< The self-signed certificate extensions
}REQ_CREATE_ROOT_CA;
DECLARE_ASN1_ITEM(REQ_CREATE_ROOT_CA)

/*!
	This strucutre represents a CHILD CA creation request
*/
typedef struct st_REQ_CREATE_CHILD_CA
{
	GEN_PRIVATE_KEY * privKey;	//!< The private key
	X509_NAME * dn;				//!< The DN
}REQ_CREATE_CHILD_CA;
DECLARE_ASN1_ITEM(REQ_CREATE_CHILD_CA)

/*!
  This class is the representation of GEN_PRIVATE_KEY
*/
class GenPrivateKey : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	GenPrivateKey();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	GenPrivateKey(const GenPrivateKey & other);

	/*! \brief This is the destructor.
	 */
	~GenPrivateKey();


	/*! \brief This function converts a GEN_PRIVATE_KEY to a GenPrivateKey.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const GEN_PRIVATE_KEY * Datas);

	/*! \brief This function converts a GenPrivateKey to a GEN_PRIVATE_KEY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(GEN_PRIVATE_KEY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of GEN_PRIVATE_KEY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets ASN1_UTF8STRING::d.keyid.
	 *  \param c_keyid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keyid(const mString & c_keyid);

	/*! \brief This function returns ASN1_UTF8STRING::d.keyid.
	 *  \return The value of ASN1_UTF8STRING::d.keyid.
	 */
	const mString & get_keyid() const;

	/*! \brief This function returns ASN1_UTF8STRING::d.keyid.
	 *  \return The value of ASN1_UTF8STRING::d.keyid.
	 */
	mString & get_keyid();


	/*! \brief This function sets ASN1_INTEGER::d.keylen.
	 *  \param c_keylen [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_keylen(unsigned long c_keylen);

	/*! \brief This function returns ASN1_INTEGER::d.keylen.
	 *  \return The value of ASN1_INTEGER::d.keylen.
	 */
	unsigned long get_keylen() const;


	/*! \brief This function copies a GenPrivateKey to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const GenPrivateKey & other);
	/*! \brief This member is an empty instance of GenPrivateKey.
	 */
	static GenPrivateKey EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	mString * m_keyid;
	unsigned long * m_keylen;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_ENTRY_INFO
*/
class EntityEntryInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityEntryInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityEntryInfo(const EntityEntryInfo & other);

	/*! \brief This is the destructor.
	 */
	~EntityEntryInfo();


	/*! \brief This function converts a ENTITY_ENTRY_INFO to a EntityEntryInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_ENTRY_INFO * Datas);

	/*! \brief This function converts a EntityEntryInfo to a ENTITY_ENTRY_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_ENTRY_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_ENTRY_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::Loaded.
	 *  \param c_loaded [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_loaded(unsigned long c_loaded);

	/*! \brief This function returns ASN1_INTEGER::Loaded.
	 *  \return The value of ASN1_INTEGER::Loaded.
	 */
	unsigned long get_loaded() const;


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_INTEGER::Type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::Type.
	 *  \return The value of ASN1_INTEGER::Type.
	 */
	unsigned long get_type() const;


	/*! \brief This function sets X509::certificate.
	 *  \param c_certificate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certificate(const PKI_CERT & c_certificate);

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	const PKI_CERT & get_certificate() const;

	/*! \brief This function returns X509::certificate.
	 *  \return The value of X509::certificate.
	 */
	PKI_CERT & get_certificate();


	/*! \brief This function copies a EntityEntryInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityEntryInfo & other);
	/*! \brief This member is an empty instance of EntityEntryInfo.
	 */
	static EntityEntryInfo EmptyInstance;

private:
	unsigned long m_loaded;
	mString m_name;
	unsigned long m_type;
	PKI_CERT m_certificate;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PKI_CREATION_REQ
*/
class PkiCreationReq : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PkiCreationReq();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PkiCreationReq(const PkiCreationReq & other);

	/*! \brief This is the destructor.
	 */
	~PkiCreationReq();


	/*! \brief This function converts a PKI_CREATION_REQ to a PkiCreationReq.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKI_CREATION_REQ * Datas);

	/*! \brief This function converts a PkiCreationReq to a PKI_CREATION_REQ.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKI_CREATION_REQ ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PKI_CREATION_REQ.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets CREATE_PKI_USER_REQUEST::AdminCreate.
	 *  \param c_admincreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_admincreate(const CreatePkiUserRequest & c_admincreate);

	/*! \brief This function returns CREATE_PKI_USER_REQUEST::AdminCreate.
	 *  \return The value of CREATE_PKI_USER_REQUEST::AdminCreate.
	 */
	const CreatePkiUserRequest & get_admincreate() const;

	/*! \brief This function returns CREATE_PKI_USER_REQUEST::AdminCreate.
	 *  \return The value of CREATE_PKI_USER_REQUEST::AdminCreate.
	 */
	CreatePkiUserRequest & get_admincreate();


	/*! \brief This function sets ASN1_UTF8STRING::Email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::Email.
	 *  \return The value of ASN1_UTF8STRING::Email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::Email.
	 *  \return The value of ASN1_UTF8STRING::Email.
	 */
	mString & get_email();


	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets GEN_PRIVATE_KEY::entities_ca.
	 *  \param c_entitiesCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitiesCa(const GenPrivateKey & c_entitiesCa);

	/*! \brief This function returns GEN_PRIVATE_KEY::entities_ca.
	 *  \return The value of GEN_PRIVATE_KEY::entities_ca.
	 */
	const GenPrivateKey & get_entitiesCa() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::entities_ca.
	 *  \return The value of GEN_PRIVATE_KEY::entities_ca.
	 */
	GenPrivateKey & get_entitiesCa();


	/*! \brief This function sets GEN_PRIVATE_KEY::entity_key.
	 *  \param c_entityKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityKey(const GenPrivateKey & c_entityKey);

	/*! \brief This function returns GEN_PRIVATE_KEY::entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::entity_key.
	 */
	const GenPrivateKey & get_entityKey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::entity_key.
	 */
	GenPrivateKey & get_entityKey();


	/*! \brief This function sets GEN_PRIVATE_KEY::ocsp_ca.
	 *  \param c_ocspCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspCa(const GenPrivateKey & c_ocspCa);

	/*! \brief This function returns GEN_PRIVATE_KEY::ocsp_ca.
	 *  \return The value of GEN_PRIVATE_KEY::ocsp_ca.
	 */
	const GenPrivateKey & get_ocspCa() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::ocsp_ca.
	 *  \return The value of GEN_PRIVATE_KEY::ocsp_ca.
	 */
	GenPrivateKey & get_ocspCa();


	/*! \brief This function sets GEN_PRIVATE_KEY::root_ca.
	 *  \param c_rootCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rootCa(const GenPrivateKey & c_rootCa);

	/*! \brief This function returns GEN_PRIVATE_KEY::root_ca.
	 *  \return The value of GEN_PRIVATE_KEY::root_ca.
	 */
	const GenPrivateKey & get_rootCa() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::root_ca.
	 *  \return The value of GEN_PRIVATE_KEY::root_ca.
	 */
	GenPrivateKey & get_rootCa();


	/*! \brief This function sets GEN_PRIVATE_KEY::users_ca.
	 *  \param c_usersCa [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_usersCa(const GenPrivateKey & c_usersCa);

	/*! \brief This function returns GEN_PRIVATE_KEY::users_ca.
	 *  \return The value of GEN_PRIVATE_KEY::users_ca.
	 */
	const GenPrivateKey & get_usersCa() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::users_ca.
	 *  \return The value of GEN_PRIVATE_KEY::users_ca.
	 */
	GenPrivateKey & get_usersCa();


	/*! \brief This function sets ASN1_INTEGER::validity.
	 *  \param c_validity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_validity(unsigned long c_validity);

	/*! \brief This function returns ASN1_INTEGER::validity.
	 *  \return The value of ASN1_INTEGER::validity.
	 */
	unsigned long get_validity() const;


	/*! \brief This function copies a PkiCreationReq to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PkiCreationReq & other);
	/*! \brief This member is an empty instance of PkiCreationReq.
	 */
	static PkiCreationReq EmptyInstance;

private:
	CreatePkiUserRequest m_admincreate;
	mString m_email;
	X509_NAME * m_dn;
	GenPrivateKey m_entitiesCa;
	GenPrivateKey m_entityKey;
	GenPrivateKey m_ocspCa;
	GenPrivateKey m_rootCa;
	GenPrivateKey m_usersCa;
	unsigned long m_validity;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUB_CREATION_REQ
*/
class PubCreationReq : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PubCreationReq();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PubCreationReq(const PubCreationReq & other);

	/*! \brief This is the destructor.
	 */
	~PubCreationReq();


	/*! \brief This function converts a PUB_CREATION_REQ to a PubCreationReq.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUB_CREATION_REQ * Datas);

	/*! \brief This function converts a PubCreationReq to a PUB_CREATION_REQ.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUB_CREATION_REQ ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUB_CREATION_REQ.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets GEN_PRIVATE_KEY::entity_key.
	 *  \param c_entityKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityKey(const GenPrivateKey & c_entityKey);

	/*! \brief This function returns GEN_PRIVATE_KEY::entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::entity_key.
	 */
	const GenPrivateKey & get_entityKey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::entity_key.
	 */
	GenPrivateKey & get_entityKey();


	/*! \brief This function sets GEN_PRIVATE_KEY::ocsp_key.
	 *  \param c_ocspKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspKey(const GenPrivateKey & c_ocspKey);

	/*! \brief This function returns GEN_PRIVATE_KEY::ocsp_key.
	 *  \return The value of GEN_PRIVATE_KEY::ocsp_key.
	 */
	const GenPrivateKey & get_ocspKey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::ocsp_key.
	 *  \return The value of GEN_PRIVATE_KEY::ocsp_key.
	 */
	GenPrivateKey & get_ocspKey();


	/*! \brief This function copies a PubCreationReq to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PubCreationReq & other);
	/*! \brief This member is an empty instance of PubCreationReq.
	 */
	static PubCreationReq EmptyInstance;

private:
	GenPrivateKey m_entityKey;
	GenPrivateKey m_ocspKey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CREATION_DATAS
*/
class EntityCreationDatas : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityCreationDatas();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityCreationDatas(const EntityCreationDatas & other);

	/*! \brief This is the destructor.
	 */
	~EntityCreationDatas();


	/*! \brief This function converts a ENTITY_CREATION_DATAS to a EntityCreationDatas.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CREATION_DATAS * Datas);

	/*! \brief This function converts a EntityCreationDatas to a ENTITY_CREATION_DATAS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CREATION_DATAS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CREATION_DATAS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets GEN_PRIVATE_KEY::d.entity_key.
	 *  \param c_entityKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityKey(const GenPrivateKey & c_entityKey);

	/*! \brief This function returns GEN_PRIVATE_KEY::d.entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::d.entity_key.
	 */
	const GenPrivateKey & get_entityKey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::d.entity_key.
	 *  \return The value of GEN_PRIVATE_KEY::d.entity_key.
	 */
	GenPrivateKey & get_entityKey();


	/*! \brief This function sets PKI_CREATION_REQ::d.pki_create.
	 *  \param c_pkiCreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkiCreate(const PkiCreationReq & c_pkiCreate);

	/*! \brief This function returns PKI_CREATION_REQ::d.pki_create.
	 *  \return The value of PKI_CREATION_REQ::d.pki_create.
	 */
	const PkiCreationReq & get_pkiCreate() const;

	/*! \brief This function returns PKI_CREATION_REQ::d.pki_create.
	 *  \return The value of PKI_CREATION_REQ::d.pki_create.
	 */
	PkiCreationReq & get_pkiCreate();


	/*! \brief This function sets PUB_CREATION_REQ::d.pub_create.
	 *  \param c_pubCreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubCreate(const PubCreationReq & c_pubCreate);

	/*! \brief This function returns PUB_CREATION_REQ::d.pub_create.
	 *  \return The value of PUB_CREATION_REQ::d.pub_create.
	 */
	const PubCreationReq & get_pubCreate() const;

	/*! \brief This function returns PUB_CREATION_REQ::d.pub_create.
	 *  \return The value of PUB_CREATION_REQ::d.pub_create.
	 */
	PubCreationReq & get_pubCreate();


	/*! \brief This function copies a EntityCreationDatas to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityCreationDatas & other);
	/*! \brief This member is an empty instance of EntityCreationDatas.
	 */
	static EntityCreationDatas EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	GenPrivateKey * m_entityKey;
	PkiCreationReq * m_pkiCreate;
	PubCreationReq * m_pubCreate;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CREATION_REQ
*/
class EntityCreationReq : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityCreationReq();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityCreationReq(const EntityCreationReq & other);

	/*! \brief This is the destructor.
	 */
	~EntityCreationReq();


	/*! \brief This function converts a ENTITY_CREATION_REQ to a EntityCreationReq.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CREATION_REQ * Datas);

	/*! \brief This function converts a EntityCreationReq to a ENTITY_CREATION_REQ.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CREATION_REQ ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CREATION_REQ.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_CREATION_DATAS::Datas.
	 *  \param c_datas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_datas(const EntityCreationDatas & c_datas);

	/*! \brief This function returns ENTITY_CREATION_DATAS::Datas.
	 *  \return The value of ENTITY_CREATION_DATAS::Datas.
	 */
	const EntityCreationDatas & get_datas() const;

	/*! \brief This function returns ENTITY_CREATION_DATAS::Datas.
	 *  \return The value of ENTITY_CREATION_DATAS::Datas.
	 */
	EntityCreationDatas & get_datas();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function copies a EntityCreationReq to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityCreationReq & other);
	/*! \brief This member is an empty instance of EntityCreationReq.
	 */
	static EntityCreationReq EmptyInstance;

private:
	EntityCreationDatas m_datas;
	mString m_name;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of PUB_ENTITY_CREATION_RESP
*/
class PubEntityCreationResp : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	PubEntityCreationResp();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	PubEntityCreationResp(const PubEntityCreationResp & other);

	/*! \brief This is the destructor.
	 */
	~PubEntityCreationResp();


	/*! \brief This function converts a PUB_ENTITY_CREATION_RESP to a PubEntityCreationResp.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PUB_ENTITY_CREATION_RESP * Datas);

	/*! \brief This function converts a PubEntityCreationResp to a PUB_ENTITY_CREATION_RESP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PUB_ENTITY_CREATION_RESP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of PUB_ENTITY_CREATION_RESP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_PUBKEY::entity_key.
	 *  \param c_entityKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityKey(const X509_PUBKEY * c_entityKey);

	/*! \brief This function returns X509_PUBKEY::entity_key.
	 *  \return The value of X509_PUBKEY::entity_key.
	 */
	const X509_PUBKEY * get_entityKey() const;

	/*! \brief This function returns X509_PUBKEY::entity_key.
	 *  \return The value of X509_PUBKEY::entity_key.
	 */
	X509_PUBKEY * get_entityKey();


	/*! \brief This function sets X509_PUBKEY::ocsp_key.
	 *  \param c_ocspKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspKey(const X509_PUBKEY * c_ocspKey);

	/*! \brief This function returns X509_PUBKEY::ocsp_key.
	 *  \return The value of X509_PUBKEY::ocsp_key.
	 */
	const X509_PUBKEY * get_ocspKey() const;

	/*! \brief This function returns X509_PUBKEY::ocsp_key.
	 *  \return The value of X509_PUBKEY::ocsp_key.
	 */
	X509_PUBKEY * get_ocspKey();


	/*! \brief This function copies a PubEntityCreationResp to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const PubEntityCreationResp & other);
	/*! \brief This member is an empty instance of PubEntityCreationResp.
	 */
	static PubEntityCreationResp EmptyInstance;

private:
	X509_PUBKEY * m_entityKey;
	X509_PUBKEY * m_ocspKey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_CREATION_RESP
*/
class EntityCreationResp : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityCreationResp();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityCreationResp(const EntityCreationResp & other);

	/*! \brief This is the destructor.
	 */
	~EntityCreationResp();


	/*! \brief This function converts a ENTITY_CREATION_RESP to a EntityCreationResp.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_CREATION_RESP * Datas);

	/*! \brief This function converts a EntityCreationResp to a ENTITY_CREATION_RESP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_CREATION_RESP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_CREATION_RESP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509_PUBKEY::d.entity_pub_key.
	 *  \param c_entityPubKey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityPubKey(const X509_PUBKEY * c_entityPubKey);

	/*! \brief This function returns X509_PUBKEY::d.entity_pub_key.
	 *  \return The value of X509_PUBKEY::d.entity_pub_key.
	 */
	const X509_PUBKEY * get_entityPubKey() const;

	/*! \brief This function returns X509_PUBKEY::d.entity_pub_key.
	 *  \return The value of X509_PUBKEY::d.entity_pub_key.
	 */
	X509_PUBKEY * get_entityPubKey();


	/*! \brief This function sets CREATE_PKI_USER_RESPONSE::d.pki_create.
	 *  \param c_pkiCreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pkiCreate(const CreatePkiUserResponse & c_pkiCreate);

	/*! \brief This function returns CREATE_PKI_USER_RESPONSE::d.pki_create.
	 *  \return The value of CREATE_PKI_USER_RESPONSE::d.pki_create.
	 */
	const CreatePkiUserResponse & get_pkiCreate() const;

	/*! \brief This function returns CREATE_PKI_USER_RESPONSE::d.pki_create.
	 *  \return The value of CREATE_PKI_USER_RESPONSE::d.pki_create.
	 */
	CreatePkiUserResponse & get_pkiCreate();


	/*! \brief This function sets PUB_ENTITY_CREATION_RESP::d.pub_create.
	 *  \param c_pubCreate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_pubCreate(const PubEntityCreationResp & c_pubCreate);

	/*! \brief This function returns PUB_ENTITY_CREATION_RESP::d.pub_create.
	 *  \return The value of PUB_ENTITY_CREATION_RESP::d.pub_create.
	 */
	const PubEntityCreationResp & get_pubCreate() const;

	/*! \brief This function returns PUB_ENTITY_CREATION_RESP::d.pub_create.
	 *  \return The value of PUB_ENTITY_CREATION_RESP::d.pub_create.
	 */
	PubEntityCreationResp & get_pubCreate();


	/*! \brief This function copies a EntityCreationResp to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityCreationResp & other);
	/*! \brief This member is an empty instance of EntityCreationResp.
	 */
	static EntityCreationResp EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	X509_PUBKEY * m_entityPubKey;
	CreatePkiUserResponse * m_pkiCreate;
	PubEntityCreationResp * m_pubCreate;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_REQ_PUB
*/
class EntitySignatureReqPub : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureReqPub();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureReqPub(const EntitySignatureReqPub & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureReqPub();


	/*! \brief This function converts a ENTITY_SIGNATURE_REQ_PUB to a EntitySignatureReqPub.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_REQ_PUB * Datas);

	/*! \brief This function converts a EntitySignatureReqPub to a ENTITY_SIGNATURE_REQ_PUB.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_REQ_PUB ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_REQ_PUB.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_PUBKEY::EntityPubKey.
	 *  \param c_entitypubkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitypubkey(const X509_PUBKEY * c_entitypubkey);

	/*! \brief This function returns X509_PUBKEY::EntityPubKey.
	 *  \return The value of X509_PUBKEY::EntityPubKey.
	 */
	const X509_PUBKEY * get_entitypubkey() const;

	/*! \brief This function returns X509_PUBKEY::EntityPubKey.
	 *  \return The value of X509_PUBKEY::EntityPubKey.
	 */
	X509_PUBKEY * get_entitypubkey();


	/*! \brief This function sets X509_PUBKEY::OcspPubKey.
	 *  \param c_ocsppubkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocsppubkey(const X509_PUBKEY * c_ocsppubkey);

	/*! \brief This function returns X509_PUBKEY::OcspPubKey.
	 *  \return The value of X509_PUBKEY::OcspPubKey.
	 */
	const X509_PUBKEY * get_ocsppubkey() const;

	/*! \brief This function returns X509_PUBKEY::OcspPubKey.
	 *  \return The value of X509_PUBKEY::OcspPubKey.
	 */
	X509_PUBKEY * get_ocsppubkey();


	/*! \brief This function copies a EntitySignatureReqPub to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureReqPub & other);
	/*! \brief This member is an empty instance of EntitySignatureReqPub.
	 */
	static EntitySignatureReqPub EmptyInstance;

private:
	X509_PUBKEY * m_entitypubkey;
	X509_PUBKEY * m_ocsppubkey;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_REQ_REP
*/
class EntitySignatureReqRep : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureReqRep();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureReqRep(const EntitySignatureReqRep & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureReqRep();


	/*! \brief This function converts a ENTITY_SIGNATURE_REQ_REP to a EntitySignatureReqRep.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_REQ_REP * Datas);

	/*! \brief This function converts a EntitySignatureReqRep to a ENTITY_SIGNATURE_REQ_REP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_REQ_REP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_REQ_REP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Address.
	 *  \param c_address [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_address(const mString & c_address);

	/*! \brief This function returns ASN1_UTF8STRING::Address.
	 *  \return The value of ASN1_UTF8STRING::Address.
	 */
	const mString & get_address() const;

	/*! \brief This function returns ASN1_UTF8STRING::Address.
	 *  \return The value of ASN1_UTF8STRING::Address.
	 */
	mString & get_address();


	/*! \brief This function sets X509_PUBKEY::EntityPubKey.
	 *  \param c_entitypubkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitypubkey(const X509_PUBKEY * c_entitypubkey);

	/*! \brief This function returns X509_PUBKEY::EntityPubKey.
	 *  \return The value of X509_PUBKEY::EntityPubKey.
	 */
	const X509_PUBKEY * get_entitypubkey() const;

	/*! \brief This function returns X509_PUBKEY::EntityPubKey.
	 *  \return The value of X509_PUBKEY::EntityPubKey.
	 */
	X509_PUBKEY * get_entitypubkey();


	/*! \brief This function sets ASN1_INTEGER::Port.
	 *  \param c_port [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_port(unsigned long c_port);

	/*! \brief This function returns ASN1_INTEGER::Port.
	 *  \return The value of ASN1_INTEGER::Port.
	 */
	unsigned long get_port() const;


	/*! \brief This function copies a EntitySignatureReqRep to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureReqRep & other);
	/*! \brief This member is an empty instance of EntitySignatureReqRep.
	 */
	static EntitySignatureReqRep EmptyInstance;

private:
	mString m_address;
	X509_PUBKEY * m_entitypubkey;
	unsigned long m_port;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_REQ_BODY
*/
class EntitySignatureReqBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureReqBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureReqBody(const EntitySignatureReqBody & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureReqBody();


	/*! \brief This function converts a ENTITY_SIGNATURE_REQ_BODY to a EntitySignatureReqBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_REQ_BODY * Datas);

	/*! \brief This function converts a EntitySignatureReqBody to a ENTITY_SIGNATURE_REQ_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_REQ_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_REQ_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509_PUBKEY::d.EntityPubKey.
	 *  \param c_entitypubkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitypubkey(const X509_PUBKEY * c_entitypubkey);

	/*! \brief This function returns X509_PUBKEY::d.EntityPubKey.
	 *  \return The value of X509_PUBKEY::d.EntityPubKey.
	 */
	const X509_PUBKEY * get_entitypubkey() const;

	/*! \brief This function returns X509_PUBKEY::d.EntityPubKey.
	 *  \return The value of X509_PUBKEY::d.EntityPubKey.
	 */
	X509_PUBKEY * get_entitypubkey();


	/*! \brief This function sets ASN1_NULL::d.other.
	 *  \param c_other [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_other(const ASN1_NULL * c_other);

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	const ASN1_NULL * get_other() const;

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	ASN1_NULL * get_other();


	/*! \brief This function sets ENTITY_SIGNATURE_REQ_PUB::d.sign_pub.
	 *  \param c_signPub [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signPub(const EntitySignatureReqPub & c_signPub);

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_PUB::d.sign_pub.
	 *  \return The value of ENTITY_SIGNATURE_REQ_PUB::d.sign_pub.
	 */
	const EntitySignatureReqPub & get_signPub() const;

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_PUB::d.sign_pub.
	 *  \return The value of ENTITY_SIGNATURE_REQ_PUB::d.sign_pub.
	 */
	EntitySignatureReqPub & get_signPub();


	/*! \brief This function sets ENTITY_SIGNATURE_REQ_REP::d.sign_rep.
	 *  \param c_signRep [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signRep(const EntitySignatureReqRep & c_signRep);

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_REP::d.sign_rep.
	 *  \return The value of ENTITY_SIGNATURE_REQ_REP::d.sign_rep.
	 */
	const EntitySignatureReqRep & get_signRep() const;

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_REP::d.sign_rep.
	 *  \return The value of ENTITY_SIGNATURE_REQ_REP::d.sign_rep.
	 */
	EntitySignatureReqRep & get_signRep();


	/*! \brief This function copies a EntitySignatureReqBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureReqBody & other);
	/*! \brief This member is an empty instance of EntitySignatureReqBody.
	 */
	static EntitySignatureReqBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	X509_PUBKEY * m_entitypubkey;
	ASN1_NULL * m_other;
	EntitySignatureReqPub * m_signPub;
	EntitySignatureReqRep * m_signRep;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_REQ
*/
class EntitySignatureReq : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureReq();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureReq(const EntitySignatureReq & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureReq();


	/*! \brief This function converts a ENTITY_SIGNATURE_REQ to a EntitySignatureReq.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_REQ * Datas);

	/*! \brief This function converts a EntitySignatureReq to a ENTITY_SIGNATURE_REQ.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_REQ ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_REQ.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Email.
	 *  \param c_email [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_email(const mString & c_email);

	/*! \brief This function returns ASN1_UTF8STRING::Email.
	 *  \return The value of ASN1_UTF8STRING::Email.
	 */
	const mString & get_email() const;

	/*! \brief This function returns ASN1_UTF8STRING::Email.
	 *  \return The value of ASN1_UTF8STRING::Email.
	 */
	mString & get_email();


	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets ENTITY_SIGNATURE_REQ_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const EntitySignatureReqBody & c_body);

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_BODY::body.
	 *  \return The value of ENTITY_SIGNATURE_REQ_BODY::body.
	 */
	const EntitySignatureReqBody & get_body() const;

	/*! \brief This function returns ENTITY_SIGNATURE_REQ_BODY::body.
	 *  \return The value of ENTITY_SIGNATURE_REQ_BODY::body.
	 */
	EntitySignatureReqBody & get_body();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a EntitySignatureReq to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureReq & other);
	/*! \brief This member is an empty instance of EntitySignatureReq.
	 */
	static EntitySignatureReq EmptyInstance;

private:
	mString m_email;
	mString m_name;
	EntitySignatureReqBody m_body;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_RESP_PUB
*/
class EntitySignatureRespPub : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureRespPub();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureRespPub(const EntitySignatureRespPub & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureRespPub();


	/*! \brief This function converts a ENTITY_SIGNATURE_RESP_PUB to a EntitySignatureRespPub.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_RESP_PUB * Datas);

	/*! \brief This function converts a EntitySignatureRespPub to a ENTITY_SIGNATURE_RESP_PUB.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_RESP_PUB ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_RESP_PUB.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509::EntityCert.
	 *  \param c_entitycert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitycert(const PKI_CERT & c_entitycert);

	/*! \brief This function returns X509::EntityCert.
	 *  \return The value of X509::EntityCert.
	 */
	const PKI_CERT & get_entitycert() const;

	/*! \brief This function returns X509::EntityCert.
	 *  \return The value of X509::EntityCert.
	 */
	PKI_CERT & get_entitycert();


	/*! \brief This function sets X509::OcspCert.
	 *  \param c_ocspcert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ocspcert(const PKI_CERT & c_ocspcert);

	/*! \brief This function returns X509::OcspCert.
	 *  \return The value of X509::OcspCert.
	 */
	const PKI_CERT & get_ocspcert() const;

	/*! \brief This function returns X509::OcspCert.
	 *  \return The value of X509::OcspCert.
	 */
	PKI_CERT & get_ocspcert();


	/*! \brief This function copies a EntitySignatureRespPub to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureRespPub & other);
	/*! \brief This member is an empty instance of EntitySignatureRespPub.
	 */
	static EntitySignatureRespPub EmptyInstance;

private:
	PKI_CERT m_entitycert;
	PKI_CERT m_ocspcert;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_RESP_BODY
*/
class EntitySignatureRespBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureRespBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureRespBody(const EntitySignatureRespBody & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureRespBody();


	/*! \brief This function converts a ENTITY_SIGNATURE_RESP_BODY to a EntitySignatureRespBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_RESP_BODY * Datas);

	/*! \brief This function converts a EntitySignatureRespBody to a ENTITY_SIGNATURE_RESP_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_RESP_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_RESP_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509::d.EntityCert.
	 *  \param c_entitycert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitycert(const PKI_CERT & c_entitycert);

	/*! \brief This function returns X509::d.EntityCert.
	 *  \return The value of X509::d.EntityCert.
	 */
	const PKI_CERT & get_entitycert() const;

	/*! \brief This function returns X509::d.EntityCert.
	 *  \return The value of X509::d.EntityCert.
	 */
	PKI_CERT & get_entitycert();


	/*! \brief This function sets ASN1_NULL::d.other.
	 *  \param c_other [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_other(const ASN1_NULL * c_other);

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	const ASN1_NULL * get_other() const;

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	ASN1_NULL * get_other();


	/*! \brief This function sets ENTITY_SIGNATURE_RESP_PUB::d.sign_pub.
	 *  \param c_signPub [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signPub(const EntitySignatureRespPub & c_signPub);

	/*! \brief This function returns ENTITY_SIGNATURE_RESP_PUB::d.sign_pub.
	 *  \return The value of ENTITY_SIGNATURE_RESP_PUB::d.sign_pub.
	 */
	const EntitySignatureRespPub & get_signPub() const;

	/*! \brief This function returns ENTITY_SIGNATURE_RESP_PUB::d.sign_pub.
	 *  \return The value of ENTITY_SIGNATURE_RESP_PUB::d.sign_pub.
	 */
	EntitySignatureRespPub & get_signPub();


	/*! \brief This function copies a EntitySignatureRespBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureRespBody & other);
	/*! \brief This member is an empty instance of EntitySignatureRespBody.
	 */
	static EntitySignatureRespBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	PKI_CERT * m_entitycert;
	ASN1_NULL * m_other;
	EntitySignatureRespPub * m_signPub;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_SIGNATURE_RESP
*/
class EntitySignatureResp : public NewPKIObject, public NewPKISignCryptObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntitySignatureResp();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntitySignatureResp(const EntitySignatureResp & other);

	/*! \brief This is the destructor.
	 */
	~EntitySignatureResp();


	/*! \brief This function converts a ENTITY_SIGNATURE_RESP to a EntitySignatureResp.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_SIGNATURE_RESP * Datas);

	/*! \brief This function converts a EntitySignatureResp to a ENTITY_SIGNATURE_RESP.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_SIGNATURE_RESP ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_SIGNATURE_RESP.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_SIGNATURE_RESP_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const EntitySignatureRespBody & c_body);

	/*! \brief This function returns ENTITY_SIGNATURE_RESP_BODY::body.
	 *  \return The value of ENTITY_SIGNATURE_RESP_BODY::body.
	 */
	const EntitySignatureRespBody & get_body() const;

	/*! \brief This function returns ENTITY_SIGNATURE_RESP_BODY::body.
	 *  \return The value of ENTITY_SIGNATURE_RESP_BODY::body.
	 */
	EntitySignatureRespBody & get_body();


	/*! \brief This function sets INTERNAL_PKI_CA::cas.
	 *  \param c_cas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cas(const InternalPkiCa & c_cas);

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	const InternalPkiCa & get_cas() const;

	/*! \brief This function returns INTERNAL_PKI_CA::cas.
	 *  \return The value of INTERNAL_PKI_CA::cas.
	 */
	InternalPkiCa & get_cas();


	/*! \brief This function sets ENTITY_CONF_CRYPTED::conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const EntityConfCrypted & c_conf);

	/*! \brief This function returns ENTITY_CONF_CRYPTED::conf.
	 *  \return The value of ENTITY_CONF_CRYPTED::conf.
	 */
	const EntityConfCrypted & get_conf() const;

	/*! \brief This function returns ENTITY_CONF_CRYPTED::conf.
	 *  \return The value of ENTITY_CONF_CRYPTED::conf.
	 */
	EntityConfCrypted & get_conf();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function converts the local instance to a Asn1EncryptSign.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const;

	/*! \brief This function loads the local instance from a Asn1EncryptSign.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \return true on success, false on failure.
	 */
	bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);


	/*! \brief This function copies a EntitySignatureResp to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntitySignatureResp & other);
	/*! \brief This member is an empty instance of EntitySignatureResp.
	 */
	static EntitySignatureResp EmptyInstance;

private:
	EntitySignatureRespBody m_body;
	InternalPkiCa m_cas;
	EntityConfCrypted m_conf;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_INIT_REQ
*/
class EntityInitReq : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityInitReq();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityInitReq(const EntityInitReq & other);

	/*! \brief This is the destructor.
	 */
	~EntityInitReq();


	/*! \brief This function converts a ENTITY_INIT_REQ to a EntityInitReq.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_INIT_REQ * Datas);

	/*! \brief This function converts a EntityInitReq to a ENTITY_INIT_REQ.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_INIT_REQ ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_INIT_REQ.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ENTITY_SIGNATURE_RESP::sign_resp.
	 *  \param c_signResp [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signResp(const EntitySignatureResp & c_signResp);

	/*! \brief This function returns ENTITY_SIGNATURE_RESP::sign_resp.
	 *  \return The value of ENTITY_SIGNATURE_RESP::sign_resp.
	 */
	const EntitySignatureResp & get_signResp() const;

	/*! \brief This function returns ENTITY_SIGNATURE_RESP::sign_resp.
	 *  \return The value of ENTITY_SIGNATURE_RESP::sign_resp.
	 */
	EntitySignatureResp & get_signResp();


	/*! \brief This function copies a EntityInitReq to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityInitReq & other);
	/*! \brief This member is an empty instance of EntityInitReq.
	 */
	static EntityInitReq EmptyInstance;

private:
	mString m_name;
	EntitySignatureResp m_signResp;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_LINK_INFO
*/
class EntityLinkInfo : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityLinkInfo();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityLinkInfo(const EntityLinkInfo & other);

	/*! \brief This is the destructor.
	 */
	~EntityLinkInfo();


	/*! \brief This function converts a ENTITY_LINK_INFO to a EntityLinkInfo.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_LINK_INFO * Datas);

	/*! \brief This function converts a EntityLinkInfo to a ENTITY_LINK_INFO.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_LINK_INFO ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_LINK_INFO.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::Name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::Name.
	 *  \return The value of ASN1_UTF8STRING::Name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_INTEGER::Type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::Type.
	 *  \return The value of ASN1_INTEGER::Type.
	 */
	unsigned long get_type() const;


	/*! \brief This function sets ASN1_BIT_STRING::flags.
	 *  \param c_flags [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_flags(const ASN1_BIT_STRING * c_flags);

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	const ASN1_BIT_STRING * get_flags() const;

	/*! \brief This function returns ASN1_BIT_STRING::flags.
	 *  \return The value of ASN1_BIT_STRING::flags.
	 */
	ASN1_BIT_STRING * get_flags();


	/*! \brief This function copies a EntityLinkInfo to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityLinkInfo & other);
	/*! \brief This member is an empty instance of EntityLinkInfo.
	 */
	static EntityLinkInfo EmptyInstance;

private:
	mString m_name;
	unsigned long m_type;
	ASN1_BIT_STRING * m_flags;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ENTITY_LINKS
*/
class EntityLinks : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	EntityLinks();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	EntityLinks(const EntityLinks & other);

	/*! \brief This is the destructor.
	 */
	~EntityLinks();


	/*! \brief This function converts a ENTITY_LINKS to a EntityLinks.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ENTITY_LINKS * Datas);

	/*! \brief This function converts a EntityLinks to a ENTITY_LINKS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ENTITY_LINKS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ENTITY_LINKS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ENTITY_LINK_INFO::dsts.
	 *  \param c_dsts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dsts(const mVector< EntityLinkInfo > & c_dsts);

	/*! \brief This function returns ENTITY_LINK_INFO::dsts.
	 *  \return The value of ENTITY_LINK_INFO::dsts.
	 */
	const mVector< EntityLinkInfo > & get_dsts() const;

	/*! \brief This function returns ENTITY_LINK_INFO::dsts.
	 *  \return The value of ENTITY_LINK_INFO::dsts.
	 */
	mVector< EntityLinkInfo > & get_dsts();


	/*! \brief This function sets ENTITY_LINK_INFO::src.
	 *  \param c_src [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_src(const EntityLinkInfo & c_src);

	/*! \brief This function returns ENTITY_LINK_INFO::src.
	 *  \return The value of ENTITY_LINK_INFO::src.
	 */
	const EntityLinkInfo & get_src() const;

	/*! \brief This function returns ENTITY_LINK_INFO::src.
	 *  \return The value of ENTITY_LINK_INFO::src.
	 */
	EntityLinkInfo & get_src();


	/*! \brief This function copies a EntityLinks to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const EntityLinks & other);
	/*! \brief This member is an empty instance of EntityLinks.
	 */
	static EntityLinks EmptyInstance;

private:
	mVector< EntityLinkInfo > m_dsts;
	EntityLinkInfo m_src;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of REQ_CREATE_ROOT_CA
*/
class ReqCreateRootCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ReqCreateRootCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ReqCreateRootCa(const ReqCreateRootCa & other);

	/*! \brief This is the destructor.
	 */
	~ReqCreateRootCa();


	/*! \brief This function converts a REQ_CREATE_ROOT_CA to a ReqCreateRootCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const REQ_CREATE_ROOT_CA * Datas);

	/*! \brief This function converts a ReqCreateRootCa to a REQ_CREATE_ROOT_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(REQ_CREATE_ROOT_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of REQ_CREATE_ROOT_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets EXTENSION_VALUE::extensions.
	 *  \param c_extensions [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_extensions(const mVector< ExtensionValue > & c_extensions);

	/*! \brief This function returns EXTENSION_VALUE::extensions.
	 *  \return The value of EXTENSION_VALUE::extensions.
	 */
	const mVector< ExtensionValue > & get_extensions() const;

	/*! \brief This function returns EXTENSION_VALUE::extensions.
	 *  \return The value of EXTENSION_VALUE::extensions.
	 */
	mVector< ExtensionValue > & get_extensions();


	/*! \brief This function sets GEN_PRIVATE_KEY::privKey.
	 *  \param c_privkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_privkey(const GenPrivateKey & c_privkey);

	/*! \brief This function returns GEN_PRIVATE_KEY::privKey.
	 *  \return The value of GEN_PRIVATE_KEY::privKey.
	 */
	const GenPrivateKey & get_privkey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::privKey.
	 *  \return The value of GEN_PRIVATE_KEY::privKey.
	 */
	GenPrivateKey & get_privkey();


	/*! \brief This function sets ASN1_INTEGER::validity.
	 *  \param c_validity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_validity(unsigned long c_validity);

	/*! \brief This function returns ASN1_INTEGER::validity.
	 *  \return The value of ASN1_INTEGER::validity.
	 */
	unsigned long get_validity() const;


	/*! \brief This function copies a ReqCreateRootCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ReqCreateRootCa & other);
	/*! \brief This member is an empty instance of ReqCreateRootCa.
	 */
	static ReqCreateRootCa EmptyInstance;

private:
	X509_NAME * m_dn;
	mVector< ExtensionValue > m_extensions;
	GenPrivateKey m_privkey;
	unsigned long m_validity;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of REQ_CREATE_CHILD_CA
*/
class ReqCreateChildCa : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	ReqCreateChildCa();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	ReqCreateChildCa(const ReqCreateChildCa & other);

	/*! \brief This is the destructor.
	 */
	~ReqCreateChildCa();


	/*! \brief This function converts a REQ_CREATE_CHILD_CA to a ReqCreateChildCa.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const REQ_CREATE_CHILD_CA * Datas);

	/*! \brief This function converts a ReqCreateChildCa to a REQ_CREATE_CHILD_CA.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(REQ_CREATE_CHILD_CA ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of REQ_CREATE_CHILD_CA.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_NAME::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::dn.
	 *  \return The value of X509_NAME::dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets GEN_PRIVATE_KEY::privKey.
	 *  \param c_privkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_privkey(const GenPrivateKey & c_privkey);

	/*! \brief This function returns GEN_PRIVATE_KEY::privKey.
	 *  \return The value of GEN_PRIVATE_KEY::privKey.
	 */
	const GenPrivateKey & get_privkey() const;

	/*! \brief This function returns GEN_PRIVATE_KEY::privKey.
	 *  \return The value of GEN_PRIVATE_KEY::privKey.
	 */
	GenPrivateKey & get_privkey();


	/*! \brief This function copies a ReqCreateChildCa to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const ReqCreateChildCa & other);
	/*! \brief This member is an empty instance of ReqCreateChildCa.
	 */
	static ReqCreateChildCa EmptyInstance;

private:
	X509_NAME * m_dn;
	GenPrivateKey m_privkey;
	void resetAll();
	void freeAll();
protected:
};


bool IsLinkAllowed(int EntityType, int LinkType);
bool IsLinkReciprocal(int EntityType, int LinkType);
EntityLinks * get_EntityLink(mVector<EntityLinks> & Links, const mString & Name);
EntityLinkInfo * get_LinkToEntity(EntityLinks & EntityLink, const mString & Name);
bool add_LinkToEntity(mVector<EntityLinks> & Links, EntityLinks & SrcEntity, const mString & TargetName);
bool del_LinkToEntity(mVector<EntityLinks> & Links, EntityLinks & SrcEntity, const mString & TargetName);

#endif // #ifndef ASN1ENTITY_H
