/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1HELPER_H
#define ASN1HELPER_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/asn1.h>
#include <openssl/pkcs12.h>
#include <openssl/asn1t.h>
#include <PKI_ERR.h>
#include <mString.h>

/*!
	This class represents an ASN1_OCTET_STRING
*/
class Asn1OctetString : public mBuffer
{
public:
	/*! \brief This is the constructor.
	 */
	Asn1OctetString();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	Asn1OctetString(const Asn1OctetString & other);

	/*! \brief This is the destructor.
	 */
	virtual ~Asn1OctetString();

	/*! \brief This function converts a ASN1_OCTET_STRING to a Asn1OctetString.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ASN1_OCTET_STRING *Datas);

	/*! \brief This function converts a Asn1OctetString to a ASN1_OCTET_STRING.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ASN1_OCTET_STRING **Datas) const;

	/*! \brief This member is an empty instance of Asn1OctetString.
	 */
	static Asn1OctetString EmptyInstance;
};


#define ASN1_STRING_SET(stdest, stsource) \
if(stsource) \
{ \
	if(!*stdest) *stdest = ASN1_STRING_new(); \
	if(*stdest) \
	{ \
		ASN1_STRING_set(*stdest, stsource, strlen(stsource)); \
	} \
}

#define ASN1_INTEGER_SET(stdest, stsource) \
if(!*stdest) *stdest = ASN1_INTEGER_new(); \
if(*stdest) \
{ \
	ASN1_INTEGER_set(*stdest, stsource); \
}

#define ASN1_STRING_GET(stname) (char*)((stname && (stname)->data && (stname)->length)?(stname)->data:NULL)
#define ASN1_INTEGER_GET(stname) ((stname)?ASN1_INTEGER_get(stname):0)

/*!
	This structure represents some signed and crypted datas
*/
typedef struct st_ASN1_ENCRYPT_SIGN
{
	ASN1_OBJECT * sym_algo;				//!< The symetrical encryption algo
	ASN1_OCTET_STRING * crypteddatas;	//!< The symetrically crypted datas
	ASN1_OCTET_STRING * sessionkey;		//!< The asymetrically crypted symetrical key
	X509_SIG * sig;						//!< The signature
}ASN1_ENCRYPT_SIGN;
DECLARE_ASN1_ITEM(ASN1_ENCRYPT_SIGN)


/*!
  This class is the representation of ASN1_ENCRYPT_SIGN
*/
class Asn1EncryptSign : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	Asn1EncryptSign();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	Asn1EncryptSign(const Asn1EncryptSign & other);

	/*! \brief This is the destructor.
	 */
	~Asn1EncryptSign();


	/*! \brief This function converts a ASN1_ENCRYPT_SIGN to a Asn1EncryptSign.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ASN1_ENCRYPT_SIGN * Datas);

	/*! \brief This function converts a Asn1EncryptSign to a ASN1_ENCRYPT_SIGN.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ASN1_ENCRYPT_SIGN ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ASN1_ENCRYPT_SIGN.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_OCTET_STRING::crypteddatas.
	 *  \param c_crypteddatas [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crypteddatas(const Asn1OctetString & c_crypteddatas);

	/*! \brief This function returns ASN1_OCTET_STRING::crypteddatas.
	 *  \return The value of ASN1_OCTET_STRING::crypteddatas.
	 */
	const Asn1OctetString & get_crypteddatas() const;

	/*! \brief This function returns ASN1_OCTET_STRING::crypteddatas.
	 *  \return The value of ASN1_OCTET_STRING::crypteddatas.
	 */
	Asn1OctetString & get_crypteddatas();


	/*! \brief This function sets ASN1_OCTET_STRING::sessionkey.
	 *  \param c_sessionkey [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sessionkey(const Asn1OctetString & c_sessionkey);

	/*! \brief This function returns ASN1_OCTET_STRING::sessionkey.
	 *  \return The value of ASN1_OCTET_STRING::sessionkey.
	 */
	const Asn1OctetString & get_sessionkey() const;

	/*! \brief This function returns ASN1_OCTET_STRING::sessionkey.
	 *  \return The value of ASN1_OCTET_STRING::sessionkey.
	 */
	Asn1OctetString & get_sessionkey();


	/*! \brief This function sets X509_SIG::sig.
	 *  \param c_sig [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sig(const X509_SIG * c_sig);

	/*! \brief This function returns X509_SIG::sig.
	 *  \return The value of X509_SIG::sig.
	 */
	const X509_SIG * get_sig() const;

	/*! \brief This function returns X509_SIG::sig.
	 *  \return The value of X509_SIG::sig.
	 */
	X509_SIG * get_sig();


	/*! \brief This function sets ASN1_OBJECT::sym_algo.
	 *  \param c_symAlgo [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_symAlgo(const ASN1_OBJECT * c_symAlgo);

	/*! \brief This function returns ASN1_OBJECT::sym_algo.
	 *  \return The value of ASN1_OBJECT::sym_algo.
	 */
	const ASN1_OBJECT * get_symAlgo() const;

	/*! \brief This function returns ASN1_OBJECT::sym_algo.
	 *  \return The value of ASN1_OBJECT::sym_algo.
	 */
	ASN1_OBJECT * get_symAlgo();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a Asn1EncryptSign to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const Asn1EncryptSign & other);
	/*! \brief This member is an empty instance of Asn1EncryptSign.
	 */
	static Asn1EncryptSign EmptyInstance;

private:
	Asn1OctetString m_crypteddatas;
	Asn1OctetString m_sessionkey;
	X509_SIG * m_sig;
	ASN1_OBJECT * m_symAlgo;
	void resetAll();
	void freeAll();
protected:
};



/*!
	This class is the model for encryption.
*/
class NewPKISignCryptObject
{
public:
	
	/*! \brief The constructor.
	 */
	NewPKISignCryptObject()
	{
	}
	/*! \brief The destructor.
	 */
	virtual ~NewPKISignCryptObject()
	{
	}

	/*! \brief This function does the signature and encipherment.
	 *  \param cryptinfo [OUT] The result.
	 *  \param sig_pkey [IN] The signature private key.
	 *  \param crypt_pkey [IN] The encipherment public key.
	 *  \param sig_md [IN] The hash algorythm.
	 *  \param crypt_cypher [IN] The symetrical encipherment hash.
	 *  \return true on success, false on failure.
	 */
	virtual bool to_SignEncrypt(Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher) const =0;

	/*! \brief This function decrypts datas and verifies the signature.
	 *  \param cryptinfo [IN] The signed and crypted datas.
	 *  \param sig_pkey [IN] The signature verification public key.
	 *  \param crypt_pkey [IN] The decipher private key.
	 *  \return true on success, false on failure.
	 */
	virtual bool from_SignEncrypt(const Asn1EncryptSign & cryptinfo, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey) =0;
};


/*! \brief This function does the signature and encipherment.
 *  \param cryptinfo [OUT] The result.
 *  \param it [IN] The item.
 *  \param a [IN] The datas.
 *  \param sig_pkey [IN] The signature private key.
 *  \param crypt_pkey [IN] The encipherment public key.
 *  \param sig_md [IN] The hash algorythm.
 *  \param crypt_cypher [IN] The symetrical encipherment hash.
 *  \return true on success, false on failure.
 */
bool Private_toSignEncrypt(Asn1EncryptSign & cryptinfo, const ASN1_ITEM * it, ASN1_VALUE * a, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey, const EVP_MD * sig_md, const EVP_CIPHER * crypt_cypher);

/*! \brief This function decrypts datas and verifies the signature.
 *  \param cryptinfo [IN] The signed and crypted datas.
 *  \param it [IN] The item.
 *  \param a [OUT] The datas.
 *  \param sig_pkey [IN] The signature verification public key.
 *  \param crypt_pkey [IN] The decipher private key.
 *  \return true on success, false on failure.
 */
bool Private_fromSignEncrypt(const Asn1EncryptSign & cryptinfo, const ASN1_ITEM * it, ASN1_VALUE ** a, const EVP_PKEY * sig_pkey, const EVP_PKEY * crypt_pkey);

#endif // #ifndef ASN1HELPER_H
