/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1LOGS_H
#define ASN1LOGS_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/evp.h>
#include <openssl/pem.h>
#include <mString.h>
#include "Asn1Helper.h"

/*!
  This structure represents the body of a Log.
*/
typedef struct st_LOG_ENTRY_BODY
{
	ASN1_INTEGER * log_status;		//!< The log status
	ASN1_INTEGER * log_type;		//!< The type of log
	ASN1_UTF8STRING * user;			//!< The user that generated the log
	ASN1_UTF8STRING * object_name;	//!< An optional obect name
	ASN1_UTF8STRING * error;		//!< An optional error message
	ASN1_INTEGER * log_date;		//!< The date of logging
}LOG_ENTRY_BODY;
DECLARE_ASN1_ITEM(LOG_ENTRY_BODY)



/*!
  This structure represents a Log signature.
*/
typedef struct st_LOG_ENTRY_SIG
{
	ASN1_BIT_STRING * signature;		//!< The signature
	X509_ALGOR * sig_alg;				//!< The signature algo
}LOG_ENTRY_SIG;
DECLARE_ASN1_ITEM(LOG_ENTRY_SIG)

/*!
  This structure represents a Log entry.
*/
typedef struct st_LOG_ENTRY
{
	ASN1_INTEGER * log_id;			//!< The log id
	LOG_ENTRY_BODY * body;			//!< The log entry body
	LOG_ENTRY_SIG * sig;			//!< The log entry signature
}LOG_ENTRY;
DECLARE_ASN1_ITEM(LOG_ENTRY)

/*!
  This structure is used to enumerate logs
*/
typedef struct st_ADMIN_REQ_ENUM_LOGS
{
	ASN1_INTEGER * index;			//!< The index
	ASN1_INTEGER * max;				//!< The maximum number of logs to return
	ASN1_UTF8STRING * object;		//!< The object name
	ASN1_INTEGER * type;			//!< The log type
	ASN1_INTEGER * date_from;		//!< The starting date of logs
	ASN1_INTEGER * date_to;			//!< The ending date of logs
	ASN1_INTEGER * status;			//!< The log status
	ASN1_UTF8STRING * user;			//!< The user that generated the log
}ADMIN_REQ_ENUM_LOGS;
DECLARE_ASN1_ITEM(ADMIN_REQ_ENUM_LOGS)


/*!
  This enumeration defines the possible type of logs
*/
typedef enum LOG_MESSAGE_TYPE
{
	LOG_MESSAGE_TYPE_USER_LOGIN				= 1,	//!< User login
	LOG_MESSAGE_TYPE_USER_ADD				= 2,	//!< The creation of a user
	LOG_MESSAGE_TYPE_USER_MOD				= 3,	//!< The modification of a user
	LOG_MESSAGE_TYPE_PWD_MOD				= 4,	//!< The modification of a user's paswword
	LOG_MESSAGE_TYPE_ENTITY_ADD				= 5,	//!< The creation of a entity
	LOG_MESSAGE_TYPE_ENTITY_DEL				= 6,	//!< The deletion of a ca
	LOG_MESSAGE_TYPE_SIGN_ENTITY			= 7,	//!< The signature of an entity certificate
	LOG_MESSAGE_TYPE_USER_REVOCATION		= 8,	//!< The revocation of an user certificate
	LOG_MESSAGE_TYPE_ENTITY_REVOCATION		= 9,	//!< The revocation of an entity certificate
	LOG_MESSAGE_TYPE_CONFIG_PUSH			= 10,	//!< The push of the configuration on a repository
	LOG_MESSAGE_TYPE_SET_OFFLINE			= 11,	//!< Set offline status
	LOG_MESSAGE_TYPE_GET_CONFIGURATION		= 12,	//!< Get configuration
	LOG_MESSAGE_TYPE_CONFIG_IMPORT			= 13,	//!< The importation of the configuration in the repository
	LOG_MESSAGE_TYPE_ENTITY_CONFIG_GET		= 14,	//!< The exportation of an entity conf
	LOG_MESSAGE_TYPE_ENTITY_INIT			= 15,	//!< The initialisation of an entity
	LOG_MESSAGE_TYPE_SET_REPOSITORIES		= 16,	//!< The modification of the repositories
	LOG_MESSAGE_TYPE_ENTITY_LOAD			= 17,	//!< The loading of an entity
	LOG_MESSAGE_TYPE_LOCK_RESOURCE			= 18,	//!< The locking of a resource
	LOG_MESSAGE_TYPE_UNLOCK_RESOURCE		= 19,	//!< The unlocking of a resource
	LOG_MESSAGE_TYPE_SET_ENTITY_ACL			= 20,	//!< The modification of an entity acl
	LOG_MESSAGE_TYPE_SET_ENTITY_MAIL_CONF	= 21,	//!< The modification of an entity mail configuration
	LOG_MESSAGE_TYPE_SET_ENTITY_CONF		= 22,	//!< The modification of an entity configuration
	LOG_MESSAGE_TYPE_SEND_ADMIN_MAIL		= 23,	//!< Sending an admin notification mail
	LOG_MESSAGE_TYPE_SEND_MAIL				= 24,	//!< Sending a notification mail
	LOG_MESSAGE_TYPE_ADD_ADMIN_MAIL_QUEUE	= 25,	//!< Adding an admin notification mail to queue
	LOG_MESSAGE_TYPE_ADD_MAIL_QUEUE			= 26,	//!< Adding a notification mail to queue
	LOG_MESSAGE_TYPE_SET_ENTITY_AUDITS		= 27,	//!< The modification of an entity audits
	LOG_MESSAGE_TYPE_SET_ENTITIES_LINKS		= 28,	//!< The modification of the entities links
	LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF		= 29,	//!< The retreival of my conf from a rep
	LOG_MESSAGE_TYPE_OBJECTS_IMPORT			= 30,	//!< The importation of the CA objects in the repository
	LOG_MESSAGE_TYPE_OBJECTS_PUSH			= 31,	//!< The push of the CA objects on a repository
	LOG_MESSAGE_TYPE_CREATE_ROOT_CA			= 32,	//!< Creating a ROOT CA
	LOG_MESSAGE_TYPE_GET_MY_REQS			= 33,	//!< An enity is retreiving its requests
	LOG_MESSAGE_TYPE_SIGN_CSR				= 34,	//!< An administrator is signing a req from a CA
	LOG_MESSAGE_TYPE_GEN_CRL				= 35,	//!< A crl Generation
	LOG_MESSAGE_TYPE_CREATE_CHILD_CA		= 36,	//!< Creating a Child CA
	LOG_MESSAGE_TYPE_IMPORT_CHILD_CA_CERT	= 37,	//!< Importing child CA cert
	LOG_MESSAGE_TYPE_CERT_REVOCATION		= 38,	//!< The revocation of a certificate
	LOG_MESSAGE_TYPE_SET_GROUPS				= 39,	//!< The modification of the groups
	LOG_MESSAGE_TYPE_ADD_GROUP				= 40,	//!< Adding a group
	LOG_MESSAGE_TYPE_DEL_GROUP				= 41,	//!< Deleting a group
	LOG_MESSAGE_TYPE_REN_GROUP				= 42,	//!< Renaming a group
	LOG_MESSAGE_TYPE_GET_LOCAL_CONF			= 43,	//!< When a user request the entity conf
	LOG_MESSAGE_TYPE_IMPORT_PROFILE			= 44,	//!< When a user imports a profile
	LOG_MESSAGE_TYPE_GET_MY_RESPS			= 45,	//!< An entity is retreiving its responses
	LOG_MESSAGE_TYPE_INSERT_REQ				= 46,	//!< An enity is retreiving its requests
	LOG_MESSAGE_TYPE_INSERT_RESP			= 47,	//!< An entity is retreiving its responses
	LOG_MESSAGE_TYPE_CERT_REQUEST			= 48,	//!< A new cert request for a profile
	LOG_MESSAGE_TYPE_DELETE_PKCS12			= 49,	//!< Delete a pkcs12
	LOG_MESSAGE_TYPE_NEW_REQUEST			= 50,	//!< A new request is received
	LOG_MESSAGE_TYPE_CA_CERT				= 51,	//!< A new certification request
	LOG_MESSAGE_TYPE_CA_REV					= 52,	//!< A new revocation request
	LOG_MESSAGE_TYPE_LOADING_PUB_METH		= 53,	//!< Loading publication method
	LOG_MESSAGE_TYPE_CERT_PUBLICATION		= 54,	//!< A cert publication request
	LOG_MESSAGE_TYPE_REV_PUBLICATION		= 55,	//!< A rev publication request
	LOG_MESSAGE_TYPE_CRL_PUBLICATION		= 56,	//!< A crl publication request
	LOG_MESSAGE_TYPE_LDAP_CONNECTION		= 57,	//!< A ldap connction
	LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_START	= 58,	//!< A ldap synch start
	LOG_MESSAGE_TYPE_CERT_REV_RESP			= 59,	//!< The response for a revocation
	LOG_MESSAGE_TYPE_CERT_SIG_RESP			= 60,	//!< The response for a signature
	LOG_MESSAGE_TYPE_CERT_RNW_RESP			= 61,	//!< The response for a renewal
	LOG_MESSAGE_TYPE_LDAP_AUTO_SYNCH_IMPRT	= 62,	//!< A ldap synch profile import
	LOG_MESSAGE_TYPE_OCSP_START				= 63,	//!< Starting a OCSP responder
	LOG_MESSAGE_TYPE_OCSP_STOP				= 64,	//!< Stoping a OCSP responder
	LOG_MESSAGE_TYPE_OCSP_REQUEST			= 65,	//!< A new OCSP request
	LOG_MESSAGE_TYPE_ADMIN_REQUEST			= 66,	//!< A new Admin request
	LOG_MESSAGE_TYPE_DELETE_RESPONSE		= 67,	//!< A response deletion request
	LOG_MESSAGE_TYPE_PROFILE_CHANGE_UID		= 68,	//!< The request for changing a profile's UID
	LOG_MESSAGE_TYPE_PROFILE_CHANGE_OWNER	= 69,	//!< The request for changing a profile's OWNER
	LOG_MESSAGE_TYPE_PROFILE_CHANGE_DN		= 70,	//!< The request for changing a profile's DN
	LOG_MESSAGE_TYPE_DELETE_PROFILE			= 71,	//!< A profile deletion request
	LOG_MESSAGE_TYPE_ENTITY_UNLOAD			= 72,	//!< Unloading an enity
	LOG_MESSAGE_TYPE_DELETE_REQUEST			= 72,	//!< A cert request deletion request
	LOG_MESSAGE_TYPE_IMPORT_CA_P12			= 73,	//!< Importing a CA PKCS#12
	LOG_MESSAGE_TYPE_EE_VAL_REQUEST			= 74,	//!< A EE validation request
	LOG_MESSAGE_TYPE_DN_VALIDATION			= 75,	//!< The validation of a DN
	LOG_MESSAGE_TYPE_WEB_USER_CREATE		= 76,	//!< The creation of a web user
	LOG_MESSAGE_TYPE_WEB_USER_LOGIN			= 77,	//!< The login of a web user
	LOG_MESSAGE_TYPE_EE_DN_PUBLISH			= 78,	//!< A EE DN publication request
	LOG_MESSAGE_TYPE_PROFILE_ACCEPT			= 79,	//!< A profile acceptation
	LOG_MESSAGE_TYPE_PROFILE_REJECT			= 80,	//!< A profile rejection
	LOG_MESSAGE_TYPE_EE_DN_REMOVE			= 81,	//!< A EE DN removal request
	LOG_MESSAGE_TYPE_CERT_SUSPENSION		= 82,	//!< The suspension of a certificate
	LOG_MESSAGE_TYPE_CERT_UNSUSPENSION		= 83,	//!< The unsuspension of a certificate
	LOG_MESSAGE_TYPE_CERT_SUSP_RESP			= 84,	//!< The response for a suspension
	LOG_MESSAGE_TYPE_CERT_UNSUSP_RESP		= 85,	//!< The response for an unsuspension
	LOG_MESSAGE_TYPE_CA_SUSP				= 86,	//!< A new suspension request
	LOG_MESSAGE_TYPE_CA_UNSUSP				= 87,	//!< A new unsuspension request
	LOG_MESSAGE_TYPE_EE_UPD_CERT_STATE		= 88,	//!< The update of a certificate state
	LOG_MESSAGE_TYPE_EE_CERT_PUBLISH		= 89	//!< The publication of a certificate
};


/*!
  This enumeration defines the possible status of logs
*/
typedef enum LOG_MESSAGE_STATUS
{
	LOG_STATUS_TYPE_REQUEST = 1,	//!< Logging a request event
	LOG_STATUS_TYPE_SUCCESS = 2,	//!< Logging a success event
	LOG_STATUS_TYPE_FAILURE = 3		//!< Logging an error event
};


#define LOG_NO_OBJECTID ((unsigned long)~0)		//!< There is no object ID

/*!
  This class is the representation of LOG_ENTRY_BODY
*/
class LogEntryBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	LogEntryBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	LogEntryBody(const LogEntryBody & other);

	/*! \brief This is the destructor.
	 */
	~LogEntryBody();


	/*! \brief This function converts a LOG_ENTRY_BODY to a LogEntryBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const LOG_ENTRY_BODY * Datas);

	/*! \brief This function converts a LogEntryBody to a LOG_ENTRY_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(LOG_ENTRY_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of LOG_ENTRY_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::error.
	 *  \param c_error [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_error(const mString & c_error);

	/*! \brief This function returns ASN1_UTF8STRING::error.
	 *  \return The value of ASN1_UTF8STRING::error.
	 */
	const mString & get_error() const;

	/*! \brief This function returns ASN1_UTF8STRING::error.
	 *  \return The value of ASN1_UTF8STRING::error.
	 */
	mString & get_error();


	/*! \brief This function sets ASN1_INTEGER::log_date.
	 *  \param c_logDate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logDate(unsigned long c_logDate);

	/*! \brief This function returns ASN1_INTEGER::log_date.
	 *  \return The value of ASN1_INTEGER::log_date.
	 */
	unsigned long get_logDate() const;


	/*! \brief This function sets ASN1_INTEGER::log_status.
	 *  \param c_logStatus [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logStatus(unsigned long c_logStatus);

	/*! \brief This function returns ASN1_INTEGER::log_status.
	 *  \return The value of ASN1_INTEGER::log_status.
	 */
	unsigned long get_logStatus() const;


	/*! \brief This function sets ASN1_INTEGER::log_type.
	 *  \param c_logType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logType(unsigned long c_logType);

	/*! \brief This function returns ASN1_INTEGER::log_type.
	 *  \return The value of ASN1_INTEGER::log_type.
	 */
	unsigned long get_logType() const;


	/*! \brief This function sets ASN1_UTF8STRING::object_name.
	 *  \param c_objectName [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_objectName(const mString & c_objectName);

	/*! \brief This function returns ASN1_UTF8STRING::object_name.
	 *  \return The value of ASN1_UTF8STRING::object_name.
	 */
	const mString & get_objectName() const;

	/*! \brief This function returns ASN1_UTF8STRING::object_name.
	 *  \return The value of ASN1_UTF8STRING::object_name.
	 */
	mString & get_objectName();


	/*! \brief This function sets ASN1_UTF8STRING::user.
	 *  \param c_user [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_user(const mString & c_user);

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	const mString & get_user() const;

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	mString & get_user();


	/*! \brief This function copies a LogEntryBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const LogEntryBody & other);
	/*! \brief This member is an empty instance of LogEntryBody.
	 */
	static LogEntryBody EmptyInstance;

private:
	mString m_error;
	unsigned long m_logDate;
	unsigned long m_logStatus;
	unsigned long m_logType;
	mString m_objectName;
	mString m_user;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of LOG_ENTRY_SIG
*/
class LogEntrySig : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	LogEntrySig();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	LogEntrySig(const LogEntrySig & other);

	/*! \brief This is the destructor.
	 */
	~LogEntrySig();


	/*! \brief This function converts a LOG_ENTRY_SIG to a LogEntrySig.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const LOG_ENTRY_SIG * Datas);

	/*! \brief This function converts a LogEntrySig to a LOG_ENTRY_SIG.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(LOG_ENTRY_SIG ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of LOG_ENTRY_SIG.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets X509_ALGOR::sig_alg.
	 *  \param c_sigAlg [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sigAlg(const X509_ALGOR * c_sigAlg);

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	const X509_ALGOR * get_sigAlg() const;

	/*! \brief This function returns X509_ALGOR::sig_alg.
	 *  \return The value of X509_ALGOR::sig_alg.
	 */
	X509_ALGOR * get_sigAlg();


	/*! \brief This function sets ASN1_BIT_STRING::signature.
	 *  \param c_signature [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signature(const ASN1_BIT_STRING * c_signature);

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	const ASN1_BIT_STRING * get_signature() const;

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	ASN1_BIT_STRING * get_signature();


	/*! \brief This funciton generates the PEM representation of the local instance.
	 *  \param PemDatas [OUT] the PEM output.
	 *  \return true on success, false on failure.
	 */
	bool to_PEM(mString & PemDatas) const;

	/*! \brief This function loads the local instance from a PEM representation.
	 *  \param PemDatas [IN] the PEM input.
	 *  \return true on success, false on failure.
	 */
	bool from_PEM(const mString & PemDatas);


	/*! \brief This function copies a LogEntrySig to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const LogEntrySig & other);
	/*! \brief This member is an empty instance of LogEntrySig.
	 */
	static LogEntrySig EmptyInstance;

private:
	X509_ALGOR * m_sigAlg;
	ASN1_BIT_STRING * m_signature;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of LOG_ENTRY
*/
class LogEntry : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	LogEntry();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	LogEntry(const LogEntry & other);

	/*! \brief This is the destructor.
	 */
	~LogEntry();


	/*! \brief This function converts a LOG_ENTRY to a LogEntry.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const LOG_ENTRY * Datas);

	/*! \brief This function converts a LogEntry to a LOG_ENTRY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(LOG_ENTRY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of LOG_ENTRY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets LOG_ENTRY_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const LogEntryBody & c_body);

	/*! \brief This function returns LOG_ENTRY_BODY::body.
	 *  \return The value of LOG_ENTRY_BODY::body.
	 */
	const LogEntryBody & get_body() const;

	/*! \brief This function returns LOG_ENTRY_BODY::body.
	 *  \return The value of LOG_ENTRY_BODY::body.
	 */
	LogEntryBody & get_body();


	/*! \brief This function sets ASN1_INTEGER::log_id.
	 *  \param c_logId [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logId(unsigned long c_logId);

	/*! \brief This function returns ASN1_INTEGER::log_id.
	 *  \return The value of ASN1_INTEGER::log_id.
	 */
	unsigned long get_logId() const;


	/*! \brief This function sets LOG_ENTRY_SIG::sig.
	 *  \param c_sig [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sig(const LogEntrySig & c_sig);

	/*! \brief This function returns LOG_ENTRY_SIG::sig.
	 *  \return The value of LOG_ENTRY_SIG::sig.
	 */
	const LogEntrySig & get_sig() const;

	/*! \brief This function returns LOG_ENTRY_SIG::sig.
	 *  \return The value of LOG_ENTRY_SIG::sig.
	 */
	LogEntrySig & get_sig();


	/*! \brief This function copies a LogEntry to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const LogEntry & other);
	/*! \brief This member is an empty instance of LogEntry.
	 */
	static LogEntry EmptyInstance;

private:
	LogEntryBody m_body;
	unsigned long m_logId;
	LogEntrySig m_sig;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_REQ_ENUM_LOGS
*/
class AdminReqEnumLogs : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminReqEnumLogs();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminReqEnumLogs(const AdminReqEnumLogs & other);

	/*! \brief This is the destructor.
	 */
	~AdminReqEnumLogs();


	/*! \brief This function converts a ADMIN_REQ_ENUM_LOGS to a AdminReqEnumLogs.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_REQ_ENUM_LOGS * Datas);

	/*! \brief This function converts a AdminReqEnumLogs to a ADMIN_REQ_ENUM_LOGS.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_REQ_ENUM_LOGS ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_REQ_ENUM_LOGS.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::date_from.
	 *  \param c_dateFrom [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dateFrom(unsigned long c_dateFrom);

	/*! \brief This function returns ASN1_INTEGER::date_from.
	 *  \return The value of ASN1_INTEGER::date_from.
	 */
	unsigned long get_dateFrom() const;


	/*! \brief This function sets ASN1_INTEGER::date_to.
	 *  \param c_dateTo [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dateTo(unsigned long c_dateTo);

	/*! \brief This function returns ASN1_INTEGER::date_to.
	 *  \return The value of ASN1_INTEGER::date_to.
	 */
	unsigned long get_dateTo() const;


	/*! \brief This function sets ASN1_INTEGER::index.
	 *  \param c_index [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_index(unsigned long c_index);

	/*! \brief This function returns ASN1_INTEGER::index.
	 *  \return The value of ASN1_INTEGER::index.
	 */
	unsigned long get_index() const;


	/*! \brief This function sets ASN1_INTEGER::max.
	 *  \param c_max [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_max(unsigned long c_max);

	/*! \brief This function returns ASN1_INTEGER::max.
	 *  \return The value of ASN1_INTEGER::max.
	 */
	unsigned long get_max() const;


	/*! \brief This function sets ASN1_UTF8STRING::object.
	 *  \param c_object [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_object(const mString & c_object);

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	const mString & get_object() const;

	/*! \brief This function returns ASN1_UTF8STRING::object.
	 *  \return The value of ASN1_UTF8STRING::object.
	 */
	mString & get_object();


	/*! \brief This function sets ASN1_INTEGER::status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::status.
	 *  \return The value of ASN1_INTEGER::status.
	 */
	unsigned long get_status() const;


	/*! \brief This function sets ASN1_INTEGER::type.
	 *  \param c_type [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_type(unsigned long c_type);

	/*! \brief This function returns ASN1_INTEGER::type.
	 *  \return The value of ASN1_INTEGER::type.
	 */
	unsigned long get_type() const;


	/*! \brief This function sets ASN1_UTF8STRING::user.
	 *  \param c_user [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_user(const mString & c_user);

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	const mString & get_user() const;

	/*! \brief This function returns ASN1_UTF8STRING::user.
	 *  \return The value of ASN1_UTF8STRING::user.
	 */
	mString & get_user();


	/*! \brief This function copies a AdminReqEnumLogs to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminReqEnumLogs & other);
	/*! \brief This member is an empty instance of AdminReqEnumLogs.
	 */
	static AdminReqEnumLogs EmptyInstance;

private:
	unsigned long m_dateFrom;
	unsigned long m_dateTo;
	unsigned long m_index;
	unsigned long m_max;
	mString m_object;
	unsigned long m_status;
	unsigned long m_type;
	mString m_user;
	void resetAll();
	void freeAll();
protected:
};


/*! \brief This function returns the string representation of a log type.
 *  \param log_type [IN] The log type.
 *  \return The string representation of the log type.
 */
char * ASN1_logs_get_TypeString(LOG_MESSAGE_TYPE log_type);	

/*! \brief This function returns the string representation of a log status.
 *  \param log_status [IN] The log status.
 *  \return The string representation of the log status.
 */
char * ASN1_logs_get_StatusString(LOG_MESSAGE_STATUS log_status);


#define LOG_ENTRY_sign(x,pkey) \
	ASN1_item_sign(LogEntryBody::get_ASN1_ITEM(), x->sig->sig_alg, NULL, x->sig->signature, (char *)x->body, pkey, EVP_sha1())

#define LOG_ENTRY_verify(x,pkey) \
	ASN1_item_verify(LogEntryBody::get_ASN1_ITEM(), x->sig->sig_alg, x->sig->signature,(char *)x->body, pkey)

#endif // #ifndef ASN1LOGS_H
