/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1User.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(USER_INFO) = {
	ASN1_SIMPLE(USER_INFO, Name, ASN1_UTF8STRING),
	ASN1_SIMPLE(USER_INFO, Activated, ASN1_INTEGER),
	ASN1_SIMPLE(USER_INFO, Flags, ASN1_INTEGER),
}ASN1_SEQUENCE_END(USER_INFO)
ASN1_SEQUENCE(USER_ENTRY_CERT) = {
	ASN1_SIMPLE(USER_ENTRY_CERT, user_cert, X509),
	ASN1_SIMPLE(USER_ENTRY_CERT, admin, ASN1_INTEGER),
}ASN1_SEQUENCE_END(USER_ENTRY_CERT)
ASN1_SEQUENCE(USER_ENTRY) = {
	ASN1_SIMPLE(USER_ENTRY, user_id, ASN1_INTEGER),
	ASN1_SIMPLE(USER_ENTRY, user_info, USER_INFO),
}ASN1_SEQUENCE_END(USER_ENTRY)
ASN1_SEQUENCE(CHANGE_USER_PASSWD) = {
	ASN1_SIMPLE(CHANGE_USER_PASSWD, user_id, ASN1_INTEGER),
	ASN1_SIMPLE(CHANGE_USER_PASSWD, Password, ASN1_UTF8STRING),
}ASN1_SEQUENCE_END(CHANGE_USER_PASSWD)
ASN1_SEQUENCE(CHANGE_PASSWD) = {
	ASN1_SIMPLE(CHANGE_PASSWD, Password, ASN1_UTF8STRING),
}ASN1_SEQUENCE_END(CHANGE_PASSWD)
ASN1_SEQUENCE(PKI_USER_KEY_SOFT) = {
	ASN1_SIMPLE(PKI_USER_KEY_SOFT, keylen, ASN1_INTEGER),
	ASN1_SIMPLE(PKI_USER_KEY_SOFT, password, ASN1_UTF8STRING),
}ASN1_SEQUENCE_END(PKI_USER_KEY_SOFT)
ASN1_CHOICE(PKI_USER_KEY) = {
	ASN1_EXP(PKI_USER_KEY, d.softkey, PKI_USER_KEY_SOFT, PKI_USER_KEY_TYPE_SOFTKEY),
	ASN1_EXP(PKI_USER_KEY, d.pubkey, X509_PUBKEY, PKI_USER_KEY_TYPE_PUBKEY),
} ASN1_CHOICE_END(PKI_USER_KEY)
ASN1_SEQUENCE(CREATE_PKI_USER_REQUEST) = {
	ASN1_SIMPLE(CREATE_PKI_USER_REQUEST, pkiadmin, ASN1_INTEGER),
	ASN1_SIMPLE(CREATE_PKI_USER_REQUEST, cn, ASN1_UTF8STRING),
	ASN1_SIMPLE(CREATE_PKI_USER_REQUEST, email, ASN1_UTF8STRING),
	ASN1_SIMPLE(CREATE_PKI_USER_REQUEST, ukey, PKI_USER_KEY),
}ASN1_SEQUENCE_END(CREATE_PKI_USER_REQUEST)
ASN1_CHOICE(CREATE_PKI_USER_RESPONSE) = {
	ASN1_EXP(CREATE_PKI_USER_RESPONSE, d.p12, PKCS12, PKI_USER_KEY_TYPE_SOFTKEY),
	ASN1_EXP(CREATE_PKI_USER_RESPONSE, d.p7b, PKCS7, PKI_USER_KEY_TYPE_PUBKEY),
} ASN1_CHOICE_END(CREATE_PKI_USER_RESPONSE)
UserInfo UserInfo::EmptyInstance;
bool UserInfo::set_activated(unsigned long c_activated)
{
	m_activated = c_activated;
	return true;
}

unsigned long UserInfo::get_activated() const
{
	return m_activated;
}

bool UserInfo::set_flags(unsigned long c_flags)
{
	m_flags = c_flags;
	return true;
}

unsigned long UserInfo::get_flags() const
{
	return m_flags;
}

bool UserInfo::set_name(const mString & c_name)
{
	m_name = c_name;
	return true;
}

const mString & UserInfo::get_name() const
{
	return m_name;
}

mString & UserInfo::get_name()
{
	return m_name;
}

UserInfo::UserInfo():NewPKIObject()
{
	resetAll();
}

UserInfo::UserInfo(const UserInfo & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

UserInfo::~UserInfo()
{
	Clear();
}

void UserInfo::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void UserInfo::freeAll()
{
}

void UserInfo::resetAll()
{
	m_activated = 0;
	m_flags = 0;
	m_name = "";
}

bool UserInfo::load_Datas(const USER_INFO * Datas)
{
	Clear();
	if(Datas->Activated)
	{
		m_activated = ASN1_INTEGER_GET(Datas->Activated);
	}
	if(Datas->Flags)
	{
		m_flags = ASN1_INTEGER_GET(Datas->Flags);
	}
	if(Datas->Name)
	{
		m_name = Datas->Name;
	}
	m_isOk=true;
	return true;
}

bool UserInfo::give_Datas(USER_INFO ** Datas) const
{
	if(!(*Datas) && !(*Datas = (USER_INFO*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Activated && !((*Datas)->Activated = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Activated, m_activated) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Activated);
		(*Datas)->Activated = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->Flags && !((*Datas)->Flags = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->Flags, m_flags) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->Flags);
		(*Datas)->Flags = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->Name && !((*Datas)->Name = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_name.c_ASN1_UTF8STRING(&(*Datas)->Name))
	{
		ASN1_UTF8STRING_free((*Datas)->Name);
		(*Datas)->Name = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool UserInfo::operator=(const UserInfo & other)
{
	Clear();
	m_activated = other.m_activated;
	m_flags = other.m_flags;
	m_name = other.m_name;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * UserInfo::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(USER_INFO);
}
UserEntryCert UserEntryCert::EmptyInstance;
bool UserEntryCert::set_admin(unsigned long c_admin)
{
	m_admin = c_admin;
	return true;
}

unsigned long UserEntryCert::get_admin() const
{
	return m_admin;
}

bool UserEntryCert::set_userCert(const PKI_CERT & c_userCert)
{
	m_userCert = c_userCert;
	return true;
}

const PKI_CERT & UserEntryCert::get_userCert() const
{
	return m_userCert;
}

PKI_CERT & UserEntryCert::get_userCert()
{
	return m_userCert;
}

UserEntryCert::UserEntryCert():NewPKIObject()
{
	resetAll();
}

UserEntryCert::UserEntryCert(const UserEntryCert & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

UserEntryCert::~UserEntryCert()
{
	Clear();
}

void UserEntryCert::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void UserEntryCert::freeAll()
{
}

void UserEntryCert::resetAll()
{
	m_admin = 0;
	m_userCert.Clear();
}

bool UserEntryCert::load_Datas(const USER_ENTRY_CERT * Datas)
{
	Clear();
	if(Datas->admin)
	{
		m_admin = ASN1_INTEGER_GET(Datas->admin);
	}
	if(Datas->user_cert)
	{
		if(!m_userCert.load_Datas(Datas->user_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool UserEntryCert::give_Datas(USER_ENTRY_CERT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (USER_ENTRY_CERT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->admin && !((*Datas)->admin = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->admin, m_admin) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->admin);
		(*Datas)->admin = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->user_cert && !((*Datas)->user_cert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_userCert.give_Datas(&(*Datas)->user_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->user_cert, ASN1_ITEM_rptr(X509));
		(*Datas)->user_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool UserEntryCert::operator=(const UserEntryCert & other)
{
	Clear();
	m_admin = other.m_admin;
	m_userCert = other.m_userCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * UserEntryCert::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(USER_ENTRY_CERT);
}
UserEntry UserEntry::EmptyInstance;
bool UserEntry::set_userId(unsigned long c_userId)
{
	m_userId = c_userId;
	return true;
}

unsigned long UserEntry::get_userId() const
{
	return m_userId;
}

bool UserEntry::set_userInfo(const UserInfo & c_userInfo)
{
	m_userInfo = c_userInfo;
	return true;
}

const UserInfo & UserEntry::get_userInfo() const
{
	return m_userInfo;
}

UserInfo & UserEntry::get_userInfo()
{
	return m_userInfo;
}

UserEntry::UserEntry():NewPKIObject()
{
	resetAll();
}

UserEntry::UserEntry(const UserEntry & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

UserEntry::~UserEntry()
{
	Clear();
}

void UserEntry::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void UserEntry::freeAll()
{
}

void UserEntry::resetAll()
{
	m_userId = 0;
	m_userInfo.Clear();
}

bool UserEntry::load_Datas(const USER_ENTRY * Datas)
{
	Clear();
	if(Datas->user_id)
	{
		m_userId = ASN1_INTEGER_GET(Datas->user_id);
	}
	if(Datas->user_info)
	{
		if(!m_userInfo.load_Datas(Datas->user_info))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool UserEntry::give_Datas(USER_ENTRY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (USER_ENTRY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->user_id && !((*Datas)->user_id = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->user_id, m_userId) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->user_id);
		(*Datas)->user_id = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->user_info && !((*Datas)->user_info = (USER_INFO*)ASN1_item_new(ASN1_ITEM_rptr(USER_INFO))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_userInfo.give_Datas(&(*Datas)->user_info))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->user_info, ASN1_ITEM_rptr(USER_INFO));
		(*Datas)->user_info = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool UserEntry::operator=(const UserEntry & other)
{
	Clear();
	m_userId = other.m_userId;
	m_userInfo = other.m_userInfo;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * UserEntry::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(USER_ENTRY);
}
ChangeUserPasswd ChangeUserPasswd::EmptyInstance;
bool ChangeUserPasswd::set_password(const mString & c_password)
{
	m_password = c_password;
	return true;
}

const mString & ChangeUserPasswd::get_password() const
{
	return m_password;
}

mString & ChangeUserPasswd::get_password()
{
	return m_password;
}

bool ChangeUserPasswd::set_userId(unsigned long c_userId)
{
	m_userId = c_userId;
	return true;
}

unsigned long ChangeUserPasswd::get_userId() const
{
	return m_userId;
}

ChangeUserPasswd::ChangeUserPasswd():NewPKIObject()
{
	resetAll();
}

ChangeUserPasswd::ChangeUserPasswd(const ChangeUserPasswd & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

ChangeUserPasswd::~ChangeUserPasswd()
{
	Clear();
}

void ChangeUserPasswd::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void ChangeUserPasswd::freeAll()
{
}

void ChangeUserPasswd::resetAll()
{
	m_password = "";
	m_userId = 0;
}

bool ChangeUserPasswd::load_Datas(const CHANGE_USER_PASSWD * Datas)
{
	Clear();
	if(Datas->Password)
	{
		m_password = Datas->Password;
	}
	if(Datas->user_id)
	{
		m_userId = ASN1_INTEGER_GET(Datas->user_id);
	}
	m_isOk=true;
	return true;
}

bool ChangeUserPasswd::give_Datas(CHANGE_USER_PASSWD ** Datas) const
{
	if(!(*Datas) && !(*Datas = (CHANGE_USER_PASSWD*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Password && !((*Datas)->Password = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_password.c_ASN1_UTF8STRING(&(*Datas)->Password))
	{
		ASN1_UTF8STRING_free((*Datas)->Password);
		(*Datas)->Password = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->user_id && !((*Datas)->user_id = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->user_id, m_userId) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->user_id);
		(*Datas)->user_id = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool ChangeUserPasswd::operator=(const ChangeUserPasswd & other)
{
	Clear();
	m_password = other.m_password;
	m_userId = other.m_userId;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * ChangeUserPasswd::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(CHANGE_USER_PASSWD);
}
ChangePasswd ChangePasswd::EmptyInstance;
bool ChangePasswd::set_password(const mString & c_password)
{
	m_password = c_password;
	return true;
}

const mString & ChangePasswd::get_password() const
{
	return m_password;
}

mString & ChangePasswd::get_password()
{
	return m_password;
}

ChangePasswd::ChangePasswd():NewPKIObject()
{
	resetAll();
}

ChangePasswd::ChangePasswd(const ChangePasswd & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

ChangePasswd::~ChangePasswd()
{
	Clear();
}

void ChangePasswd::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void ChangePasswd::freeAll()
{
}

void ChangePasswd::resetAll()
{
	m_password = "";
}

bool ChangePasswd::load_Datas(const CHANGE_PASSWD * Datas)
{
	Clear();
	if(Datas->Password)
	{
		m_password = Datas->Password;
	}
	m_isOk=true;
	return true;
}

bool ChangePasswd::give_Datas(CHANGE_PASSWD ** Datas) const
{
	if(!(*Datas) && !(*Datas = (CHANGE_PASSWD*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->Password && !((*Datas)->Password = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_password.c_ASN1_UTF8STRING(&(*Datas)->Password))
	{
		ASN1_UTF8STRING_free((*Datas)->Password);
		(*Datas)->Password = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool ChangePasswd::operator=(const ChangePasswd & other)
{
	Clear();
	m_password = other.m_password;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * ChangePasswd::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(CHANGE_PASSWD);
}
PkiUserKeySoft PkiUserKeySoft::EmptyInstance;
bool PkiUserKeySoft::set_keylen(unsigned long c_keylen)
{
	m_keylen = c_keylen;
	return true;
}

unsigned long PkiUserKeySoft::get_keylen() const
{
	return m_keylen;
}

bool PkiUserKeySoft::set_password(const mString & c_password)
{
	m_password = c_password;
	return true;
}

const mString & PkiUserKeySoft::get_password() const
{
	return m_password;
}

mString & PkiUserKeySoft::get_password()
{
	return m_password;
}

PkiUserKeySoft::PkiUserKeySoft():NewPKIObject()
{
	resetAll();
}

PkiUserKeySoft::PkiUserKeySoft(const PkiUserKeySoft & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

PkiUserKeySoft::~PkiUserKeySoft()
{
	Clear();
}

void PkiUserKeySoft::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PkiUserKeySoft::freeAll()
{
}

void PkiUserKeySoft::resetAll()
{
	m_keylen = 0;
	m_password = "";
}

bool PkiUserKeySoft::load_Datas(const PKI_USER_KEY_SOFT * Datas)
{
	Clear();
	if(Datas->keylen)
	{
		m_keylen = ASN1_INTEGER_GET(Datas->keylen);
	}
	if(Datas->password)
	{
		m_password = Datas->password;
	}
	m_isOk=true;
	return true;
}

bool PkiUserKeySoft::give_Datas(PKI_USER_KEY_SOFT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PKI_USER_KEY_SOFT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->keylen && !((*Datas)->keylen = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->keylen, m_keylen) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->keylen);
		(*Datas)->keylen = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->password && !((*Datas)->password = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_password.c_ASN1_UTF8STRING(&(*Datas)->password))
	{
		ASN1_UTF8STRING_free((*Datas)->password);
		(*Datas)->password = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool PkiUserKeySoft::operator=(const PkiUserKeySoft & other)
{
	Clear();
	m_keylen = other.m_keylen;
	m_password = other.m_password;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PkiUserKeySoft::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PKI_USER_KEY_SOFT);
}
PkiUserKey PkiUserKey::EmptyInstance;
bool PkiUserKey::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int PkiUserKey::get_type() const
{
	 return m_type;
}

bool PkiUserKey::set_pubkey(const X509_PUBKEY * c_pubkey)
{
	if(m_type != 1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(m_pubkey)
		ASN1_item_free((ASN1_VALUE*)m_pubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	m_pubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_pubkey);
	if(!m_pubkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_isOk=true;
	return true;
}

const X509_PUBKEY * PkiUserKey::get_pubkey() const
{
	if((m_type != 1) || !m_pubkey)
	{
		return NULL;
	}
	if(!m_pubkey)
		((PkiUserKey*)this)->m_pubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_pubkey;
}

X509_PUBKEY * PkiUserKey::get_pubkey()
{
	if((m_type != 1) || !m_pubkey)
	{
		return NULL;
	}
	if(!m_pubkey)
		m_pubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_pubkey;
}

bool PkiUserKey::set_softkey(const PkiUserKeySoft & c_softkey)
{
	if(m_type != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_softkey) = c_softkey;
	m_isOk=true;
	return true;
}

const PkiUserKeySoft & PkiUserKey::get_softkey() const
{
	if((m_type != 0) || !m_softkey)
	{
		return PkiUserKeySoft::EmptyInstance;
	}
	return (*m_softkey);
}

PkiUserKeySoft & PkiUserKey::get_softkey()
{
	if((m_type != 0) || !m_softkey)
	{
		return PkiUserKeySoft::EmptyInstance;
	}
	return (*m_softkey);
}

bool PkiUserKey::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 1:
			m_pubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!m_pubkey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 0:
			m_softkey = new PkiUserKeySoft();
			if(!m_softkey)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

PkiUserKey::PkiUserKey():NewPKIObject()
{
	resetAll();
}

PkiUserKey::PkiUserKey(const PkiUserKey & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

PkiUserKey::~PkiUserKey()
{
	Clear();
}

void PkiUserKey::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void PkiUserKey::freeAll()
{
	if(m_pubkey)
	{
		ASN1_item_free((ASN1_VALUE*)m_pubkey, ASN1_ITEM_rptr(X509_PUBKEY));
	}
	if(m_softkey)
	{
		delete m_softkey;
	}
}

void PkiUserKey::resetAll()
{
	m_type = -1;
	m_pubkey = NULL;
	m_softkey = NULL;
}

bool PkiUserKey::load_Datas(const PKI_USER_KEY * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 1:
			if(Datas->d.pubkey)
			{
				if(m_pubkey)
					ASN1_item_free((ASN1_VALUE*)m_pubkey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_pubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->d.pubkey);
				if(!m_pubkey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			break;
		case 0:
			if(Datas->d.softkey)
			{
				if(!(*m_softkey).load_Datas(Datas->d.softkey))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool PkiUserKey::give_Datas(PKI_USER_KEY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (PKI_USER_KEY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 1:
			if(m_pubkey)
			{
				if((*Datas)->d.pubkey)
					ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pubkey, ASN1_ITEM_rptr(X509_PUBKEY));
				if(!((*Datas)->d.pubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_pubkey)))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
			}
			else
			{
				if(!(*Datas)->d.pubkey)
				{
					(*Datas)->d.pubkey = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
					if(!(*Datas)->d.pubkey)
					{
						NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
						return false;
					}
				}
			}
			break;
		case 0:
			if(!((*Datas)->d.softkey = (PKI_USER_KEY_SOFT*)ASN1_item_new(ASN1_ITEM_rptr(PKI_USER_KEY_SOFT))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_softkey).give_Datas(&(*Datas)->d.softkey))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.softkey, ASN1_ITEM_rptr(PKI_USER_KEY_SOFT));
				(*Datas)->d.softkey = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool PkiUserKey::operator=(const PkiUserKey & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 1:
			if(other.m_pubkey)
			{
				if(m_pubkey)
					ASN1_item_free((ASN1_VALUE*)m_pubkey, ASN1_ITEM_rptr(X509_PUBKEY));
				m_pubkey = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_pubkey);
				if(!m_pubkey)
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 0:
			if(!other.m_softkey)
				break;
			(*m_softkey) = *(other.m_softkey);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * PkiUserKey::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(PKI_USER_KEY);
}
CreatePkiUserRequest CreatePkiUserRequest::EmptyInstance;
bool CreatePkiUserRequest::set_cn(const mString & c_cn)
{
	m_cn = c_cn;
	return true;
}

const mString & CreatePkiUserRequest::get_cn() const
{
	return m_cn;
}

mString & CreatePkiUserRequest::get_cn()
{
	return m_cn;
}

bool CreatePkiUserRequest::set_email(const mString & c_email)
{
	m_email = c_email;
	return true;
}

const mString & CreatePkiUserRequest::get_email() const
{
	return m_email;
}

mString & CreatePkiUserRequest::get_email()
{
	return m_email;
}

bool CreatePkiUserRequest::set_pkiadmin(unsigned long c_pkiadmin)
{
	m_pkiadmin = c_pkiadmin;
	return true;
}

unsigned long CreatePkiUserRequest::get_pkiadmin() const
{
	return m_pkiadmin;
}

bool CreatePkiUserRequest::set_ukey(const PkiUserKey & c_ukey)
{
	m_ukey = c_ukey;
	return true;
}

const PkiUserKey & CreatePkiUserRequest::get_ukey() const
{
	return m_ukey;
}

PkiUserKey & CreatePkiUserRequest::get_ukey()
{
	return m_ukey;
}

CreatePkiUserRequest::CreatePkiUserRequest():NewPKIObject()
{
	resetAll();
}

CreatePkiUserRequest::CreatePkiUserRequest(const CreatePkiUserRequest & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

CreatePkiUserRequest::~CreatePkiUserRequest()
{
	Clear();
}

void CreatePkiUserRequest::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void CreatePkiUserRequest::freeAll()
{
}

void CreatePkiUserRequest::resetAll()
{
	m_cn = "";
	m_email = "";
	m_pkiadmin = 0;
	m_ukey.Clear();
}

bool CreatePkiUserRequest::load_Datas(const CREATE_PKI_USER_REQUEST * Datas)
{
	Clear();
	if(Datas->cn)
	{
		m_cn = Datas->cn;
	}
	if(Datas->email)
	{
		m_email = Datas->email;
	}
	if(Datas->pkiadmin)
	{
		m_pkiadmin = ASN1_INTEGER_GET(Datas->pkiadmin);
	}
	if(Datas->ukey)
	{
		if(!m_ukey.load_Datas(Datas->ukey))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool CreatePkiUserRequest::give_Datas(CREATE_PKI_USER_REQUEST ** Datas) const
{
	if(!(*Datas) && !(*Datas = (CREATE_PKI_USER_REQUEST*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->cn && !((*Datas)->cn = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cn.c_ASN1_UTF8STRING(&(*Datas)->cn))
	{
		ASN1_UTF8STRING_free((*Datas)->cn);
		(*Datas)->cn = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->email && !((*Datas)->email = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_email.c_ASN1_UTF8STRING(&(*Datas)->email))
	{
		ASN1_UTF8STRING_free((*Datas)->email);
		(*Datas)->email = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->pkiadmin && !((*Datas)->pkiadmin = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->pkiadmin, m_pkiadmin) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->pkiadmin);
		(*Datas)->pkiadmin = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->ukey && !((*Datas)->ukey = (PKI_USER_KEY*)ASN1_item_new(ASN1_ITEM_rptr(PKI_USER_KEY))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_ukey.give_Datas(&(*Datas)->ukey))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->ukey, ASN1_ITEM_rptr(PKI_USER_KEY));
		(*Datas)->ukey = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool CreatePkiUserRequest::operator=(const CreatePkiUserRequest & other)
{
	Clear();
	m_cn = other.m_cn;
	m_email = other.m_email;
	m_pkiadmin = other.m_pkiadmin;
	m_ukey = other.m_ukey;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * CreatePkiUserRequest::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(CREATE_PKI_USER_REQUEST);
}
CreatePkiUserResponse CreatePkiUserResponse::EmptyInstance;
bool CreatePkiUserResponse::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int CreatePkiUserResponse::get_type() const
{
	 return m_type;
}

bool CreatePkiUserResponse::set_p12(const PKI_PKCS12 & c_p12)
{
	if(m_type != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_p12) = c_p12;
	m_isOk=true;
	return true;
}

const PKI_PKCS12 & CreatePkiUserResponse::get_p12() const
{
	if((m_type != 0) || !m_p12)
	{
		return PKI_PKCS12::EmptyInstance;
	}
	return (*m_p12);
}

PKI_PKCS12 & CreatePkiUserResponse::get_p12()
{
	if((m_type != 0) || !m_p12)
	{
		return PKI_PKCS12::EmptyInstance;
	}
	return (*m_p12);
}

bool CreatePkiUserResponse::set_p7b(const PKI_P7B & c_p7b)
{
	if(m_type != 1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_p7b) = c_p7b;
	m_isOk=true;
	return true;
}

const PKI_P7B & CreatePkiUserResponse::get_p7b() const
{
	if((m_type != 1) || !m_p7b)
	{
		return PKI_P7B::EmptyInstance;
	}
	return (*m_p7b);
}

PKI_P7B & CreatePkiUserResponse::get_p7b()
{
	if((m_type != 1) || !m_p7b)
	{
		return PKI_P7B::EmptyInstance;
	}
	return (*m_p7b);
}

bool CreatePkiUserResponse::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 0:
			m_p12 = new PKI_PKCS12();
			if(!m_p12)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 1:
			m_p7b = new PKI_P7B();
			if(!m_p7b)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

CreatePkiUserResponse::CreatePkiUserResponse():NewPKIObject()
{
	resetAll();
}

CreatePkiUserResponse::CreatePkiUserResponse(const CreatePkiUserResponse & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

CreatePkiUserResponse::~CreatePkiUserResponse()
{
	Clear();
}

void CreatePkiUserResponse::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void CreatePkiUserResponse::freeAll()
{
	if(m_p12)
	{
		delete m_p12;
	}
	if(m_p7b)
	{
		delete m_p7b;
	}
}

void CreatePkiUserResponse::resetAll()
{
	m_type = -1;
	m_p12 = NULL;
	m_p7b = NULL;
}

bool CreatePkiUserResponse::load_Datas(const CREATE_PKI_USER_RESPONSE * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 0:
			if(Datas->d.p12)
			{
				if(!(*m_p12).load_Datas(Datas->d.p12))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 1:
			if(Datas->d.p7b)
			{
				if(!(*m_p7b).load_Datas(Datas->d.p7b))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool CreatePkiUserResponse::give_Datas(CREATE_PKI_USER_RESPONSE ** Datas) const
{
	if(!(*Datas) && !(*Datas = (CREATE_PKI_USER_RESPONSE*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 0:
			if(!((*Datas)->d.p12 = (PKCS12*)ASN1_item_new(ASN1_ITEM_rptr(PKCS12))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_p12).give_Datas(&(*Datas)->d.p12))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.p12, ASN1_ITEM_rptr(PKCS12));
				(*Datas)->d.p12 = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 1:
			if(!((*Datas)->d.p7b = (PKCS7*)ASN1_item_new(ASN1_ITEM_rptr(PKCS7))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_p7b).give_Datas(&(*Datas)->d.p7b))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.p7b, ASN1_ITEM_rptr(PKCS7));
				(*Datas)->d.p7b = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool CreatePkiUserResponse::operator=(const CreatePkiUserResponse & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 0:
			if(!other.m_p12)
				break;
			(*m_p12) = *(other.m_p12);
			break;
		case 1:
			if(!other.m_p7b)
				break;
			(*m_p7b) = *(other.m_p7b);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * CreatePkiUserResponse::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(CREATE_PKI_USER_RESPONSE);
}
