/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CERT.cpp: implementation of the PKI_CERT class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_CERT.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PKI_CERT PKI_CERT::EmptyInstance;

PKI_CERT::PKI_CERT(const PKI_CERT & other)
{
	Reset();
	*this = other;
}

PKI_CERT::PKI_CERT(const char * PEM_CER, const PKI_RSA & PrivateKey)
{
	Reset();
	if(!SetCert(PEM_CER))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
	if(PrivateKey && !SetPrivateKey(PrivateKey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_CERT::PKI_CERT(const X509 *certificat, const PKI_RSA & PrivateKey)
{
	Reset();
	if(!SetCert(certificat))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
	if(PrivateKey && !SetPrivateKey(PrivateKey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_CERT::PKI_CERT()
{
	Reset();
}

PKI_CERT::~PKI_CERT()
{
	Clear();
}

void PKI_CERT::Clear()
{
	if(cert)
		X509_free(cert);
	if(public_key)
		EVP_PKEY_free(public_key);
	Reset();
}


bool PKI_CERT::SetCert(const char *PEM_CER)
{
	Clear();

	if(!PEM_CER)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	cert=X509_new();
	if(!cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	if(!StringToX509(PEM_CER))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!LoadDatas())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
		return false;
	}

	return true;
}

bool PKI_CERT::SetCert(const X509 *certificate)
{
	CRYPTO_w_lock(CRYPTO_LOCK_X509);
	if(!certificate)
	{
		CRYPTO_w_unlock(CRYPTO_LOCK_X509);
		Clear();
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	// We ignore the same certificate
	if(certificate == cert)
	{
		CRYPTO_w_unlock(CRYPTO_LOCK_X509);
		return true;
	}
//	CRYPTO_add(&((X509 *)certificate)->references, 1, CRYPTO_LOCK_X509);
	((X509 *)certificate)->references++;
	CRYPTO_w_unlock(CRYPTO_LOCK_X509);

	Clear();
	cert = (X509 *)certificate;

	if(!LoadDatas())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


bool PKI_CERT::StringToX509(const char * pem)
{
	int dsize=0;
	unsigned char *data=NULL;
	unsigned char *tmp_data=NULL;
	PEM_DER der_cvrt;
	BIO * sout;

	if(*pem == '-')
	{
		sout = BIO_new_mem_buf((char*)pem, strlen(pem));
		if(!sout)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
		if(!PEM_read_bio_X509(sout, &cert, NULL, NULL))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			BIO_free_all(sout);
			return false;
		}
		BIO_free_all(sout);
		return true;
	}

	if(!der_cvrt.Pem2Der(pem, strlen(pem), (char **)&data, &dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	tmp_data=data;
	if(!d2i_X509(&cert,&data,dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(tmp_data);
		return false;
	}
	free(tmp_data);
	return true;
}

bool PKI_CERT::X509ToString(const X509 * certificate)
{
	int dsize=0;
	unsigned char *p,*data=NULL;
	
	if ((dsize=i2d_X509((X509*)certificate,NULL)) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CSR_X509REQ);
		return false;
	}

	/* dzise + 8 bytes are needed */
	data=(unsigned char *)malloc(dsize+20);
	if (!data)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p=data;
	if((dsize=i2d_X509((X509*)certificate,&p))<0)
	{
		free(data);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CSR_X509REQ);
		return false;
	}

	if(!pem_cert.FromDER(data, dsize))
	{
		free(data);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	free(data);
	return true;
}

bool PKI_CERT::LoadAll()
{
	if(cert && !leDN.EntriesCount() && 
		!LoadDN())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(cert && !IssuerDN.EntriesCount() && 
		!LoadIssuerDN())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(cert && !InternalExts.GetExts().EntriesCount() && 
		!InternalExts.Load(cert->cert_info->extensions))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(cert && !m_Thumprint.size())
	{
		((PKI_CERT*)this)->LoadThumbprint();
	}
	if(cert && !pem_cert.size() &&
		!X509ToString(cert))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool PKI_CERT::LoadDN()
{
	X509_NAME * name;
	name=X509_get_subject_name(cert);
	if(!name)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!leDN.From_X509_NAME(name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

bool PKI_CERT::LoadIssuerDN()
{
	X509_NAME * name;
	name=X509_get_issuer_name(cert);
	if(!name)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!IssuerDN.From_X509_NAME(name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	return true;
}

const HashTable_Dn & PKI_CERT::GetCertDN() const
{
	if(cert && !leDN.EntriesCount())
	{
		if(!((PKI_CERT*)this)->LoadDN())
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
	}
	return leDN;
}

X509 * PKI_CERT::GetX509(bool Duplicate) const
{
	if(!cert)
		return NULL;
	if(Duplicate)
	{
		CRYPTO_add(&cert->references, 1, CRYPTO_LOCK_X509);
	}
	return cert;
}

const PKI_RSA & PKI_CERT::GetPrivateKey() const
{
	return rsakey;
}

const mString & PKI_CERT::GetCertPEM() const
{
	if(cert && !pem_cert.size())
	{
		if(!((PKI_CERT*)this)->X509ToString(cert))
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
	}
	return pem_cert;
}

time_t PKI_CERT::GetStartDate() const
{
	if(!cert) return 0;
	ASN1_UTCTIME *tm;
	tm=X509_get_notBefore(cert);
	if(!tm) return 0;
	return ASN1_TIME_to_timet((char*)tm->data);
}

time_t PKI_CERT::GetEndDate() const
{
	if(!cert) return 0;
	ASN1_UTCTIME *tm;
	tm=X509_get_notAfter(cert);
	if(!tm) return 0;
	return ASN1_TIME_to_timet((char*)tm->data);
}


bool PKI_CERT::operator=( const PKI_CERT & other)
{
	// Trying to copy myself on me
	if(&other.cert == &cert)
		return true;

	Clear();
	if(!other.cert)
		return false;

	CRYPTO_add(&other.cert->references, 1, CRYPTO_LOCK_X509);
	cert = other.cert;

	if( !(public_key = X509_get_pubkey(cert)) )
	{
		Clear();
		return false;
	}

	leDN = other.leDN;
	IssuerDN = other.leDN;
	InternalExts = other.InternalExts;
	pem_cert = other.pem_cert;
	rsakey = other.rsakey;
	m_Thumprint = other.m_Thumprint;
	m_Name = other.m_Name;
	return true;
}


bool PKI_CERT::operator==( const PKI_CERT & other ) const
{
	if(!cert || !other.cert) return false;

	return (GetCertPEM() == other.GetCertPEM());
}

bool PKI_CERT::operator==( const X509 * other ) const
{
	if(!cert || !other) return false;

	return (X509_cmp(cert, other) == 0);
}

bool PKI_CERT::operator!=( const X509 * other ) const
{
	if(!cert || !other) return false;

	return (X509_cmp(cert, other) != 0);
}

bool PKI_CERT::operator==( const X509_PUBKEY * pubkey ) const
{
	if(!cert || !pubkey) return false;

	if(cert->cert_info->key->public_key->length ==
		pubkey->public_key->length &&
		memcmp(cert->cert_info->key->public_key->data,
		pubkey->public_key->data, 
		pubkey->public_key->length) == 0)
	{
		return true;
	}
	return false;
}

const EVP_PKEY * PKI_CERT::GetPublicKey() const
{
	return public_key;
}

bool PKI_CERT::SetPrivateKey(const PKI_RSA & PrivateKey, bool CheckKey)
{
	if(PrivateKey)
	{
		rsakey = PrivateKey;
		if(!rsakey)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}

		if (CheckKey && cert && X509_check_private_key(cert, (EVP_PKEY*)rsakey.GetRsaKey()) < 0)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
			return false;
		}		
	}
	return true;
}

bool PKI_CERT::SignCSR(PKI_CERT & ResultCert, const PKI_CSR & ReqCSR, 
					   const HashTable_String * Exts, int Days, 
					   long Serial, const char * md,
					   bool keepCsrExts, bool csrExtsOverwrite, 
					   bool check_sig) const
{
	X509_REQ * req;

	X509 *ret=NULL;
	X509_NAME *subject=NULL;
	X509_CINF *cert_info;
	X509V3_CTX ctx;
	const EVP_PKEY * private_key;
	HashTable_String ExpandedExts;

	EVP_PKEY * pktmp;
	const EVP_MD * digest;
	

	//We get our own private key for signature
	private_key = rsakey.GetRsaKey();
	if(!private_key)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
		return false;
	}

	//We get the digest
	if (md)
		digest=(EVP_MD*)EVP_get_digestbyname(md);
	else
		digest=(EVP_MD *)EVP_md5();

	if (!digest)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We get the CSR
	req=ReqCSR.GetX509_REQ();
	if(!req)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(check_sig)
	{
		//Set public key
		pktmp=X509_REQ_get_pubkey(req);
		if(!pktmp)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if (X509_REQ_verify(req,pktmp) <= 0)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			EVP_PKEY_free(pktmp);
			return false;
		}
		EVP_PKEY_free(pktmp);
	}

	// We get the csr's dn
	subject=X509_REQ_get_subject_name(req);
	if(!subject)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	if ((ret=X509_new()) == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	cert_info=ret->cert_info;

	// Make it an X509 v3 certificate.
	if (!X509_set_version(ret,2))
	{	
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	//Set serial
	if (!ASN1_INTEGER_set(cert_info->serialNumber,Serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}
	
	//Set issuer
	if (!X509_set_issuer_name(ret,X509_get_subject_name(cert)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	//Set start time
	X509_gmtime_adj(X509_get_notBefore(ret),0);

	//Set end time
	X509_gmtime_adj(X509_get_notAfter(ret),(long)60*60*24*Days);


	//Set subject
	if (!X509_set_subject_name(ret,subject))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	//Set public key
	pktmp=X509_REQ_get_pubkey(req);
	if(!pktmp)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}
	if (!X509_set_pubkey(ret,pktmp))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		EVP_PKEY_free(pktmp);
		return false;
	}
	EVP_PKEY_free(pktmp);


	//Set the extensions
	X509V3_set_ctx(&ctx, cert, ret, req, NULL, 0);
	if(!PKI_EXT::Add_CertExtensions(Exts, &ctx, keepCsrExts, csrExtsOverwrite, ret))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	//We sign the certificate
	if (!X509_sign(ret, (EVP_PKEY*)private_key, digest))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	if(!ResultCert.SetCert(ret))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	X509_free(ret);

	return true;
}

bool PKI_CERT::CreateSelfSigned(const PKI_CSR & ReqCSR, const HashTable_String *Exts, int Days, long Serial)
{
	X509_NAME *subject=NULL;
	X509 *ret=NULL;
	X509_CINF *cert_info;
	const EVP_PKEY * private_key;
	EVP_PKEY * pktmp;
	const EVP_MD * digest;
	X509_REQ * req;
	X509V3_CTX ctx;

	//We get the CSR's private key to assign it to us
	const PKI_RSA & reqprivkey = ReqCSR.GetRsaKey();
	if(!reqprivkey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
		return false;
	}

	//We get our own private key for signature
	private_key = reqprivkey.GetRsaKey();
	if(!private_key)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
		return false;
	}

	//We get the digest
	digest=(EVP_MD *)EVP_sha1();
	if (!digest)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//We get the CSR
	req=ReqCSR.GetX509_REQ();
	if(!req)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}



	// We get the DN from the CSR
	subject=X509_REQ_get_subject_name(req);

	if ((ret=X509_new()) == NULL)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	cert_info=ret->cert_info;

	/* Make it an X509 v3 certificate. */
	if (!X509_set_version(ret,2))
	{	
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CERTIFY_CSR);
		X509_free(ret);
		return false;
	}

	if (!ASN1_INTEGER_set(cert_info->serialNumber,Serial))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CERTIFY_CSR);
		X509_free(ret);
		return false;
	}
	
	if (!X509_set_issuer_name(ret,subject))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CERTIFY_CSR);
		X509_free(ret);
		return false;
	}

	X509_gmtime_adj(X509_get_notBefore(ret),0);

	X509_gmtime_adj(X509_get_notAfter(ret),(long)60*60*24*Days);

	if (!X509_set_subject_name(ret,subject))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CERTIFY_CSR);
		X509_free(ret);
		return false;
	}

	//Set public key
	pktmp=X509_REQ_get_pubkey(req);
	if(!pktmp)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}
	if (!X509_set_pubkey(ret,pktmp))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		EVP_PKEY_free(pktmp);
		return false;
	}
	EVP_PKEY_free(pktmp);

	/* Lets add the extensions, if there are any */
	/* Free the current entries if any, there should not
	 * be any I believe */
	if (cert_info->extensions != NULL)
	{
		sk_X509_EXTENSION_pop_free(cert_info->extensions,X509_EXTENSION_free);
		cert_info->extensions = NULL;
	}

	
	X509V3_set_ctx(&ctx, ret, ret, req, NULL, 0);

	if(Exts && !InternalExts.Add_CertExtensions(Exts, &ctx, false, false, ret))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		X509_free(ret);
		return false;
	}

	if (!X509_sign(ret, (EVP_PKEY*)private_key, digest))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CERTIFY_CSR);
		X509_free(ret);
		return false;
	}

	if(!SetCert(ret))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!SetPrivateKey(reqprivkey, false))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	X509_free(ret);

	return true;
}

PKI_CERT::operator int() const
{
	if(!cert)
		return 0;
	else
		return 1;
}


unsigned long PKI_CERT::GetSerial() const
{
	ASN1_INTEGER *bs;
	bs=X509_get_serialNumber(cert);
	if(bs)
		return ASN1_INTEGER_get(bs);
	else
		return 0;
}

const char * PKI_CERT::GetSignatureAlg() const
{
	int nid;
	nid = OBJ_obj2nid(cert->cert_info->signature->algorithm);
	if(nid == NID_undef) return NULL;
	return (char*)OBJ_nid2sn(nid);
}

const HashTable_Dn & PKI_CERT::GetIssuerDN() const
{
	if(cert && !IssuerDN.EntriesCount())
	{
		if(!((PKI_CERT*)this)->LoadIssuerDN())
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
	}
	return IssuerDN;
}

unsigned long PKI_CERT::GetKeyLen() const
{
	if(!public_key) return 0;
	return BN_num_bits(public_key->pkey.rsa->n);
}

bool PKI_CERT::LoadDatas()
{
	char * name;
	if( !(public_key = X509_get_pubkey(cert)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CSR_X509REQ);
		return false;
	}

	if (rsakey && !X509_check_private_key(cert, (EVP_PKEY*)rsakey.GetRsaKey()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MISMATCH_KEYS);
		return false;
	}

	if(!cert->name)
	{
		name = X509_NAME_oneline(cert->cert_info->subject, NULL, 0);
		if(!name)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		m_Name = name;
		free(name);
	}
	else
	{
		m_Name = cert->name;
	}

	return true;
}

const HashTable_String & PKI_CERT::GetExtensions() const
{
	if(cert && !InternalExts.GetExts().EntriesCount())
	{
		if(!((PKI_CERT*)this)->InternalExts.Load(cert->cert_info->extensions))
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
	}
	return InternalExts.GetExts();
}

const char * PKI_CERT::GetThumprintAlgo() const
{
	int nid;
	nid = OBJ_obj2nid(cert->sig_alg->algorithm);
	if(nid == NID_undef) return NULL;
	return (char*)OBJ_nid2sn(nid);
}

const char * PKI_CERT::GetThumprint() const
{
	if(cert && !m_Thumprint.size())
	{
		((PKI_CERT*)this)->LoadThumbprint();
	}
	return m_Thumprint.c_str();
}

void PKI_CERT::LoadThumbprint()
{
	char * Thumprint;
	unsigned char *s;
	int i, n;

	BIO * membio = BIO_new(BIO_s_mem());
	if(!membio) return;


	n=cert->signature->length;
	s=cert->signature->data;
	for (i=0; i<n; i++)
	{
		if ((i%18) == 0 && i)
			BIO_write(membio,"\r\n",2);
	
		BIO_printf(membio,"%02x%s",s[i], ((i+1) == n)?"":":");
	}


	Thumprint = (char *)malloc(BIO_number_written(membio) + 1);
	if(!Thumprint)
	{
		BIO_free_all(membio);
		return;
	}

	BIO_read(membio, Thumprint, BIO_number_written(membio));
	Thumprint[BIO_number_written(membio)]=0;
	BIO_free_all(membio);
	m_Thumprint = Thumprint;
	free(Thumprint);
}

unsigned long PKI_CERT::GetVersion() const
{
	return X509_get_version(cert);
}

const char * PKI_CERT::GetStringName() const
{
	if(!m_Name.size())
		return "";
	else
		return m_Name.c_str();
}


const X509_PUBKEY * PKI_CERT::GetX509_PUBKEY() const
{
	if(!cert)
		return NULL;
	else
		return cert->cert_info->key;
}

void PKI_CERT::Reset()
{
	cert = NULL;
	leDN.Clear();
	IssuerDN.Clear();
	InternalExts.Clear();
	pem_cert = "";
	public_key = NULL;
	rsakey.Clear();
	m_Thumprint = "";
	m_Name = "";
}

bool PKI_CERT::load_Datas(const X509 *certificate)
{
	return SetCert(certificate);
}

bool PKI_CERT::give_Datas(X509 **Cert) const
{
	if(*Cert)
	{
		X509_free(*Cert);
	}
	if(!cert)
	{
		*Cert = NULL;
	}
	else
	{
		*Cert = GetX509(true);
		if(!*Cert)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
		}
	}

	return true;
}

bool PKI_CERT::LoadFromFile(const mString &File)
{
	FILE * fp;
	X509 * x509;

	fp = fopen(File.c_str(), "rb");
	if(!fp)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		ERR_add_error_data(2, File.c_str(), strerror(errno));
		return false;
	}
	if( !(x509 = d2i_X509_fp(fp, NULL)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ERR_add_error_data(1, File.c_str());
		fclose(fp);
		return false;
	}
	fclose(fp);
	if(!SetCert(x509))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ERR_add_error_data(1, File.c_str());
		X509_free(x509);
		return false;
	}
	X509_free(x509);
	return true;
}
