/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKCS12.cpp: implementation of the PKCS12 class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_P7B.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PKI_P7B PKI_P7B::EmptyInstance;

PKI_P7B::PKI_P7B(const PKI_P7B & other)
{
	m_Certs = SKM_sk_new_null(X509);
	m_p7b = NULL;
	*this = other;
}

PKI_P7B::PKI_P7B()
{
	m_Certs = SKM_sk_new_null(X509);
	m_p7b = NULL;
}

PKI_P7B::~PKI_P7B()
{
	Clear();
	if(m_Certs) 
		SKM_sk_pop_free(X509, m_Certs, X509_free);
}

void PKI_P7B::Clear(bool CertsList)
{
	X509 * currCert;
	if(m_p7b)
	{
		PKCS7_free(m_p7b);
		m_p7b = NULL;
	}
	m_Pkcs7Pem = "";
	if(CertsList && m_Certs)
	{
		while( (currCert = SKM_sk_pop(X509, m_Certs)) )
		{
			X509_free(currCert);
		}
	}
}

bool PKI_P7B::Generate()
{
	Clear(false);

	X509 * currCert;
	int i;

	if(m_p7b)
		PKCS7_free(m_p7b);

	m_p7b = PKCS7_new();
	if(!m_p7b)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC); 
		return false;
	}

	if(PKCS7_set_type(m_p7b, NID_pkcs7_signed) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT); 
		return false;
	}

	for(i=0; i<SKM_sk_num(X509, m_Certs); i++)
	{
		currCert = SKM_sk_value(X509, m_Certs, i);
		if(!currCert)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
		PKCS7_add_certificate(m_p7b, currCert);
	}

	if(!PKCS7ToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;	
}

bool PKI_P7B::StringToPKCS7(const char * p7pem)
{
	int dsize=0;
	unsigned char *data=NULL;
	unsigned char *tmp_data=NULL;
	PEM_DER der_cvrt;
	BIO * sout;

	if(*p7pem == '-')
	{
		sout = BIO_new_mem_buf((char*)p7pem, strlen(p7pem));
		if(!sout)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
		if(!PEM_read_bio_PKCS7(sout, &m_p7b, NULL, NULL))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			BIO_free_all(sout);
			return false;
		}
		BIO_free_all(sout);
		return true;
	}

	if(!der_cvrt.Pem2Der(p7pem, strlen(p7pem), (char **)&data, &dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	tmp_data=data;
	if(!d2i_PKCS7(&m_p7b,&data,dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		free(tmp_data);
		return false;
	}
	free(tmp_data);
	return true;
}

bool PKI_P7B::PKCS7ToString()
{
	int n;
	PEM_DER pem_cvrt;
	char * b;
	unsigned char * p;

	n=i2d_PKCS7(m_p7b,NULL);
	if(!n)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	b=(char *)malloc(n);
	if (!b)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p=(unsigned char *)b;

	n = i2d_PKCS7(m_p7b,&p);
	if(!n)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GEN_P12);
		free(b);
		return false;
	}

	

	if(!m_Pkcs7Pem.FromDER((unsigned char*)b, n))
	{
		free(b);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	free(b);

	return true;
}

const mString & PKI_P7B::GetPemP7B() const
{
	return m_Pkcs7Pem;
}

PKCS7 * PKI_P7B::GetPKCS7(bool Duplicate) const
{
	if(Duplicate && m_p7b)
	{
		return PKCS7_dup(m_p7b);
	}
	
	return m_p7b;
}

bool PKI_P7B::AddCert(X509 * Cert)
{
	if(!Cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	CRYPTO_add(&Cert->references,1,CRYPTO_LOCK_X509);
	
	if(SKM_sk_push(X509, m_Certs, Cert) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		X509_free(Cert);
		return false;
	}

	return true;
}

bool PKI_P7B::LoadDatas()
{
	X509 * currCert;
	int i;

	if(!PKCS7_type_is_signed(m_p7b))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	if(!m_p7b->d.sign || !m_p7b->d.sign->cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	for(i=0; i<SKM_sk_num(X509, m_p7b->d.sign->cert); i++)
	{
		currCert = SKM_sk_value(X509, m_p7b->d.sign->cert, i);
		if(!currCert)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
		CRYPTO_add(&currCert->references,1,CRYPTO_LOCK_X509);
		
		if(SKM_sk_push(X509, m_Certs, currCert) < 0)
		{
			X509_free(currCert);		
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}

	if(!PKCS7ToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool PKI_P7B::Load(const char * p7pem)
{
	if(!p7pem)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!StringToPKCS7(p7pem))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		return false;
	}

	if(!LoadDatas())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		return false;
	}

	return true;
}

bool PKI_P7B::Load(const PKCS7 * p7)
{
	Clear();

	if(!p7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(m_p7b)
		PKCS7_free(m_p7b);
	m_p7b = PKCS7_dup((PKCS7*)p7);
	if(!m_p7b)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}

	if(!LoadDatas())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		return false;
	}
	
	return true;	
}

int PKI_P7B::GetNumCert() const
{
	return SKM_sk_num(X509, m_Certs);
}

X509 * PKI_P7B::GetCert(int index, bool Duplicate) const
{
	X509 * cert;
	
	cert = SKM_sk_value(X509, m_Certs, index);
	if(!cert)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return NULL;
	}
	if(Duplicate)
		CRYPTO_add(&cert->references,1,CRYPTO_LOCK_X509);
		
	return cert;
}

bool PKI_P7B::load_Datas(const PKCS7 *c_p7)
{
	if(!Load(c_p7))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool PKI_P7B::give_Datas(PKCS7 **c_p7) const
{
	if(*c_p7)
		PKCS7_free(*c_p7);

	if(!m_p7b)
	{
		*c_p7 = NULL;
	}
	else
	{
		if(!(*c_p7 = GetPKCS7(true)) )
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

PKI_P7B::operator int() const
{
	return m_p7b?1:0;
}

bool PKI_P7B::operator=(const PKI_P7B & other)
{
	// Trying to copy myself on me
	if(&other.m_p7b == &m_p7b)
		return true;

	Clear();
	if(!other.m_p7b)
		return false;

	return Load(other.m_p7b);
}

