/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_PKCS12.h: interface for the PKCS12 class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_PKCS12_H
#define PKI_PKCS12_H

#include <PKI_ERR.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <HashTable/HashTable_String.h>
#include <PKI_RSA.h>
#include <PKI_CERT.h>

#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <openssl/pkcs12.h>


/*!
	This class represents a PKCS12
*/
class PKI_PKCS12  
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_PKCS12();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other pksc12 to copy from.
	 */
	PKI_PKCS12(const PKI_PKCS12 & other);

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_PKCS12();

	/*! \brief This function clears the internal datas.
	 *  \param all [IN] Whether or not to clear the certificates.
	 */
	void Clear(bool all = true);

	/*! \brief This function generates the P12.
	 *  \param passphrase [IN] The P12 passphrase.
	 *  \return true on success, false on failure.
	 */
	bool Generate(const char * passphrase);

	/*! \brief This function loads a P12 from a file.
	 *  \param Filename [IN] The filename to load.
	 *  \param Password [IN] The p12's password.
	 *  \return true on success, false on failure.
	 */
	bool LoadFromFile(const char * Filename, const char * Password);

	/*! \brief This function loads a P12.
	 *  \param strP12 [IN] The p12 datas.
	 *  \param Password [IN] The optional p12's password.
	 *  \return true on success, false on failure.
	 */
	bool Load(const mString & strP12, const char * Password);

	/*! \brief This function reloads a P12.
	 *  \param Password [IN] The optional p12's password.
	 *  \return true on success, false on failure.
	 */
	bool Reload(const char * Password);

	/*! \brief This function sets the end user cert.
	 *  \param EndUserCert [IN] The end user cert.
	 */
	void SetEndUserCert(const PKI_CERT & EndUserCert);

	/*! \brief This function set the end user key.
	 *  \param EndUserKey [IN] The end user key.
	 */
	void SetEndUserKey(const PKI_RSA & EndUserKey);

	/*! \brief This function sets the parent certificates.
	 *  \param ParentCerts [IN] The parent certs.
	 */
	void SetParentCerts(const HashTable_String & ParentCerts);

	/*! \brief This function returns the end user key.
	 *  \return The end user key.
	 */
	const PKI_RSA & GetEndUserKey() const;

	/*! \brief This function returns the end user cert.
	 *  \return The end user cert.
	 */
	const PKI_CERT & GetEndUserCert() const;

	/*! \brief This function returns the parent certs.
	 *  \return The parent certs.
	 */
	const HashTable_String & GetParentCerts() const;

	/*! \brief This function returns the p12 in PEM format.
	 *  \return The p12 in PEM format.
	 */
	const mString & GetPemPKCS12() const;
	
	/*! \brief This function returns the PKCS12.
	 *  \param Duplicate [IN] Should the PKCS12 be duplicated, then freed by calling function.
	 *  \return The PKCS12.
	 */
	PKCS12 * GetPKCS12(bool Duplicate = false) const;

	/*! \brief This function copies the internal pkcs12.
	 *  \param c_p12 [OUT] The pkcs12.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(PKCS12 ** c_p12) const;
	
	/*! \brief This function sets the pkcs12.
	 *  \param c_p12 [IN] A pointer to an OpenSSL PKCS12 certificate.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const PKCS12 * c_p12);

	/*! \brief This operator copies one PKI_CERT into another.
	 *  \param other [IN] The other certificate to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_PKCS12 &other );

	/*! \brief This member is an empty instance of this class.
	 */
	static PKI_PKCS12 EmptyInstance;

	/*! \brief This operator is used to know if the class is set.
	 *  \return return 1 when PKCS12 is operable and 0 when not.
	 */
	operator int() const;
private:
	mString Pkcs12Pem;
	PKCS12 *p12;
	bool PKCS12ToString();
	PKI_CERT m_EndUserCert;
	PKI_RSA m_EndUserKey;
	HashTable_String m_ParentCerts;
	
	bool Private_Load(const char *Password, bool LoadAll = true);

	int dump_certs_keys_p12 (STACK_OF(X509) * CertStore, PKCS12 *p12, const char *pass, int passlen);
	int dump_certs_pkeys_bags (STACK_OF(X509) * CertStore, STACK_OF(PKCS12_SAFEBAG) *bags, const char *pass, int passlen);
	int dump_certs_pkeys_bag (STACK_OF(X509) * CertStore, PKCS12_SAFEBAG *bag, const char *pass, int passlen);
};

#endif // !defined(PKCS12_H)
