/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PkiClient.h: interface for the PkiClient class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKICLIENT_H
#define PKICLIENT_H

#include <stdio.h>
#include <openssl/ssl.h>
#include <time.h>

#include <ASN1/Asn1Req.h>
#include <ASN1/Asn1Resp.h>

#include <PKI_ERR.h>
#include <NewPKI.h>
#include <UnixWinSock.h>
#include <newpki_threads.h>
#include <HashTable/HashTable_String.h>
#include <HashTable/HashTable_Dn.h>
#include <PKI_CERT.h>

#include "SslConnection.h"

#if defined(_WIN32) && !defined(NO_BIO_MS_SSL)
	#include "SspiConnection.h"
#else
	#define PCCERT_CONTEXT void *
#endif
#include <mString.h>
#include <vector>
using namespace std;


typedef void (PROC_WAIT_CALLBACK) ();

class PkiClient
{
public:
	PkiClient(PROC_WAIT_CALLBACK * WaitFunction);
	virtual ~PkiClient();
	bool Connect(const mString & Server, unsigned int Port, const PKI_CERT & AuthCert, PCCERT_CONTEXT MsAuthCert);
	bool Reconnect();
	void CloseConnection();
	void GetPeerCertificate(PKI_CERT & PeerCert);
	void GetEntityCert(PKI_CERT & entity_cert);
	unsigned int GetRemotePort();
	const mString & GetRemoteServer();
	const char * GetError();
	void SetReadTimeOut(int ReadTimeOut);
	static void CancelAllConnections();
	static unsigned long GetSentBytes();
	static unsigned long GetRecvBytes();
	const AdminReqLogin & GetLoginInfo();
	bool IsConnected();
	

	bool UserLogin(const AdminReqLogin & Login, int & UserType);
	bool EnumEntities(mVector<EntityEntryInfo> & entities);
	bool CreateEntity(const EntityCreationReq & cre_entity, EntityCreationResp & creation_resp);
	bool DeleteEntity(const mString & Name);
	bool LoadEntity(const mString & Name);
	bool UnloadEntity(const mString & Name);
	bool EnumLOG(const AdminReqEnumLogs & Filters, mVector<LogEntry> & Logs);
	bool GetLogsCount(const AdminReqEnumLogs & Filters, unsigned long & LogsCount);
	bool EnumUsers(mVector<UserEntry> & users);
	bool CreateUser(const UserInfo & user);
	bool UpdateUser(const UserEntry & user);
	bool ChangeUserPassword(const ChangeUserPasswd & Password);
	bool ChangePassword(const ChangePasswd & Password);
	bool SignEntity(const EntitySignatureReq & sign_entity, EntitySignatureResp & resp);
	bool EnumUsersCert(mVector<UserEntryCert> & users_cert);
	bool CreatePkiUser(const CreatePkiUserRequest & req, CreatePkiUserResponse & resp);
	bool GetOfflineState(bool & offline);
	bool SetOfflineState(bool offline);
	bool PushConfiguration(const ExportedPkiConf &  conf);
	bool RevokeEntityCert(unsigned long serial);
	bool RevokeUserCert(unsigned long serial);
	bool GetMyACL(ASN1_BIT_STRING **acl);
	bool GetConfiguration(ExportedPkiConf & conf);
	bool SearchLdap(const mString & Filters, mVector<LdapResult> & Results);
	bool DeletePkcs12(unsigned long CertId);
	bool GetCaP7b(PKI_P7B & p7b);
	bool SynchObjects(const ObjectsListSynchro & local, ObjectsListSynchro & remote);
	bool GenerateCRL();
	bool RequestCertificate(const NewpkiRequestCert & cert_request, unsigned long & ReqId);
	bool GetLogsType(mVector<unsigned long> & LogsType);
	bool GetEntityLogsType(int EntityType, mVector<unsigned long> & LogsType);
	bool SetRepositoriesList(const mVector<RepEntryInfo> & repositories);
	bool GetRepositoriesList(mVector<RepEntryInfo> & repositories);
	bool InitEntity(const EntityInitReq & init_entity);
	bool GetMyConf(EntityConfCrypted & myConf);
	bool GetEntitiesLinks(mVector<EntityLinks> & links);
	bool SetEntitiesLinks(const mVector<EntityLinks> & links);
	bool ChangeProfileDN(unsigned long ProfileId, const X509_NAME * dn);
	bool DeleteProfile(unsigned long ProfileId);
	bool ChangeProfileOwner(unsigned long ProfileId, unsigned long group_id);
	bool ChangeProfileUID(unsigned long ProfileId, const mString & uid);
	bool DeleteRequest(unsigned long ReqId);

	bool GetLocalConf(EntityConf & conf);
	bool ImportProfile(const NewpkiProfile & Profile, unsigned long & ProfileId);
	bool CheckLogsIntegrity();
	bool GetEntityConf(const PKI_CERT & entity_cert, EntityConfBody & conf);
	bool SetEntityConf(const PKI_CERT & entity_cert, const EntityConfBody & conf);
	bool GetEntityAcl(const PKI_CERT & entity_cert, X509Acl & acl);
	bool SetEntityAcl(const PKI_CERT & entity_cert, const mVector<AclEntry> & acl_entries);
	bool GetEntityAudits(const PKI_CERT & entity_cert, mVector<EntityAuditEntry> & audits);
	bool SetEntityAudits(const PKI_CERT & entity_cert, const mVector<EntityAuditEntry> & audits);
	bool GetEntityMailConf(const PKI_CERT & entity_cert, EmailConf & mail_conf);
	bool SetEntityMailConf(const PKI_CERT & entity_cert, const EmailConf & mail_conf);
	bool ResourceUnlock(const mString & ResourceName);
	bool ResourceLock(const mString & ResourceName);
	bool AdminSendMail(const MailDatas & mail);
	bool GetMailConf(EmailConf & mail_conf);
	bool GetCaStatus(unsigned long & status);
	bool CreateCaRoot(const ReqCreateRootCa & ca_info);
	bool EnumCERT(long index, long num, CERT_STATE state, mVector<InternalCaCert> & certs);
	bool EnumCrls(long index, long num, mVector<PKI_CRL> & crls);
	bool SignCSR(const PKI_CSR & req, int Days, const mString & uid, PKI_P7B & p7b);
	bool CreateCaChild(const ReqCreateChildCa & ca_info, PKI_CSR & csr);
	bool ImportChildCaCert(const PKI_P7B & certs);
	bool ImportCaP12(const PKI_PKCS12 & p12, const mString & password);
	bool RenameGroup(unsigned long id, const mString & Name);
	bool DelGroup(unsigned long id);
	bool AddGroup(const mString & Name);
	bool SetGroups(const mVector<UsersGroup> & Groups);
	bool GetGroups(mVector<UsersGroup> & Groups);
	bool RevokeCert(unsigned long serial);
	bool SuspendCert(unsigned long serial);
	bool UnsuspendCert(unsigned long serial);
	bool PushWaitingObject(const WaitingNewpkiObject & obj, int & Status);
	bool SendRequest(const CryptedNewpkiRequest & req);
	bool GetMyRequests(const TransactionIds & transactions_ids, CryptedNewpkiRequests & objects);
	bool SendResponse(const CryptedNewpkiResponse & resp);
	bool DeleteResponse(const Asn1OctetString & transactionID);
	bool GetMyResponses(const TransactionIds & transactions_ids, CryptedNewpkiResponses & objects);
	bool EnumProfiles(long index, long num, int state, const mString & filter, mVector<NewpkiProfileDatas> & Profiles);
	bool AcceptProfile(unsigned long profile_id);
	bool RejectProfile(unsigned long profile_id, const mString & reason);
	X509_NAME * GetDnValidation(const mString & Id);
	bool SetDnValidation(const mString & Id, const X509_NAME * dn);
	bool EnumEeUsers(long index, long num, int state, const mString & filter, mVector<NewpkiEeUser> & Users);

	bool WebUserChangePassword(unsigned long user_id, const mString & oldPassword, const mString & Password);
	bool WebUserLogin(const mString & email, const mString & password, unsigned long & UserId);
	bool WebUserCreate(const mString & email, const mString & password, const X509_NAME * dn);
	bool WebUserActivate(const mString & otp);
	bool WebUserRequestCertificate(const NewpkiEeRequestCert & cert_request);
	bool WebUserRevokeCertificate(const WebuserRevokeCert & rev_request);
	bool WebUserEnumCerts(unsigned long user_id, mVector<NewpkiProfileDatasCert> & certs);
	bool WebUserGetDn(unsigned long user_id, HashTable_Dn & dn);
private:
	static void AddRecvBytes(unsigned long bytes);
	static void AddSentBytes(unsigned long bytes);
	void Disconnect();
	void ClearErrors();

	class RequestDatas
	{
	public:
		RequestDatas(const AdminRequest & request, AdminResponse & response):m_request(request), m_response(response)
		{
		}
		~RequestDatas()
		{}
		bool m_Finished;
		bool m_Success;
		PkiClient * me_this;
		const AdminRequest & m_request;
		AdminResponse & m_response;
	};
	
	class ConnectionParam
	{
	public:
		ConnectionParam(){}
		~ConnectionParam(){}
		PkiClient * me_this;
		mString Server;
		unsigned short Port;
		bool Finished;
		bool Success;
		PKI_CERT m_AuthCert;
		PCCERT_CONTEXT m_MsAuthCert;
	};

	PROC_WAIT_CALLBACK * m_WaitFunction;

	#define INIT_NET_PARAM(type) \
		ClearErrors(); \
		if(!m_connection) \
		{ \
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_NO_CONNECTION); \
			PackThreadErrors(); \
			return false; \
		} \
		AdminRequest Request; \
		AdminResponse Response; \
		RequestDatas datas(Request, Response); \
		datas.me_this = this; \
		if(!Request.get_body().set_type(type)) \
		{ \
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT); \
			PackThreadErrors(); \
			return false; \
	} \
	Request.set_isOK();
	
	#define NET_PARAM datas
	#define NET_PARAM_RESPONSE Response






	bool DoNetworkExchange(RequestDatas * ClientParam);
	static void ThreadProcSocket(const NewpkiThread * Thread, void * param);
	bool DoBufferSend(RequestDatas * ClientParam);

	bool DoNetworkConnection(ConnectionParam * ConnParam);
	static void ThreadProcConnection(const NewpkiThread * Thread, void *param);
	bool DoConnection(ConnectionParam * ConnParam);

	Connection * m_connection;
	ConnectionParam m_ConnParam;
	vector<ErrorEntry> errors;

	static void add_session(const mString & key, SSL_SESSION* sess);
	static SSL_SESSION * get_session(const mString & key);
	static void del_session(const mString & key);
	static void purge_sessions();
	static void generate_id(mString & id, const mString & Server, unsigned int Port, const PKI_CERT & AuthCert);

	void PackThreadErrors();
	PKI_CERT m_PeerCert;
	mString str_Error;
	int m_ReadTimeOut;
	static vector<SOCKET> ListAllConnections;
	static CriticalSection ListAllConnectionsLock;
	static CriticalSection DatasCounterLock;
	static unsigned long m_RcvDatas;
	static unsigned long m_SndDatas;
	AdminReqLogin m_loginInfo;

	static SessionsCache m_clientSessions;
};

#endif // !defined(PKICLIENT_H)
