/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "SslConnection.h"

SslConnection::SslConnection(SessionsCache & cache):Connection(),m_cache(cache)
{
	m_ssl = NULL;
	m_ctx = NULL;
	m_connection = NULL;
	m_enabledCache = false;

	SSL_METHOD * meth;
	meth = SSLv23_client_method();
	if(!meth)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();
		return;
	}
	m_ctx = SSL_CTX_new (meth);
	if(!m_ctx)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();
		return;
	}
	SSL_CTX_set_verify(m_ctx, SSL_VERIFY_PEER|SSL_VERIFY_CLIENT_ONCE, verify_callback);
	SSL_CTX_set_session_cache_mode(m_ctx, SSL_SESS_CACHE_OFF);

}

SslConnection::~SslConnection()
{
	if(m_ctx)
	{
		SSL_CTX_free(m_ctx);
	}
	close();
}

int SslConnection::verify_callback(int ok, X509_STORE_CTX *ctx)
{
	return 1;
}

bool SslConnection::set_certificate(const PKI_CERT & cert)
{
	m_dn = cert.GetStringName();

	if(!SSL_CTX_use_certificate(m_ctx, cert.GetX509()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!SSL_CTX_use_PrivateKey(m_ctx, (EVP_PKEY*)cert.GetPrivateKey().GetRsaKey()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool SslConnection::connect()
{
	close();
	if(!m_ctx)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	int ret;
	mString sessid;
	SSL_SESSION * sess;
	SSL_SESSION * sessNew;

	if(!do_connection())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if( !(m_ssl=SSL_new(m_ctx)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		close();
		return false;
	}

	SSL_set_mode(m_ssl, SSL_MODE_AUTO_RETRY);

	SSL_set_fd(m_ssl,m_socket);

	m_connection = BIO_new(BIO_f_ssl());
	if(!m_connection)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		close();
		return false;
	}

	if(BIO_set_ssl(m_connection, m_ssl, BIO_CLOSE) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		close();
		return false;
	}

	if(BIO_set_ssl_mode(m_connection, 1) <= 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		close();
		return false;
	}

	if(m_enabledCache)
	{
		m_cache.generate_id(sessid, m_host, m_dn, m_port);
		if( (sess = m_cache.get_session(sessid)) )
		{
			SSL_set_session(m_ssl, sess);
		}
	}
	else
	{
		sess = NULL;
	}

	ERR_clear_error();
	if( (ret = SSL_connect(m_ssl)) <= 0)
	{
		switch(SSL_get_error(m_ssl, ret))
		{
			case SSL_ERROR_SSL:
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				break;
			case SSL_ERROR_SYSCALL:
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				ERR_add_error_data(1, strerror(errno));
				break;
			default:
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				break;
		}
		close();
		return false;
	}

	sessNew = SSL_get_session(m_ssl);
	if(m_enabledCache)
	{
		if(sessNew->session_id_length)
		{
			if(sess)
			{

				if(sess->session_id_length != sessNew->session_id_length ||
					memcmp(sess->session_id, sessNew->session_id, sessNew->session_id_length) != 0)
				{
					m_cache.delete_session(sessid);
					m_cache.add_session(sessid, sessNew);
				}
				SSL_SESSION_free(sess);
			}
			else
			{
				m_cache.add_session(sessid, sessNew);
			}
		}
		else if(sess)
		{
			m_cache.delete_session(sessid);
			SSL_SESSION_free(sess);
		}
	}
	return true;
}

void SslConnection::close()
{
	if(m_ssl)
	{
		SSL_shutdown(m_ssl);
		m_ssl = NULL;
	}
	if(m_connection)
	{
		BIO_free_all(m_connection);
		m_connection = NULL;
	}
	do_close();
}

void SslConnection::enable_Cache(bool enabled)
{
	m_enabledCache = enabled;
}


bool SslConnection::GetPeerCertificate(PKI_CERT &PeerCert)
{
	X509 * cert;
	if(!m_ssl)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	if(! (cert = SSL_get_peer_certificate(m_ssl)) )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!PeerCert.SetCert(cert))
	{
		X509_free(cert);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	X509_free(cert);
	return true;
}

bool SslConnection::SendRequest(const AdminRequest &request, AdminResponse &response, int ReadTimeout)
{
	if(!do_send(m_connection, request, response, ReadTimeout))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}
