/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef NEWPKI_THREADS_H
#define NEWPKI_THREADS_H

#ifdef _WIN32
#pragma warning(disable:4786)
#endif



#include <stdlib.h>
#include "CriticalSection.h"
#include <map>
using namespace std;


#ifdef _WIN32
	#include <windows.h>
	#include <process.h>
#else
	#include <unistd.h>
	#include <pthread.h>
#endif
#include <CriticalSection.h>


#ifdef _WIN32
	#define NEWPKI_THREAD_ID unsigned long
	#define NEWPKI_THREAD_NULL	0
#else
	#define NEWPKI_THREAD_ID pthread_t
	#define NEWPKI_THREAD_NULL	0
#endif


class NewpkiThread;

typedef void (THREAD_START_ROUTINE) (const NewpkiThread * myThread, void * param);


/*!
	This class implements threads
*/
class NewpkiThread
{
public:
	/*! \brief This is the constructor.
	 */
	NewpkiThread();

	/*! \brief This is the destructor.
	 */
	~NewpkiThread();

	/*! \brief This function intialize the class (the thread isn't created until Start is called).
	 *  \param StartRoutine [IN] The start routine to call when the thread is created.
	 *  \param Param [IN] A value to pass back to StartRoutine when the thread is created.
	 */
	void Create(THREAD_START_ROUTINE * StartRoutine, void * Param);

	/*! \brief This function creates and starts the thread.
	 *  \return true on success, false on failure.
	 */
	bool Start();

	/*! \brief This function stops the thread.
	 */
	void Stop();

	/*! \brief This function makes the calling thread sleep.
	 *  \param ms_sleep [IN] The ammount of time to sleep for in Millisecondes.
	 */
	static void Sleep(time_t ms_sleep);

	/*! \brief This function makes the calling thread sleep ms_sleep, or until the tread should stop.
	 *  \param ms_sleep [IN] The ammount of time to sleep for in secondes.
	 *  \return true if the thread wasn't interrupted, false if the thread was interrupted.
	 */
	bool SleepInterrupt(time_t s_sleep) const;

	/*! \brief This function returns the thread id of the calling function.
	 *  \return The thread id of the calling function.
	 */
	static NEWPKI_THREAD_ID CurrentThreadId();

	/*! \brief This function returns the id of the created thread.
	 *  \return The thread id or NEWPKI_THREAD_NULL when thread hasn't been started yet.
	 */
	NEWPKI_THREAD_ID GetThreadId() const;

	/*! \brief This function returns true is SpawnStop has been called.
	 *  \return true is SpawnStop has been called.
	 */
	bool ShouldStop() const;

	/*! \brief This function returns true if Start has been successfully called.
	 *  \return true if Start has been successfully called.
	 */
	bool IsRunning() const;

	/*! \brief This function yells to the threads to really start.
	 */
	static void SignalStart();

	/*! \brief This function yells to the thread to stop.
	 */
	static void SignalStop();
private:
	THREAD_START_ROUTINE * m_StartRoutine;
	void * m_Param;
	NEWPKI_THREAD_ID m_thread_id;
	bool m_ShouldStop;
	bool m_IsRunning;
#ifdef _WIN32
	static void tmp_ThreadProc(void * Param);
#else
	static void * tmp_ThreadProc(void * Param);
#endif
	bool DoStart();
	static map< NewpkiThread*, bool > m_threads;
	static CriticalSection m_threadsLock;
	static bool m_isSignaledStarted;
	static bool m_isSignaledStopped;
};


#endif

