// publication_ldap.cpp : Defines the entry point for the DLL application.
//

#include "publication_ldap.h"


#include <PKI_PLUG_PUB.h>

#include <mString.h>
#ifdef _WIN32
	#include <windows.h>
	#include <winldap.h>
#else
	#include <ldap.h>
#endif



#ifdef _WIN32
	#define LDAP_RC_TYPE unsigned long
	#define LDAP_LAST_ERROR LdapGetLastError()
	struct l_timeval timeout;
#else
	#define LDAP_RC_TYPE int
	#define LDAP_LAST_ERROR errno
	struct timeval timeout;
	#define stricmp strcasecmp
	#define strupr(x) \
	{ \
		size_t dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) (x)[dtlen]=toupper((x)[dtlen]); \
	}
	#define strlwr(x) \
	{ \
		size_t dtlen; \
		for(dtlen=0; dtlen<strlen(x); dtlen++) (x)[dtlen]=tolower((x)[dtlen]); \
	}
#endif



int GetRDN(const HashTable_String & Options, const mString & ldap_uid, const PKI_CERT & Cert, mString & RDN);
int SearchLadp(const HashTable_String & Options, const PKI_CERT & Cert, const mString & LdapBase, mString & RDN);
int DoModification(const HashTable_String & Options, LDAPMod ** Mods, const mString & RDN);
int DoSearch(const HashTable_String & Options, const mString & LdapBase, const mString & Filters, LDAPMessage ** pMsg);
int ProcessResult(const HashTable_String & Options, LDAPMessage * pMsg, const HashTable_Dn & Dn, mString & RDN);
const char * GetFixedName(char *Obj);

static int InitLib(const HashTable_String & Options);
static int ClearLib(const HashTable_String & Options);
static int OnNewCertificate(const HashTable_String & Options, const PKI_CERT & Certificate, const PKI_P7B & P7B, const mString & ldap_uid);
static int OnNewRevocation(const HashTable_String & Options, const PKI_CERT & Certificate, const mString & ldap_uid);
static int OnNewCrl(const HashTable_String & Options, const PKI_CRL & Crl);
static const char * GetLibLastError();




static PKI_PLUG_PUB st_functions={
	InitLib,
	ClearLib,
	OnNewCertificate,
	OnNewRevocation,
	OnNewCrl,
	GetLibLastError
};

extern "C" void * m_GetFunctions()
{
	return &st_functions;
}


mString LastError;
LDAP * m_Connection = NULL;


static int InitLib(const HashTable_String & Options)
{
	ERR_clear_error();
	
	LastError = "";

	if(m_Connection)
	{
		ldap_unbind_s(m_Connection);
		m_Connection = NULL;
	}


	LDAP_RC_TYPE LdapRet;
	
	const char * Server;
	const char * strPort;
	unsigned int Port;
	const char * Username;
	const char * Password;
	int protoVersion;

	Server = Options.Get("Server");
	strPort = Options.Get("Port");
	if(!Server || !*Server || !Options.Get("Port"))
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "Server");
		return 0;
	}
	sscanf(strPort, "%d", &Port);
	if(!strPort || !*strPort || !Port)
	{

		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "Port");
		return 0;
	}
	Username = Options.Get("Username");
	Password = Options.Get("Password");




	m_Connection = ldap_init((char*)Server, Port);
	if(!m_Connection)
	{
		LastError = ldap_err2string(LDAP_LAST_ERROR);
		return 0;
	}

	LdapRet = ldap_bind_s(m_Connection, (char*)Username, (char*)Password, LDAP_AUTH_SIMPLE);
	if(LdapRet != LDAP_SUCCESS)
	{
		/* Wrong protocol version try another one */
		if(LdapRet == LDAP_PROTOCOL_ERROR)
		{
			/* Get protocol version */
			LdapRet = ldap_get_option(m_Connection, LDAP_OPT_PROTOCOL_VERSION, &protoVersion);
			if(LdapRet != LDAP_SUCCESS)
			{
				LastError = ldap_err2string(LdapRet);
				ldap_unbind_s(m_Connection);
				m_Connection = NULL;
				return 0;
			}
			/* Set protocol version */
			protoVersion = protoVersion == LDAP_VERSION3 ? LDAP_VERSION2 : LDAP_VERSION3;
			LdapRet = ldap_set_option(m_Connection, LDAP_OPT_PROTOCOL_VERSION, &protoVersion);
			if(LdapRet != LDAP_SUCCESS)
			{
				LastError = ldap_err2string(LdapRet);
				ldap_unbind_s(m_Connection);
				m_Connection = NULL;
				return 0;
			}
			/* Retry to connect */
			LdapRet = ldap_bind_s(m_Connection, (char*)Username, (char*)Password, LDAP_AUTH_SIMPLE);
			if(LdapRet != LDAP_SUCCESS)
			{
				LastError = ldap_err2string(LdapRet);
				ldap_unbind_s(m_Connection);
				m_Connection = NULL;
				return 0;
			}
		}
		else
		{
			LastError = ldap_err2string(LdapRet);
			ldap_unbind_s(m_Connection);
			m_Connection = NULL;
			return 0;
		}
	}
	return 1;
}

static int ClearLib(const HashTable_String & Options)
{
	ERR_clear_error();
	LastError = "";
	
	if(m_Connection)
	{
		ldap_unbind_s(m_Connection);
		m_Connection = NULL;
	}
	ERR_remove_state(0);
	return 1;
}

static int OnNewCertificate(const HashTable_String & Options, const PKI_CERT & Certificate, const PKI_P7B & P7B, const mString & ldap_uid)
{
	ERR_clear_error();
	LastError = "";
	
	if(!m_Connection)
	{
		if(!InitLib(Options))
		{
			return 0;
		}
	}
	
	mBuffer DerCert;
	char * PemCert = NULL;
	PKI_CERT lCertificate;
	PKI_P7B lP7B;
	
	mString CertAttr;
	mString Format;
	bool Binary;
	mString RDN;

	if(!lCertificate.load_Datas(Certificate.GetX509()))
	{
		return 0;
	}

	if(!lP7B.load_Datas(P7B.GetPKCS7()))
	{
		return 0;
	}


	CertAttr = Options.Get("CertAttr");
	if(!CertAttr.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "CertAttr");
		return 0;
	}
	Format = Options.Get("Format");
	if(!Format.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "Format");
		return 0;
	}

	// We first get the RDN corresponding to th UID
	if(!GetRDN(Options, ldap_uid, lCertificate, RDN))
	{
		return 0;
	}
		
	Binary = (CertAttr.find(";binary",0) != -1);

	if(Format == "X509")
	{
		// Is it binary ?
		if(Binary)
		{
			if(!lCertificate.GetCertPEM().ToDER(DerCert))
			{
				LastError = NEWPKIerrGetStr(ERROR_BAD_DATAS);
				return 0;
			}
		}
		else
		{
			PemCert = (char*)lCertificate.GetCertPEM().c_str();
		}
	}
	else if(Format == "PKCS7")
	{
		// Is it binary ?
		if(Binary)
		{
			if(!lP7B.GetPemP7B().ToDER(DerCert))
			{
				LastError = NEWPKIerrGetStr(ERROR_BAD_DATAS);
				return 0;
			}
		}
		else
		{
			PemCert = (char*)lP7B.GetPemP7B().c_str();
		}
	}
	else
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "Format (PKCS7 or X509)");
		return 0;
	}

	LDAPMod Mod;
	struct berval DerValue={DerCert.get_BufferLen(), (char*)DerCert.get_Buffer()};
	struct berval * DerValues[2]={&DerValue, NULL};
	char * PemValues[2]={PemCert, NULL};




	//Mod.mod_type = "usercertificate;binary";
	Mod.mod_type = (char*)CertAttr.c_str();


	Mod.mod_op = LDAP_MOD_REPLACE;
	// Setting the binary datas
	if(DerCert.get_BufferLen())
	{
		Mod.mod_op |= LDAP_MOD_BVALUES;
		Mod.mod_vals.modv_bvals = DerValues;
	}
	else
	{
		// Setting the text datas
		Mod.mod_vals.modv_strvals = PemValues;
	}

	LDAPMod * NewEntries[2] = {&Mod, NULL};

	if(!DoModification(Options, NewEntries, RDN))
	{
		return 0;
	}
	return 1;
}

static int OnNewRevocation(const HashTable_String & Options, const PKI_CERT & Certificate, const mString & ldap_uid)
{
	ERR_clear_error();
	LastError = "";
	
	if(!m_Connection)
	{
		if(!InitLib(Options))
		{
			return 0;
		}
	}
	
	mString CertAttr;
	mString RDN;

	CertAttr = Options.Get("CertAttr");
	if(!CertAttr.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "CertAttr");
		return 0;
	}

	// We first get the RDN corresponding to th UID
	if(!GetRDN(Options, ldap_uid, Certificate, RDN))
	{
		return 0;
	}
	
	LDAPMod Mod;
	Mod.mod_type = (char*)CertAttr.c_str();

	Mod.mod_op = LDAP_MOD_DELETE;
	Mod.mod_vals.modv_bvals = NULL;

	LDAPMod * NewEntries[2] = {&Mod, NULL};

	if(!DoModification(Options, NewEntries, RDN))
	{
		return 0;
	}
	return 1;
}

static int OnNewCrl(const HashTable_String & Options, const PKI_CRL & Crl)
{
	ERR_clear_error();
	LastError = "";

	if(!m_Connection)
	{
		if(!InitLib(Options))
		{
			return 0;
		}
	}
	
	mBuffer DerCrl;
	mString CrlAttr;
	mString RDN;


	CrlAttr = Options.Get("CrlAttr");
	if(!CrlAttr.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "CrlAttr");
		return 0;
	}
	RDN = Options.Get("RDN");
	if(!RDN.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "RDN");
		return 0;
	}

	if(!Crl.GetPemCRL().ToDER(DerCrl))
	{
		LastError = NEWPKIerrGetStr(ERROR_BAD_DATAS);
		return 0;
	}

	
	LDAPMod Mod;
	struct berval DerValue={DerCrl.get_BufferLen(), (char*)DerCrl.get_Buffer()};
	struct berval * DerValues[2]={&DerValue, NULL};

	Mod.mod_type = (char*)CrlAttr.c_str();

	Mod.mod_op = LDAP_MOD_REPLACE | LDAP_MOD_BVALUES;
	Mod.mod_vals.modv_bvals = DerValues;
	LDAPMod * NewEntries[2] = {&Mod, NULL};

	if(!DoModification(Options, NewEntries, RDN))
	{
		return 0;
	}
	return 1;
}

static const char * GetLibLastError()
{
	return LastError.c_str();
}



int GetRDN(const HashTable_String & Options, const mString & ldap_uid, const PKI_CERT & Cert, mString & RDN)
{
	mString UidAttrName;
	mString Filters;
	mString LdapBase;
	LDAPMessage * pMsg;
	LDAPMessage * currMsg;
	char * Name;
	const char * utf8;

	
	LdapBase = Options.Get("Base");
	if(!LdapBase.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "Base");
		return 0;
	}


	// We don't have an UID we're going to try to find the entry
	// matchinh its certificate's DN
	if(!ldap_uid.size())
	{
		return SearchLadp(Options, Cert, LdapBase, RDN);
	}

	UidAttrName = Options.Get("UidAttr");
	if(!UidAttrName.size())
	{
		LastError.sprintf("%s : %s", NEWPKIerrGetStr(ERROR_BAD_DATAS), "UidAttr");
		return 0;
	}

	Filters = "(&(";
	Filters += UidAttrName;
	Filters += "=";
	Filters += ldap_uid;
	Filters += "))";

	if(!DoSearch(Options, LdapBase, Filters, &pMsg))
	{
		return 0;
	}

	if(!ldap_count_entries(m_Connection, pMsg))
	{
		ldap_msgfree(pMsg);
		return 0;
	}

	// Get first entry
	currMsg = ldap_first_entry(m_Connection, pMsg);
	if(!currMsg)
	{
		ldap_msgfree(pMsg);
		LastError = ldap_err2string(LDAP_LAST_ERROR);
		return 0;
	}
	Name = ldap_get_dn(m_Connection, currMsg);
	if(!Name)
	{
		ldap_msgfree(pMsg);
		LastError = ldap_err2string(LDAP_LAST_ERROR);
		return 0;
	}
	utf8 = Options.Get("UTF-8");
	if(!utf8 || !*utf8 || *utf8 == '0')
		RDN = Name;
	else
		mString::Encode("UTF-8", "ISO-8859-1",  Name, RDN);
	ldap_memfree(Name);
	ldap_msgfree(pMsg);
	return 1;
}

int DoSearch(const HashTable_String & Options, const mString & LdapBase, const mString & Filters, LDAPMessage ** pMsg)
{
	LDAP_RC_TYPE LdapRet;
	const char * utf8;
	mString strFilters;

	utf8 = Options.Get("UTF-8");
	if(!utf8 || !*utf8 || *utf8 == '0')
		strFilters = Filters;
	else
		mString::Encode("ISO-8859-1", "UTF-8", Filters, strFilters);

	

	*pMsg=NULL;
	if((LdapRet = ldap_search_s(m_Connection, (char*)LdapBase.c_str(), LDAP_SCOPE_SUBTREE, (char*)strFilters.c_str(), NULL, 0, pMsg)) != LDAP_SUCCESS)
	{
		if(*pMsg)
		{
			ldap_msgfree(*pMsg);
			*pMsg=NULL;
		}
		if(LdapRet == LDAP_SERVER_DOWN)
		{
			if(!InitLib(Options))
			{
				return 0;
			}

			if((LdapRet = ldap_search_s(m_Connection, (char*)LdapBase.c_str(), LDAP_SCOPE_SUBTREE, (char*)strFilters.c_str(), NULL, 0, pMsg)) != LDAP_SUCCESS)
			{
				if(*pMsg)
				{
					ldap_msgfree(*pMsg);
					*pMsg=NULL;
				}
				LastError = ldap_err2string(LdapRet);
				return 0;
			}
		}
		else
		{
			LastError = ldap_err2string(LdapRet);
			return 0;
		}
	}

	return 1;
}

int SearchLadp(const HashTable_String & Options, const PKI_CERT & Cert, const mString & LdapBase, mString & RDN)
{
	const char * value;
	int pos;
	mString Filters;
	LDAPMessage * pMsg;


	// We first try the email address
	pos = Cert.GetCertDN().SeekEntryName("emailAddress", HASHTABLE_NOT_FOUND);
	if(pos != HASHTABLE_NOT_FOUND)
	{
		value = Cert.GetCertDN().Get(pos);
	}
	else
	{
		value = Cert.GetExtensions().Get("subjectAltName");
	}
	if(value)
	{
		Filters = "(&(mail=";
		Filters += value;
		Filters += "))";

		if(!DoSearch(Options, LdapBase, Filters, &pMsg))
			return 0;

		if(ProcessResult(Options, pMsg, Cert.GetCertDN(), RDN))
		{
			ldap_msgfree(pMsg);
			return 1;
		}
		ldap_msgfree(pMsg);
	}

	// We then try the common name
	pos = Cert.GetCertDN().SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
	if(pos != HASHTABLE_NOT_FOUND)
	{
		value = Cert.GetCertDN().Get(pos);
	}
	else
	{
		return 0;
	}
	if(value)
	{
		Filters = "(&(cn=";
		Filters += value;
		Filters += "))";

		if(!DoSearch(Options, LdapBase, Filters, &pMsg))
			return 0;

		if(ProcessResult(Options, pMsg, Cert.GetCertDN(), RDN))
		{
			ldap_msgfree(pMsg);
			return 1;
		}
		ldap_msgfree(pMsg);
	}

	return 0;
}

int DoModification(const HashTable_String & Options, LDAPMod ** Mods, const mString & RDN)
{
	LDAP_RC_TYPE LdapRet;
	mString strRDN;
	const char * utf8;

	// If we don't have the user ldap_uid
	// we are going to search for it
	if(!RDN.size())
	{
		LastError = NEWPKIerrGetStr(ERROR_BAD_DATAS);
		return 0;
	}

	utf8 = Options.Get("UTF-8");
	if(!utf8 || !*utf8 || *utf8 == '0')
		strRDN = RDN;
	else
		mString::Encode("ISO-8859-1", "UTF-8", RDN, strRDN);


	if((LdapRet = ldap_modify_s(m_Connection, (char*)strRDN.c_str(), Mods)) != LDAP_SUCCESS)
	{
		if(LdapRet == LDAP_SERVER_DOWN)
		{
			if(!InitLib(Options))
			{
				return 0;
			}

			if((LdapRet = ldap_modify_s(m_Connection, (char*)strRDN.c_str(), Mods)) != LDAP_SUCCESS)
			{
				LastError = ldap_err2string(LdapRet);
				return 0;
			}
		}
		else
		{
			LastError = ldap_err2string(LdapRet);
			return 0;
		}
	}

	return 1;
}


const char * GetFixedName(char *Obj)
{
	int nid;

	if(stricmp(Obj, "mail")==0) 
		return "emailAddress";

	nid = OBJ_txt2nid(Obj);
	if(nid == NID_undef)
	{
		strupr((char*)Obj);
		nid = OBJ_txt2nid(Obj);
		if(nid == NID_undef)
		{
			strlwr((char*)Obj);
			((char*)Obj)[0] = toupper(Obj[0]);
			nid = OBJ_txt2nid(Obj);
		}
	}
	if(nid == NID_undef)
		return NULL;

	return OBJ_nid2ln(nid);
}

int ProcessResult(const HashTable_String & Options, LDAPMessage * pMsg, const HashTable_Dn & Dn, mString & RDN)
{
	char * Name;
	LDAPMessage* currMsg;
	char * attrName;
	char ** attrValue;
	BerElement* ptr;
	int currNumMatch;
	int maxNumMatch;
	LDAPMessage* maxNumMatchMsg;
	const char * FixedName;
	const char * Value;
	int pos;
	const char * utf8;

	utf8 = Options.Get("UTF-8");	


	switch(ldap_count_entries(m_Connection, pMsg))
	{
		case 0:
			return 0;
			break;

		case 1:
			// we have our rdn
			currMsg = ldap_first_entry(m_Connection, pMsg);
			if(!currMsg)
			{
				return 0;
			}
			Name = ldap_get_dn(m_Connection, currMsg);
			if(!Name)
			{
				return 0;
			}
			if(!utf8 || !*utf8 || *utf8 == '0')
				RDN = Name;
			else
				mString::Encode("UTF-8", "ISO-8859-1",  Name, RDN);
			ldap_memfree(Name);
			return 1;
			break;
		
		default:
			// more than one entry matching let's see which one
			// matches the most the DN

			maxNumMatch = 0;
			maxNumMatchMsg = NULL;
			
			// For each ettribute of each entry, we try to find
			// the name/value in the DN, if we find it, the current
			// entry get one point.
			// The entry that has the most points wins !
			for(currMsg = ldap_first_entry(m_Connection, pMsg); 
				currMsg; 
				currMsg = ldap_next_entry(m_Connection, currMsg))
			{
				currNumMatch=0;

				// For each attribute
				for(	attrName = ldap_first_attribute(m_Connection, currMsg, &ptr);
						attrName;
						attrName = ldap_next_attribute(m_Connection, currMsg, ptr)
					)
				{
					if( strstr(attrName, ";binary") || !(FixedName = GetFixedName(attrName)) )
					{
						ldap_memfree(attrName);
						continue;
					}
					if( (pos = Dn.SeekEntryName(FixedName, HASHTABLE_NOT_FOUND)) != HASHTABLE_NOT_FOUND )
					{
						attrValue = ldap_get_values(m_Connection, currMsg, attrName);
						if(attrValue)
						{
							Value = Dn.Get(pos);
							if(Value)
							{
								if(stricmp(Value, *attrValue) == 0)
								{
									currNumMatch++;
								}
							}
							ldap_value_free(attrValue);
						}
					}
					ldap_memfree(attrName);
				}
				if(currNumMatch >= maxNumMatch)
				{
					maxNumMatchMsg = currMsg;
					maxNumMatch = currNumMatch;
				}
			}
			// Did we find an entry ?
			if(maxNumMatchMsg)
			{
				Name = ldap_get_dn(m_Connection, maxNumMatchMsg);
				if(!Name)
				{
					return 0;
				}
				if(!utf8 || !*utf8 || *utf8 == '0')
					RDN = Name;
				else
					mString::Encode("UTF-8", "ISO-8859-1",  Name, RDN);
				ldap_memfree(Name);
				return 1;
			}
			return 0;
			break;
	}
}

