/* ruler.c: draw and manage rulers in image windows
 */

#include "ip.h"

/* Scale a ruler ppm factor by the magnification.
 */
static float
ppm2cf( RulerInfo *rule, float ppm )
{
	int w, h;
	const int bigfac = 1000000;

	im2disp( rule->scr, bigfac, bigfac, &w, &h );

	return( (h * rule->ppm) / bigfac );
}

/* Paint the whole vertical ruler. An ealier stage sets XClipping. Pass the
 * approximate range of ruler values that need repainting.
 */
static void
paint_vertical_ruler( RulerInfo *rule, int from, int to )
{
	float tstart = rule->pxsep * floor( (float) from / rule->pxsep );
	float lstart = (rule->pxsep * 5) * 
		floor( (float) from / (rule->pxsep * 5) );
	float cf = ppm2cf( rule, rule->ppm );
	float y;

	int py;
	char buf[ 256 ];

	/* Little pips.
	 */
	for( y = tstart; y < to; y += rule->pxsep ) {
		/* Translate to physical pixel coordinates.
		 */
		py = y - rule->from;

		/* Draw line.
		 */
		XDrawLine( our_display, XtWindow( rule->draw ), ruler_gc,
			rule->width - 6, py, rule->width, py );
		XDrawLine( our_display, XtWindow( rule->draw ), plain_gc,
			rule->width - 6, py+1, rule->width, py+1 );
	}

	/* Big pips.
	 */
	for( y = lstart; y < to; y += rule->pxsep * 5 ) {
		/* Translate to physical pixel coordinates.
		 */
		py = y - rule->from;

		/* Extend line.
		 */
		XDrawLine( our_display, XtWindow( rule->draw ), ruler_gc,
			0, py, rule->width, py );
		XDrawLine( our_display, XtWindow( rule->draw ), plain_gc,
			0, py+1, rule->width, py+1 );

		/* Label. Map pixels back into mm.
		 */
		im_snprintf( buf, 256, "%d", (int) (y/cf + 0.5) );
		XDrawString( our_display, XtWindow( rule->draw ), ruler_gc, 
			1, py + rule->tarea.height + 3, 
			buf, strlen( buf ) );
	}
}

/* Paint the whole horizontal ruler. An ealier stage sets XClipping. Again,
 * pass the approximate range of values to repaint.
 */
static void
paint_horizontal_ruler( RulerInfo *rule, int from, int to )
{	
	float tstart = rule->pxsep * floor( (float) from / rule->pxsep );
	float lstart = (rule->pxsep * 5) * 
		floor( (float) from / (rule->pxsep * 5) );
	float cf = ppm2cf( rule, rule->ppm );
	float x;
	int px;
	char buf[ 256 ];

	/* Little pips.
	 */
	for( x = tstart; x < to; x += rule->pxsep ) {
		/* Translate to physical pixel coordinates.
		 */
		px = x - rule->from;

		/* Draw line.
		 */
		XDrawLine( our_display, XtWindow( rule->draw ), ruler_gc,
			px, rule->width - 6, px, rule->width );
		XDrawLine( our_display, XtWindow( rule->draw ), plain_gc,
			px+1, rule->width - 6, px+1, rule->width );
	}

	/* Big pips.
	 */
	for( x = lstart; x < to; x += rule->pxsep * 5 ) {
		/* Translate to physical pixel coordinates.
		 */
		px = x - rule->from;

		/* Extend line.
		 */
		XDrawLine( our_display, XtWindow( rule->draw ), ruler_gc,
			px, 0, px, rule->width );
		XDrawLine( our_display, XtWindow( rule->draw ), plain_gc,
			px+1, 0, px+1, rule->width );

		/* Label. Map pixels back into mm.
		 */
		im_snprintf( buf, 256, "%d", (int) (x/cf + 0.5) );
		XDrawString( our_display, XtWindow( rule->draw ), ruler_gc, 
			px + 3, rule->tarea.height + 1, 
			buf, strlen( buf ) );
	}
}

/* Paint ruler tick.
 */
static void
paint_ruler_tick( RulerInfo *rule )
{	
	switch( rule->type ) {
	case RULER_HORIZONTAL:
		XDrawLine( our_display, XtWindow( rule->draw ), tick_gc,
			rule->tpos - rule->from, 0,
			rule->tpos - rule->from, rule->width );
		break;

	case RULER_VERTICAL:
		XDrawLine( our_display, XtWindow( rule->draw ), tick_gc,
			0, rule->tpos - rule->from,
			rule->width, rule->tpos - rule->from );
		break;

	default:
		assert( FALSE );
	}
}

/* Light the tick in the ruler.
 */
static void
light_ruler_tick( RulerInfo *rule )
{	
	/* Only if already snuffed.
	 */
	if( rule->enabled && !rule->tick ) {
		paint_ruler_tick( rule );
		rule->tick = True;
	}
}

/* Snuff the tick in the ruler.
 */
static void
snuff_ruler_tick( RulerInfo *rule )
{	
	/* Only if already lit.
	 */
	if( rule->enabled && rule->tick ) {
		paint_ruler_tick( rule );
		rule->tick = False;
	}
}

/* Refresh the tick in the ruler.
 */
static void
refresh_ruler_tick( RulerInfo *rule )
{	
	/* Only if already lit.
	 */
	if( rule->enabled && rule->tick )
		paint_ruler_tick( rule );
}

/* Find the visible area for a ruler.
 */
static void
find_ruler_visible( RulerInfo *rule, Rect *r )
{	
	switch( rule->type ) {
	case RULER_HORIZONTAL:
		r->left = rule->from;
		r->width = rule->space;
		r->top = 0;
		r->height = rule->width;
		break;

	case RULER_VERTICAL:
		r->left = 0;
		r->width = rule->width;
		r->top = rule->from;
		r->height = rule->space;
		break;
	
	default:
		assert( FALSE );
	}
}

/* Decode a ruler angle.
 */
static char *
decode_angle( enum ruler_angle a )
{	
	switch( a ) {
	case RULER_HORIZONTAL:
		return( "horizontal" );
	
	case RULER_VERTICAL:
		return( "vertical" );

	default:
		assert( FALSE );

		/* Keep gcc happy.
		 */
		return( NULL );
	}
}

/* Repaint an area of a ruler.
 */
static void
repaint_ruler_region( RulerInfo *rule, Rect *r )
{	
	Rect damage, visible;
	XRectangle exp_rect[ 1 ];

	switch( rule->type ) {
	case RULER_VERTICAL:
		/* Intersect with the section of the ruler we know to be 
		 * visible.
		 */
		find_ruler_visible( rule, &visible );
		im_rect_intersectrect( r, &visible, &damage );

		/* Make damage list.
		 */
		exp_rect[ 0 ].x = damage.left;
		exp_rect[ 0 ].y = damage.top - rule->from;
		exp_rect[ 0 ].width = damage.width;
		exp_rect[ 0 ].height = damage.height;

		/* Set clipping.
		 */
		XSetClipRectangles( our_display, ruler_gc,
			0, 0, exp_rect, 1, Unsorted );
		XSetClipRectangles( our_display, tick_gc,
			0, 0, exp_rect, 1, Unsorted );

		/* Repaint ruler section.
		 */
		paint_vertical_ruler( rule, 
			damage.top, damage.top + damage.height );

		/* Refresh tick. 
		 */
		refresh_ruler_tick( rule );

		/* Clear clipping.
		 */
		XSetClipMask( our_display, ruler_gc, None );
		XSetClipMask( our_display, tick_gc, None );

		break;

	case RULER_HORIZONTAL:
		/* Intersect with the section of the ruler we know to be 
		 * visible.
		 */
		find_ruler_visible( rule, &visible );
		im_rect_intersectrect( r, &visible, &damage );

		/* Make damage list.
		 */
		exp_rect[ 0 ].x = damage.left - rule->from;
		exp_rect[ 0 ].y = damage.top;
		exp_rect[ 0 ].width = damage.width;
		exp_rect[ 0 ].height = damage.height;

		/* Set clipping.
		 */
		XSetClipRectangles( our_display, ruler_gc,
			0, 0, exp_rect, 1, Unsorted );
		XSetClipRectangles( our_display, tick_gc,
			0, 0, exp_rect, 1, Unsorted );

		/* Repaint ruler section.
		 */
		paint_horizontal_ruler( rule,
			damage.left, damage.left + damage.width );

		/* Refresh tick. 
		 */
		refresh_ruler_tick( rule );

		/* Clear clipping.
		 */
		XSetClipMask( our_display, ruler_gc, None );
		XSetClipMask( our_display, tick_gc, None );

		break;

	default:
		assert( FALSE );
	}
}

/* Repaint event in a ruler.
 */
/*ARGSUSED*/
static void
repaint_ruler( Widget wid, XtPointer client, XtPointer call ) 
{	
	decl( RulerInfo *, rule, client );
	decl( XmDrawingAreaCallbackStruct *, cb, call );
	struct display *scr = rule->scr;
	Symbol *sym = scr->sym;

	/* Check the repaint .. is it kosher? We occasionally get exposes
	 * half way through deletes. Ignore these, as we might have closed the
	 * image we are trying to display from.
	 */
	if( sym->dest || !rule->enabled )
		return;

	/* Check event class.
	 */
	if( cb->event->type == Expose && cb->reason == XmCR_EXPOSE ) {
		/* Expose event has arrived. Get ready to repaint.
		 */
		Rect r;

		/* Make exposure rect.
		 */
		r.left = cb->event->xexpose.x;
		r.top = cb->event->xexpose.y;
		r.width = cb->event->xexpose.width; 
		r.height = cb->event->xexpose.height;

		/* Transform the damage into virtual ruler coordinates.
		 */
		switch( rule->type ) {
		case RULER_HORIZONTAL:
			r.left += scr->vis.left;
			break;

		case RULER_VERTICAL:
			r.top += scr->vis.top;
			break;

		default:
			assert( FALSE );
		}

		/* Repair damage. 
		 */
		repaint_ruler_region( rule, &r );
	}
}

/* New xy position for the scr we are representing. Try to scroll to the new
 * place.
 */
void
scroll_ruler( RulerInfo *rule, int left, int top )
{	
	Rect old, new, overlap, dirty;
	
	/* Sanity.
	 */
	if( !rule->enabled )
		return;

	switch( rule->type ) {
	case RULER_HORIZONTAL:
		/* Test for no change: return immediately if there's 
		 * nothing to do.
		 */
		if( rule->from == left )
			return;

		/* Fill out old and new position rects and intersect.
		 */
		find_ruler_visible( rule, &old );
		find_ruler_visible( rule, &new );
		new.left = left;
		im_rect_intersectrect( &old, &new, &overlap );

		/* Are there some old pixels we can reuse?
		 */
		if( !im_rect_isempty( &overlap ) ) {
			/* Yes: blit to new place.
			 */
			XCopyArea( our_display, XtWindow( rule->draw ), 
				XtWindow( rule->draw ), ruler_gc,
				overlap.left - old.left, overlap.top,
				overlap.width, overlap.height,
				overlap.left - new.left, overlap.top );

			/* Update ruler variables.
			 */
			rule->from = left;
			wait_for_end_of_scroll( XtWindow( rule->draw ) ); 

			/* Calculate dirty area.
			 */
			dirty.width = rule->space - overlap.width;
			dirty.height = rule->width;
			dirty.top = 0;
			if( overlap.left == left ) 
				/* Rect to IM_RECT_RIGHT of overlap needs painting.
				 */
				dirty.left = overlap.left + overlap.width;
			else 
				/* Rect to left of overlap needs painting.
				 */
				dirty.left = left;

			/* Blank out dirty area and repaint.
			 */
			XClearArea( our_display, XtWindow( rule->draw ),
				dirty.left - left, dirty.top,
				dirty.width, dirty.height,
				False );
			repaint_ruler_region( rule, &dirty );
		}
		else {
			/* No reusable pixels. Redraw whole ruler.
			 */
			rule->from = left;
			force_ruler_repaint( rule );
		}

		break;

	case RULER_VERTICAL:
		/* Test for no change: return immediately if there's 
		 * nothing to do.
		 */
		if( rule->from == top )
			return;

		/* Fill out old and new position rects and intersect.
		 */
		find_ruler_visible( rule, &old );
		find_ruler_visible( rule, &new );
		new.top = top;
		im_rect_intersectrect( &old, &new, &overlap );

		/* Are there some old pixels we can reuse?
		 */
		if( !im_rect_isempty( &overlap ) ) {
			/* Yes: blit to new place.
			 */
			XCopyArea( our_display, XtWindow( rule->draw ), 
				XtWindow( rule->draw ), ruler_gc,
				overlap.left, overlap.top - old.top,
				overlap.width, overlap.height,
				overlap.left, overlap.top - new.top );
			
			/* Update ruler variables.
			 */
			rule->from = top;
			wait_for_end_of_scroll( XtWindow( rule->draw ) ); 

			/* Calculate dirty area.
			 */
			dirty.height = rule->space - overlap.height;
			dirty.width = rule->width;
			dirty.left = 0;
			if( overlap.top == top ) 
				/* Rect to IM_RECT_RIGHT of overlap needs painting.
				 */
				dirty.top = overlap.top + overlap.height;
			else 
				/* Rect to left of overlap needs painting.
				 */
				dirty.top = top;

			/* Blank out dirty area and repaint.
			 */
			XClearArea( our_display, XtWindow( rule->draw ),
				dirty.left, dirty.top - top,
				dirty.width, dirty.height,
				False );
			repaint_ruler_region( rule, &dirty );
		}
		else {
			/* No reusable pixels. Redraw whole ruler.
			 */
			rule->from = top;
			force_ruler_repaint( rule );
		}

		break;

	default:
		assert( FALSE );
	}
}

/* Our scr has scrolled: update the rulers.
 */
void
update_rulers( struct display *scr )
{	
	if( scr->gdraw ) {
		scroll_ruler( scr->hrule, scr->vis.left, scr->vis.top );
		scroll_ruler( scr->vrule, scr->vis.left, scr->vis.top );
	}
}

/* Read the ruler DrawingArea size and rejig.
 */
void
rethink_ruler( RulerInfo *rule )
{	
	int mmsep;
	Dimension w, h;
	Arg args[ MAX_ARGS ];
	int n;

	/* Nothing if no ruler yet.
	 */
	if( !rule->draw )
		return;

	/* Read size out of widget. 
	 */
	n = 0;
	XtSetArg( args[ n ], XmNwidth, &w ); n++;
	XtSetArg( args[ n ], XmNheight, &h ); n++;
	XtGetValues( rule->draw, args, n );

	/* Set up draw fields.
	 */
	switch( rule->type ) {
	case RULER_VERTICAL:
		rule->space = h;
		rule->from = rule->scr->vis.top;
		break;

	case RULER_HORIZONTAL:
		rule->space = w;
		rule->from = rule->scr->vis.left;
		break;

	default:
		assert( FALSE );
	}

	/* Aim for one mark every 10 pixels. How many mm is that? 
	 * Round up to whole number of mm.
	 */
	mmsep = ceil( 10.0 / ppm2cf( rule, rule->ppm ) );

	/* How many pixels is that?
	 */
	rule->pxsep = (float) mmsep * ppm2cf( rule, rule->ppm );
}

/* Resize event in a ruler.
 */
/*ARGSUSED*/
static void
resize_ruler( Widget wid, XtPointer client, XtPointer call )
{
	decl( RulerInfo *, rule, client );
	decl( XmDrawingAreaCallbackStruct *, cb, call );

	/* Make sure this is a resize event in an enabled ruler.
	 */
	if( rule->enabled && cb->reason == XmCR_RESIZE )
		rethink_ruler( rule );
}

/* Graphics expose in a ruler.
 */
/*ARGSUSED*/
static void
gr_exp_ruler( Widget wid, XtPointer rp, XEvent *ev, Boolean *cont )
{	
	decl( RulerInfo *, rule, rp );
	Rect r;

	if( ev->type == GraphicsExpose ) {
		/* Make exposure rect.
		 */
		r.left = ev->xgraphicsexpose.x;
		r.top = ev->xgraphicsexpose.y;
		r.width = ev->xgraphicsexpose.width; 
		r.height = ev->xgraphicsexpose.height;

		/* Transform the damage into virtual canvas coordinates.
		 */
		switch( rule->type ) {
		case RULER_HORIZONTAL:
			r.left += rule->from;
			break;

		case RULER_VERTICAL:
			r.top += rule->from;
			break;

		default:
			assert( FALSE );
		}

		/* Repair damage.
		 */
		repaint_ruler_region( rule, &r );
	}
}

/* Make a ruler.
 */
RulerInfo *
build_ruler( Widget draw, struct display *scr, enum ruler_angle type )
{	
	RulerInfo *rule = IM_NEW( NULL, RulerInfo );
	int dirn, asc, des;
	int sz;
	XCharStruct overall;
	Arg args[ MAX_ARGS ];
	int n;

	/* Compute size of sample string. If we were to draw the string at 0x0,
	 * tarea would just enclose it. Parent uses this information in layout 
	 * of ruler.
	 */
	XTextExtents( ruler_font, "8888", 4, &dirn, &asc, &des, &overall );
	rule->tarea.width = overall.rbearing + overall.lbearing;
	rule->tarea.height = overall.ascent + overall.descent;
	rule->tarea.left = -overall.lbearing;
	rule->tarea.top = -overall.ascent;

	/* How wide (or high) should our ruler be? Big enough to enclose 
	 * the labeling. Make sure both rulers are the same size - looks funny
	 * otherwise. Note magic number - this looks ok for 8-point text, but
	 * might look slightly funny for other sizes.
	 */
	sz = IM_MAX( rule->tarea.height, rule->tarea.width ) + 3;
	n = 0;
	if( type == RULER_HORIZONTAL ) {
		XtSetArg( args[ n ], XmNheight, sz ); n++;
	}
	else {
		XtSetArg( args[ n ], XmNwidth, sz ); n++;
	}
	XtSetValues( draw, args, n );

	/* Set fields - rest set on expose/resize. This is the only place that
	 * rule->width is set.
	 */
	rule->width = sz;
	rule->draw = draw;
	rule->scr = scr;
	rule->enabled = True;
	rule->tick = False;
	rule->type = type;

	rule->ppm = -1.0;

	/* Check for silly values!
	 */
	if( rule->ppm <= 0.0 || rule->ppm > 1000.0 )
		rule->ppm = 1.0;

	/* Attach callbacks.
	 */
	XtAddCallback( draw, XmNexposeCallback, repaint_ruler, rule );
	XtAddCallback( draw, XmNresizeCallback, resize_ruler, rule );
	XtAddEventHandler( draw, None, True, gr_exp_ruler, rule );

	return( rule );
}

/* Destroy a ruler. Called on kill_screen().
 */
void 
destroy_ruler( RulerInfo *rule )
{
	FREE( rule );
}

/* Hide the rulers.
 */
void
hide_rulers( struct display *scr )
{	
	Arg args[ MAX_ARGS ];
	int n;

	/* Stop interactive updates.
	 */
	scr->hrule->enabled = False;
	scr->vrule->enabled = False;

	/* Unmanage, link window area back to edges of MainWindow.
	 */
	n = 0;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
	XtSetValues( scr->dframe, args, n );
	XtUnmanageChild( scr->hrule->draw );
	XtUnmanageChild( scr->vrule->draw );
}

/* Show the rulers.
 */
void
show_rulers( struct display *scr )
{	
	Arg args[ MAX_ARGS ];
	int n;

	/* Manage, link window area back to edges of rulers.
	 */
	XtManageChild( scr->hrule->draw );
	XtManageChild( scr->vrule->draw );
	n = 0;
	XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNtopWidget, scr->hrule->draw ); n++;
	XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
	XtSetArg( args[ n ], XmNleftWidget, scr->vrule->draw ); n++;
	XtSetValues( scr->dframe, args, n );
	force_resize( scr );

	/* Rethink all ruler variables and force a ruler repaint.
	 */
	rethink_ruler( scr->hrule );
	rethink_ruler( scr->vrule );
	force_ruler_repaint( scr->hrule );
	force_ruler_repaint( scr->vrule );

	/* Start interactive updates.
	 */
	scr->hrule->enabled = True;
	scr->vrule->enabled = True;
}

/* Move ruler ticks.
 */
void
update_ticks( struct display *scr, int x, int y )
{	
	snuff_ruler_tick( scr->hrule );
	snuff_ruler_tick( scr->vrule );
	scr->hrule->tpos = x;
	scr->vrule->tpos = y;
	light_ruler_tick( scr->hrule );
	light_ruler_tick( scr->vrule );
}

/* Force a complete repaint of a ruler. Rather than asking for exposes in
 * XClearArea, repaint the whole thing ourselves. Less delay!
 */
void
force_ruler_repaint( RulerInfo *rule )
{	
	if( rule->enabled && XtWindow( rule->draw ) ) {
		Rect visible;

		find_ruler_visible( rule, &visible );
		XClearArea( our_display, XtWindow( rule->draw ),
			0, 0, 0, 0, False );
		repaint_ruler_region( rule, &visible );
	}
}

/* Set ruler resolution field and redisplay. Check for silly values!
 */
Bool
set_ruler_ppm( RulerInfo *rule, double ppm )
{	
	/* Check for silly values!
	 */
	if( ppm <= 0.01 || ppm > 1000.0 ) {
		errors( "Pixels-per-millimetre setting of %g is silly!\n"
			"Setting not changed", ppm );
		return( False );
	}

	/* Refresh.
	 */
	rule->ppm = ppm;
	rethink_ruler( rule );
	force_ruler_repaint( rule );

	return( True );
}

/* Read current ruler resolution.
 */
double
get_ruler_ppm( RulerInfo *rule )
{	
	return( rule->ppm );
}
